﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/lmem/lmem_FrameHeap.h>

#include "testLMem_ThreadSafe.h"

namespace {

/**
 * @brief   フレームヒープのテストフィクスチャです。
 */
class FrameHeapThreadSafeTest : public LMemThreadSafeTest
{
protected:

    nn::lmem::HeapCommonHead m_CommonHead;

    /**
     * @brief   テスト開始時に毎回呼び出される関数です。
     */
    virtual void SetUp()
    {
        InitializeCommmon();
    }

    /**
     * @brief   テスト終了時に毎回呼び出される関数です。
     */
    virtual void TearDown()
    {
       nn::lmem::DestroyFrameHeap(m_HeapHandle);
       std::free(m_Heap);
    }

    /**
     * @brief   ヘッダ領域を与えずにフレームヒープを初期化します。
     */
    void InitializeFrameHeap()
    {
        m_HeapHandle = nn::lmem::CreateFrameHeap(m_Heap, m_HeapSize, nn::lmem::CreationOption_ThreadSafe | nn::lmem::CreationOption_DebugFill);
        ASSERT_NOT_NULL(m_HeapHandle);
    }

    /**
     * @brief   ヘッダ領域を与えてフレームヒープを初期化します。
     */
    void InitializeFrameHeapWithHead()
    {
        m_HeapHandle = nn::lmem::CreateFrameHeap(m_Heap, m_HeapSize, nn::lmem::CreationOption_ThreadSafe | nn::lmem::CreationOption_DebugFill, &m_CommonHead);
        ASSERT_NOT_NULL(m_HeapHandle);
    }
};

// =============
// スレッド関数
// =============

/**
 * @brief   メモリブロックの確保を行います。
 * @details また、既に確保済みのメモリブロックが破壊されていないか確認します。
 */
void AllocateFrameHeap(void* arg)
{
    CheckStruct* ptr = reinterpret_cast<CheckStruct*>(arg);

    const int BlockCount = 1024;                    // 確保するメモリブロックの数
    const size_t AllocateSize = sizeof(nn::Bit64);  // 確保するメモリブロックのサイズ
    nn::Bit64* memoryBlocks[BlockCount];            // 確保したメモリブロックのアドレス

    for(int i = 0; i < BlockCount; ++i)
    {
        memoryBlocks[i] = reinterpret_cast<nn::Bit64*>(nn::lmem::AllocateFromFrameHeap(ptr->handle, AllocateSize));
        ASSERT_NOT_NULL(memoryBlocks[i]);
        *memoryBlocks[i] = ptr->fill;
        for(int j = 0; j <= i; ++j)
        {
            // 確保済みの領域が誤って別スレッドから上書きされていないか確認
            ASSERT_EQ(*memoryBlocks[j], ptr->fill);
        }
    }
}

/**
 * @brief   後方からのメモリ確保と解放を繰り返しメモリアクセスする領域を重複させます。
 */
void AllocateFrameHeapFromTail(void* arg)
{
    CheckStruct* ptr = reinterpret_cast<CheckStruct*>(arg);

    const int RepeatCount = 2048;                       // 確保と解放を繰り返す回数
    const size_t AllocateSize = sizeof(nn::Bit64);      // 確保するメモリブロックのサイズ
    nn::Bit64* pMemoryBlock;                            // 確保したメモリブロックのアドレス

    for(int i = 0; i < RepeatCount; ++i)
    {
        pMemoryBlock = reinterpret_cast<nn::Bit64*>(nn::lmem::AllocateFromFrameHeap(ptr->handle, AllocateSize, -nn::lmem::DefaultAlignment));
        if(pMemoryBlock != nullptr)
        {
            *pMemoryBlock = ptr->fill;
            ASSERT_EQ(*pMemoryBlock, ptr->fill);
            nn::lmem::FreeToFrameHeap(ptr->handle, nn::lmem::FreeMode_Rear);
        }
    }
}

/**
 * @brief   ヒープ領域を少しずつ縮小させていきます。
 */
void AdjustFrameHeap(void* arg)
{
    CheckStruct* ptr = reinterpret_cast<CheckStruct*>(arg);

    const int RepeatCount = 512;                // Adjust を繰り返す回数
    const int FirstAllocateSize = 1024 * 1024;  // 最初に確保するサイズ
    const int DecreaseSize = 128;               // 1 回の試行で減らしていくヒープ領域のサイズ
    size_t allocateSize;                        // 確保するメモリブロックのサイズ
    void* pMemoryBlock;                         // 確保したメモリブロックのアドレス
    nn::lmem::MemoryRange range;

    allocateSize = FirstAllocateSize - DecreaseSize;
    for(int i = 0; i < RepeatCount; ++i)
    {
        // まず Adjust でヒープ領域が全て消えてしまわないように前方からメモリを確保する
        pMemoryBlock = nn::lmem::AllocateFromFrameHeap(ptr->handle, allocateSize);
        ASSERT_NOT_NULL(pMemoryBlock);

        // Adjust してヒープ領域を少し縮める
        range = nn::lmem::AdjustFrameHeap(ptr->handle, nn::lmem::AdjustMode_Tail);

        nn::lmem::FreeToFrameHeap(ptr->handle, nn::lmem::FreeMode_Front);
        if(range.size != 0)
        {
            allocateSize -= DecreaseSize;
        }
    }
}

/**
 * @brief   ヒープ領域を前方から少しずつ縮小させていきます。
 */
void AdjustFrameHeapFromHead(void* arg)
{
    nn::lmem::HeapHandle handle = reinterpret_cast<nn::lmem::HeapHandle>(arg);

    const int RepeatCount = 512;                // Adjust を繰り返す回数
    const int FirstAllocateSize = 1024 * 1024;  // 最初に確保するサイズ
    const int DecreaseSize = 128;               // 1 回の試行で減らしていくヒープ領域のサイズ
    size_t allocateSize;                        // 確保するメモリブロックのサイズ
    void* pMemoryBlock;                         // 確保したメモリブロックのアドレス
    nn::lmem::MemoryRange range;

    allocateSize = FirstAllocateSize - DecreaseSize;
    for(int i = 0; i < RepeatCount; ++i)
    {
        // まず Adjust でヒープ領域が全て消えてしまわないように後方からメモリを確保する
        pMemoryBlock = nn::lmem::AllocateFromFrameHeap(handle, allocateSize, -nn::lmem::DefaultAlignment);
        ASSERT_NOT_NULL(pMemoryBlock);

        // Adjust してヒープ領域を少し縮める
        range = nn::lmem::AdjustFrameHeap(handle, nn::lmem::AdjustMode_Head);
        ASSERT_NE(range.size, static_cast<size_t>(0));

        nn::lmem::FreeToFrameHeap(handle, nn::lmem::FreeMode_Rear);
        allocateSize -= DecreaseSize;
    }
}

/**
 * @brief   ヒープの先頭アドレスの取得を繰り返します。
 */
void GetStartAddressFrameHeap(void* arg)
{
    nn::lmem::HeapHandle handle = reinterpret_cast<nn::lmem::HeapHandle>(arg);

    const int RepeatCount = 2048;       // GetStartAddress を繰り返す回数

    for(int i = 0; i < RepeatCount; ++i)
    {
        nn::lmem::GetStartAddress(handle);
    }
}

// =======
// テスト
// =======

/**
 * @brief   複数スレッドを作成し、確保を繰り返します。
 */

TEST_F(FrameHeapThreadSafeTest, AllocateFrameHeap)
{
    const int FunctionCount = 8;
    nn::os::ThreadFunction functions[FunctionCount] = { AllocateFrameHeap,
                                                        AllocateFrameHeap,
                                                        AllocateFrameHeap,
                                                        AllocateFrameHeap,
                                                        AllocateFrameHeap,
                                                        AllocateFrameHeap,
                                                        AllocateFrameHeap,
                                                        AllocateFrameHeap };

    InitializeFrameHeap();

    DoMultiThreadTestWithCheckStruct(functions, FunctionCount);
}

/**
 * @brief   複数スレッドで後方からの確保とヒープの縮小を繰り返します。
 */
TEST_F(FrameHeapThreadSafeTest, AdjustFrameHeap)
{
    const int FunctionCount = 2;
    nn::os::ThreadFunction functions[FunctionCount] = { AllocateFrameHeapFromTail,
                                                        AdjustFrameHeap };

    InitializeFrameHeap();

    DoMultiThreadTestWithCheckStruct(functions, FunctionCount);
}

/**
 * @brief   複数スレッドで前方からのヒープの縮小と先頭アドレスの取得を繰り返します。
 */
TEST_F(FrameHeapThreadSafeTest, GetStartAddressFrameHeap)
{
    const int FunctionCount = 8;
    nn::os::ThreadFunction functions[FunctionCount] = { AdjustFrameHeapFromHead,
                                                        GetStartAddressFrameHeap,
                                                        GetStartAddressFrameHeap,
                                                        GetStartAddressFrameHeap,
                                                        GetStartAddressFrameHeap,
                                                        GetStartAddressFrameHeap,
                                                        GetStartAddressFrameHeap,
                                                        GetStartAddressFrameHeap };

    InitializeFrameHeapWithHead();

    DoMultiThreadTest(functions, FunctionCount);
}

}   // unnamed namespace
