﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/lmem/lmem_Common.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/lmem/lmem_UnitHeap.h>

#include <random>

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)

// デバッグ時のみ有効
#define NN_TEST_MEM_DEBUG

#endif

// =============
// マクロ
// =============

#define ASSERT_NULL(p) ASSERT_TRUE(p == nullptr)
#define ASSERT_NOT_NULL(p) ASSERT_TRUE(p != nullptr)

// =============
// static 関数
// =============

namespace {

/**
 * @brief   拡張ヒープのテストで利用するテストフィクスチャです。
 */
class ExpHeapAlignmentTest : public testing::Test
{
protected:

    const size_t m_HeapSize;            // グローバルヒープのサイズ
    const int m_MinAlignment;           // ヒープの最小アライメント。4 バイト

    // ヒープの最大アライメント
    // 実際の ExpHeap にはアライメント上限はないが、テストでは確認可能な範囲を設定する
    // 1MB とする
    const int m_MaxAlignment;

    uintptr_t m_Heap;                   // グローバルヒープのポインタ
    nn::lmem::HeapHandle m_HeapHandle;   // ユニットヒープのヒープハンドル

    ExpHeapAlignmentTest() : m_HeapSize(nn::os::MemoryBlockUnitSize * 4), m_MinAlignment(4), m_MaxAlignment(1024 * 1024 ){}

    /**
     * @brief テスト開始時に毎回呼び出される関数です。
     */
    virtual void SetUp()
    {
        m_Heap = reinterpret_cast<uintptr_t>(std::malloc(m_HeapSize));
    }

    /**
     * @brief テスト終了時に毎回呼び出される関数です。
     */
    virtual void TearDown()
    {
        std::free(reinterpret_cast<void*>(m_Heap));
    }

    /**
     * @brief   メモリが確保できたら、EixtCode を返します。
     * @details 1つの TEST で複数の DEATH テストを行うための関数です。
     */
    void DoTestAlignment(int alignment)
    {
        const size_t AllocateSize = 64;
        // 不正なアライメントであれば、 AllocateFromExpHeap() の内部の ASSERT で落ちる
        void* pMemoryAddress = nn::lmem::AllocateFromExpHeap(m_HeapHandle, AllocateSize, alignment);
        ASSERT_NOT_NULL(pMemoryAddress);
        nn::lmem::FreeToExpHeap(m_HeapHandle, pMemoryAddress);

        // 正常終了
        exit(0);
    }
};

/**
 * @brief   テストに与えるユニットヒープに関するパラメータです。
 */
enum UnitHeapTestParam
{
    UnitHeapTestParam_Head,     //!< ヒープの先頭にヒープ管理領域がある
    UnitHeapTestParam_Tail,     //!< ヒープの末尾にヒープ管理領域がある
    UnitHeapTestParam_Outside   //!< ヒープ領域外にヒープ管理領域がある
};

/**
 * @brief   ユニットヒープのテストで利用するテストフィクスチャです。
 */
class UnitHeapAlignmentTest : public testing::TestWithParam<int>
{
protected:

    const size_t m_HeapSize;                // グローバルヒープのサイズ
    const int m_MinAlignment;               // ユニットヒープの最小アライメント
    uintptr_t m_Heap;                       // グローバルヒープのポインタ
    nn::lmem::HeapHandle m_HeapHandle;      // ユニットヒープのヒープハンドル
    nn::lmem::HeapCommonHead m_HeapHead;    // ヒープ管理領域（ヒープ共通ヘッダ）
    int m_TestParam;                        // テスト時に渡されるパラメータ（UnitHeapTestParam列挙型）
    size_t m_HeadSize;                      // ヒープ内部に確保されている管理領域のサイズ

    UnitHeapAlignmentTest() : m_HeapSize(nn::os::MemoryBlockUnitSize * 4), m_MinAlignment(4){}

    /**
     * @brief テスト開始時に毎回呼び出される関数です。
     */
    virtual void SetUp()
    {
        m_Heap = reinterpret_cast<uintptr_t>(std::malloc(m_HeapSize));
    }

    /**
     * @brief テスト開始時に毎回呼び出される関数です。
     */
    void SetUpParam()
    {
        m_TestParam = GetParam();
        if( m_TestParam == UnitHeapTestParam_Outside)
        {
            m_HeadSize = 0;
        }
        else
        {
            m_HeadSize = sizeof(nn::lmem::HeapCommonHead);
        }
    }

    /**
     * @brief テスト終了時に毎回呼び出される関数です。
     */
    virtual void TearDown()
    {
        std::free(reinterpret_cast<void*>(m_Heap));
    }

};

typedef ExpHeapAlignmentTest ExpHeapAlignmentDeathTest;

}   // unnamed namespace


// ====================
// テスト - 拡張ヒープ
// ====================

// 確保されるメモリブロックのアドレスが、適切なアライメントで確保されるか
// 指定のアライメントで確保した場合、確保アドレスがアライメントに従う、
// かつ指定のアライメントで確保しなかった場合、確保アドレスがアライメントに従わなければ、OK
TEST_F(ExpHeapAlignmentTest, Allocate)
{
    const size_t AllocateSize = 16;
    void* pMemoryAddress;

    for(int alignment = 4; alignment <= m_MaxAlignment; alignment <<= 1)
    {
        for(int sign = -1; sign <= 1; sign += 2)
        {
            // 拡張ヒープの作成
            m_HeapHandle = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_NoOption);
            ASSERT_NOT_NULL(m_HeapHandle);

            if ( alignment != m_MinAlignment )
            {
                // 指定のアライメントで確保しない場合に、
                // アライメントが外れるようにアドレスを調整する
                pMemoryAddress = nn::lmem::AllocateFromExpHeap(m_HeapHandle, AllocateSize, m_MinAlignment * sign);
                ASSERT_NOT_NULL(pMemoryAddress);
                nn::lmem::FreeToExpHeap(m_HeapHandle, pMemoryAddress);

                if( (reinterpret_cast<uintptr_t>(pMemoryAddress) & (alignment - 1)) == 0)
                {
                    bool getsPadding = false;
                    // アライメントがずれる確保サイズを探す
                    for(size_t i = 1; static_cast<int>(i) < m_MaxAlignment; ++i)
                    {
                        void* pPaddingAddress = nn::lmem::AllocateFromExpHeap(m_HeapHandle, i, m_MinAlignment * sign);
                        ASSERT_NOT_NULL(pPaddingAddress);
                        pMemoryAddress = nn::lmem::AllocateFromExpHeap(m_HeapHandle, AllocateSize, m_MinAlignment * sign);
                        ASSERT_NOT_NULL(pMemoryAddress);
                        nn::lmem::FreeToExpHeap(m_HeapHandle, pMemoryAddress);
                        if( (reinterpret_cast<uintptr_t>(pMemoryAddress) & (alignment - 1)) != 0)
                        {
                            getsPadding = true;
                            break;
                        }
                        nn::lmem::FreeToExpHeap(m_HeapHandle, pPaddingAddress);
                    }
                    EXPECT_TRUE(getsPadding);   // アライメントがずれるサイズは必ずあるので、ここは true になるはず
                }

                // アライメントが外れるようにアドレスを調整したので、
                // 単純に確保しただけではアライメントが合わない
                pMemoryAddress = nn::lmem::AllocateFromExpHeap(m_HeapHandle, AllocateSize, m_MinAlignment * sign);
                EXPECT_NE( (reinterpret_cast<uintptr_t>(pMemoryAddress) & (alignment - 1)), static_cast<uintptr_t>(0) );
                nn::lmem::FreeToExpHeap(m_HeapHandle, pMemoryAddress);
            }

            // アライメントを指定すると、指定したアライメントに合ったアドレスでメモリを確保する
            pMemoryAddress = nn::lmem::AllocateFromExpHeap(m_HeapHandle, AllocateSize, alignment * sign);
            EXPECT_EQ( (reinterpret_cast<uintptr_t>(pMemoryAddress) & (alignment - 1)), static_cast<uintptr_t>(0) );
            nn::lmem::FreeToExpHeap(m_HeapHandle, pMemoryAddress);

            nn::lmem::DestroyExpHeap(m_HeapHandle);
        }
    }
}

// ランダムにメモリを確保する
// ランダムなサイズと 2 のべき乗アライメントでメモリが確保できれば、OK
TEST_F(ExpHeapAlignmentTest, RandomAllocate)
{
    const int Seed = 12345;
    const int MaxRandomAllocateSize = 1024;
    // 4～1Mまでのアライメントの配列
    const int AlignCount = 38;
    int pAlignment[AlignCount] = {4, 8, 16, 32, 64, 128, 256, 512, 1024, 2048, 4096, 8192,
                                  16384, 32768, 65536, 131072, 262144, 524288, 1048576,
                                  -4, -8, -16, -32, -64, -128, -256, -512, -1024, -2048, -4096, -8192,
                                  -16384, -32768, -65536, -131072, -262144, -524288, -1048576};
    std::mt19937 engine(Seed);
    std::uniform_int_distribution<uint32_t> distributionForAllocate(0, MaxRandomAllocateSize - 1);
    std::uniform_int_distribution<uint32_t> distributionForAlign(0, AlignCount - 1);

    NN_LOG("Seed : %d\n", Seed);

    // 拡張ヒープの作成
    m_HeapHandle = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_NoOption);
    ASSERT_NOT_NULL(m_HeapHandle);

    // アライメントをランダムにシャッフルし、ランダムな値でメモリ確保を繰り返す
    const int SetNum = 3;
    for(int i = 0; i < SetNum; ++i)
    {
        // アライメントの値をシャッフル
        for(int j = 0; j < AlignCount - 1; ++j)
        {
            int swapNum = distributionForAlign(engine);
            std::swap(pAlignment[j], pAlignment[swapNum]);
        }

        for(int j = 0; j < AlignCount; ++j)
        {
            size_t allocateSize = distributionForAllocate(engine);
            NN_LOG("allocate size: %zu alignment: %d\n", allocateSize, pAlignment[j]);
            void* pAddress = nn::lmem::AllocateFromExpHeap(m_HeapHandle, allocateSize, pAlignment[j]);
            ASSERT_EQ( (reinterpret_cast<uintptr_t>(pAddress) & (std::abs(pAlignment[j]) - 1)), static_cast<uintptr_t>(0) );
            ASSERT_TRUE(nn::lmem::CheckExpHeap(m_HeapHandle, nn::lmem::ErrorOption_NoOption));
        }
    }
    nn::lmem::DestroyExpHeap(m_HeapHandle);
}

// ========================
// テスト - ユニットヒープ
// ========================

// アライメント
// ヒープ開始アドレス、ヒープサイズ、ユニットサイズ、アライメントの4要素それぞれを変更させながら、
// 各要素が違う値で正常にヒープが確保されていれば、OK
TEST_P(UnitHeapAlignmentTest, Simple)
{
    SetUpParam();

    const size_t HeapSize = 16384 + m_HeadSize; // 16KB + 管理領域のサイズ
    void* memoryAddress;

    for(uintptr_t addr = static_cast<uintptr_t>(m_Heap); addr <= static_cast<uintptr_t>(m_Heap + 16); ++addr )
    {
        for(size_t size = HeapSize - 16; size <= HeapSize; size += 4)
        {
            for(size_t unitSize = 0; unitSize <= 96; unitSize += 4)
            {
                for(int alignment = m_MinAlignment; alignment <= 4096; alignment <<= 1)
                {
                    if( !( alignment >= sizeof(nn::lmem::detail::UnitHead) &&
                           alignment % sizeof(nn::lmem::detail::UnitHead) == 0 &&
                           unitSize >= sizeof(nn::lmem::detail::UnitHead) )
                       )
                    {
                        continue;
                    }
                    // ヒープの作成とメモリ確保
                    if(m_TestParam == UnitHeapTestParam_Head)
                    {
                        m_HeapHandle = nn::lmem::CreateUnitHeap(reinterpret_cast<void*>(addr), size, unitSize, nn::lmem::CreationOption_NoOption, alignment, nn::lmem::InfoPlacement_Head);
                    }
                    else if(m_TestParam == UnitHeapTestParam_Tail)
                    {
                        m_HeapHandle = nn::lmem::CreateUnitHeap(reinterpret_cast<void*>(addr), size, unitSize, nn::lmem::CreationOption_NoOption, alignment, nn::lmem::InfoPlacement_Tail);
                    }
                    else if(m_TestParam == UnitHeapTestParam_Outside)
                    {
                        m_HeapHandle = nn::lmem::CreateUnitHeap(reinterpret_cast<void*>(addr), size, unitSize, nn::lmem::CreationOption_NoOption, alignment, &m_HeapHead);
                    }
                    ASSERT_NOT_NULL(m_HeapHandle);
                    EXPECT_EQ(nn::lmem::GetUnitHeapAlignment(m_HeapHandle), alignment);
                    memoryAddress = nn::lmem::AllocateFromUnitHeap(m_HeapHandle);
                    EXPECT_EQ(reinterpret_cast<uintptr_t>(memoryAddress) % alignment, 0);
                    nn::lmem::FreeToUnitHeap(m_HeapHandle, memoryAddress);
                    nn::lmem::DestroyUnitHeap(m_HeapHandle);       // ヒープの破棄
                }
            }
        }
    }
}

INSTANTIATE_TEST_CASE_P(HeapHeadPosition,
                        UnitHeapAlignmentTest,
                        testing::Values(UnitHeapTestParam_Head, UnitHeapTestParam_Tail, UnitHeapTestParam_Outside));

// =============
// DEATH テスト
// =============

// ASSERT で落ちるテストは、 Debug ビルド、 Develop ビルドでのみ有効
#if defined(NN_TEST_MEM_DEBUG)

// 不正アライメント
// 2のべき乗以外の値をアライメントとして用いた場合に、 ASSERT で落ちれば、OK
TEST_F(ExpHeapAlignmentDeathTest, Assert)
{
    const int CheckAlignRange = 16;    // テストするアライメントの範囲

    // 拡張ヒープの作成
    m_HeapHandle = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(m_Heap), m_HeapSize, nn::lmem::CreationOption_NoOption);
    ASSERT_NOT_NULL(m_HeapHandle);

    for(int alignment = 0; alignment <= CheckAlignRange; ++alignment)
    {
        if(!(alignment & (alignment - 1)) && alignment != 0)
        {
            // アライメントが 2 のべき乗であるなら、 doTestAlignment() は exit(0) で終了する
#if defined(NN_BUILD_CONFIG_OS_WIN32)
            EXPECT_EXIT(DoTestAlignment(alignment), testing::ExitedWithCode(0), "");
            EXPECT_EXIT(DoTestAlignment(-alignment), testing::ExitedWithCode(0), "");
#endif
        }
        else
        {
             // アライメントが 2 のべき乗でないなら、 ASSERT で落ちる
            EXPECT_DEATH_IF_SUPPORTED( DoTestAlignment(alignment), "" );
            EXPECT_DEATH_IF_SUPPORTED( DoTestAlignment(-alignment), "" );
        }
    }

    nn::lmem::DestroyExpHeap(m_HeapHandle);
}

#endif  // #if defined(NN_TEST_MEM_DEBUG)
