﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <string>
#include <vector>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>
#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/fs.h>
#include <nn/kvdb/kvdb_FileKeyValueCache.h>

namespace {
    struct Key
    {
        char data[12];

        bool operator == (const Key& key) const
        {
            return std::memcmp(data, key.data, sizeof(data)) == 0;
        }
    };

    typedef nn::kvdb::FileKeyValueCache<Key, 16> TestKeyValueCache;

    char g_Buffer[TestKeyValueCache::RequiredBufferSize];
    const size_t ValueBufferSize = 1024 * 1024;
    char g_ValueBuffer[ValueBufferSize];

    void* Allocate(size_t size)
    {
        return std::malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        return std::free(p);
    }

    class KvdbBasicTest : public testing::Test
    {
    protected:
        static void SetUpTestCase()
        {
            nn::fs::SetAllocator(Allocate, Deallocate);
            NN_ABORT_UNLESS(nn::fs::MountHostRoot().IsSuccess());
        }

        static void TearDownTestCase()
        {
            nn::fs::UnmountHostRoot();
        }

        virtual void SetUp()
        {
            auto workingPath = std::string(nnt::GetHostArgv()[1]) + "/work";

            nn::fs::DeleteDirectoryRecursively(workingPath.c_str());
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CreateDirectory(workingPath.c_str()));

            m_WorkingDirectoryPath = workingPath;
        }

        virtual void TearDown()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::DeleteDirectoryRecursively(GetWorkingDirectoryPath()));
        }

        const char* GetWorkingDirectoryPath()
        {
            return m_WorkingDirectoryPath.c_str();
        }

        std::vector<Key> CreateRandomKeyList(int count)
        {
            std::vector<Key> keyList;
            for (int i = 0; i < count; i++)
            {
                keyList.push_back(CreateRandomKey());
            }

            return keyList;
        }

        std::vector<std::vector<char>> CreateRandomValueList(int count)
        {
            std::vector<std::vector<char>> valueList;
            for (int i = 0; i < count; i++)
            {
                valueList.push_back(CreateRandomValue());
            }

            return valueList;
        }

        Key CreateRandomKey()
        {
            Key key;
            for (auto& d : key.data)
            {
                d = static_cast<char>(m_RandomEngine());
            }

            return key;
        }

        std::vector<char> CreateRandomValue()
        {
            std::vector<char> value;

            size_t size = m_RandomEngine() % ValueBufferSize;
            for (size_t i = 0; i < size; i++)
            {
                value.push_back(static_cast<char>(m_RandomEngine()));
            }

            return value;
        }

    private:
        std::string m_WorkingDirectoryPath;
        std::mt19937_64 m_RandomEngine;
    };
}

TEST_F(KvdbBasicTest, Basic)
{
    auto path = GetWorkingDirectoryPath();

    NNT_EXPECT_RESULT_FAILURE(nn::kvdb::ResultDatabaseNotCreated, TestKeyValueCache::Verify(path));
    NNT_EXPECT_RESULT_SUCCESS(TestKeyValueCache::Create(path));
    NNT_EXPECT_RESULT_SUCCESS(TestKeyValueCache::Verify(path));

    auto keyList1 = CreateRandomKeyList(TestKeyValueCache::MaxEntryCount);
    auto valueList1 = CreateRandomValueList(TestKeyValueCache::MaxEntryCount);

    {
        TestKeyValueCache cache;
        NNT_EXPECT_RESULT_SUCCESS(cache.Initialize(path, g_Buffer, TestKeyValueCache::RequiredBufferSize));

        for (int i = 0; i < TestKeyValueCache::MaxEntryCount; i++)
        {
            auto& key = keyList1[i];
            auto& value = valueList1[i];
            size_t outSize;

            NNT_EXPECT_RESULT_FAILURE(nn::kvdb::ResultKeyNotFound, cache.Get(&outSize, g_ValueBuffer, ValueBufferSize, key));
            NNT_EXPECT_RESULT_SUCCESS(cache.Put(key, value.data(), value.size()));
            NNT_EXPECT_RESULT_SUCCESS(cache.Get(&outSize, g_ValueBuffer, ValueBufferSize, key));
            EXPECT_TRUE(std::memcmp(g_ValueBuffer, value.data(), outSize) == 0);
        }

        NNT_EXPECT_RESULT_SUCCESS(cache.Flush());
    }

    {
        TestKeyValueCache cache;
        NNT_EXPECT_RESULT_SUCCESS(cache.Initialize(path, g_Buffer, TestKeyValueCache::RequiredBufferSize));

        for (int i = 0; i < TestKeyValueCache::MaxEntryCount; i++)
        {
            auto& key = keyList1[i];
            auto& value = valueList1[i];
            size_t outSize;

            NNT_EXPECT_RESULT_SUCCESS(cache.Get(&outSize, g_ValueBuffer, ValueBufferSize, key));
            EXPECT_TRUE(std::memcmp(g_ValueBuffer, value.data(), outSize) == 0);
        }

        NNT_EXPECT_RESULT_SUCCESS(cache.Flush());
    }

    auto keyList2 = CreateRandomKeyList(TestKeyValueCache::MaxEntryCount);
    auto valueList2 = CreateRandomValueList(TestKeyValueCache::MaxEntryCount);

    {
        TestKeyValueCache cache;
        NNT_EXPECT_RESULT_SUCCESS(cache.Initialize(path, g_Buffer, TestKeyValueCache::RequiredBufferSize));

        for (int i = 0; i < TestKeyValueCache::MaxEntryCount; i++)
        {
            auto& key = keyList2[i];
            auto& value = valueList2[i];
            size_t outSize;

            NNT_EXPECT_RESULT_FAILURE(nn::kvdb::ResultKeyNotFound, cache.Get(&outSize, g_ValueBuffer, ValueBufferSize, key));
            NNT_EXPECT_RESULT_SUCCESS(cache.Put(key, value.data(), value.size()));
            NNT_EXPECT_RESULT_SUCCESS(cache.Get(&outSize, g_ValueBuffer, ValueBufferSize, key));
            EXPECT_TRUE(std::memcmp(g_ValueBuffer, value.data(), outSize) == 0);

            auto& oldKey = keyList1[i];
            NNT_EXPECT_RESULT_FAILURE(nn::kvdb::ResultKeyNotFound, cache.Get(&outSize, g_ValueBuffer, ValueBufferSize, oldKey));
        }

        for (int i = 0; i < TestKeyValueCache::MaxEntryCount; i++)
        {
            auto& key = keyList2[i];
            auto& value = valueList2[i];
            size_t outSize;

            NNT_EXPECT_RESULT_SUCCESS(cache.Get(&outSize, g_ValueBuffer, ValueBufferSize, key));
            EXPECT_TRUE(std::memcmp(g_ValueBuffer, value.data(), outSize) == 0);
        }

        for (int i = 0; i < TestKeyValueCache::MaxEntryCount; i++)
        {
            auto& oldKey = keyList1[i];
            size_t outSize;
            NNT_EXPECT_RESULT_FAILURE(nn::kvdb::ResultKeyNotFound, cache.Get(&outSize, g_ValueBuffer, ValueBufferSize, oldKey));
        }

        NNT_EXPECT_RESULT_SUCCESS(cache.Flush());
    }

    {
        TestKeyValueCache cache;
        NNT_EXPECT_RESULT_SUCCESS(cache.Initialize(path, g_Buffer, TestKeyValueCache::RequiredBufferSize));

        for (int i = 0; i < TestKeyValueCache::MaxEntryCount; i++)
        {
            size_t outSize;
            auto& touchKey = keyList2[i];
            auto& touchValue = valueList2[i];

            if (i == 0)
            {
                NNT_EXPECT_RESULT_SUCCESS(cache.Get(&outSize, g_ValueBuffer, ValueBufferSize, touchKey));
            }

            auto& key = keyList1[i];
            auto& value = valueList1[i];
            NNT_EXPECT_RESULT_SUCCESS(cache.Put(key, value.data(), value.size()));
            NNT_EXPECT_RESULT_SUCCESS(cache.Get(&outSize, g_ValueBuffer, ValueBufferSize, key));
            EXPECT_TRUE(std::memcmp(g_ValueBuffer, value.data(), outSize) == 0);

            if (i == 0)
            {
                NNT_EXPECT_RESULT_SUCCESS(cache.Get(&outSize, g_ValueBuffer, ValueBufferSize, touchKey));
                EXPECT_TRUE(std::memcmp(g_ValueBuffer, touchValue.data(), outSize) == 0);

                auto popedKey = keyList2[i + 1];
                NNT_EXPECT_RESULT_FAILURE(nn::kvdb::ResultKeyNotFound, cache.Get(&outSize, g_ValueBuffer, ValueBufferSize, popedKey));
            }
        }

        NNT_EXPECT_RESULT_SUCCESS(cache.Flush());
    }
}
