﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/kpr/kpr_KeyCodeComposer.h>
#include <nnt/nntest.h>

//!< キーコードの組み立てモードは初期化直後に既定値になっているか
TEST(KeyCodeComposerSuite, ModeTest1)
{
    ::nn::kpr::KeyCodeComposer composer;
    EXPECT_EQ(::nn::kpr::KeyCodeComposerMode_Default, composer.GetMode());
}

//!< キーコードの組み立てモードを変更できるか
TEST(KeyCodeComposerSuite, ModeTest2)
{
    ::nn::kpr::KeyCodeComposer composer;

    composer.SetMode(::nn::kpr::KeyCodeComposerMode_DeadKey);
    EXPECT_EQ(
        ::nn::kpr::KeyCodeComposerMode_DeadKey, composer.GetMode());

    composer.SetMode(::nn::kpr::KeyCodeComposerMode_RomajiHiragana);
    EXPECT_EQ(
        ::nn::kpr::KeyCodeComposerMode_RomajiHiragana, composer.GetMode());

    composer.SetMode(::nn::kpr::KeyCodeComposerMode_RomajiKatakana);
    EXPECT_EQ(
        ::nn::kpr::KeyCodeComposerMode_RomajiKatakana, composer.GetMode());
}

//!< キーコードの組み立てモードの切り替えは組み立て中キーコード列を破棄するか
TEST(KeyCodeComposerSuite, ModeTest3)
{
    const char* names[] = {
        "DeadKey",
        "RomajiHiragana",
        "RomajiKatakana",
    };

    const auto keyCode = static_cast<uint16_t>('a');

    ::nn::kpr::KeyCodeComposer composer;

    for (const char* (&name) : names)
    {
        SCOPED_TRACE(::testing::Message() << "Mode: " << name);

        const auto mode = static_cast<::nn::kpr::KeyCodeComposerMode>(
            (&name - names) + 1);

        const auto anotherMode = static_cast<::nn::kpr::KeyCodeComposerMode>(
            (&name - names + 1) % NN_ARRAY_SIZE(names) + 1);

        composer.SetMode(::nn::kpr::KeyCodeComposerMode_Default);
        composer.Enqueue(&keyCode, 1);
        EXPECT_EQ(1, composer.GetCompositionCount());

        composer.SetMode(mode);
        EXPECT_EQ(0, composer.GetCompositionCount());

        composer.Enqueue(&keyCode, 1);
        EXPECT_EQ(1, composer.GetCompositionCount());

        composer.SetMode(mode);
        EXPECT_EQ(1, composer.GetCompositionCount());

        composer.SetMode(anotherMode);
        EXPECT_EQ(0, composer.GetCompositionCount());

        composer.Enqueue(&keyCode, 1);
        EXPECT_EQ(1, composer.GetCompositionCount());

        composer.SetMode(mode);
        EXPECT_EQ(0, composer.GetCompositionCount());
    }
}

//!< 組み立て中キーコード列は初期化直後に空になっているか
TEST(KeyCodeComposerSuite, CompositionTest1)
{
    ::nn::kpr::KeyCodeComposer composer;
    EXPECT_EQ(0, composer.GetCompositionCount());
    EXPECT_EQ(0, composer.GetDequeueableCount());
}

//!< 組み立て中キーコード列を正しく取得できるか
TEST(KeyCodeComposerSuite, CompositionTest2)
{
    ::nn::kpr::KeyCodeComposer composer;
    composer.SetMode(::nn::kpr::KeyCodeComposerMode_RomajiHiragana);

    auto keyCode = uint16_t();

    uint16_t buffer[8] = {};

    keyCode = static_cast<uint16_t>('n');
    EXPECT_EQ(1, composer.Enqueue(&keyCode, 1));
    EXPECT_EQ(0, composer.GetDequeueableCount());
    EXPECT_EQ(1, composer.GetCompositionCount());
    EXPECT_EQ(1, composer.GetComposition(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>('n'), buffer[0]);

    keyCode = static_cast<uint16_t>('k');
    EXPECT_EQ(1, composer.Enqueue(&keyCode, 1));
    EXPECT_EQ(1, composer.GetDequeueableCount());
    EXPECT_EQ(2, composer.GetCompositionCount());
    EXPECT_EQ(2, composer.GetComposition(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>(0x3093), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>('k'), buffer[1]);

    EXPECT_EQ(1, composer.Dequeue(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>(0x3093), buffer[0]);
    EXPECT_EQ(0, composer.GetDequeueableCount());
    EXPECT_EQ(1, composer.GetCompositionCount());
    EXPECT_EQ(1, composer.GetComposition(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>('k'), buffer[0]);

    keyCode = static_cast<uint16_t>('a');
    EXPECT_EQ(1, composer.Enqueue(&keyCode, 1));
    EXPECT_EQ(1, composer.GetDequeueableCount());
    EXPECT_EQ(1, composer.GetCompositionCount());
    EXPECT_EQ(1, composer.GetComposition(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>(0x304B), buffer[0]);

    EXPECT_EQ(1, composer.Dequeue(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>(0x304B), buffer[0]);
    EXPECT_EQ(0, composer.GetDequeueableCount());
    EXPECT_EQ(0, composer.GetCompositionCount());

    keyCode = static_cast<uint16_t>('k');
    EXPECT_EQ(1, composer.Enqueue(&keyCode, 1));
    EXPECT_EQ(0, composer.GetDequeueableCount());
    EXPECT_EQ(1, composer.GetCompositionCount());
    EXPECT_EQ(1, composer.GetComposition(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>('k'), buffer[0]);
}

//!< 組み立て中キーコード列が最大長付近にある際に正しく取得できるか
TEST(KeyCodeComposerSuite, CompositionTest3)
{
    ::nn::kpr::KeyCodeComposer composer;
    composer.SetMode(::nn::kpr::KeyCodeComposerMode_RomajiHiragana);

    const uint16_t keyCodes[] = { 'l', 'l', 't' };
    EXPECT_EQ(3, composer.Enqueue(keyCodes, 3));

    const auto keyCode = static_cast<uint16_t>('s');
    EXPECT_EQ(1, composer.Enqueue(&keyCode, 1));

    uint16_t buffer[8] = {};
    EXPECT_EQ(1, composer.GetDequeueableCount());
    EXPECT_EQ(4, composer.GetCompositionCount());
    EXPECT_EQ(4, composer.GetComposition(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>('l'), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>('l'), buffer[1]);
    EXPECT_EQ(static_cast<uint16_t>('t'), buffer[2]);
    EXPECT_EQ(static_cast<uint16_t>('s'), buffer[3]);

    EXPECT_EQ(1, composer.Dequeue(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>('l'), buffer[0]);
    EXPECT_EQ(0, composer.GetDequeueableCount());
    EXPECT_EQ(3, composer.GetCompositionCount());
    EXPECT_EQ(3, composer.GetComposition(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>('l'), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>('t'), buffer[1]);
    EXPECT_EQ(static_cast<uint16_t>('s'), buffer[2]);
}

//!< キーコード列の追加が正しく処理されるか
TEST(KeyCodeComposerSuite, EnqueueKeyCodesTest1)
{
    ::nn::kpr::KeyCodeComposer composer;
    composer.SetMode(::nn::kpr::KeyCodeComposerMode_RomajiKatakana);

    const uint16_t keyCodes[] = { 'n', 'd', 'a' };
    EXPECT_EQ(2, composer.Enqueue(keyCodes, 3));
    EXPECT_EQ(1, composer.GetDequeueableCount());

    EXPECT_EQ(0, composer.Enqueue(&keyCodes[2], 1));

    auto keyCode = uint16_t();
    EXPECT_EQ(1, composer.Dequeue(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>(0x30F3), keyCode);
    EXPECT_EQ(0, composer.GetDequeueableCount());

    EXPECT_EQ(1, composer.Enqueue(&keyCodes[2], 1));

    EXPECT_EQ(1, composer.Dequeue(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>(0x30C0), keyCode);
    EXPECT_EQ(0, composer.GetDequeueableCount());
}

//!< 組み立てモードが既定値の場合にキーコードが正しく処理されるか
TEST(KeyCodeComposerSuite, DefaultModeTest1)
{
    ::nn::kpr::KeyCodeComposer composer;

    auto keyCode = uint16_t();

    keyCode = static_cast<uint16_t>(0x0300);
    EXPECT_EQ(1, composer.Enqueue(&keyCode, 1));
    EXPECT_EQ(1, composer.GetDequeueableCount());
    keyCode = uint16_t();
    EXPECT_EQ(1, composer.Dequeue(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>(0x0300), keyCode);

    keyCode = static_cast<uint16_t>('k');
    EXPECT_EQ(1, composer.Enqueue(&keyCode, 1));
    EXPECT_EQ(1, composer.GetDequeueableCount());
    keyCode = uint16_t();
    EXPECT_EQ(1, composer.Dequeue(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>('k'), keyCode);
}

//!< デッドキーが正しく処理されるか
TEST(KeyCodeComposerSuite, DeadKeyTest1)
{
    ::nn::kpr::KeyCodeComposer composer;
    composer.SetMode(::nn::kpr::KeyCodeComposerMode_DeadKey);

    auto keyCode = uint16_t();

    keyCode = static_cast<uint16_t>(0x0300);
    EXPECT_EQ(1, composer.Enqueue(&keyCode, 1));
    EXPECT_EQ(0, composer.GetDequeueableCount());

    uint16_t buffer[8] = {};

    keyCode = static_cast<uint16_t>(0x0301);
    EXPECT_EQ(1, composer.Enqueue(&keyCode, 1));
    EXPECT_EQ(2, composer.Dequeue(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>(0x0060), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>(0x00B4), buffer[1]);

    keyCode = static_cast<uint16_t>(0x0302);
    EXPECT_EQ(1, composer.Enqueue(&keyCode, 1));
    EXPECT_EQ(0, composer.GetDequeueableCount());

    keyCode = static_cast<uint16_t>('k');
    EXPECT_EQ(1, composer.Enqueue(&keyCode, 1));
    EXPECT_EQ(2, composer.Dequeue(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>(0x005E), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>('k'), buffer[1]);

    keyCode = static_cast<uint16_t>(0x0303);
    EXPECT_EQ(1, composer.Enqueue(&keyCode, 1));
    EXPECT_EQ(0, composer.GetDequeueableCount());

    keyCode = static_cast<uint16_t>('n');
    EXPECT_EQ(1, composer.Enqueue(&keyCode, 1));
    EXPECT_EQ(1, composer.Dequeue(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>(0x00F1), buffer[0]);

    keyCode = static_cast<uint16_t>('0');
    EXPECT_EQ(1, composer.Enqueue(&keyCode, 1));
    EXPECT_EQ(1, composer.Dequeue(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>('0'), buffer[0]);
}

//!< ローマ字変換（ひらがな）が正しく処理されるか
TEST(KeyCodeComposerSuite, RomajiHiraganaTest1)
{
    ::nn::kpr::KeyCodeComposer composer;
    composer.SetMode(::nn::kpr::KeyCodeComposerMode_RomajiHiragana);

    uint16_t buffer[8] = {};

    const uint16_t keyCodes1[] = { 'n', '0' };
    EXPECT_EQ(2, composer.Enqueue(keyCodes1, 2));
    EXPECT_EQ(2, composer.Dequeue(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>(0x3093), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>('0'), buffer[1]);

    const uint16_t keyCodes2[] = { 'n', 'l' };
    EXPECT_EQ(2, composer.Enqueue(keyCodes2, 2));
    EXPECT_EQ(1, composer.Dequeue(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>(0x3093), buffer[0]);

    const uint16_t keyCodes3[] = { 'l', 't', 's' };
    EXPECT_EQ(3, composer.Enqueue(keyCodes3, 3));
    EXPECT_EQ(1, composer.Dequeue(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>('l'), buffer[0]);

    const uint16_t keyCodes4[] = { 'a' };
    EXPECT_EQ(1, composer.Enqueue(keyCodes4, 1));
    EXPECT_EQ(2, composer.Dequeue(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>(0x3063), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>(0x3041), buffer[1]);

    const uint16_t keyCodes5[] = { 'w', 'w', 'h', 'a' };
    EXPECT_EQ(4, composer.Enqueue(keyCodes5, 4));
    EXPECT_EQ(3, composer.Dequeue(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>(0x3063), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>(0x3046), buffer[1]);
    EXPECT_EQ(static_cast<uint16_t>(0x3041), buffer[2]);
}

//!< ローマ字変換（カタカナ）が正しく処理されるか
TEST(KeyCodeComposerSuite, RomajiKatakanaTest1)
{
    ::nn::kpr::KeyCodeComposer composer;
    composer.SetMode(::nn::kpr::KeyCodeComposerMode_RomajiKatakana);

    uint16_t buffer[8] = {};

    const uint16_t keyCodes1[] = { 'n', '!' };
    EXPECT_EQ(2, composer.Enqueue(keyCodes1, 2));
    EXPECT_EQ(2, composer.Dequeue(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>(0x30F3), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>('!'), buffer[1]);

    const uint16_t keyCodes2[] = { 'n', 'x' };
    EXPECT_EQ(2, composer.Enqueue(keyCodes2, 2));
    EXPECT_EQ(1, composer.Dequeue(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>(0x30F3), buffer[0]);

    const uint16_t keyCodes3[] = { 'x', 'w', 'h' };
    EXPECT_EQ(3, composer.Enqueue(keyCodes3, 3));
    EXPECT_EQ(1, composer.Dequeue(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>('x'), buffer[0]);

    const uint16_t keyCodes4[] = { 'a' };
    EXPECT_EQ(1, composer.Enqueue(keyCodes4, 1));
    EXPECT_EQ(2, composer.Dequeue(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>(0x30A5), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>(0x30A1), buffer[1]);

    const uint16_t keyCodes5[] = { 'c', 'c', 'y', 'a' };
    EXPECT_EQ(4, composer.Enqueue(keyCodes5, 4));
    EXPECT_EQ(3, composer.Dequeue(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>(0x30C3), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>(0x30C1), buffer[1]);
    EXPECT_EQ(static_cast<uint16_t>(0x30E3), buffer[2]);
}

//!< キーコード列の削除が正しく処理されるか
TEST(KeyCodeComposerSuite, RemoveKeyCodesTest1)
{
    ::nn::kpr::KeyCodeComposer composer;
    composer.SetMode(::nn::kpr::KeyCodeComposerMode_RomajiKatakana);

    uint16_t buffer[8] = {};

    EXPECT_EQ(0, composer.Remove(buffer, NN_ARRAY_SIZE(buffer)));

    const uint16_t keyCodes1[] = { 'n', 'y' };
    EXPECT_EQ(2, composer.Enqueue(keyCodes1, NN_ARRAY_SIZE(keyCodes1)));
    EXPECT_EQ(2, composer.Remove(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>('n'), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>('y'), buffer[1]);
    EXPECT_EQ(0, composer.GetCompositionCount());

    const uint16_t keyCodes2[] = { 'l', 'l', 'w', 'h' };
    EXPECT_EQ(4, composer.Enqueue(keyCodes2, NN_ARRAY_SIZE(keyCodes2)));
    EXPECT_EQ(1, composer.Remove(buffer, 1));
    EXPECT_EQ(static_cast<uint16_t>('h'), buffer[0]);
    EXPECT_EQ(1, composer.GetDequeueableCount());
    EXPECT_EQ(3, composer.GetCompositionCount());

    EXPECT_EQ(2, composer.Remove(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>('l'), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>('w'), buffer[1]);
    EXPECT_EQ(1, composer.GetDequeueableCount());
    EXPECT_EQ(1, composer.GetCompositionCount());

    EXPECT_EQ(1, composer.Dequeue(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>('l'), buffer[0]);
}

//!< キーコード列の明示的な確定が正しく処理されるか
TEST(KeyCodeComposerSuite, FlushKeyCodesTest1)
{
    ::nn::kpr::KeyCodeComposer composer;

    auto keyCode = uint16_t();

    composer.SetMode(::nn::kpr::KeyCodeComposerMode_DeadKey);
    composer.Flush();
    keyCode = static_cast<uint16_t>(0x0308);
    EXPECT_EQ(1, composer.Enqueue(&keyCode, 1));
    composer.Flush();
    EXPECT_EQ(1, composer.Dequeue(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>(0x00A8), keyCode);

    composer.SetMode(::nn::kpr::KeyCodeComposerMode_RomajiHiragana);
    composer.Flush();
    keyCode = static_cast<uint16_t>('n');
    EXPECT_EQ(1, composer.Enqueue(&keyCode, 1));
    composer.Flush();
    EXPECT_EQ(1, composer.Dequeue(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>(0x3093), keyCode);

    composer.SetMode(::nn::kpr::KeyCodeComposerMode_RomajiKatakana);
    composer.Flush();
    keyCode = static_cast<uint16_t>('n');
    EXPECT_EQ(1, composer.Enqueue(&keyCode, 1));
    composer.Flush();
    EXPECT_EQ(1, composer.Dequeue(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>(0x30F3), keyCode);

    const uint16_t keyCodes1[] = { 'x', 'x', 't', 's' };
    EXPECT_EQ(4, composer.Enqueue(keyCodes1, 4));
    composer.Flush();
    EXPECT_EQ(1, composer.Dequeue(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>('x'), keyCode);
    EXPECT_EQ(1, composer.Dequeue(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>('x'), keyCode);
    EXPECT_EQ(1, composer.Dequeue(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>('t'), keyCode);
    EXPECT_EQ(1, composer.Dequeue(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>('s'), keyCode);
}

//!< キーコード列の破棄が正しく処理されるか
TEST(KeyCodeComposerSuite, ClearTest1)
{
    ::nn::kpr::KeyCodeComposer composer;
    composer.SetMode(::nn::kpr::KeyCodeComposerMode_RomajiHiragana);

    const uint16_t keyCodes1[] = { 'g' };
    EXPECT_EQ(1, composer.Enqueue(keyCodes1, 1));

    composer.Clear();
    EXPECT_EQ(0, composer.GetCompositionCount());
    EXPECT_EQ(0, composer.GetDequeueableCount());

    const uint16_t keyCodes2[] = { 'm' };
    EXPECT_EQ(1, composer.Enqueue(keyCodes2, 1));
    EXPECT_EQ(1, composer.GetCompositionCount());
    EXPECT_EQ(0, composer.GetDequeueableCount());

    const uint16_t keyCodes3[] = { 'i' };
    EXPECT_EQ(1, composer.Enqueue(keyCodes3, 1));
    EXPECT_EQ(1, composer.GetCompositionCount());
    EXPECT_EQ(1, composer.GetDequeueableCount());

    uint16_t buffer[8] = {};
    EXPECT_EQ(1, composer.Dequeue(buffer, NN_ARRAY_SIZE(buffer)));
    EXPECT_EQ(static_cast<uint16_t>(0x307F), buffer[0]);
}
