﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>

#include "TestVm.h"
#include "TestVmInterpreter.h"
#include "TestVmByJit.h"
#include <memory>

namespace {

template <size_t StackCount>
void RunImpl(nn::jit::testvm::Instruction* instructions, size_t instructionCount, nn::jit::testvm::RunTestVm runTestVm, int32_t expected)
{
    if (!runTestVm)
    {
        return;
    }

    using namespace nn::jit::testvm;
    int32_t stack[StackCount] = {};
    MachineContext c = {};
    c.program.instructions = instructions;
    c.program.instructionCount = instructionCount;
    nn::jit::testvm::ResolveLabel(&c.program);
    c.state.stack = stack;
    c.state.stackCountMax = StackCount;

    runTestVm(&c);

    ASSERT_EQ(1, c.state.stackCount);
    ASSERT_EQ(expected, c.state.stack[0]);
}

template <size_t StackCount, size_t N>
void RunImpl(nn::jit::testvm::Instruction (&instructions)[N], nn::jit::testvm::RunTestVm runTestVm, int32_t expected)
{
    RunImpl<StackCount>(instructions, N, runTestVm, expected);
}

}

class RunVmTest
    : public testing::TestWithParam<nn::jit::testvm::RunTestVm>
{
};

TEST_P(RunVmTest, Add)
{
    using namespace nn::jit::testvm;
    Instruction instructions[] =
    {
        4,
        6,
        InstructionKind::Add,
        InstructionKind::End,
    };
    RunImpl<10>(instructions, GetParam(), 10);
}

TEST_P(RunVmTest, Dup)
{
    using namespace nn::jit::testvm;
    Instruction instructions[] = {
        4,
        InstructionKind::Dup,
        InstructionKind::Add,
        InstructionKind::End,
    };
    RunImpl<10>(instructions, GetParam(), 8);
}

TEST_P(RunVmTest, Swap)
{
    using namespace nn::jit::testvm;
    Instruction instructions[] = {
        4,
        6,
        InstructionKind::Swap,
        InstructionKind::Pop,
        InstructionKind::End,
    };
    RunImpl<10>(instructions, GetParam(), 6);
}

TEST_P(RunVmTest, Read)
{
    using namespace nn::jit::testvm;
    Instruction instructions[] = {
        3,
        2,
        1,
        0,
        Instruction::Read(4),
        InstructionKind::Swap,
        InstructionKind::Pop,
        InstructionKind::Swap,
        InstructionKind::Pop,
        InstructionKind::Swap,
        InstructionKind::Pop,
        InstructionKind::Swap,
        InstructionKind::Pop,
        InstructionKind::End,
    };
    RunImpl<10>(instructions, GetParam(), 3);
}

TEST_P(RunVmTest, Write)
{
    using namespace nn::jit::testvm;
    Instruction instructions[] = {
        3,
        2,
        1,
        0,
        Instruction::Write(4),
        InstructionKind::Pop,
        InstructionKind::Pop,
        InstructionKind::End,
    };
    RunImpl<10>(instructions, GetParam(), 0);
}

TEST_P(RunVmTest, JmpR)
{
    using namespace nn::jit::testvm;
    Instruction instructions[] =
    {
        Instruction::PushLabel(0), InstructionKind::JmpR,
        4,
        InstructionKind::End,

        Instruction::Label(0),
        6,
        InstructionKind::End,
    };
    RunImpl<10>(instructions, GetParam(), 6);
}

TEST_P(RunVmTest, JmpRIf0_0)
{
    using namespace nn::jit::testvm;
    Instruction instructions[] =
    {
        0,
        Instruction::PushLabel(0), InstructionKind::JmpRIf0,
        4,
        InstructionKind::End,

        Instruction::Label(0),
        6,
        InstructionKind::End,
    };
    RunImpl<10>(instructions, GetParam(), 6);
}

TEST_P(RunVmTest, JmpRIf0_Not0)
{
    using namespace nn::jit::testvm;
    Instruction instructions[] =
    {
        1,
        Instruction::PushLabel(0), InstructionKind::JmpRIf0,
        4,
        InstructionKind::End,

        Instruction::Label(0),
        6,
        InstructionKind::End,
    };
    RunImpl<10>(instructions, GetParam(), 4);
}

TEST_P(RunVmTest, Not0)
{
    using namespace nn::jit::testvm;
    Instruction instructions[] =
    {
        0,
        InstructionKind::Not,
        InstructionKind::End,
    };
    RunImpl<10>(instructions, GetParam(), 1);
}

TEST_P(RunVmTest, Not1)
{
    using namespace nn::jit::testvm;
    Instruction instructions[] =
    {
        1,
        InstructionKind::Not,
        InstructionKind::End,
    };
    RunImpl<10>(instructions, GetParam(), 0);
}

TEST_P(RunVmTest, Call)
{
    using namespace nn::jit::testvm;
    Instruction instructions[] =
    {
        10,
        Instruction::PushLabel(0), InstructionKind::Call,
        InstructionKind::End,

        Instruction::Label(0),
        InstructionKind::Swap,
        InstructionKind::Dup,
        InstructionKind::Mul,
        InstructionKind::Swap,
        InstructionKind::Ret,
    };
    RunImpl<10>(instructions, GetParam(), 100);
}

TEST_P(RunVmTest, SquareSum)
{
    using namespace nn::jit::testvm;
    const auto MaxCount = 100;
    auto square = [](int32_t x)
    {
        return x * x;
    };
    auto answer = [&]()
    {
        int32_t ret = 0;
        for (int32_t i = MaxCount; i > 0; --i)
        {
            ret += square(i);
        }
        return ret;
    }();
    Instruction instructions[] =
    {
        0,
        MaxCount,
        Instruction::PushLabel(1), InstructionKind::JmpR,

        Instruction::Label(0),
        InstructionKind::Pop,
        InstructionKind::End,

        Instruction::Label(1),
        InstructionKind::Dup, Instruction::PushLabel(0), InstructionKind::JmpRIf0,
        InstructionKind::Swap,
        Instruction::Read(2),
        Instruction::PushLabel(2), InstructionKind::Call,
        InstructionKind::Add,
        InstructionKind::Swap,
        -1,
        InstructionKind::Add,
        Instruction::PushLabel(1), InstructionKind::JmpR,

        Instruction::Label(2),
        InstructionKind::Swap,
        InstructionKind::Dup,
        InstructionKind::Mul,
        InstructionKind::Swap,
        InstructionKind::Ret,
    };
    RunImpl<10>(instructions, GetParam(), answer);
}

TEST_P(RunVmTest, BigLoop)
{
    using namespace nn::jit::testvm;
    int Loop = 10000;
    int N = 10000;
    std::unique_ptr<Instruction[]> instructions(new Instruction[N + 100]);
    auto p = &instructions[0];
    *p++ = Loop;
    *p++ = Instruction::Label(0);
    *p++ = InstructionKind::Dup;
    *p++ = Instruction::PushLabel(1); *p++ = InstructionKind::Call;
    *p++ = InstructionKind::Pop;
    *p++ = 1;
    *p++ = InstructionKind::Neg;
    *p++ = InstructionKind::Add;
    *p++ = InstructionKind::Dup;
    *p++ = InstructionKind::Not;
    *p++ = Instruction::PushLabel(0); *p++ = InstructionKind::JmpRIf0;
    *p++ = InstructionKind::End;

    *p++ = Instruction::Label(1);
    for (int i = 0; i < N; ++i)
    {
        *p++ = InstructionKind::Swap;
    }
    *p++ = InstructionKind::Swap;
    *p++ = InstructionKind::Dup;
    *p++ = InstructionKind::Mul;
    *p++ = InstructionKind::Swap;
    *p++ = InstructionKind::Ret;
    RunImpl<10>(instructions.get(), p - instructions.get(), GetParam(), 0);
}

namespace {

int32_t Ackermann(int32_t m, int32_t n) NN_NOEXCEPT
{
    if (m == 0)
    {
        return n + 1;
    }
    if (n == 0)
    {
        return Ackermann(m - 1, 1);
    }
    return Ackermann(m - 1, Ackermann(m, n - 1));
}

}

TEST_P(RunVmTest, Ackermann)
{
    using namespace nn::jit::testvm;
    const auto M = 3;
    const auto N = 5;
    Instruction instructions[] =
    {
        // 0:
        M,
        N,
        Instruction::PushLabel(0), InstructionKind::Call,
        InstructionKind::End,

        // 5:
        Instruction::Label(0), // m, n, LR
        Instruction::Read(3),  // m, n, LR, m
        InstructionKind::Not,  // m, n, LR, !m
        Instruction::PushLabel(1), InstructionKind::JmpRIf0, // m, n, LR

        // if m = 0
        // 10:
        InstructionKind::Swap, // m, LR, n
        1,
        InstructionKind::Add,  // m, LR, n + 1
        Instruction::Write(3), // n + 1, LR
        InstructionKind::Ret,

        // else
        // 15:
        Instruction::Label(1), // m, n, LR
        Instruction::Read(2),  // m, n, LR, n
        InstructionKind::Not,  // m, n, LR, !n
        Instruction::PushLabel(2), InstructionKind::JmpRIf0, // m, n, LR

        // if n = 0
        // 20:
        InstructionKind::Swap, // m, LR, n
        InstructionKind::Pop,  // m, LR
        InstructionKind::Swap, // LR, m
        -1,
        InstructionKind::Add,  // LR, m - 1
        // 25:
        1,                     // LR, m - 1, 1
        Instruction::PushLabel(0), InstructionKind::Call, // LR, Ack(m - 1, 1)
        InstructionKind::Swap, // Ack(m - 1, 1), LR
        InstructionKind::Ret,

        // else
        // 30:
        Instruction::Label(2), // m, n, LR
        Instruction::Read(3),  // m, n, LR, m
        Instruction::Read(3),  // m, n, LR, m, n
        -1,
        InstructionKind::Add,  // m, n, LR, m, n - 1
        // 35:
        Instruction::PushLabel(0), InstructionKind::Call, // m, n, LR, Ack(m, n - 1)
        Instruction::Read(4),  // m, n, LR, Ack(m, n - 1), m
        -1,
        InstructionKind::Add,  // m, n, LR, Ack(m, n - 1), m - 1
        // 40:
        InstructionKind::Swap, // m, n, LR, m - 1, Ack(m, n - 1)
        Instruction::PushLabel(0), InstructionKind::Call, // m, n, LR, Ack(m - 1, Ack(m, n - 1))
        Instruction::Write(4), // Ack(m - 1, Ack(m, n - 1)), n, LR
        InstructionKind::Swap, // Ack(m - 1, Ack(m, n - 1)), LR, n
        // 45:
        InstructionKind::Pop,  // Ack(m - 1, Ack(m, n - 1)), LR
        InstructionKind::Ret,
    };
    RunImpl<1000>(instructions, GetParam(), Ackermann(M, N));
}

INSTANTIATE_TEST_CASE_P(TestVmTest, RunVmTest, ::testing::Values(
    nn::jit::testvm::RunTestVmByInterpreter,
    nn::jit::testvm::RunTestVmByJit
));
