﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "TestVmInterpreter.h"

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

namespace nn { namespace jit { namespace testvm {

int32_t Interpreter::Pop() NN_NOEXCEPT
{
    auto& n = m_pContext->state.stackCount;
    if (n == 0)
    {
        return 0;
    }
    --n;
    auto ret = m_pContext->state.stack[n];
    m_pContext->state.stack[n] = 0xCCCCCCCC;
    return ret;
}

void Interpreter::Push(int32_t x) NN_NOEXCEPT
{
    auto& n = m_pContext->state.stackCount;
    if (n == m_pContext->state.stackCountMax)
    {
        return;
    }
    m_pContext->state.stack[n++] = x;
}

void Interpreter::Jump(int32_t pc) NN_NOEXCEPT
{
    m_pContext->pc = pc;
}
void Interpreter::JumpRelative(int32_t d) NN_NOEXCEPT
{
    Jump(m_pContext->pc + d - 1);
}

bool Interpreter::Step1() NN_NOEXCEPT
{
    auto& pc = m_pContext->pc;
    #if 0
        NN_LOG("pc=%2d d=%2d", static_cast<int>(pc), static_cast<int>(m_pContext->state.stackCount));
        NN_LOG("  ");
        for (auto i = 0u; i < m_pContext->state.stackCount; ++i)
        {
            NN_LOG("[%d]=%2d ", i, static_cast<int>(m_pContext->state.stack[i]));
        }
        NN_LOG("\n");
    #endif
    if (!(pc < m_pContext->program.instructionCount))
    {
        return false;
    }
    auto inst = m_pContext->program.instructions[pc];
    ++pc;
    switch (inst.kind)
    {
        case InstructionKind::End:
        {
            --pc;
            return false;
        }
        case InstructionKind::Nop:
        case InstructionKind::Label:
        {
            return true;
        }
        case InstructionKind::Imm:
        {
            Push(inst.imm);
            return true;
        }
        case InstructionKind::Read:
        {
            auto x = m_pContext->state.stack[m_pContext->state.stackCount - inst.imm];
            Push(x);
            return true;
        }
        case InstructionKind::Write:
        {
            auto& ref = m_pContext->state.stack[m_pContext->state.stackCount - inst.imm];
            auto x = Pop();
            ref = x;
            return true;
        }
        case InstructionKind::Pop:
        {
            Pop();
            return true;
        }
        case InstructionKind::Dup:
        {
            auto tmp = Pop();
            Push(tmp);
            Push(tmp);
            return true;
        }
        case InstructionKind::Swap:
        {
            auto x = Pop();
            auto y = Pop();
            Push(x);
            Push(y);
            return true;
        }
        case InstructionKind::Add:
        {
            auto x = Pop();
            auto y = Pop();
            Push(x + y);
            return true;
        }
        case InstructionKind::Mul:
        {
            auto x = Pop();
            auto y = Pop();
            Push(x * y);
            return true;
        }
        case InstructionKind::Neg:
        {
            auto x = Pop();
            Push(-x);
            return true;
        }
        case InstructionKind::Not:
        {
            auto x = Pop();
            Push(x == 0 ? 1 : 0);
            return true;
        }
        case InstructionKind::JmpR:
        {
            auto d = Pop();
            JumpRelative(d);
            return true;
        }
        case InstructionKind::JmpRIf0:
        {
            auto d = Pop();
            auto x = Pop();
            if (x == 0)
            {
                JumpRelative(d);
            }
            return true;
        }
        case InstructionKind::Call:
        {
            auto d = Pop();
            Push(pc);
            JumpRelative(d - 1);
            return true;
        }
        case InstructionKind::Ret:
        {
            auto newPc = Pop();
            Jump(newPc);
            return true;
        }
        default: NN_UNEXPECTED_DEFAULT;
    }
} // NOLINT(impl/function_size)

void Interpreter::Run() NN_NOEXCEPT
{
    while (Step1())
    {
    }
}


void RunTestVmByInterpreter(MachineContext* pContext) NN_NOEXCEPT
{
    Interpreter(pContext).Run();
}

}}}
