﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Utility.h"

#include <random>
#include <thread>
#include <mutex>
#include <nn/os.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Server.h>
#include <nn/svc/svc_ThreadLocalRegion.h>
#include <nn/util/util_BitUtil.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/sf/hipc/detail/sf_HipcMessageBufferAccessor2.h>

namespace nn { namespace jit {

namespace {

class RandomGenerator
{
private:

    std::mt19937 m_Random;

public:

    RandomGenerator() NN_NOEXCEPT
    {
        Bit64 seed;
        auto result = nn::svc::GetInfo(&seed, svc::InfoType_Random, nn::svc::Handle(0), 2);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        m_Random.seed(seed);
    }

    uint64_t Generate(uint64_t randMax) NN_NOEXCEPT
    {
        std::uniform_int_distribution<uint64_t> rand(0, randMax);
        return rand(m_Random);
    }

};

RandomGenerator g_RandomGenerator;

}

AslrAllocator::AslrAllocator(nn::svc::Handle process) NN_NOEXCEPT
{
    Bit64 value;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_HeapRegionAddress, process, 0));
    m_HeapBegin = value;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_HeapRegionSize, process, 0));
    m_HeapSize = value;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_ReservedRegionAddress, process, 0));
    m_RsvBegin = value;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_ReservedRegionSize, process, 0));
    m_RsvSize = value;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_AslrRegionAddress, process, 0));
    m_AslrBegin = value;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_AslrRegionSize, process, 0));
    m_AslrSize = value;
}

uint64_t AslrAllocator::GetAslrRegion(uint64_t mappingSize) NN_NOEXCEPT
{
    if (mappingSize > m_AslrSize)
    {
        return 0;
    }
    for (int i = 0; i < 512; i++)
    {
        uint64_t addr = m_AslrBegin + g_RandomGenerator.Generate((m_AslrSize - mappingSize) / os::MemoryPageSize) * os::MemoryPageSize;

        if (m_HeapSize && !(addr + mappingSize - 1 < m_HeapBegin || m_HeapBegin + m_HeapSize - 1 < addr))
        {
            continue;
        }

        if (m_RsvSize && !(addr + mappingSize - 1 < m_RsvBegin || m_RsvBegin + m_RsvSize - 1 < addr))
        {
            continue;
        }
        return addr;
    }

    return 0;
}

#if 1
svc::Handle DuplicateHandle(svc::Handle handle) NN_NOEXCEPT
{
    static std::mutex mutex;
    static NN_ALIGNAS(4096) char clientBuffer[4096];

    std::lock_guard<decltype(mutex)> lk(mutex);

    svc::Handle serverHandle;
    svc::Handle clientHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(svc::CreateSession(&serverHandle, &clientHandle, false, 0));
    NN_UTIL_SCOPE_EXIT
    {
        svc::CloseHandle(serverHandle);
        svc::CloseHandle(clientHandle);
    };

    using namespace nn::sf::hipc::detail;
    {
        HipcMessageHeaderInfo headerInfo = {};
        headerInfo.copyHandleCount = 1;
        const HipcMessageDataInfo dataInfo = MakeHipcMessageDataInfo(headerInfo);
        HipcMessageWriter w(dataInfo, reinterpret_cast<void*>(clientBuffer), 256);
        w.SetHeader(0);
        w.SetCopyHandle(0, handle.GetPrintableBits());
        svc::Handle e;
        NN_ABORT_UNLESS_RESULT_SUCCESS(svc::SendAsyncRequestWithUserBuffer(&e, reinterpret_cast<uintptr_t>(clientBuffer), 4096, clientHandle));
        svc::CloseHandle(e);
    }
    {
        int32_t n;
        NN_ABORT_UNLESS_RESULT_SUCCESS(svc::ReplyAndReceive(&n, &serverHandle, 1, svc::INVALID_HANDLE_VALUE, -1));
        HipcMessageReader r;
        r.Initialize(svc::GetThreadLocalRegion()->messageBuffer);
        NN_SDK_ASSERT(r.GetHipcMessageHeaderInfo().baseInfo.copyHandleCount == 1);
        return svc::Handle(r.GetCopyHandle(0));
    }
}
#else
svc::Handle DuplicateHandle(svc::Handle handle) NN_NOEXCEPT
{
    svc::Handle serverHandle;
    svc::Handle clientHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(svc::CreateSession(&serverHandle, &clientHandle, false, 0));

    using namespace nn::sf::hipc::detail;
    HipcMessageHeaderInfo headerInfo = {};
    headerInfo.copyHandleCount = 1;
    const HipcMessageDataInfo dataInfo = MakeHipcMessageDataInfo(headerInfo);

    svc::Handle processHandle;
    std::thread th([serverHandle, &dataInfo, &processHandle]()
    {
        int32_t n;
        NN_ABORT_UNLESS_RESULT_SUCCESS(svc::ReplyAndReceive(&n, &serverHandle, 1, svc::INVALID_HANDLE_VALUE, -1));
        HipcMessageReader r;
        r.Initialize(svc::GetThreadLocalRegion()->messageBuffer);
        NN_SDK_ASSERT(r.GetHipcMessageHeaderInfo().baseInfo.copyHandleCount == 1);
        processHandle = svc::Handle(r.GetCopyHandle(0));
        svc::CloseHandle(serverHandle);
    });

    {
        HipcMessageWriter w(dataInfo, svc::GetThreadLocalRegion()->messageBuffer, 256);
        w.SetHeader(0);
        w.SetCopyHandle(0, handle.GetPrintableBits());
        svc::SendSyncRequest(clientHandle);
    }

    th.join();
    svc::CloseHandle(clientHandle);
    return processHandle;
}
#endif

namespace {

    template <typename F>
    static void IterateCacheLine(uintptr_t addr, size_t size, F f) NN_NOEXCEPT
    {
        uintptr_t cacheTypeRegister;
        asm volatile ("mrs %0, ctr_el0" : "=r" (cacheTypeRegister));
        size_t cacheLineSize = 4 << ((cacheTypeRegister >> 16) & 0xF);
        auto startAddr = reinterpret_cast<uintptr_t>(addr) & ~(cacheLineSize - 1);
        auto endAddr = reinterpret_cast<uintptr_t>(addr) + size;
        for (auto p = startAddr; p < endAddr; p += cacheLineSize)
        {
            f(p);
        }
    }

}

void StoreDataCacheAndMemoryBarrier(uintptr_t addr, size_t size) NN_NOEXCEPT
{
    //データキャッシュ store
    IterateCacheLine(addr, size, [](uintptr_t p)
    {
        asm volatile ("dc cvau, %0"::"r"(p):"memory");
    });
    // ish メモリバリア
    asm volatile ("dsb ish":::"memory");
}

void InvalidateInstructionCacheAndMemoryBarrier(uintptr_t addr, size_t size) NN_NOEXCEPT
{
    // 命令キャッシュ無効化
    IterateCacheLine(addr, size, [](uintptr_t p)
    {
        asm volatile ("ic ivau, %0"::"r"(p):"memory");
    });
    // ish メモリバリア
    asm volatile ("dsb ish":::"memory");
    // パイプラインフラッシュ
    asm volatile ("isb":::"memory");
}

}}
