﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <functional>
#include <vector>

#include <nn/irsensor.h>

#include "AttachmentEvaluationView.h"

class MenuItem
{
public:
    typedef std::function<void(std::stringstream&)> ReaderType;
    typedef std::function<void(int8_t)> WriterType;

public:
    MenuItem(const char* name, ReaderType reader, WriterType writer);

    void Increment(int8_t delta) const;
    void Read(std::stringstream& sstr) const;

    const char* GetName() const;

private:
    std::string m_Name;
    ReaderType m_Reader;
    WriterType m_Writer;
};

class IrSensorModeStatistics
{
public:
    static const int ResetIntervalsInFrame  = 60 * 3;
    static const int UpdateIntervalsInFrame = 20;

    static const float ExpectedMomentModeFramerate;
    static const float ExpectedClusteringModeFramerate;
    static const float ExpectedHandAnalysisFramerate;

    uint32_t     m_FramerateCounter{ 0 };
    nn::os::Tick m_FramerateFirstTick{ 0 };
    int64_t      m_FramerateFirstSample{ 0 };
    float        m_FramerateComputation{ 0.0f };
    float        m_PacketDropPercentage{ 0.0f };

    void Update(const int64_t* pCurrentSamplingNumber, const float ExpectedFrameRate);
};

class IrSensorModeState
{
public:
    IrSensorModeState(IrSensorMode* pNextProcessor, int* pMenuSelection, nn::irsensor::IrCameraHandle irCameraHandle);
    virtual ~IrSensorModeState() = default;

    static void AddCommonReadWriteMenu(std::vector<MenuItem>* pMenu, ::nn::irsensor::IrCameraConfig* pConfig, ::nn::TimeSpanType exposureMin, ::nn::TimeSpanType exposureMax);
    static void AddCommonReadOnlyMenu(std::vector<MenuItem>* pMenu, int64_t* pSamplingNumber, nn::irsensor::IrCameraAmbientNoiseLevel* pAmbientNoiseLevel);
    static void AddStatisticsMenu(std::vector<MenuItem>* pMenu, float* pPacketLoss);

    const std::vector<MenuItem>& GetReadWriteMenu();
    const std::vector<MenuItem>& GetReadOnlyMenu();

    void Stop();
    void HandleResult(nn::Result result);
    virtual void Start() = 0;
    virtual void Update() = 0;
    virtual void Record(const RecordInfo& recordInfo, const PathInfo& pathInfo, int clusterCountMax, GraphicsSystem* pGraphicsSystem) = 0;
    virtual void Reset() = 0;
    virtual void Render(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, nn::gfx::CommandBuffer* pCommandBuffer, int index) = 0;

protected:
    std::vector<MenuItem> m_ReadWriteMenu;
    std::vector<MenuItem> m_ReadOnlyMenu;

    nn::irsensor::IrCameraHandle   m_IrCameraHandle;
    IrSensorModeStatistics  m_Statistics;

    AttachmentInput*  m_pInput;
    AttachmentOutput* m_pOutput;
    AttachmentLogger* m_pLogger;
};
