﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ImageTransferModeState.h"

#include <algorithm>
#include <sstream>

#include <nn/os.h>
#include <nn/nn_Assert.h>

namespace
{

struct ImageProperties
{
public:
    ImageProperties(int width, int height, int size, int workBufferSize, const char* name)
        : Width(width)
        , Height(height)
        , Size(size)
        , WorkBufferSize(workBufferSize)
        , Name(name)
    {}

public:
    int         Width;
    int         Height;
    int         Size;
    int         WorkBufferSize;
    const char* Name;
};

NN_STATIC_ASSERT(nn::irsensor::ImageTransferProcessorFormat_320x240 == 0);
NN_STATIC_ASSERT(nn::irsensor::ImageTransferProcessorFormat_160x120 == 1);
NN_STATIC_ASSERT(nn::irsensor::ImageTransferProcessorFormat_80x60 == 2);

ImageProperties Properties[] ={
    ImageProperties(
        nn::irsensor::IrCameraImageWidth,
        nn::irsensor::IrCameraImageHeight,
        nn::irsensor::ImageTransferProcessorImageSize320x240,
        nn::irsensor::ImageTransferProcessorWorkBufferSize320x240,
        "320x240"
    ),
    ImageProperties(
        nn::irsensor::IrCameraImageWidth >> nn::irsensor::ImageTransferProcessorFormat_160x120,
        nn::irsensor::IrCameraImageHeight >> nn::irsensor::ImageTransferProcessorFormat_160x120,
        nn::irsensor::ImageTransferProcessorImageSize160x120,
        nn::irsensor::ImageTransferProcessorWorkBufferSize160x120,
        "160x120"
    ),
    ImageProperties(
        nn::irsensor::IrCameraImageWidth >> nn::irsensor::ImageTransferProcessorFormat_80x60,
        nn::irsensor::IrCameraImageHeight >> nn::irsensor::ImageTransferProcessorFormat_80x60,
        nn::irsensor::ImageTransferProcessorImageSize80x60,
        nn::irsensor::ImageTransferProcessorWorkBufferSize80x60,
        "80x60"
    )
};

}

ImageTransferModeState::ImageTransferModeState(IrSensorMode* pNextProcessor, int* pMenuSelection, nn::irsensor::IrCameraHandle irCameraHandle, void* pWorkMemory, GraphicsSystem *pGraphicsSystem)
    : IrSensorModeState(pNextProcessor, pMenuSelection, irCameraHandle)
    , m_pImageTransferWorkBuffer(pWorkMemory)
    , m_pImageCopyWorkBuffer(static_cast<uint8_t*>(pWorkMemory) + nn::irsensor::ImageTransferProcessorWorkBufferSize320x240)
    , m_pGraphicsSystem(pGraphicsSystem)
{
    m_PreviousSamplingNumber = 0;
    {
        nn::irsensor::GetImageTransferProcessorDefaultConfig(&m_ImageTransferProcessorConfig);

        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_Read);
        info.SetSize(nn::irsensor::ImageTransferProcessorImageSize320x240);
        pGraphicsSystem->AllocateBuffer(&m_Buffer, &info);

        for (int i = 0; i < NN_ARRAY_SIZE(m_Textures); i++)
        {
            CreateTexture(pGraphicsSystem, Properties[i].Width, Properties[i].Height, nn::gfx::ImageFormat_R8_Unorm, &m_Textures[i]);
        }
        CreateSampler(pGraphicsSystem, &m_Sampler);
    }

    nn::irsensor::ImageTransferProcessorConfig* pImageTransferConfig = &m_ImageTransferProcessorConfig;
    AddCommonReadWriteMenu(&m_ReadWriteMenu,
        &pImageTransferConfig->irCameraConfig,
        nn::irsensor::ImageTransferProcessorExposureTimeMin,
        nn::irsensor::ImageTransferProcessorExposureTimeMax
    );

    m_ReadWriteMenu.emplace_back("Format",
        [pImageTransferConfig](std::stringstream& sstr) {
            sstr << Properties[pImageTransferConfig->format].Name;
        },
        [pImageTransferConfig](int8_t delta) {
            int8_t format = static_cast<int8_t>(pImageTransferConfig->format) + static_cast<int8_t>(delta);
            format = std::max(std::min(format, static_cast<int8_t>(nn::irsensor::ImageTransferProcessorFormat_80x60)), static_cast<int8_t>(nn::irsensor::ImageTransferProcessorFormat_320x240));
            pImageTransferConfig->format = static_cast<nn::irsensor::ImageTransferProcessorFormat>(format);
        }
    );

    AddCommonReadOnlyMenu(&m_ReadOnlyMenu, &m_ImageTransferProcessorState.samplingNumber, &m_ImageTransferProcessorState.ambientNoiseLevel);

}

ImageTransferModeState::~ImageTransferModeState()
{
    m_pGraphicsSystem->FreeBuffer(&m_Buffer);

    for (int i = 0; i < NN_ARRAY_SIZE(m_Textures); i++)
    {
        m_pGraphicsSystem->FreeTexture(&m_Textures[i].texture);
        m_Textures[i].view.Finalize(&m_pGraphicsSystem->GetDevice());
    }

    m_Sampler.sampler.Finalize(&m_pGraphicsSystem->GetDevice());
}

void ImageTransferModeState::Start()
{
    nn::irsensor::RunImageTransferProcessor(m_IrCameraHandle, m_ImageTransferProcessorConfig, static_cast<void*>(m_pImageTransferWorkBuffer), Properties[m_ImageTransferProcessorConfig.format].WorkBufferSize);
    m_Format = m_ImageTransferProcessorConfig.format;
    // バッファをクリアしておく
    uint8_t* sourcePixels = m_Buffer.Map<uint8_t>();
    memset(sourcePixels, 0, Properties[m_Format].Size);
    m_Buffer.FlushMappedRange(0, nn::irsensor::ImageTransferProcessorImageSize320x240);
    m_Buffer.Unmap();
}

namespace
{

void CopyBufferToImage(nn::gfx::Texture* pTexture, const nn::gfx::Buffer* pBuffer, int width, int height, nn::gfx::CommandBuffer* pCommandBuffer)
{
    NN_ASSERT_NOT_NULL(pTexture);
    NN_ASSERT_NOT_NULL(pBuffer);
    NN_ASSERT_NOT_NULL(pCommandBuffer);

    nn::gfx::TextureCopyRegion dstRegion;
    dstRegion.SetDefault();
    dstRegion.SetDepth(1);

    dstRegion.SetWidth(width);
    dstRegion.SetHeight(height);
    pCommandBuffer->CopyBufferToImage(pTexture, dstRegion, pBuffer, 0);
}

void CopyRawImageToBuffer(const nn::gfx::Buffer* pBuffer,
    void* copyBuffer,
    nn::irsensor::ImageTransferProcessorFormat format) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pBuffer);

    size_t writeSize = Properties[format].Size;

    uint8_t* sourcePixels = pBuffer->Map<uint8_t>();
    memcpy(sourcePixels, copyBuffer, Properties[format].Size);
    pBuffer->FlushMappedRange(0, writeSize);
    pBuffer->Unmap();
}
}

void ImageTransferModeState::CopyImageBuffer(nn::gfx::CommandBuffer* pCommandBuffer)
{
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    CopyBufferToImage(&m_Textures[m_Format].texture, &m_Buffer,
        Properties[m_Format].Width, Properties[m_Format].Height,
        pCommandBuffer);
}

void ImageTransferModeState::Update()
{
    nn::Result result = nn::irsensor::GetImageTransferProcessorState(&m_ImageTransferProcessorState, m_pImageCopyWorkBuffer, Properties[m_Format].Size, m_IrCameraHandle);
    HandleResult(result);
    if (m_PreviousSamplingNumber < m_ImageTransferProcessorState.samplingNumber)
    {
        // データ更新があった場合はバッファコピー
        m_PreviousSamplingNumber = m_ImageTransferProcessorState.samplingNumber;
        CopyRawImageToBuffer(&m_Buffer, m_pImageCopyWorkBuffer, m_Format);
    }
}

void ImageTransferModeState::RenderImageTransferProcessorState(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer,
    nn::gfx::CommandBuffer* pCommandBuffer,
    const int screenIndex) NN_NOEXCEPT
{
    NN_UNUSED(screenIndex);

    NN_ASSERT_NOT_NULL(pPrimitiveRenderer);
    NN_ASSERT_NOT_NULL(pCommandBuffer);

    const nn::util::Uint8x4 White ={ { 255, 255, 255, 255 } };

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer,
        nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // グリッドを描画
    nn::util::Vector3fType corner;
    nn::util::Vector3fType size;

    // 比が3:2 になるように計算
    const float xStart = -0.95f, xEnd = 0.0f;
    const float yStart = 0.72f, yEnd = -0.4f;

    nn::util::VectorSet(&corner, xStart, yStart, 0.0f);
    nn::util::VectorSet(&size, xEnd - xStart, yEnd - yStart, 0.0f);

    pPrimitiveRenderer->SetColor(White);
    pPrimitiveRenderer->DrawQuad(
        pCommandBuffer,
        corner,
        size,
        m_Textures[m_Format].descriptor,
        m_Sampler.descriptor);
}

void ImageTransferModeState::Record(const RecordInfo& recordInfo, const PathInfo& pathInfo, int clusterCountMax, GraphicsSystem* pGraphicsSystem)
{
    NN_UNUSED(recordInfo);
    NN_UNUSED(pathInfo);
    NN_UNUSED(clusterCountMax);
    NN_UNUSED(pGraphicsSystem);
}

void ImageTransferModeState::Reset()
{

}

void ImageTransferModeState::Render(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, nn::gfx::CommandBuffer* pCommandBuffer, int index)
{
    CopyImageBuffer(pCommandBuffer);
    RenderImageTransferProcessorState(pPrimitiveRenderer, pCommandBuffer, index);
}
