﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/hid/hid_ControllerSupport.h>

#include "ClusteringModeState.h"
#include "ImageTransferModeState.h"

#include "AttachmentEvaluationView.h"

#include <nn/time.h>
#include <sstream>

const int AttachmentVersionNameSizeMax = 16; //!< アタッチメントバージョン名の最大長
const int AttachmentVersionCountMax = 4; //!< アタッチメントバージョンの種類

const char AttachmentVersionName[AttachmentVersionCountMax][AttachmentVersionNameSizeMax] =
{
    "DP1",
    "PreDP2",
    "DP2",
    "DP2.5",
};

const int CameraVersionNameSizeMax = 16; //!< カメラバージョン名の最大長
const int CameraVersionCountMax = 8; //!< カメラの種類
const char CameraVersionName[CameraVersionCountMax][CameraVersionNameSizeMax] =
{
    "Typ0",
    "Typ1",
    "Typ2",
    "Typ3",
    "Typ4",
    "Typ5",
    "Typ6",
    "Typ7"
};

const int AttachmentIdMax = 1000; //!< アタッチメント ID の最大値

void DrawSafetyFrame(AttachmentInput* pInput,
    GraphicsSystem* pGraphicsSystem)
{
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();
    nn::gfx::CommandBuffer* pCommandBuffer = &pGraphicsSystem->GetCommandBuffer();

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
    pPrimitiveRenderer->SetLineWidth(3.0f);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // 比が3:2 になるように計算
    const float xStart = -0.95f, xEnd = 0.0f;
    const float yStart = 0.72f, yEnd = -0.4f;

    nn::util::Vector3fType begin;
    nn::util::Vector3fType end;

    const nn::util::Unorm8x4 Red = { { 128, 0, 0, 255 } };
    const nn::util::Unorm8x4 Yellow = { { 128, 128, 0, 255 } };

    int safetyX = 15;
    int safetyY = 15;
    for (auto i = 0; i < 2; i++)
    {
        if (i == 0)
        {
            pPrimitiveRenderer->SetColor(Red);
        }
        else
        {
            pPrimitiveRenderer->SetColor(Yellow);
            safetyX = pInput->safetyFramePixelX;
            safetyY = pInput->safetyFramePixelY;

        }
        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * safetyX / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * safetyY / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        nn::util::VectorSet(&end,
            xStart + (xEnd - xStart) * (nn::irsensor::IrCameraImageWidth - safetyX) / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * safetyY / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * (nn::irsensor::IrCameraImageWidth - safetyX) / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * (nn::irsensor::IrCameraImageHeight - safetyY) / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
        nn::util::VectorSet(&end,
            xStart + (xEnd - xStart) * safetyX / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * (nn::irsensor::IrCameraImageHeight - safetyY) / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * safetyX / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * safetyY / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    }
}

void DrawExpectedClusters(
    AttachmentInput* pInput,
    AttachmentOutput* pOutput,
    GraphicsSystem* pGraphicsSystem)
{
    if (pOutput == nullptr)
    {
        return;
    }

    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();
    nn::gfx::CommandBuffer* pCommandBuffer = &pGraphicsSystem->GetCommandBuffer();

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);
    pPrimitiveRenderer->SetLineWidth(2.0f);

    // 比が3:2 になるように計算
    const float xStart = -0.95f, xEnd = 0.0f;
    const float yStart = 0.72f, yEnd = -0.4f;

    nn::util::Vector3fType begin;
    nn::util::Vector3fType end;

    const nn::util::Unorm8x4 Gray = { { 128, 128, 128, 255 } };
    const nn::util::Unorm8x4 Blue = { { 0, 0, 128, 255 } };

    for (auto i = 0; i < pInput->objectCount; i++)
    {
        if (pOutput->clusters[i].isValid)
        {
            pPrimitiveRenderer->SetColor(Blue);
        }
        else
        {
            pPrimitiveRenderer->SetColor(Gray);
        }

        int x = pInput->expectedClusters[i].idealClusteringData.bound.x + pInput->clusteringConfig.windowOfInterest.x;
        int y = pInput->expectedClusters[i].idealClusteringData.bound.y + pInput->clusteringConfig.windowOfInterest.y;
        int width = pInput->expectedClusters[i].idealClusteringData.bound.width;
        int height = pInput->expectedClusters[i].idealClusteringData.bound.height;

        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * x / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * y / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        nn::util::VectorSet(&end,
            xStart + (xEnd - xStart) * (x + width - 1) / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * y / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * (x + width - 1) / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * (y + height - 1) / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
        nn::util::VectorSet(&end,
            xStart + (xEnd - xStart) * x / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * (y + height - 1) / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * x / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * y / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    }
}

void DrawMeasuredClusters(
    AttachmentInput* pInput,
    AttachmentOutput* pOutput,
    GraphicsSystem* pGraphicsSystem)
{
    if (pOutput == nullptr)
    {
        return;
    }

    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();
    nn::gfx::CommandBuffer* pCommandBuffer = &pGraphicsSystem->GetCommandBuffer();

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
    pPrimitiveRenderer->SetLineWidth(2.0f);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);


    // 比が3:2 になるように計算
    const float xStart = -0.95f, xEnd = 0.0f;
    const float yStart = 0.72f, yEnd = -0.4f;

    nn::util::Vector3fType begin;
    nn::util::Vector3fType end;

    const nn::util::Unorm8x4 DeepBlue = { { 0, 64, 192, 255 } };

    for (auto i = 0; i < pInput->objectCount; i++)
    {
        if (pOutput->clusters[i].isValid)
        {
            pPrimitiveRenderer->SetColor(DeepBlue);

            int x = pOutput->clusters[i].measuredClusteringData.bound.x + pInput->clusteringConfig.windowOfInterest.x;
            int y = pOutput->clusters[i].measuredClusteringData.bound.y + pInput->clusteringConfig.windowOfInterest.y;
            int width = pOutput->clusters[i].measuredClusteringData.bound.width;
            int height = pOutput->clusters[i].measuredClusteringData.bound.height;

            nn::util::VectorSet(&begin,
                xStart + (xEnd - xStart) * x / static_cast<float>(nn::irsensor::IrCameraImageWidth),
                yStart + (yEnd - yStart) * y / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
            nn::util::VectorSet(&end,
                xStart + (xEnd - xStart) * (x + width - 1) / static_cast<float>(nn::irsensor::IrCameraImageWidth),
                yStart + (yEnd - yStart) * y / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
            pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
            nn::util::VectorSet(&begin,
                xStart + (xEnd - xStart) * (x + width - 1) / static_cast<float>(nn::irsensor::IrCameraImageWidth),
                yStart + (yEnd - yStart) * (y + height - 1) / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
            pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
            nn::util::VectorSet(&end,
                xStart + (xEnd - xStart) * x / static_cast<float>(nn::irsensor::IrCameraImageWidth),
                yStart + (yEnd - yStart) * (y + height - 1) / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
            pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
            nn::util::VectorSet(&begin,
                xStart + (xEnd - xStart) * x / static_cast<float>(nn::irsensor::IrCameraImageWidth),
                yStart + (yEnd - yStart) * y / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
            pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
        }
    }
}

void DrawCameraParameters(const std::vector<MenuItem>& menu,
    const float offsetX, const float offsetY,
    int selectIndex,
    nn::gfx::util::DebugFontTextWriter* pTextWriter)
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    const nn::util::Unorm8x4 White ={ { 255, 255, 255, 255 } };
    const nn::util::Unorm8x4 Red ={ { 255, 0, 0, 255 } };

    pTextWriter->SetScale(1.0f, 1.0f);

    for (size_t index = 0; index < menu.size(); index++)
    {
        pTextWriter->SetTextColor(((size_t) selectIndex == index) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + index * 30.0f);

        std::stringstream text;
        text << menu[index].GetName() << ": ";
        menu[index].Read(text);

        pTextWriter->Print(text.str().c_str());
    }
}

void DrawRecordFinishedNotification(
    const float offsetX, const float offsetY,
    nn::gfx::util::DebugFontTextWriter* pTextWriter)
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    const nn::util::Unorm8x4 Yellow ={ { 255, 255, 0, 255 } };

    pTextWriter->SetScale(1.0f, 1.0f);
    pTextWriter->SetTextColor(Yellow);
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print("Save Log...");
}

void DrawOutputData(
    const float offsetX, const float offsetY,
    AttachmentOutput* pOutput,
    nn::gfx::util::DebugFontTextWriter* pTextWriter)
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    const nn::util::Unorm8x4 White ={ { 255, 255, 255, 255 } };
    const int fontSize = 12;
    auto index = 0;

    pTextWriter->SetScale(0.8f, 0.8f);
    pTextWriter->SetTextColor(White);
    pTextWriter->SetCursor(offsetX, offsetY + fontSize * index++);
    pTextWriter->Print("[SixAxis] X:%8.5f(%0.2f[deg]) Y:%8.5f(%0.2f[deg]) Z:%8.5f(%0.2f[deg])",
        pOutput->sixAxisData.acceleration.x, pOutput->sixAxisDegreeData.GetX(),
        pOutput->sixAxisData.acceleration.y, pOutput->sixAxisDegreeData.GetY(),
        pOutput->sixAxisData.acceleration.z, pOutput->sixAxisDegreeData.GetZ());
    pTextWriter->SetCursor(offsetX, offsetY + fontSize * index++);
    pTextWriter->Print("[OutMostFrame] Xmin:%d Ymin:%d Xmax:%d Ymax:%d", pOutput->outMostFrame.x, pOutput->outMostFrame.y,
        pOutput->outMostFrame.x + pOutput->outMostFrame.width, pOutput->outMostFrame.y + pOutput->outMostFrame.height);
    pTextWriter->SetCursor(offsetX, offsetY + fontSize * index++);
    pTextWriter->Print("[CenterPoint] X:%0.1f Y:%0.1f", pOutput->centerPoint.x, pOutput->centerPoint.y);
    for (auto i = 0; i< pOutput->objectCount; i++)
    {
        pTextWriter->SetCursor(offsetX, offsetY + fontSize * index++);
        pTextWriter->Print("[Cluster %2d] Xmin:%03d Ymin:%03d Xmax:%03d Ymax:%03d",
            i,
            pOutput->clusters[i].measuredClusteringData.bound.x,
            pOutput->clusters[i].measuredClusteringData.bound.y,
            pOutput->clusters[i].measuredClusteringData.bound.x + pOutput->clusters[i].measuredClusteringData.bound.width,
            pOutput->clusters[i].measuredClusteringData.bound.y + pOutput->clusters[i].measuredClusteringData.bound.height);
    }
}

void CreateSampler(GraphicsSystem* pGraphicsSystem, Sampler* pSampler)
{
    nn::gfx::Sampler::InfoType samplerInfo;
    samplerInfo.SetDefault();
    samplerInfo.SetFilterMode(nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint);
    samplerInfo.SetAddressU(nn::gfx::TextureAddressMode_Mirror);
    samplerInfo.SetAddressV(nn::gfx::TextureAddressMode_Mirror);
    samplerInfo.SetAddressW(nn::gfx::TextureAddressMode_Mirror);
    pSampler->sampler.Initialize(&pGraphicsSystem->GetDevice(), samplerInfo);

    pGraphicsSystem->RegisterSamplerSlot(&pSampler->descriptor, pSampler->sampler);
}

void CreateTexture(GraphicsSystem* pGraphicsSystem, int width, int height, nn::gfx::ImageFormat format, Texture* pTexture)
{
    {
        nn::gfx::Texture::InfoType textureInfo;

        textureInfo.SetDefault();
        textureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
        textureInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        textureInfo.SetMipCount(1);
        textureInfo.SetTileMode(nn::gfx::TileMode_Linear);
        textureInfo.SetWidth(width);
        textureInfo.SetHeight(height);
        textureInfo.SetImageFormat(format);

        pGraphicsSystem->AllocateTexture(&pTexture->texture, &textureInfo);
    }

    {
        nn::gfx::TextureView::InfoType viewInfo;
        viewInfo.SetDefault();
        viewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
        viewInfo.SetImageFormat(format);
        viewInfo.SetTexturePtr(&pTexture->texture);
        viewInfo.SetChannelMapping(nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red,
            nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red);

        pTexture->view.Initialize(&pGraphicsSystem->GetDevice(), viewInfo);
    }

    {
        pGraphicsSystem->RegisterTextureViewSlot(&pTexture->descriptor, pTexture->view);
    }
}

void AttachmentEvaluationView::DrawRecordParameters(
    const float offsetX, const float offsetY,
    int selectIndex,
    nn::gfx::util::DebugFontTextWriter* pTextWriter)
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    const nn::util::Unorm8x4 White ={ { 255, 255, 255, 255 } };
    const nn::util::Unorm8x4 Red ={ { 255, 0, 0, 255 } };

    pTextWriter->SetScale(1.0f, 1.0f);

    int index = 0;
    pTextWriter->SetTextColor((selectIndex == index) ? Red : White);
    pTextWriter->SetCursor(offsetX, offsetY + index++ * 30.0f);
    pTextWriter->Print("Measure Camera Id : %s", CameraVersionName[m_RecordInfo.measuredCameraId]);
    if (strcmp(m_pAttachmentInput->attachmentName, "Cal") != 0)
    {
        pTextWriter->SetTextColor((selectIndex == index) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + index++ * 30.0f);
        pTextWriter->Print("Attachment Version : %s", AttachmentVersionName[m_RecordInfo.attachmentVersionId]);
        pTextWriter->SetTextColor((selectIndex == index) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + index++ * 30.0f);
        pTextWriter->Print("Attachment Id : %d", m_RecordInfo.attachmentId);
    }
}

AttachmentEvaluationView::AttachmentEvaluationView()
    : m_NpadId(0)
    , m_MenuSelection(0)
    , m_RecordMenuIndex(RecordMenu_Camera)
    , m_pGraphicsSystem(nullptr)
    , m_pImageBuffer(nullptr)
    , m_PreviousPadState()
    , m_CurrentProcessor(IrSensorMode::IrSensorMode_Clustering)
    , m_NextProcessor(IrSensorMode::IrSensorMode_Clustering)
    , m_IrCameraHandle()
    , m_pAttachmentInput(nullptr)
    , m_AttachmentOutput()
    , m_AttachmentLogger()
    , m_ViewMode(ViewMode_Record)
    , m_RecordInfo()
    , m_RecordFinishedCounter(0)
    , m_IsCaptureCalled(false)
    , m_IsCaptured(false)
{
    for (auto i = 0; i < IrSensorMode_Count; i++)
    {
        m_ModeStates[i] = nullptr;
    }

    // 時刻ライブラリ初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());
}

AttachmentEvaluationView::~AttachmentEvaluationView()
{
    // 時刻ライブラリの終了
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Finalize());
}

void AttachmentEvaluationView::Initialize(nn::hid::NpadIdType id, GraphicsSystem* pGraphicsSystem, void* pImageTransferWorkMemory, AttachmentInput* pInput)
{
    m_NpadId = id;
    m_pGraphicsSystem = pGraphicsSystem;
    m_pImageBuffer = pImageTransferWorkMemory;
    m_pAttachmentInput = pInput;

    // 6軸のハンドル取得
    int handleCount = nn::hid::GetSixAxisSensorHandles(&m_SixAxisHandles[0],
        nn::hid::NpadSixAxisSensorHandleCountMax,
        m_NpadId,
        nn::hid::NpadStyleJoyDual::Mask);
    NN_LOG("HandleCount(JoyDual)=%d\n", handleCount);
    nn::hid::StartSixAxisSensor(m_SixAxisHandles[1]);

    //IR カメラのハンドルの取得
    m_IrCameraHandle = nn::irsensor::GetIrCameraHandle(m_NpadId);

    //IR カメラの初期化
    nn::irsensor::Initialize(m_IrCameraHandle);
    NN_LOG("NpadPlayerNumber(%d)\n", m_NpadId);

    // IrSensor を Clustering として初期化
    m_ModeStates[IrSensorMode_Clustering] = new ClusteringModeState(&m_NextProcessor, &m_MenuSelection, m_IrCameraHandle, m_pAttachmentInput, &m_AttachmentOutput, &m_AttachmentLogger);
    m_ModeStates[IrSensorMode_ImageTransfer] = new ImageTransferModeState(&m_NextProcessor, &m_MenuSelection, m_IrCameraHandle, pImageTransferWorkMemory, pGraphicsSystem);

    m_CurrentProcessor = m_NextProcessor = IrSensorMode_Clustering;

    m_ModeStates[m_CurrentProcessor]->Start();
}

void AttachmentEvaluationView::Finalize()
{
    // View の終了処理
    m_ModeStates[m_CurrentProcessor]->Stop();
    delete m_ModeStates[IrSensorMode_Clustering];
    delete m_ModeStates[IrSensorMode_ImageTransfer];
    nn::irsensor::Finalize(m_IrCameraHandle);
}


void AttachmentEvaluationView::Update() NN_NOEXCEPT
{
    if (!ProcessHidInput())
    {
        // return false;
        return;
    }
    m_ModeStates[m_CurrentProcessor]->Update();
}

void AttachmentEvaluationView::Draw(GraphicsSystem* pGraphicsSystem, nn::gfx::util::DebugFontTextWriter* pTextWriter) NN_NOEXCEPT
{
    // クラスタ情報の表示
    m_ModeStates[m_CurrentProcessor]->Render(&pGraphicsSystem->GetPrimitiveRenderer(), &pGraphicsSystem->GetCommandBuffer(), 0);

    // 安全フレームの表示
    DrawSafetyFrame(m_pAttachmentInput, pGraphicsSystem);

    // 期待値クラスタの表示
    DrawExpectedClusters(m_pAttachmentInput, &m_AttachmentOutput, pGraphicsSystem);

    // ワーストデータの表示
    DrawMeasuredClusters(m_pAttachmentInput, &m_AttachmentOutput, pGraphicsSystem);

    // データ類のログ表示
    DrawOutputData(40.f, 520.f, &m_AttachmentOutput, pTextWriter);

    switch (m_ViewMode)
    {
    case ViewMode_Edit:
        {
            auto rwMenu = m_ModeStates[m_NextProcessor]->GetReadWriteMenu();
            auto rMenu = m_ModeStates[m_NextProcessor]->GetReadOnlyMenu();

            int selectIndex = GetMenuIndex();
            DrawCameraParameters(rwMenu, 700.0f, 50.f + 360.0f,
                selectIndex, pTextWriter);
            DrawCameraParameters(rMenu, 970.f, 50.f + 360.f,
                -1, pTextWriter);
        }
        break;
    case ViewMode_Record:
        {
            DrawRecordParameters(700.0f, 80.f, m_RecordMenuIndex, pTextWriter);

            if (m_RecordFinishedCounter > 0)
            {
                DrawRecordFinishedNotification(1100.f, 12.f, pTextWriter);
                m_RecordFinishedCounter--;
            }
        }
        break;
    default:
        break;
    }
}

bool AttachmentEvaluationView::IsFrameCaptured() NN_NOEXCEPT
{
    return m_IsCaptured;
}

void AttachmentEvaluationView::CaptureFrame() NN_NOEXCEPT
{
    if (m_IsCaptureCalled)
    {
        m_IsCaptureCalled = false;
        m_pGraphicsSystem->CaptureFrame();
        m_IsCaptured = true;
    }
}

bool AttachmentEvaluationView::ProcessHidInput() NN_NOEXCEPT
{
    nn::hid::NpadJoyDualState currentState;
    nn::hid::NpadButtonSet buttonDownSet;

    // 押されたボタンを検出
    nn::hid::GetNpadState(&currentState, m_NpadId);
    buttonDownSet = currentState.buttons & ~m_PreviousPadState.buttons;
    m_PreviousPadState = currentState;

    // 右コンの6軸情報を更新
    nn::hid::SixAxisSensorState sixAxisState;
    nn::hid::GetSixAxisSensorState(&sixAxisState, m_SixAxisHandles[1]);
    m_AttachmentOutput.sixAxisData = sixAxisState;

    // 加速度情報からジョイコンの角度を計算
    nn::util::Vector3f accel(sixAxisState.acceleration);
    accel.Normalize();
    float degX = nn::util::RadianToDegree(nn::util::AcosEst(nn::util::Vector3f::Dot(accel, nn::util::Vector3f::UnitX())));
    float degY = nn::util::RadianToDegree(nn::util::AcosEst(nn::util::Vector3f::Dot(accel, nn::util::Vector3f::UnitY())));
    float degZ = nn::util::RadianToDegree(nn::util::AcosEst(nn::util::Vector3f::Dot(accel, -nn::util::Vector3f::UnitZ())));
    m_AttachmentOutput.sixAxisDegreeData.Set(degX, degY, degZ);

    // Toggle ViewMode
    if ((buttonDownSet & nn::hid::NpadButton::StickR::Mask).IsAnyOn())
    {
        m_ViewMode = static_cast<ViewMode>((m_ViewMode + 1) % ViewMode_CountMax);
    }

    switch (m_ViewMode)
    {
    case ViewMode_Edit:
        {
            // Menu Item Selection
            if ((buttonDownSet & nn::hid::NpadButton::Down::Mask).IsAnyOn() || (buttonDownSet & nn::hid::NpadButton::Up::Mask).IsAnyOn())
            {
                int8_t delta = ((buttonDownSet & nn::hid::NpadButton::Down::Mask).IsAnyOn()) ? 1 : -1;
                m_MenuSelection = static_cast<int>(m_MenuSelection + delta);
            }

            // Menu Action
            if ((buttonDownSet & nn::hid::NpadButton::Left::Mask).IsAnyOn() || (buttonDownSet & nn::hid::NpadButton::Right::Mask).IsAnyOn())
            {
                int8_t delta = ((buttonDownSet & nn::hid::NpadButton::Right::Mask).IsAnyOn()) ? 1 : -1;

                const std::vector<MenuItem> menu = this->m_ModeStates[m_NextProcessor]->GetReadWriteMenu();
                const int menuIndex = GetMenuIndex();
                if (menuIndex >= 0)
                {
                    menu[menuIndex].Increment(delta);
                }
            }

            if ((buttonDownSet & nn::hid::NpadButton::L::Mask).IsAnyOn())
            {
                m_ModeStates[m_CurrentProcessor]->Stop();
                m_ModeStates[m_NextProcessor]->Start();
                m_CurrentProcessor = m_NextProcessor;
            }
        }
        break;
    case ViewMode_Record:
        {
            // Menu Item Selection
            if ((buttonDownSet & nn::hid::NpadButton::Down::Mask).IsAnyOn() || (buttonDownSet & nn::hid::NpadButton::Up::Mask).IsAnyOn())
            {
                int8_t delta = ((buttonDownSet & nn::hid::NpadButton::Down::Mask).IsAnyOn()) ? 1 : -1;
                if (std::strcmp(m_pAttachmentInput->attachmentName, "Cal") == 0)
                {
                    m_RecordMenuIndex = static_cast<RecordMenu>((m_RecordMenuIndex + delta + RecordMenu_AttachmentVersion) % RecordMenu_AttachmentVersion);
                }
                else
                {
                    m_RecordMenuIndex = static_cast<RecordMenu>((m_RecordMenuIndex + delta + RecordMenu_CountMax) % RecordMenu_CountMax);
                }
            }

            // Menu Action
            if ((buttonDownSet & nn::hid::NpadButton::Left::Mask).IsAnyOn() || (buttonDownSet & nn::hid::NpadButton::Right::Mask).IsAnyOn())
            {
                int8_t delta = ((buttonDownSet & nn::hid::NpadButton::Right::Mask).IsAnyOn()) ? 1 : -1;

                switch(m_RecordMenuIndex)
                {
                case RecordMenu_Camera:
                    {
                        m_RecordInfo.measuredCameraId = (m_RecordInfo.measuredCameraId + delta + CameraVersionCountMax) % CameraVersionCountMax;
                    }
                    break;
                case RecordMenu_AttachmentVersion:
                    {
                        m_RecordInfo.attachmentVersionId = (m_RecordInfo.attachmentVersionId + delta + AttachmentVersionCountMax) % AttachmentVersionCountMax;
                    }
                    break;
                case RecordMenu_AttachmentId:
                    {
                        m_RecordInfo.attachmentId = (m_RecordInfo.attachmentId + delta + AttachmentIdMax) % AttachmentIdMax;
                    }
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }
            }

            // Start recording
            if ((buttonDownSet & nn::hid::NpadButton::L::Mask).IsAnyOn())
            {
                m_IsCaptureCalled = true;
            }

            if (m_IsCaptured)
            {
                m_IsCaptured = false;
                // ファイル名の生成
                nn::time::PosixTime posixTime;
                nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime);
                nn::time::CalendarTime calendarTime;
                nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime));

                const int FileNameCountMax = 128;

                PathInfo info;
                if (std::strcmp(m_pAttachmentInput->attachmentName, "Cal") == 0)
                {
                    nn::util::SNPrintf(info.directoryName, FileNameCountMax,
                        "%s_Typ%d_%d%02d%02d_%02d%02d%02d",
                        m_pAttachmentInput->attachmentName,
                        m_RecordInfo.measuredCameraId,
                        calendarTime.year,
                        calendarTime.month,
                        calendarTime.day,
                        calendarTime.hour,
                        calendarTime.minute,
                        calendarTime.second);

                    nn::util::SNPrintf(info.logFileName, FileNameCountMax,
                        "%s_Typ%d_%d%02d%02d_%02d%02d%02d.%s",
                        m_pAttachmentInput->attachmentName,
                        m_RecordInfo.measuredCameraId,
                        calendarTime.year,
                        calendarTime.month,
                        calendarTime.day,
                        calendarTime.hour,
                        calendarTime.minute,
                        calendarTime.second,
                        "csv");

                    nn::util::SNPrintf(info.pngFileName, FileNameCountMax,
                        "%s_Typ%d_%d%02d%02d_%02d%02d%02d.%s",
                        m_pAttachmentInput->attachmentName,
                        m_RecordInfo.measuredCameraId,
                        calendarTime.year,
                        calendarTime.month,
                        calendarTime.day,
                        calendarTime.hour,
                        calendarTime.minute,
                        calendarTime.second,
                        "png");
                }
                else
                {
                    nn::util::SNPrintf(info.directoryName, FileNameCountMax,
                        "%s_%s_Id%04d_Typ%d_%d%02d%02d_%02d%02d%02d",
                        m_pAttachmentInput->attachmentName,
                        AttachmentVersionName[m_RecordInfo.attachmentVersionId],
                        m_RecordInfo.attachmentId,
                        m_RecordInfo.measuredCameraId,
                        calendarTime.year,
                        calendarTime.month,
                        calendarTime.day,
                        calendarTime.hour,
                        calendarTime.minute,
                        calendarTime.second);

                    nn::util::SNPrintf(info.logFileName, FileNameCountMax,
                        "%s_%s_Id%04d_Typ%d_%d%02d%02d_%02d%02d%02d.%s",
                        m_pAttachmentInput->attachmentName,
                        AttachmentVersionName[m_RecordInfo.attachmentVersionId],
                        m_RecordInfo.attachmentId,
                        m_RecordInfo.measuredCameraId,
                        calendarTime.year,
                        calendarTime.month,
                        calendarTime.day,
                        calendarTime.hour,
                        calendarTime.minute,
                        calendarTime.second,
                        "csv");

                    nn::util::SNPrintf(info.pngFileName, FileNameCountMax,
                        "%s_%s_Id%04d_Typ%d_%d%02d%02d_%02d%02d%02d.%s",
                        m_pAttachmentInput->attachmentName,
                        AttachmentVersionName[m_RecordInfo.attachmentVersionId],
                        m_RecordInfo.attachmentId,
                        m_RecordInfo.measuredCameraId,
                        calendarTime.year,
                        calendarTime.month,
                        calendarTime.day,
                        calendarTime.hour,
                        calendarTime.minute,
                        calendarTime.second,
                        "png");
                }
                m_ModeStates[m_CurrentProcessor]->Record(m_RecordInfo, info, m_pAttachmentInput->objectCount, m_pGraphicsSystem);
                m_RecordFinishedCounter = 100;
            }

            // Reset internal state
            if ((buttonDownSet & nn::hid::NpadButton::ZL::Mask).IsAnyOn())
            {
                m_ModeStates[m_CurrentProcessor]->Reset();
            }
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // アプリ終了
    if ((buttonDownSet & nn::hid::NpadButton::Minus::Mask).IsAnyOn())
    {
        return false;
    }

    return true;
}//NOLINT(readability/fn_size)

int AttachmentEvaluationView::GetMenuIndex() const
{
    const std::vector<MenuItem> menu = this->m_ModeStates[m_NextProcessor]->GetReadWriteMenu();
    int size = static_cast<int>(menu.size());
    int menuIndex = ((m_MenuSelection % size) + size) % size;

    NN_ASSERT(menuIndex >= 0);
    NN_ASSERT(menuIndex < size);

    return menuIndex;
}
