﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Macro.h>
#include <nn/os.h>
#include <nn/hid.h>
#include <nn/hid/hid_ControllerSupport.h>

#include "SceneBase.h"
#include "AttachmentEvaluationScene.h"

#include <sstream>
namespace
{

void DrawMenuAttachment(const float offsetX, const float offsetY,
    int attachmentCount,
    int attachmentIndex,
    GraphicsSystem* pGraphicsSystem)
{
    // Draw Menu Bar
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();
    nn::gfx::CommandBuffer* pCommandBuffer = &pGraphicsSystem->GetCommandBuffer();

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // 背景を描画
    const float xStart = -1.0f;
    const float xEnd = 1.0f;
    const float yStart = 0.9f;
    const float yEnd = 0.8f;

    // 分割する
    float xWidth = (xEnd - xStart) / attachmentCount;
    for (auto i = 0; i < attachmentCount; i++)
    {
        nn::util::Vector3fType corner;
        nn::util::VectorSet(&corner, xStart + i * xWidth, yStart, 0.0f);

        nn::util::Vector3fType size;
        nn::util::VectorSet(&size, xWidth, yEnd - yStart, 0.0f);

        const nn::util::Unorm8x4 Color[] = {
            {{ 243, 152, 0, 255 }},  //Orange
            {{ 100, 69, 40, 255 }},   //Brown
            {{ 128, 128, 128, 255 }},   //Gray
        };

        pPrimitiveRenderer->SetColor(Color[i]);
        pPrimitiveRenderer->DrawQuad(
            pCommandBuffer,
            corner,
            size);
    }

    // Write text
    nn::gfx::util::DebugFontTextWriter* pTextWriter = &pGraphicsSystem->GetDebugFont();
    const nn::util::Unorm8x4 White ={ { 255, 255, 255, 255 } };
    const nn::util::Unorm8x4 Red ={ { 255, 0, 0, 255 } };

    pTextWriter->SetScale(1.2f, 1.2f);

    int offsetWidthX = 1280 / attachmentCount;
    int offset = 30;
    for (auto i = 0; i < attachmentCount; i++)
    {
        if (attachmentIndex == i)
        {
            pTextWriter->SetTextColor(Red);
        }
        else
        {
            pTextWriter->SetTextColor(White);
        }
        pTextWriter->SetCursor(offsetX + i * offsetWidthX + offset, offsetY);
        if (i == attachmentCount - 1)
        {
            pTextWriter->Print("Calibration Tool");
        }
        else
        {
            pTextWriter->Print("Attachment : %d", i);
        }
    }
}

void DrawMenuToolState(const float offsetX, const float offsetY,
    GraphicsSystem* pGraphicsSystem)
{
    // Draw Menu Bar
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();
    nn::gfx::CommandBuffer* pCommandBuffer = &pGraphicsSystem->GetCommandBuffer();

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // 背景を描画
    const float xStart = -1.0f;
    const float xEnd = 1.0f;
    const float yStart = 1.0f;
    const float yEnd = 0.9f;
    nn::util::Vector3fType corner;
    nn::util::VectorSet(&corner, xStart, yStart, 0.0f);

    nn::util::Vector3fType size;
    nn::util::VectorSet(&size, xEnd - xStart, yEnd - yStart, 0.0f);

    const nn::util::Unorm8x4 Green = { { 0, 32, 0, 255 } };
    pPrimitiveRenderer->SetColor(Green);
    pPrimitiveRenderer->DrawQuad(
        pCommandBuffer,
        corner,
        size);

    // Write text
    nn::gfx::util::DebugFontTextWriter* pTextWriter = &pGraphicsSystem->GetDebugFont();
    const nn::util::Unorm8x4 White ={ { 255, 255, 255, 255 } };

    pTextWriter->SetScale(1.2f, 1.2f);

    pTextWriter->SetTextColor(White);
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print("Attachment Evaluation Tool: v1.3");
}

void DrawMenu(const float offsetX, const float offsetY,
    int attachmentCount,
    int attachmentIndex,
    GraphicsSystem* pGraphicsSystem)
{
    NN_ASSERT_NOT_NULL(pGraphicsSystem);

    // 一段目メニュー
    DrawMenuToolState(offsetX, offsetY, pGraphicsSystem);
    // 二段目メニュー
    DrawMenuAttachment(offsetX, offsetY + 40, attachmentCount, attachmentIndex, pGraphicsSystem);
}

} // namespace


AttachmentEvaluationScene::AttachmentEvaluationScene()
    : m_Cursor(0)
    , m_ViewIndex(0)
    , m_View()
{
}

AttachmentEvaluationScene::~AttachmentEvaluationScene()
{
}

void AttachmentEvaluationScene::DoInitialize()
{
    // View の初期化
    m_View[m_ViewIndex].Initialize(m_NpadId, m_pGraphicsSystem, m_pImageBuffer, m_pAttachmentInput[m_ViewIndex]);
    // キャプチャの開始
    m_pGraphicsSystem->InitializeFrameCapture();
}

void AttachmentEvaluationScene::DoFinalize()
{
    // キャプチャの終了
    m_pGraphicsSystem->FinalizeFrameCapture();
    // 各 View の終了処理
    m_View[m_ViewIndex].Finalize();
}


void AttachmentEvaluationScene::DoUpdate() NN_NOEXCEPT
{
    if (!ProcessHidInput())
    {
        return;
    }

    m_View[m_ViewIndex].Update();
}

void AttachmentEvaluationScene::DoDraw() NN_NOEXCEPT
{
    DrawMenu(0.0f, 0.0f, m_AttachmentInputSize, m_ViewIndex, m_pGraphicsSystem);
    m_View[m_ViewIndex].Draw(m_pGraphicsSystem, &m_pGraphicsSystem->GetDebugFont());
}

bool AttachmentEvaluationScene::ProcessHidInput() NN_NOEXCEPT
{
    nn::hid::NpadJoyDualState currentState;
    nn::hid::NpadButtonSet buttonDownSet;

    // 押されたボタンを検出
    nn::hid::GetNpadState(&currentState, m_NpadId);
    buttonDownSet = currentState.buttons & ~m_PreviousPadState.buttons;
    m_PreviousPadState = currentState;

    // Menu Item Selection
    if ((buttonDownSet & nn::hid::NpadButton::Down::Mask).IsAnyOn())
    {
        m_MenuSelection += 1;
    };

    if ((buttonDownSet & nn::hid::NpadButton::Up::Mask).IsAnyOn())
    {
        m_MenuSelection -= 1;
    };

    // Menu Action
    if ((buttonDownSet & nn::hid::NpadButton::StickL::Mask).IsAnyOn())
    {
        int8_t delta = 1;

        m_View[m_ViewIndex].Finalize();
        m_ViewIndex = (m_ViewIndex + delta + m_AttachmentInputSize) % m_AttachmentInputSize;
        m_View[m_ViewIndex].Initialize(m_NpadId, m_pGraphicsSystem, m_pImageBuffer, m_pAttachmentInput[m_ViewIndex]);
    }

    // イベントトリガー
    if ((buttonDownSet & nn::hid::NpadButton::ZL::Mask).IsAnyOn())
    {
    }

    if ((buttonDownSet & nn::hid::NpadButton::Plus::Mask).IsAnyOn() || (buttonDownSet & nn::hid::NpadButton::Minus::Mask).IsAnyOn())
    {
        return false;
    }

    return true;

}

bool AttachmentEvaluationScene::IsFrameCaptured()
{
    return m_View[m_ViewIndex].IsFrameCaptured();
}

void AttachmentEvaluationScene::CaptureFrame()
{
    m_View[m_ViewIndex].CaptureFrame();
}
