﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <vector>

// NintendoSDK のヘッダファイルをインクルードする前に、NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET マクロを
// 定義することで、DebugFontWriter::Print() の入力文字コードを Windows のロケールのデフォルト
// (日本語の場合、CP932)に変更できます。
//#define NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nn/os.h>
#include <nn/vi.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_ControllerSupport.h>
#include <nn/err/err_ShowErrorApi.h>
#include <nn/err/err_LibraryApi.h>

#include <nn/irsensor.h>
#include <nn/irsensor/irsensor_PointingProcessorApiPrivate.h>
#include <nn/irsensor/irsensor_PointingProcessorTypesPrivate.h>
#include <nn/irsensor/irsensor_IrLedProcessorApi.h>
#include <nn/irsensor/irsensor_IrLedProcessorTypes.h>
#include <nn/irsensor/irsensor_AdaptiveClusteringProcessor.h>
#include "GraphicsSystem.h"
#include "IrsensorApiTestTool.h"
#include "HandAnalysisUtil.h"

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
#include <nn/nn_Windows.h>
#endif

namespace {

Npad::Id g_NpadIds[] =
{
    nn::hid::NpadId::No1,
    nn::hid::NpadId::No2,
    nn::hid::NpadId::No3,
    nn::hid::NpadId::No4,
    nn::hid::NpadId::No5,
    nn::hid::NpadId::No6,
    nn::hid::NpadId::No7,
    nn::hid::NpadId::No8,
    nn::hid::NpadId::Handheld,
};

const size_t NpadIdCountMax = sizeof(g_NpadIds) / sizeof(g_NpadIds[0]);

int g_Page = 0;
const int PageCountMax = (NpadIdCountMax + 1) / 2;

bool g_SamplingDropArray[NpadIdCountMax][MaxPerformanceArraySize];
int64_t g_StartSamplingNumberArray[NpadIdCountMax][MaxPerformanceArraySize];
bool g_DataDropArray[NpadIdCountMax][MaxPerformanceArraySize];
int g_SamplingDropArrayIndex[NpadIdCountMax] = {0};
int g_StartSamplingNumberArrayIndex[NpadIdCountMax] = {0};
int g_DataDropArrayIndex[NpadIdCountMax] = {0};

int g_FrameRateIndex = 0; // 初期値は60fps
int g_FrameRate = FpsTable[g_FrameRateIndex];

// 仮アプレット起動用スレッド設定
const size_t AppletThreadStackSize = 32 * 1024;
nn::os::ThreadType  g_AppletThread;
NN_OS_ALIGNAS_THREAD_STACK char g_AppletThreadStack[AppletThreadStackSize];
bool g_IsAppletThreadRunning = false;

// IR層の PLR 測定用のスレッド設定
const size_t MeasureThreadStackSize = 32 * 1024;
nn::os::ThreadType  g_MeasureThread;
NN_OS_ALIGNAS_THREAD_STACK char g_MeasureThreadStack[MeasureThreadStackSize];
bool g_IsMeasureThreadRunning = false;
nn::os::TimerEventType g_TimerEvent;

// IRセンサーのUpdate用のスレッド設定
const size_t UpdateThreadStackSize = 32 * 1024;
nn::os::ThreadType  g_UpdateThread;
NN_OS_ALIGNAS_THREAD_STACK char g_UpdateThreadStack[UpdateThreadStackSize];
bool g_IsUpdateThreadRunning = false;
nn::os::TimerEventType g_UpdateTimerEvent;
nn::os::EventType g_UpdateEvent;

#if defined(NN_BUILD_CONFIG_OS_WIN)

WNDPROC g_DefaultWndProc = 0;

LRESULT CALLBACK WndProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
    switch (uMsg)
    {
    case WM_ACTIVATEAPP:
        break;

    default:
        break;
    }

    return ::CallWindowProc(g_DefaultWndProc, hwnd, uMsg, wParam, lParam);
}
#endif

float ConvertFromPixelToTextureFactorX(float pixel) NN_NOEXCEPT
{
    NN_ASSERT_MINMAX(pixel, 0, DisplayWidth);
    return pixel / DisplayWidth * 2.0f - 1.0f;
}

float ConvertFromPixelToTextureFactorY(float pixel) NN_NOEXCEPT
{
    NN_ASSERT_MINMAX(pixel, 0, DisplayHeight);
    return (DisplayHeight - pixel) / DisplayHeight * 2.0f - 1.0f;
}

void PrintIrsensorError(nn::gfx::util::DebugFontTextWriter* pTextWriter, nn::Result result) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    if (nn::irsensor::ResultIrsensorNotReady::Includes(result))
    {
        pTextWriter->Print("Result : ResultIrsensorNotReady");
    }
    else if (nn::irsensor::ResultIrsensorUnconnected::Includes(result))
    {
        pTextWriter->Print("Result : ResultIrsensorUnconnected");
    }
    else if (nn::irsensor::ResultIrsensorUnsupported::Includes(result))
    {
        pTextWriter->Print("Result : ResultIrsensorUnsupported");
    }
    else if (nn::irsensor::ResultIrsensorDeviceError::Includes(result))
    {
        pTextWriter->Print("Result : ResultIrsensorDeviceError");
    }
    else if (nn::irsensor::ResultIrsensorFirmwareCheckIncompleted::Includes(result))
    {
        pTextWriter->Print("Result : ResultIrsensor\n       FirmwareCheckIncompleted");
    }
    else if (nn::irsensor::ResultHandAnalysisModeIncorrect::Includes(result))
    {
        pTextWriter->Print("Result : ResultHandAnalysisModeIncorrect");
    }
    else if (nn::irsensor::ResultHandAnalysisError::Includes(result))
    {
        pTextWriter->Print("Result : ResultHandAnalysisError");
    }
    else if (result.IsSuccess())
    {
        pTextWriter->Print("Result : ResultSuccess");
    }
    else
    {
        pTextWriter->Print("Result : Unknown Error");
    }
}

void InitializeCameraViewer(GraphicsSystem* pGraphicsSystem, CameraViewer* pCameraViewer) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pGraphicsSystem);
    NN_ASSERT_NOT_NULL(pCameraViewer);

    // バッファを初期化
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_Read);
    info.SetSize(nn::irsensor::ImageTransferProcessorImageSize320x240);
    pGraphicsSystem->AllocateBuffer(&pCameraViewer->copiedImageBuffer, &info);

    int width = nn::irsensor::IrCameraImageWidth;
    int height = nn::irsensor::IrCameraImageHeight;

    // それぞれのサイズの画像に対して、カラーバッファのテクスチャビューを初期化
    for (size_t i = 0; i < 5; i++)
    {
        nn::gfx::Texture::InfoType textureInfo;
        textureInfo.SetDefault();
        textureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
        textureInfo.SetWidth(width);
        textureInfo.SetHeight(height);
        textureInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        textureInfo.SetImageFormat(nn::gfx::ImageFormat_R8_Unorm);
        textureInfo.SetMipCount(1);
        textureInfo.SetTileMode(nn::gfx::TileMode_Linear);

        pGraphicsSystem->AllocateTexture(&pCameraViewer->texture[i], &textureInfo);

        // カラーバッファのテクスチャビューを初期化
        nn::gfx::TextureView::InfoType viewInfo;
        viewInfo.SetDefault();
        viewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
        viewInfo.SetImageFormat(nn::gfx::ImageFormat_R8_Unorm);
        viewInfo.SetTexturePtr(&pCameraViewer->texture[i]);
        viewInfo.SetChannelMapping(nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red,
            nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red);
        pCameraViewer->textureView[i].Initialize(&pGraphicsSystem->GetDevice(), viewInfo);

        pGraphicsSystem->RegisterTextureViewSlot(&pCameraViewer->textureDescriptor[i], pCameraViewer->textureView[i]);

        width /= 2;
        height /= 2;
    }

    // HandAnalysis の Imageモードに対して、カラーバッファのテクスチャビューを初期化
    width = nn::irsensor::IrHandAnalysisImageWidth;
    height = nn::irsensor::IrHandAnalysisImageHeight;
    nn::gfx::Texture::InfoType textureInfo;
    textureInfo.SetDefault();
    textureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
    textureInfo.SetWidth(width);
    textureInfo.SetHeight(height);
    textureInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
    textureInfo.SetImageFormat(nn::gfx::ImageFormat_R16_Unorm);
    textureInfo.SetMipCount(1);
    textureInfo.SetTileMode(nn::gfx::TileMode_Linear);

    pGraphicsSystem->AllocateTexture(&pCameraViewer->textureHand, &textureInfo);

    // カラーバッファのテクスチャビューを初期化
    nn::gfx::TextureView::InfoType viewInfo;
    viewInfo.SetDefault();
    viewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
    viewInfo.SetImageFormat(nn::gfx::ImageFormat_R16_Unorm);
    viewInfo.SetTexturePtr(&pCameraViewer->textureHand);
    viewInfo.SetChannelMapping(nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red,
        nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red);
    pCameraViewer->textureViewHand.Initialize(&pGraphicsSystem->GetDevice(), viewInfo);

    pGraphicsSystem->RegisterTextureViewSlot(&pCameraViewer->textureDescriptorHand, pCameraViewer->textureViewHand);


    nn::gfx::Sampler::InfoType samplerInfo;
    samplerInfo.SetDefault();
    samplerInfo.SetFilterMode(nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint);
    samplerInfo.SetAddressU(nn::gfx::TextureAddressMode_Mirror);
    samplerInfo.SetAddressV(nn::gfx::TextureAddressMode_Mirror);
    samplerInfo.SetAddressW(nn::gfx::TextureAddressMode_Mirror);
    pCameraViewer->sampler.Initialize(&pGraphicsSystem->GetDevice(), samplerInfo);

    pGraphicsSystem->RegisterSamplerSlot(&pCameraViewer->samplerDescriptor, pCameraViewer->sampler);
}

void FinalizeCameraViewer(GraphicsSystem* pGraphicsSystem, CameraViewer* pCameraViewer) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pGraphicsSystem);
    NN_ASSERT_NOT_NULL(pCameraViewer);

    pGraphicsSystem->FreeBuffer(&pCameraViewer->copiedImageBuffer);

    for (size_t i = 0; i < 5; i++)
    {
        pGraphicsSystem->FreeTexture(&pCameraViewer->texture[i]);
        pCameraViewer->textureView[i].Finalize(&pGraphicsSystem->GetDevice());
    }
    pGraphicsSystem->FreeTexture(&pCameraViewer->textureHand);
    pCameraViewer->textureViewHand.Finalize(&pGraphicsSystem->GetDevice());

    pCameraViewer->sampler.Finalize(&pGraphicsSystem->GetDevice());
}

void ClearCommunicationPerformanceMonitor(nn::irsensor::IrCameraHandle handle) NN_NOEXCEPT
{
    int index = handle._handle & 0x000F;

    for (auto i = 0; i < MaxPerformanceArraySize; i++)
    {
        g_SamplingDropArray[index][i] = false;
        g_StartSamplingNumberArray[index][i] = 0;
        g_DataDropArray[index][i] = false;
    }
    g_SamplingDropArrayIndex[index] = 0;
    g_StartSamplingNumberArrayIndex[index] = 0;
    g_DataDropArrayIndex[index] = 0;
}

void ClearIrStateSettings(IrState* pIrState) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pIrState);

    pIrState->nextProcessor = IrSensorMode_Moment;

    nn::irsensor::GetMomentProcessorDefaultConfig(&pIrState->momentProcessorConfig);
    nn::irsensor::Rect defaultRect = { 0, 0, ::nn::irsensor::IrCameraImageWidth, ::nn::irsensor::IrCameraImageHeight };
    pIrState->momentProcessorWoi = defaultRect;

    nn::irsensor::GetClusteringProcessorDefaultConfig(&pIrState->clusteringProcessorConfig);
    pIrState->clusteringProcessorWoi = defaultRect;

    pIrState->format = nn::irsensor::ImageTransferProcessorFormat_320x240;
    memset(pIrState->imageTransferWorkBuffer, 0, nn::irsensor::ImageTransferProcessorWorkBufferSize320x240);
    nn::irsensor::GetImageTransferProcessorDefaultConfig(&pIrState->imageTransferProcessorConfigNext);
    nn::irsensor::GetImageTransferProcessorDefaultConfig(&pIrState->imageTransferProcessorConfigCurrent);

    pIrState->handAnalysisConfigCurrent.mode = nn::irsensor::HandAnalysisMode_SilhouetteAndImage;
    pIrState->handAnalysisConfigNext.mode = nn::irsensor::HandAnalysisMode_SilhouetteAndImage;
    pIrState->adaptiveClusteringProcessorConfig.mode = nn::irsensor::AdaptiveClusteringMode_StaticFov;

    nn::irsensor::GetIrLedProcessorDefaultConfig(&pIrState->irLedProcessorConfig);
}

void ClearIrStateData(IrState* pIrState) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pIrState);

    for (auto i = 0; i < nn::irsensor::MomentProcessorStateCountMax; i++)
    {
        memset(&pIrState->momentProcessorState[i], 0, sizeof(nn::irsensor::MomentProcessorState));
    }
    pIrState->momentApiResult = nn::ResultSuccess();
    pIrState->momentApiOutCount = 0;

    for (auto i = 0; i < nn::irsensor::ClusteringProcessorStateCountMax; i++)
    {
        memset(&pIrState->clusteringProcessorState[i], 0, sizeof(nn::irsensor::ClusteringProcessorState));
    }
    pIrState->clusteringApiResult = nn::ResultSuccess();
    pIrState->clusteringApiOutCount = 0;

    pIrState->imageTransferProcessorState.samplingNumber = 0;
    pIrState->imageTransferApiResult = nn::ResultSuccess();
    pIrState->imageTransferPrevSamplingNumber = 0;
    pIrState->imageTransferFrameInterval = 0;
    pIrState->imageTransferGetDataTime = 0;

    for (auto i = 0; i < nn::irsensor::PointingProcessorStateCountMax; i++)
    {
        memset(&pIrState->pointingProcessorState[i], 0, sizeof(nn::irsensor::PointingProcessorState));
        memset(&pIrState->pointingProcessorMarkerState[i], 0, sizeof(nn::irsensor::PointingProcessorMarkerState));
    }
    pIrState->pointingApiResult = nn::ResultSuccess();
    pIrState->pointingApiOutCount = 0;

    memset(&pIrState->handAnalysisImageState, 0, sizeof(nn::irsensor::HandAnalysisImageState));
    memset(&pIrState->handAnalysisSilhouetteState, 0, sizeof(nn::irsensor::HandAnalysisSilhouetteState));
    pIrState->handAnalysisApiResult = nn::ResultSuccess();
    pIrState->handAnalysisPolygon = { 0 };

    for (auto i = 0; i < nn::irsensor::AdaptiveClusteringProcessorStateCountMax; i++)
    {
        memset(&pIrState->adaptiveClusteringProcessorState[i], 0, sizeof(nn::irsensor::AdaptiveClusteringProcessorState));
    }
    pIrState->adaptiveClusteringApiResult = nn::ResultSuccess();
    pIrState->adaptiveClusteringApiOutCount = 0;
}

void ClearDeviceState(DeviceState* pDeviceState)
{
    NN_ASSERT_NOT_NULL(pDeviceState);

    pDeviceState->padMenuSelect = 0;
    pDeviceState->padOutputMenuSelect = 0;
    pDeviceState->padInitializedMenuSelect = 0;
    pDeviceState->firmwareUpdateStatusResult = nn::ResultSuccess();
    pDeviceState->firmwareUpdateTrialCount = 0;

    pDeviceState->irCameraStatus = nn::irsensor::IrCameraStatus_Unconnected;
    pDeviceState->isCameraStatusValid = false;
    pDeviceState->irProcessorStatus = nn::irsensor::ImageProcessorStatus_Stopped;
    pDeviceState->isProcessorStatusValid = false;
    pDeviceState->padTypes = PadTypes_Npad;
    ClearIrStateSettings(&pDeviceState->irState);
    ClearIrStateData(&pDeviceState->irState);

    ClearCommunicationPerformanceMonitor(pDeviceState->irCameraHandle);
    pDeviceState->samplingDropRatio = 0.0f;
    pDeviceState->dataDropRatio = 0.0f;

    pDeviceState->updateCounter = 0;
    pDeviceState->latestSamplingNum = 0;
    pDeviceState->latestSampleCount = 0;
    pDeviceState->averagePlr = 0.0f;
    pDeviceState->maxPlr = 0;
    pDeviceState->burstDropCount = 0;
    pDeviceState->burstDropCountWorst = 0;
    pDeviceState->burstDropCountWorstAve = 0.0f;
    pDeviceState->burstDropCountWorstCurrent = 0;
    pDeviceState->burstDropCountWorstMax = 0;
    pDeviceState->frameIntervalMs = 0;
    pDeviceState->aveFrameIntervalMs = 0;
    pDeviceState->frameIntervalCount = 0;
    pDeviceState->prevIntervalUpdateTime = 0;

    // バッファをクリア
    uint8_t* destinationPixels = pDeviceState->cameraViewer.copiedImageBuffer.Map< uint8_t >();
    int imageSize = nn::irsensor::ImageTransferProcessorImageSize320x240;
    memset(destinationPixels, 0, imageSize);
    pDeviceState->cameraViewer.copiedImageBuffer.FlushMappedRange(0, imageSize);
    pDeviceState->cameraViewer.copiedImageBuffer.Unmap();
    memset(pDeviceState->cameraViewer.cameraImageBuffer, 0, imageSize);
}

void WriteToolVersion(
    nn::gfx::util::DebugFontTextWriter* pTextWriter,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    const nn::util::Unorm8x4 Green = { { 0, 255, 0, 255 } };

    pTextWriter->SetScale(0.8f, 0.8f);
    pTextWriter->SetTextColor(Green);
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print(ToolVersion);
}

void WriteFps(
    nn::gfx::util::DebugFontTextWriter* pTextWriter,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    static nn::os::Tick prevTick;

    const nn::util::Unorm8x4 Green = { { 0, 255, 0, 255 } };

    pTextWriter->SetScale(0.8f, 0.8f);
    pTextWriter->SetTextColor(Green);
    pTextWriter->SetCursor(offsetX, offsetY);

    float microsec = static_cast<float>(nn::os::ConvertToTimeSpan(nn::os::GetSystemTick() - prevTick).GetMicroSeconds());
    prevTick = nn::os::GetSystemTick();
    pTextWriter->Print("App: %0.2f [fps]", 1 / microsec * 1000000);
}

void WriteHelp(GraphicsSystem* pGraphicsSystem,
    nn::gfx::util::DebugFontTextWriter* pTextWriter,
    const IrSensorApiTestToolState irToolState,
    const NpadTypes npadType,
    const float offsetX,
    const float offsetY,
    bool isUseContinuousRunCommand) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pGraphicsSystem);
    NN_ASSERT_NOT_NULL(pTextWriter);

    const nn::util::Unorm8x4 White = { { 255, 255, 255, 255 } };
    const nn::util::Unorm8x4 Green = { { 0, 32, 0, 255 } };
    const nn::util::Unorm8x4 Cyan = { { 0, 64, 64, 255 } };
    const nn::util::Unorm8x4 Blue = { { 0, 32, 128, 255 } };
    const nn::util::Unorm8x4 Purple = { { 32, 0, 128, 255 } };
    const nn::util::Unorm8x4 Mazenta = { { 64, 0, 64, 255 } };
    const nn::util::Unorm8x4 Yellow = { { 255, 255, 0, 255 } };

    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();
    nn::gfx::CommandBuffer* pCommandBuffer = &pGraphicsSystem->GetCommandBuffer();

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // 背景を描画
    float xStart = ConvertFromPixelToTextureFactorX(offsetX);
    float xEnd = xStart + 0.45f;
    float yEnd = ConvertFromPixelToTextureFactorY(offsetY);
    float yStart = yEnd - 0.05f;

    nn::util::Vector3fType corner;
    nn::util::VectorSet(&corner, xStart, yStart, 0.0f);

    nn::util::Vector3fType size;
    nn::util::VectorSet(&size, xEnd - xStart, yEnd - yStart, 0.0f);

    if (g_Page == 0)
    {
        pPrimitiveRenderer->SetColor(Green);
    }
    else if (g_Page == 1)
    {
        pPrimitiveRenderer->SetColor(Cyan);
    }
    else if (g_Page == 2)
    {
        pPrimitiveRenderer->SetColor(Blue);
    }
    else if (g_Page == 3)
    {
        pPrimitiveRenderer->SetColor(Purple);
    }
    else if (g_Page == 4)
    {
        pPrimitiveRenderer->SetColor(Mazenta);
    }

    pPrimitiveRenderer->DrawQuad(
        pCommandBuffer,
        corner,
        size);

    pTextWriter->SetScale(0.9f, 0.9f);
    pTextWriter->SetTextColor(White);
    auto offset = 20;

    auto i = 0;
    pTextWriter->SetCursor(offsetX, offsetY + offset * i++);
    pTextWriter->Print("Help window");
    pTextWriter->SetCursor(offsetX, offsetY + offset * i++);
    if (npadType == NpadTypes_JoyRight || npadType == NpadTypes_Handheld || npadType == NpadTypes_JoyDual)
    {
        pTextWriter->Print("Press A + X + R to close app");
    }
    else if (npadType == NpadTypes_JoyLeft)
    {
        pTextWriter->Print("Press ^ + < + L to close app");
    }

    switch (irToolState)
    {
    case IrSensorApiTestToolState_Initial:
        {
            if (npadType == NpadTypes_JoyRight || npadType == NpadTypes_Handheld || npadType == NpadTypes_JoyDual)
            {
                pTextWriter->SetCursor(offsetX, offsetY + offset * i++);
                pTextWriter->Print("Press X / B to change controller type");
                pTextWriter->SetCursor(offsetX, offsetY + offset * i++);
                pTextWriter->Print("Press R to initialize");
            }
            else if (npadType == NpadTypes_JoyLeft)
            {
                pTextWriter->SetCursor(offsetX, offsetY + offset * i++);
                pTextWriter->Print("Press ^ / v to change controller type");
                pTextWriter->SetCursor(offsetX, offsetY + offset * i++);
                pTextWriter->Print("Press L to initialize");
            }
        }
        break;
    case IrSensorApiTestToolState_Initialized:
        {
            if (npadType == NpadTypes_JoyRight || npadType == NpadTypes_Handheld || npadType == NpadTypes_JoyDual)
            {
                pTextWriter->SetCursor(offsetX, offsetY + offset * i++);
                pTextWriter->Print("Press R to check ir state");
            }
            else if (npadType == NpadTypes_JoyLeft)
            {
                pTextWriter->SetCursor(offsetX, offsetY + offset * i++);
                pTextWriter->Print("Press L to check ir state");
            }
        }
        break;
    case IrSensorApiTestToolState_CameraReady:
        {
            pTextWriter->SetCursor(offsetX, offsetY + offset * i++);
            pTextWriter->Print("Press X / B to change component");
            pTextWriter->SetCursor(offsetX, offsetY + offset * i++);
            pTextWriter->Print("Press Y / A to change value");
            pTextWriter->SetCursor(offsetX, offsetY + offset * i++);
            pTextWriter->Print("Press R to run with user setting");
            pTextWriter->SetCursor(offsetX, offsetY + offset * i++);
            pTextWriter->Print("Press ZR to set default value");
            pTextWriter->SetCursor(offsetX, offsetY + offset * i++);
            pTextWriter->Print("Press + to finalize");
        }
        break;
    case IrSensorApiTestToolState_Running:
        {
            pTextWriter->SetCursor(offsetX, offsetY + offset * i++);
            pTextWriter->Print("Press X / B to change component");
            pTextWriter->SetCursor(offsetX, offsetY + offset * i++);
            pTextWriter->Print("Press Y / A to change value");
            pTextWriter->SetCursor(offsetX, offsetY + offset * i++);
            pTextWriter->SetTextColor(Yellow);
            if (isUseContinuousRunCommand)
            {
                pTextWriter->Print("Press R to run with user setting");
            }
            else
            {
                pTextWriter->Print("Press R to set output type");
            }
            pTextWriter->SetTextColor(White);
            pTextWriter->SetCursor(offsetX, offsetY + offset * i++);
            pTextWriter->Print("Press ZR to StopImageProcessor");
            pTextWriter->SetCursor(offsetX, offsetY + offset * i++);
            pTextWriter->Print("Press + to finalize");
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    pTextWriter->SetCursor(offsetX, offsetY + offset * i++);
    pTextWriter->Print("Press stick to change page");
}//NOLINT(readability/fn_size)

void WriteProcessorState(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    DeviceState* pDeviceState,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    NN_ASSERT_NOT_NULL(pDeviceState);

    const nn::util::Unorm8x4 White = { { 255, 255, 255, 255 } };
    const nn::util::Unorm8x4 Red = { { 255, 0, 0, 255 } };

    auto startOffset = 30;
    auto offset = 20;

    pTextWriter->SetScale(1.00f, 1.00f);
    pTextWriter->SetTextColor(White);

    nn::irsensor::IrCameraConfig* pConfig = nullptr;
    if (pDeviceState->irState.nextProcessor == IrSensorMode_Moment)
    {
        pConfig = &pDeviceState->irState.momentProcessorConfig.irCameraConfig;
    }
    else if (pDeviceState->irState.nextProcessor == IrSensorMode_Clustering)
    {
        pConfig = &pDeviceState->irState.clusteringProcessorConfig.irCameraConfig;
    }
    else if (pDeviceState->irState.nextProcessor == IrSensorMode_ImageTransfer)
    {
        pConfig = &pDeviceState->irState.imageTransferProcessorConfigNext.irCameraConfig;
    }

    IrState* pIrState = &pDeviceState->irState;

    auto index = 0;
    if (pDeviceState->irToolState == IrSensorApiTestToolState_CameraReady)
    {
        index = pDeviceState->padMenuSelect;
    }
    else if (pDeviceState->irToolState == IrSensorApiTestToolState_Running)
    {
        index = pDeviceState->padOutputMenuSelect - MenuOutputLength;
    }

    if (pIrState->nextProcessor == IrSensorMode_Moment)
    {
        pTextWriter->SetTextColor((index == 0) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset);
        pTextWriter->Print("Current Mode : Moment");
    }
    else if (pIrState->nextProcessor == IrSensorMode_Clustering)
    {
        pTextWriter->SetTextColor((index == 0) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset);
        pTextWriter->Print("Current Mode : Clustering");
    }
    else if (pIrState->nextProcessor == IrSensorMode_ImageTransfer)
    {
        pTextWriter->SetTextColor((index == 0) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset);
        pTextWriter->Print("Current Mode : ImageTransfer");
    }
    else if (pIrState->nextProcessor == IrSensorMode_Pointing)
    {
        pTextWriter->SetTextColor((index == 0) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset);
        pTextWriter->Print("Current Mode : Pointing");
    }
    else if (pIrState->nextProcessor == IrSensorMode_HandAnalysis)
    {
        pTextWriter->SetTextColor((index == 0) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset);
        pTextWriter->Print("Current Mode : HandAnalysis");
    }
    else if (pIrState->nextProcessor == IrSensorMode_IrLed)
    {
        pTextWriter->SetTextColor((index == 0) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset);
        pTextWriter->Print("Current Mode : IrLed");
    }
    else if (pIrState->nextProcessor == IrSensorMode_AdaptiveClustering)
    {
        pTextWriter->SetTextColor((index == 0) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset);
        pTextWriter->Print("Current Mode : AdaptiveClustering");
    }

    pTextWriter->SetTextColor((index == 1) ? Red : White);
    if (pConfig != nullptr)
    {
        pTextWriter->SetScale(0.90f, 0.90f);
        auto count = 2;
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        pTextWriter->Print("ExposureTime : %llu", pConfig->exposureTime.GetMicroSeconds());
        pTextWriter->SetTextColor((index == 2) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        switch (pConfig->lightTarget)
        {
        case nn::irsensor::IrCameraLightTarget_AllObjects:
            pTextWriter->Print("lightTarget : AllObjects");
            break;
        case nn::irsensor::IrCameraLightTarget_FarObjects:
            pTextWriter->Print("lightTarget : FarObjects");
            break;
        case nn::irsensor::IrCameraLightTarget_NearObjects:
            pTextWriter->Print("lightTarget : NearObjects");
            break;
        case nn::irsensor::IrCameraLightTarget_None:
            pTextWriter->Print("lightTarget : None");
            break;
        default:
            break;
        }

        pTextWriter->SetTextColor((index == 3) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        pTextWriter->Print("DigitalGain : %d", pConfig->gain);

        pTextWriter->SetTextColor((index == 4) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        if (pConfig->isNegativeImageUsed)
        {
            pTextWriter->Print("NegativeImage : ON");
        }
        else
        {
            pTextWriter->Print("NegativeImage : OFF");
        }
    }
}//NOLINT(readability/fn_size)

void WriteMomentProcessorState(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    nn::Result momentApiResult,
    int momentApiOutCount,
    const nn::irsensor::MomentProcessorState* pMomentState,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    NN_ASSERT_NOT_NULL(pMomentState);

    const nn::util::Unorm8x4 White = { { 255, 255, 255, 255 } };

    pTextWriter->SetScale(1.0f, 1.0f);
    pTextWriter->SetTextColor(White);
    auto i = 1;
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    PrintIrsensorError(pTextWriter, momentApiResult);

    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("Api OutCount : %d", momentApiOutCount);

    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("Sampling Number : %ld", pMomentState->samplingNumber);

    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("TimeStamp : %lld[ns]", pMomentState->timeStamp.GetNanoSeconds());

    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("AmbientNoiseLevel : %s", AmbientNoiseLevelName[pMomentState->ambientNoiseLevel]);

    // 全体ブロックのデータを計算
    nn::irsensor::Rect woi = {0, 0, nn::irsensor::IrCameraImageWidth, nn::irsensor::IrCameraImageHeight};
    auto totalStats = nn::irsensor::CalculateMomentRegionStatistic(
        pMomentState, woi, 0, 0, nn::irsensor::MomentProcessorBlockRowCount, nn::irsensor::MomentProcessorBlockColumnCount);

    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("Block Total ");
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("  averageIntensity : %.3f", totalStats.averageIntensity);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("  centroid X : %.3f", totalStats.centroid.x);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("  centroid Y : %.3f", totalStats.centroid.y);
}

void WriteProcessorParameters(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    DeviceState* pDeviceState,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    NN_ASSERT_NOT_NULL(pDeviceState);

    const nn::util::Unorm8x4 White = { { 255, 255, 255, 255 } };
    const nn::util::Unorm8x4 Red = { { 255, 0, 0, 255 } };

    pTextWriter->SetScale(0.90f, 0.90f);
    pTextWriter->SetTextColor(White);

    auto startOffset = 150;
    auto count = 0;
    auto offset = 20;
    auto index = 0;
    if (pDeviceState->irToolState == IrSensorApiTestToolState_CameraReady)
    {
        index = pDeviceState->padMenuSelect;
    }
    else if (pDeviceState->irToolState == IrSensorApiTestToolState_Running)
    {
        index = pDeviceState->padOutputMenuSelect - MenuOutputLength;
    }

    if (pDeviceState->irState.nextProcessor == IrSensorMode_Moment)
    {
        pTextWriter->SetTextColor((index == 5) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        if (pDeviceState->irState.momentProcessorConfig.preprocess == nn::irsensor::MomentProcessorPreprocess_Binarize)
        {
            pTextWriter->Print("Preprocess : Binarize");
        }
        else
        {
            pTextWriter->Print("Preprocess : Cutoff");
        }
        pTextWriter->SetTextColor((index == 6) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        pTextWriter->Print("Intensity Threshold : %d", pDeviceState->irState.momentProcessorConfig.preprocessIntensityThreshold);

        pTextWriter->SetTextColor((index == 7) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        pTextWriter->Print("WOI X : %d", pDeviceState->irState.momentProcessorConfig.windowOfInterest.x);

        pTextWriter->SetTextColor((index == 8) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        pTextWriter->Print("WOI Y : %d", pDeviceState->irState.momentProcessorConfig.windowOfInterest.y);

        pTextWriter->SetTextColor((index == 9) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        pTextWriter->Print("WOI Width : %d", pDeviceState->irState.momentProcessorConfig.windowOfInterest.width);

        pTextWriter->SetTextColor((index == 10) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        pTextWriter->Print("WOI Height : %d", pDeviceState->irState.momentProcessorConfig.windowOfInterest.height);
    }
    else if (pDeviceState->irState.nextProcessor == IrSensorMode_Clustering)
    {
        pTextWriter->SetTextColor((index == 5) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        pTextWriter->Print("Object Pixel Count Min : %d", pDeviceState->irState.clusteringProcessorConfig.objectPixelCountMin);

        pTextWriter->SetTextColor((index == 6) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        pTextWriter->Print("Object Pixel Count Max : %d", pDeviceState->irState.clusteringProcessorConfig.objectPixelCountMax);

        pTextWriter->SetTextColor((index == 7) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        pTextWriter->Print("Object Intensity Min : %d", pDeviceState->irState.clusteringProcessorConfig.objectIntensityMin);

        pTextWriter->SetTextColor((index == 8) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);

        if (pDeviceState->irState.clusteringProcessorConfig.isExternalLightFilterEnabled)
        {
            pTextWriter->Print("IsExternalLightFilterEnabled : true");
        }
        else
        {
            pTextWriter->Print("IsExternalLightFilterEnabled : false");
        }

        pTextWriter->SetTextColor((index == 9) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        pTextWriter->Print("WOI X : %d", pDeviceState->irState.clusteringProcessorConfig.windowOfInterest.x);

        pTextWriter->SetTextColor((index == 10) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        pTextWriter->Print("WOI Y : %d", pDeviceState->irState.clusteringProcessorConfig.windowOfInterest.y);

        pTextWriter->SetTextColor((index == 11) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        pTextWriter->Print("WOI Width : %d", pDeviceState->irState.clusteringProcessorConfig.windowOfInterest.width);

        pTextWriter->SetTextColor((index == 12) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        pTextWriter->Print("WOI Height : %d", pDeviceState->irState.clusteringProcessorConfig.windowOfInterest.height);
    }
    else if (pDeviceState->irState.nextProcessor == IrSensorMode_ImageTransfer)
    {
        pTextWriter->SetTextColor((index == 5) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        if (pDeviceState->irState.imageTransferProcessorConfigNext.origFormat == nn::irsensor::ImageTransferProcessorFormat_320x240)
        {
            pTextWriter->Print("OriginalFormat : 320x240");
        }
        else if (pDeviceState->irState.imageTransferProcessorConfigNext.origFormat == nn::irsensor::ImageTransferProcessorFormat_160x120)
        {
            pTextWriter->Print("OriginalFormat : 160x120");
        }
        else if (pDeviceState->irState.imageTransferProcessorConfigNext.origFormat == nn::irsensor::ImageTransferProcessorFormat_80x60)
        {
            pTextWriter->Print("OriginalFormat : 80x60");
        }
        else if (pDeviceState->irState.imageTransferProcessorConfigNext.origFormat == nn::irsensor::ImageTransferProcessorFormat_40x30)
        {
            pTextWriter->Print("OriginalFormat : 40x30");
        }
        else if (pDeviceState->irState.imageTransferProcessorConfigNext.origFormat == nn::irsensor::ImageTransferProcessorFormat_20x15)
        {
            pTextWriter->Print("OriginalFormat : 20x15");
        }
        pTextWriter->SetTextColor((index == 6) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        if (pDeviceState->irState.imageTransferProcessorConfigNext.trimmingFormat == nn::irsensor::ImageTransferProcessorFormat_320x240)
        {
            pTextWriter->Print("Format : 320x240");
        }
        else if (pDeviceState->irState.imageTransferProcessorConfigNext.trimmingFormat == nn::irsensor::ImageTransferProcessorFormat_160x120)
        {
            pTextWriter->Print("Format : 160x120");
        }
        else if (pDeviceState->irState.imageTransferProcessorConfigNext.trimmingFormat == nn::irsensor::ImageTransferProcessorFormat_80x60)
        {
            pTextWriter->Print("Format : 80x60");
        }
        else if (pDeviceState->irState.imageTransferProcessorConfigNext.trimmingFormat == nn::irsensor::ImageTransferProcessorFormat_40x30)
        {
            pTextWriter->Print("Format : 40x30");
        }
        else if (pDeviceState->irState.imageTransferProcessorConfigNext.trimmingFormat == nn::irsensor::ImageTransferProcessorFormat_20x15)
        {
            pTextWriter->Print("Format : 20x15");
        }
        pTextWriter->SetTextColor((index == 7) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        pTextWriter->Print("StartX : %d", pDeviceState->irState.imageTransferProcessorConfigNext.trimmingStartX);
        pTextWriter->SetTextColor((index == 8) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        pTextWriter->Print("StartY : %d", pDeviceState->irState.imageTransferProcessorConfigNext.trimmingStartY);
        pTextWriter->SetTextColor((index == 9) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        if (pDeviceState->irState.imageTransferProcessorConfigNext.isExternalLightFilterEnabled)
        {
            pTextWriter->Print("IsExternalLightFilterEnabled : true");
        }
        else
        {
            pTextWriter->Print("IsExternalLightFilterEnabled : false");
        }
    }
    else if (pDeviceState->irState.nextProcessor == IrSensorMode_Pointing)
    {
        pTextWriter->SetTextColor((index == 5) ? Red : White);
    }
    else if (pDeviceState->irState.nextProcessor == IrSensorMode_HandAnalysis)
    {
        pTextWriter->SetTextColor((index == 1) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        if (pDeviceState->irState.handAnalysisConfigNext.mode == nn::irsensor::HandAnalysisMode_Silhouette)
        {
            pTextWriter->Print("Mode : Silhouette");
        }
        else if (pDeviceState->irState.handAnalysisConfigNext.mode == nn::irsensor::HandAnalysisMode_Image)
        {
            pTextWriter->Print("Mode : Image");
        }
        else if (pDeviceState->irState.handAnalysisConfigNext.mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage)
        {
            pTextWriter->Print("Mode : SilhouetteAndImage");
        }
        else if (pDeviceState->irState.handAnalysisConfigNext.mode == nn::irsensor::HandAnalysisMode_SilhouetteOnly)
        {
            pTextWriter->Print("Mode : SilhouetteOnly");
        }
    }
    else if (pDeviceState->irState.nextProcessor == IrSensorMode_IrLed)
    {
        pTextWriter->SetTextColor((index == 1) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        switch (pDeviceState->irState.irLedProcessorConfig.lightTarget)
        {
        case nn::irsensor::IrCameraLightTarget_AllObjects:
            pTextWriter->Print("lightTarget : AllObjects");
            break;
        case nn::irsensor::IrCameraLightTarget_FarObjects:
            pTextWriter->Print("lightTarget : FarObjects");
            break;
        case nn::irsensor::IrCameraLightTarget_NearObjects:
            pTextWriter->Print("lightTarget : NearObjects");
            break;
        case nn::irsensor::IrCameraLightTarget_None:
            pTextWriter->Print("lightTarget : None");
            break;
        default:
            break;
        }
    }
    else if (pDeviceState->irState.nextProcessor == IrSensorMode_AdaptiveClustering)
    {
        pTextWriter->SetTextColor((index == 1) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + startOffset + offset * count++);
        if (pDeviceState->irState.adaptiveClusteringProcessorConfig.mode == nn::irsensor::AdaptiveClusteringMode_StaticFov)
        {
            pTextWriter->Print("Mode : StaticFov");
        }
        else if (pDeviceState->irState.adaptiveClusteringProcessorConfig.mode == nn::irsensor::AdaptiveClusteringMode_DynamicFov)
        {
            pTextWriter->Print("Mode : DynamicFov");
        }
    }
}//NOLINT(readability/fn_size)

void WriteClusteringProcessorState(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    nn::Result clusteringApiResult,
    int clusteringApiOutCount,
    const nn::irsensor::ClusteringProcessorState* pClusteringState,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    NN_ASSERT_NOT_NULL(pClusteringState);

    const nn::util::Unorm8x4 White = { { 255, 255, 255, 255 } };

    // 変えられない設定
    pTextWriter->SetTextColor(White);
    auto i = 1;

    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    PrintIrsensorError(pTextWriter, clusteringApiResult);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("Api OutCount : %d", clusteringApiOutCount);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("Sampling Number : %ld", pClusteringState->samplingNumber);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("TimeStamp : %lld[ns]", pClusteringState->timeStamp.GetNanoSeconds());
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("AmbientNoiseLevel : %s", AmbientNoiseLevelName[pClusteringState->ambientNoiseLevel]);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("Object Count : %d", pClusteringState->objectCount);
}

void WriteImageTransferProcessorState(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    nn::Result imageTransferApiResult,
    const nn::irsensor::ImageTransferProcessorState* pImageTransferState,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    NN_ASSERT_NOT_NULL(pImageTransferState);

    const nn::util::Unorm8x4 White = { { 255, 255, 255, 255 } };

    // 変えられない設定
    pTextWriter->SetTextColor(White);
    auto i = 1;
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    PrintIrsensorError(pTextWriter, imageTransferApiResult);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("Sampling Number : %ld", pImageTransferState->samplingNumber);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("AmbientNoiseLevel : %s", AmbientNoiseLevelName[pImageTransferState->ambientNoiseLevel]);
}

void WritePointingProcessorState(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    nn::Result pointingApiResult,
    int pointingApiOutCount,
    const nn::irsensor::PointingProcessorState* pPointingState,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    NN_ASSERT_NOT_NULL(pPointingState);

    const nn::util::Unorm8x4 White = { { 255, 255, 255, 255 } };

    // 変えられない設定
    pTextWriter->SetTextColor(White);
    auto i = 1;

    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    PrintIrsensorError(pTextWriter, pointingApiResult);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("Api OutCount : %d", pointingApiOutCount);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("Sampling Number : %ld", pPointingState->samplingNumber);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("TimeStamp : %lld[ns]", pPointingState->timeStamp.GetNanoSeconds());
}

void WriteHandAnalysisProcessorState(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    nn::irsensor::HandAnalysisMode mode,
    nn::Result handAnalysisApiResult,
    float shapeArea,
    const nn::irsensor::HandAnalysisImageState* pHandAnalysisImageState,
    const nn::irsensor::HandAnalysisSilhouetteState* pHandAnalysisSilhouetteState,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    NN_ASSERT((pHandAnalysisImageState != nullptr) || (pHandAnalysisSilhouetteState != nullptr));

    int64_t samplingNumber = 0;
    nn::irsensor::IrCameraAmbientNoiseLevel level = nn::irsensor::IrCameraAmbientNoiseLevel_Unknown;

    float shapeAreaVal = 0.0f;
    if (mode == nn::irsensor::HandAnalysisMode_Image
        || mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage)
    {
        samplingNumber = pHandAnalysisImageState->samplingNumber;
        level = pHandAnalysisImageState->ambientNoiseLevel;
    }
    if (mode == nn::irsensor::HandAnalysisMode_Silhouette
        || mode == nn::irsensor::HandAnalysisMode_SilhouetteOnly
        || mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage)
    {
        samplingNumber = pHandAnalysisSilhouetteState->samplingNumber;
        level = pHandAnalysisSilhouetteState->ambientNoiseLevel;
        shapeAreaVal = shapeArea;
    }

    const nn::util::Unorm8x4 White = { { 255, 255, 255, 255 } };

    // 変えられない設定
    pTextWriter->SetTextColor(White);
    auto i = 1;

    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    PrintIrsensorError(pTextWriter, handAnalysisApiResult);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("Sampling Number : %ld", samplingNumber);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("AmbientNoiseLevel : %s", AmbientNoiseLevelName[level]);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("Shape Area : %f", shapeAreaVal);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);

    if ((mode == nn::irsensor::HandAnalysisMode_Silhouette
        || mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage)
        && pHandAnalysisSilhouetteState->handCount > 0)
    {
        const nn::irsensor::Hand& hand = pHandAnalysisSilhouetteState->hands[0];
        int numFinger = 0;

        for (int j = 0; j < nn::irsensor::HandFinger_Count; ++j)
        {
            if (hand.fingers[j].isValid)
            {
                numFinger++;
            }
        }
        switch (numFinger)
        {
        case 0:
        case 1:
            {
                pTextWriter->Print("Shape : Rock");
            }
            break;
        case 2:
        case 3:
            {
                pTextWriter->Print("Shape : Scissors");
            }
            break;
        case 5:
        case 6:
            {
                pTextWriter->Print("Shape : Paper");
            }
            break;
        default:
            {
                pTextWriter->Print("Shape : Unknown");
            }
            break;
        }
    }
    else
    {
        pTextWriter->Print("Shape : Unknown");
    }
}

void WriteIrLedProcessorState(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    nn::Result irLedApiResult,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    const nn::util::Unorm8x4 White = { { 255, 255, 255, 255 } };

    // 変えられない設定
    pTextWriter->SetTextColor(White);
    auto i = 1;

    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    PrintIrsensorError(pTextWriter, irLedApiResult);
}

void WriteAdaptiveClusteringProcessorState(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    nn::Result adaptiveClusteringApiResult,
    const nn::irsensor::AdaptiveClusteringProcessorState* pAdaptiveClusteringState,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    NN_ASSERT(pAdaptiveClusteringState != nullptr);

    int64_t samplingNumber = pAdaptiveClusteringState->samplingNumber;
    nn::irsensor::IrCameraAmbientNoiseLevel level = pAdaptiveClusteringState->ambientNoiseLevel;
    nn::irsensor::AdaptiveClusteringAccuracyLevel accuracy = pAdaptiveClusteringState->accuracyLevel;

    const nn::util::Unorm8x4 White = { { 255, 255, 255, 255 } };

    // 変えられない設定
    pTextWriter->SetTextColor(White);
    auto i = 1;

    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    PrintIrsensorError(pTextWriter, adaptiveClusteringApiResult);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("Sampling Number : %ld", samplingNumber);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("AmbientNoiseLevel : %s", AmbientNoiseLevelName[level]);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("ElementCount : %d", pAdaptiveClusteringState->objectCount);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("AccuracyLevel : %s", AccuracyLevelName[accuracy]);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("BackgroundIntensity : %d", pAdaptiveClusteringState->backgroundIntensity);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
}

void WriteMomentProcessorState(GraphicsSystem* pGraphicsSystem,
    const nn::irsensor::Rect windowOfInterest,
    const nn::irsensor::MomentProcessorState* pMomentState,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pGraphicsSystem);
    NN_ASSERT_NOT_NULL(pMomentState);

    const nn::util::Uint8x4 White = { { 255, 255, 255, 255 } };
    const nn::util::Uint8x4 Red = { { 255, 0, 0, 255 } };

    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();
    nn::gfx::CommandBuffer* pCommandBuffer = &pGraphicsSystem->GetCommandBuffer();

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    const float FullSizeWidth = 0.6f;
    const float FullSizeHeight = 0.8f;

    float xStart = ConvertFromPixelToTextureFactorX(offsetX) + FullSizeWidth * windowOfInterest.x / static_cast<float>(nn::irsensor::IrCameraImageWidth);
    float xEnd = xStart + FullSizeWidth * windowOfInterest.width / static_cast<float>(nn::irsensor::IrCameraImageWidth);
    float yStart = ConvertFromPixelToTextureFactorY(offsetY) - FullSizeHeight
        + FullSizeHeight * windowOfInterest.y / static_cast<float>(nn::irsensor::IrCameraImageHeight);
    float yEnd = yStart + FullSizeHeight * windowOfInterest.height / static_cast<float>(nn::irsensor::IrCameraImageHeight);

    nn::util::Vector3fType begin;
    nn::util::Vector3fType end;

    size_t idx = 0;
    for (size_t y = 0; y < nn::irsensor::MomentProcessorBlockRowCount; ++y)
    {
        for (size_t x = 0; x < nn::irsensor::MomentProcessorBlockColumnCount; ++x)
        {
            nn::util::Vector3fType corner;
            uint8_t averageIntensity = static_cast<uint8_t>(pMomentState->blocks[idx].averageIntensity);

            // 各面を描画
            nn::util::VectorSet(&corner,
                xStart + (xEnd - xStart) * x / static_cast<float>(nn::irsensor::MomentProcessorBlockColumnCount),
                yStart + (yEnd - yStart) * y / static_cast<float>(nn::irsensor::MomentProcessorBlockRowCount), 0.0f);

            nn::util::Vector3fType size;
            nn::util::VectorSet(&size,
                (xEnd - xStart) / static_cast<float>(nn::irsensor::MomentProcessorBlockColumnCount),
                (yEnd - yStart) / static_cast<float>(nn::irsensor::MomentProcessorBlockRowCount), 0.0f);

            nn::util::Uint8x4 gridColor = { { averageIntensity, averageIntensity, averageIntensity, averageIntensity } };
            pPrimitiveRenderer->SetColor(gridColor);
            pPrimitiveRenderer->DrawQuad(
                pCommandBuffer,
                corner,
                size);

            // 重心点を描画
            nn::util::VectorSet(&corner,
                xStart + (xEnd - xStart) * (pMomentState->blocks[idx].centroid.x) / static_cast<float>(windowOfInterest.width),
                yStart + (yEnd - yStart) * (pMomentState->blocks[idx].centroid.y) / static_cast<float>(windowOfInterest.height), 0.0f);
            pPrimitiveRenderer->SetColor(Red);
            pPrimitiveRenderer->DrawSphere(
                pCommandBuffer,
                nns::gfx::PrimitiveRenderer::Surface_Solid,
                nns::gfx::PrimitiveRenderer::Subdiv_Coarse,
                corner,
                4.0f / static_cast<float>(nn::irsensor::IrCameraImageWidth));

            idx++;
        }
    }

    // グリッドを描画
    pPrimitiveRenderer->SetLineWidth(0.0f);

    for (size_t x = 0; x <= nn::irsensor::MomentProcessorBlockColumnCount; ++x)
    {
        float xValue = xStart + (xEnd - xStart) * x / static_cast<float>(nn::irsensor::MomentProcessorBlockColumnCount);
        nn::util::VectorSet(&begin, xValue, yStart, 0.0f);
        nn::util::VectorSet(&end, xValue, yEnd, 0.0f);
        pPrimitiveRenderer->SetColor(White);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    }
    for (size_t y = 0; y <= nn::irsensor::MomentProcessorBlockRowCount; ++y)
    {
        float yValue = yStart + (yEnd - yStart) * y / static_cast<float>(nn::irsensor::MomentProcessorBlockRowCount);
        nn::util::VectorSet(&begin, xStart, yValue, 0.0f);
        nn::util::VectorSet(&end, xEnd, yValue, 0.0f);
        pPrimitiveRenderer->SetColor(White);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    }
}

void WriteClusteringProcessorState(GraphicsSystem* pGraphicsSystem,
    const nn::irsensor::Rect windowOfInterest,
    const nn::irsensor::ClusteringProcessorState* pClusteringState,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pGraphicsSystem);
    NN_ASSERT_NOT_NULL(pClusteringState);

    const nn::util::Uint8x4 White = { { 255, 255, 255, 255 } };
    const nn::util::Uint8x4 Green = { { 0, 255, 0, 255 } };

    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();
    nn::gfx::CommandBuffer* pCommandBuffer = &pGraphicsSystem->GetCommandBuffer();

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // グリッドを描画
    nn::util::Vector3fType begin;
    nn::util::Vector3fType end;
    pPrimitiveRenderer->SetLineWidth(10.f);

    float xStart = ConvertFromPixelToTextureFactorX(offsetX);
    float xEnd = xStart + 0.6f;
    float yEnd = ConvertFromPixelToTextureFactorY(offsetY);
    float yStart = yEnd - 0.8f;

    pPrimitiveRenderer->SetLineWidth(10.f);
    pPrimitiveRenderer->SetColor(White);

    nn::util::VectorSet(&begin, xStart, yStart, 0.0f);
    nn::util::VectorSet(&end, xStart, yEnd, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin, xEnd, yEnd, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&end, xEnd, yStart, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin, xStart, yStart, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);

    // 各面を描画
    for (auto i = 0; i < pClusteringState->objectCount; ++i)
    {
        const nn::irsensor::ClusteringData* pObject = &pClusteringState->objects[i];
        float boundStartX = static_cast<float>(pObject->bound.x + windowOfInterest.x);
        float boundStartY = static_cast<float>(pObject->bound.y + windowOfInterest.y);
        float boundEndX = static_cast<float>(pObject->bound.x + pObject->bound.width + windowOfInterest.x);
        float boundEndY = static_cast<float>(pObject->bound.y + pObject->bound.height + windowOfInterest.y);
        pPrimitiveRenderer->SetColor(Green);
        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * boundStartX / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * boundStartY / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        nn::util::VectorSet(&end,
            xStart + (xEnd - xStart) * boundEndX / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * boundStartY / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * boundEndX / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * boundEndY / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
        nn::util::VectorSet(&end,
            xStart + (xEnd - xStart) * boundStartX / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * boundEndY / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * boundStartX / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * boundStartY / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    }
}

void GetBufferToImage(CameraViewer* pCameraViewer,
    IrState* pIrState,
    const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pCameraViewer);
    NN_ASSERT_NOT_NULL(pIrState);

    // Generate the linear texture w/ CPU writes
    size_t writeSize = 0;
    if (pIrState->format == nn::irsensor::ImageTransferProcessorFormat_320x240)
    {
        writeSize = nn::irsensor::ImageTransferProcessorImageSize320x240;
    }
    else if (pIrState->format == nn::irsensor::ImageTransferProcessorFormat_160x120)
    {
        writeSize = nn::irsensor::ImageTransferProcessorImageSize160x120;
    }
    else if (pIrState->format == nn::irsensor::ImageTransferProcessorFormat_80x60)
    {
        writeSize = nn::irsensor::ImageTransferProcessorImageSize80x60;
    }
    else if (pIrState->format == nn::irsensor::ImageTransferProcessorFormat_40x30)
    {
        writeSize = nn::irsensor::ImageTransferProcessorImageSize40x30;
    }
    else if (pIrState->format == nn::irsensor::ImageTransferProcessorFormat_20x15)
    {
        writeSize = nn::irsensor::ImageTransferProcessorImageSize20x15;
    }
    else
    {
        return;
    }

    uint8_t* sourcePixels = static_cast<uint8_t*>(pCameraViewer->cameraImageBuffer);

    pIrState->imageTransferApiResult = nn::irsensor::GetImageTransferProcessorState(
        &pIrState->imageTransferProcessorState,
        sourcePixels,
        writeSize,
        handle);

    if (pIrState->imageTransferPrevSamplingNumber < pIrState->imageTransferProcessorState.samplingNumber)
    {
        // フレーム更新時間を保存 (初回は無視する)
        if (pIrState->imageTransferGetDataTime != 0)
        {
            pIrState->imageTransferFrameInterval
                = (nn::os::GetSystemTick() - nn::os::Tick(pIrState->imageTransferGetDataTime)).ToTimeSpan().GetMilliSeconds();
        }
        pIrState->imageTransferGetDataTime = nn::os::GetSystemTick().GetInt64Value();

        pIrState->imageTransferPrevSamplingNumber = pIrState->imageTransferProcessorState.samplingNumber;
        uint8_t* destinationPixels = pCameraViewer->copiedImageBuffer.Map< uint8_t >();
        memcpy(destinationPixels, sourcePixels, writeSize);
        pCameraViewer->copiedImageBuffer.FlushMappedRange(0, writeSize);
        pCameraViewer->copiedImageBuffer.Unmap();
    }
}

void CopyImageBuffer(GraphicsSystem* pGraphicsSystem,
    CameraViewer* pCameraViewer,
    IrState* pIrState) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pGraphicsSystem);
    NN_ASSERT_NOT_NULL(pCameraViewer);
    NN_ASSERT_NOT_NULL(pIrState);

    nn::gfx::TextureCopyRegion dstRegion;
    dstRegion.SetDefault();
    dstRegion.SetDepth(1);
    if (pIrState->currentProcessor == IrSensorMode_ImageTransfer)
    {
        if (pIrState->format == nn::irsensor::ImageTransferProcessorFormat_320x240)
        {
            dstRegion.SetWidth(320);
            dstRegion.SetHeight(240);
            pGraphicsSystem->GetCommandBuffer().CopyBufferToImage(&pCameraViewer->texture[0], dstRegion, &pCameraViewer->copiedImageBuffer, 0);
        }
        else if (pIrState->format == nn::irsensor::ImageTransferProcessorFormat_160x120)
        {
            dstRegion.SetWidth(160);
            dstRegion.SetHeight(120);
            pGraphicsSystem->GetCommandBuffer().CopyBufferToImage(&pCameraViewer->texture[1], dstRegion, &pCameraViewer->copiedImageBuffer, 0);
        }
        else if (pIrState->format == nn::irsensor::ImageTransferProcessorFormat_80x60)
        {
            dstRegion.SetWidth(80);
            dstRegion.SetHeight(60);
            pGraphicsSystem->GetCommandBuffer().CopyBufferToImage(&pCameraViewer->texture[2], dstRegion, &pCameraViewer->copiedImageBuffer, 0);
        }
        else if (pIrState->format == nn::irsensor::ImageTransferProcessorFormat_40x30)
        {
            dstRegion.SetWidth(40);
            dstRegion.SetHeight(30);
            pGraphicsSystem->GetCommandBuffer().CopyBufferToImage(&pCameraViewer->texture[3], dstRegion, &pCameraViewer->copiedImageBuffer, 0);
        }
        else if (pIrState->format == nn::irsensor::ImageTransferProcessorFormat_20x15)
        {
            dstRegion.SetWidth(20);
            dstRegion.SetHeight(15);
            pGraphicsSystem->GetCommandBuffer().CopyBufferToImage(&pCameraViewer->texture[4], dstRegion, &pCameraViewer->copiedImageBuffer, 0);
        }
    }
    else if (pIrState->currentProcessor == IrSensorMode_HandAnalysis)
    {
        if ((pIrState->handAnalysisConfigCurrent.mode == nn::irsensor::HandAnalysisMode_Image)
            || (pIrState->handAnalysisConfigCurrent.mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage))
        {
            dstRegion.SetWidth(nn::irsensor::IrHandAnalysisImageWidth);
            dstRegion.SetHeight(nn::irsensor::IrHandAnalysisImageHeight);
            pGraphicsSystem->GetCommandBuffer().CopyBufferToImage(&pCameraViewer->textureHand, dstRegion, &pCameraViewer->copiedImageBuffer, 0);
        }
    }
    pGraphicsSystem->DrawDone();
}

void WriteImageTransferProcessorState(GraphicsSystem* pGraphicsSystem,
    CameraViewer* pCameraViewer,
    IrState* pIrState,
    OutputLevel outputLevel,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pGraphicsSystem);
    NN_ASSERT_NOT_NULL(pCameraViewer);
    NN_ASSERT_NOT_NULL(pIrState);

    const nn::util::Uint8x4 White = { { 255, 255, 255, 255 } };

    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();
    nn::gfx::CommandBuffer* pCommandBuffer = &pGraphicsSystem->GetCommandBuffer();

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer,
        nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // グリッドを描画
    nn::util::Vector3fType corner;
    nn::util::Vector3fType size;

    float xStart = ConvertFromPixelToTextureFactorX(offsetX);
    float xEnd = xStart + 0.6f;
    float yEnd = ConvertFromPixelToTextureFactorY(offsetY);
    float yStart = yEnd - 0.8f;

    if (outputLevel == OutputLevel_Detail)
    {
        auto origWidth = (nn::irsensor::IrCameraImageWidth >> pIrState->imageTransferProcessorConfigCurrent.origFormat);
        auto origHeight = (nn::irsensor::IrCameraImageHeight >> pIrState->imageTransferProcessorConfigCurrent.origFormat);
        auto trimWidth = (nn::irsensor::IrCameraImageWidth >> pIrState->imageTransferProcessorConfigCurrent.trimmingFormat);
        auto trimHeight = (nn::irsensor::IrCameraImageHeight >> pIrState->imageTransferProcessorConfigCurrent.trimmingFormat);

        nn::util::VectorSet(&corner,
            xStart + (xEnd - xStart) * pIrState->imageTransferProcessorConfigCurrent.trimmingStartX / origWidth,
            yStart + (yEnd - yStart) * (origHeight - trimHeight - pIrState->imageTransferProcessorConfigCurrent.trimmingStartY) / origHeight, 0.0f);
        nn::util::VectorSet(&size,
            (xEnd - xStart) * trimWidth / origWidth,
            (yEnd - yStart) * trimHeight / origHeight, 0.0f);
    }
    else if (outputLevel == OutputLevel_Normal)
    {
        nn::util::VectorSet(&corner, xStart, yStart, 0.0f);
        nn::util::VectorSet(&size, xEnd - xStart, yEnd - yStart, 0.0f);
    }

    pPrimitiveRenderer->SetColor(White);
    pPrimitiveRenderer->DrawQuad(
        pCommandBuffer,
        corner,
        size,
        pCameraViewer->textureDescriptor[pIrState->format],
        pCameraViewer->samplerDescriptor);
}

void WritePointingProcessorState(GraphicsSystem* pGraphicsSystem,
    const nn::irsensor::PointingProcessorState* pPointingState,
    const nn::irsensor::PointingProcessorMarkerState* pPointingMarkerState,
    OutputLevel outputLevel,
    int pointingApiOutCount,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pGraphicsSystem);
    NN_ASSERT_NOT_NULL(pPointingState);
    NN_ASSERT_NOT_NULL(pPointingMarkerState);

    const nn::util::Uint8x4 White = { { 255, 255, 255, 255 } };
    const nn::util::Uint8x4 Green = { { 0, 255, 0, 255 } };
    const nn::util::Uint8x4 Red   = { { 255, 0, 0, 255 } };

    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();
    nn::gfx::CommandBuffer* pCommandBuffer = &pGraphicsSystem->GetCommandBuffer();

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // グリッドを描画
    nn::util::Vector3fType begin;
    nn::util::Vector3fType end;

    float xStart = ConvertFromPixelToTextureFactorX(offsetX);
    float xEnd = xStart + 0.6f;
    float yEnd = ConvertFromPixelToTextureFactorY(offsetY);
    float yStart = yEnd - 0.8f;

    pPrimitiveRenderer->SetLineWidth(2.f);
    pPrimitiveRenderer->SetColor(White);

    nn::util::VectorSet(&begin, xStart, yStart, 0.0f);
    nn::util::VectorSet(&end, xStart, yEnd, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin, xEnd, yEnd, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&end, xEnd, yStart, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin, xStart, yStart, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);

    for (auto i = 0; i < pointingApiOutCount; i++)
    {
        if ((pPointingState + i)->pointingStatus == nn::irsensor::PointingStatus_DataValid)
        {
            pPrimitiveRenderer->SetColor(Green);
        }
        else
        {
            pPrimitiveRenderer->SetColor(Red);
        }
        // 中心点を表示
        nn::util::Vector3fType center;
        nn::util::VectorSet(&center,
            xStart + (xEnd - xStart) * ((pPointingState + i)->position.x + 1.0f) * 0.5f,
            yStart + (yEnd - yStart) * (1.0f - (1.0f - (pPointingState + i)->position.y) * 0.5f), 0.0f);
        pPrimitiveRenderer->DrawSphere(
            pCommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface_Solid,
            nns::gfx::PrimitiveRenderer::Subdiv_Coarse,
            center,
            4.0f / static_cast<float>(nn::irsensor::IrCameraImageWidth)
        );
    }

    if (outputLevel == OutputLevel_Detail)
    {
        // 各面を描画
        nn::irsensor::Rect windowOfInterest = { 0, 0, 320, 240 };
        for (auto i = 0; i < nn::irsensor::PointingProcessorMarkerObjectCount; ++i)
        {
            const nn::irsensor::PointingObjectData* pObject = &pPointingMarkerState->objects[i];
            float boundStartX = static_cast<float>(nn::irsensor::IrCameraImageWidth - (pObject->bound.x + windowOfInterest.x));
            float boundStartY = static_cast<float>(pObject->bound.y + windowOfInterest.y);
            float boundEndX = static_cast<float>(nn::irsensor::IrCameraImageWidth - (pObject->bound.x + pObject->bound.width + windowOfInterest.x));
            float boundEndY = static_cast<float>(pObject->bound.y + pObject->bound.height + windowOfInterest.y);
            if (pObject->isDataValid)
            {
                pPrimitiveRenderer->SetColor(Green);
                nn::util::VectorSet(&begin,
                    xStart + (xEnd - xStart) * boundStartX / static_cast<float>(nn::irsensor::IrCameraImageWidth),
                    yStart + (yEnd - yStart) * boundStartY / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
                nn::util::VectorSet(&end,
                    xStart + (xEnd - xStart) * boundEndX / static_cast<float>(nn::irsensor::IrCameraImageWidth),
                    yStart + (yEnd - yStart) * boundStartY / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
                pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
                nn::util::VectorSet(&begin,
                    xStart + (xEnd - xStart) * boundEndX / static_cast<float>(nn::irsensor::IrCameraImageWidth),
                    yStart + (yEnd - yStart) * boundEndY / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
                pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
                nn::util::VectorSet(&end,
                    xStart + (xEnd - xStart) * boundStartX / static_cast<float>(nn::irsensor::IrCameraImageWidth),
                    yStart + (yEnd - yStart) * boundEndY / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
                pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
                nn::util::VectorSet(&begin,
                    xStart + (xEnd - xStart) * boundStartX / static_cast<float>(nn::irsensor::IrCameraImageWidth),
                    yStart + (yEnd - yStart) * boundStartY / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
                pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
            }
        }
    }
}

void WriteHandAnalysisImageProcessorState(GraphicsSystem* pGraphicsSystem,
    CameraViewer* pCameraViewer,
    IrState* pIrState,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pGraphicsSystem);
    NN_ASSERT_NOT_NULL(pCameraViewer);
    NN_ASSERT_NOT_NULL(pIrState);

    const nn::util::Uint8x4 White = { { 255, 255, 255, 255 } };

    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();
    nn::gfx::CommandBuffer* pCommandBuffer = &pGraphicsSystem->GetCommandBuffer();

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer,
        nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // グリッドを描画
    nn::util::Vector3fType corner;
    nn::util::Vector3fType size;

    float xStart = ConvertFromPixelToTextureFactorX(offsetX);
    float xEnd = xStart + 0.6f;
    float yStart = ConvertFromPixelToTextureFactorY(offsetY);
    float yEnd = yStart - 0.8f;

    nn::util::VectorSet(&corner, xStart, yStart, 0.0f);
    nn::util::VectorSet(&size, xEnd - xStart, yEnd - yStart, 0.0f);

    pPrimitiveRenderer->SetColor(White);
    pPrimitiveRenderer->DrawQuad(
        pCommandBuffer,
        corner,
        size,
        pCameraViewer->textureDescriptorHand,
        pCameraViewer->samplerDescriptor);
}

void WriteHandAnalysisSilhouetteProcessorState(GraphicsSystem* pGraphicsSystem,
    const nn::irsensor::HandAnalysisSilhouetteState* pHandAnalysisSilhouetteState,
    PolygonProperties* pPolygonProperties,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pGraphicsSystem);
    NN_ASSERT_NOT_NULL(pHandAnalysisSilhouetteState);

    const nn::util::Uint8x4 Green = { { 0, 255, 0, 255 } };
    const nn::util::Uint8x4 Red ={ { 255, 0, 0, 255 } };
    const nn::util::Uint8x4 Blue ={ { 0, 0, 255, 255 } };
    const nn::util::Uint8x4 Yellow ={ { 255, 255, 0, 255 } };


    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();
    nn::gfx::CommandBuffer* pCommandBuffer = &pGraphicsSystem->GetCommandBuffer();

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // グリッドを描画
    nn::util::Vector3fType begin;
    nn::util::Vector3fType end;
    pPrimitiveRenderer->SetLineWidth(5.f);

    float xStart = ConvertFromPixelToTextureFactorX(offsetX);
    float xEnd = xStart + 0.6f;
    float yEnd = ConvertFromPixelToTextureFactorY(offsetY);
    float yStart = yEnd - 0.8f;

    // NOTE: Coordinate are normalized according to Y axis.
    // Because the image ratio is 4/3, X points needs to be scaled
    // by 3/4 to fit in image space.
    const float xscale = 0.75f;
    const float yscale = 1.0f;

    // シルエットを描画
    for (int shapeIndex = 0; shapeIndex < pHandAnalysisSilhouetteState->shapeCount; shapeIndex++)
    {
        std::vector<nn::util::Float2> silhouette;
        const nn::irsensor::Shape& shape = pHandAnalysisSilhouetteState->shapes[shapeIndex];

        nn::util::Float2 p1, p2;
        p2 = pHandAnalysisSilhouetteState->points[shape.firstPointIndex];
        pPrimitiveRenderer->SetColor(Green);
        for (auto i = 0; i < shape.pointCount - 1; ++i)
        {
            p1 = p2;
            p2 = pHandAnalysisSilhouetteState->points[(shape.firstPointIndex + i + 1) % NN_ARRAY_SIZE(pHandAnalysisSilhouetteState->points)];

            nn::util::VectorSet(&begin,
                xStart + (xEnd - xStart) * (p1.x * xscale + 1.0f) / 2.0f,
                yStart + (yEnd - yStart) * (p1.y * yscale + 1.0f) / 2.0f, 0.0f);
            nn::util::VectorSet(&end,
                xStart + (xEnd - xStart) * (p2.x * xscale + 1.0f) / 2.0f,
                yStart + (yEnd - yStart) * (p2.y * yscale + 1.0f) / 2.0f, 0.0f);

            pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);

            silhouette.push_back(p1);
        }
        silhouette.push_back(p2);

        std::vector<size_t> convexHullIndices;
        // Convex Hull を計算する
        ExtractConvexHull(convexHullIndices, silhouette);
        if (convexHullIndices.size() > 0)
        {
            p2 = pHandAnalysisSilhouetteState->points[shape.firstPointIndex + convexHullIndices[0]];
            pPrimitiveRenderer->SetColor(Yellow);
            for (size_t i = 0; i < convexHullIndices.size() - 1; ++i)
            {
                p1 = p2;
                p2 = pHandAnalysisSilhouetteState->points[shape.firstPointIndex + convexHullIndices[i + 1]];

                nn::util::VectorSet(&begin,
                    xStart + (xEnd - xStart) * (p1.x * xscale + 1.0f) / 2.0f,
                    yStart + (yEnd - yStart) * (p1.y * yscale + 1.0f) / 2.0f, 0.0f);
                nn::util::VectorSet(&end,
                    xStart + (xEnd - xStart) * (p2.x * xscale + 1.0f) / 2.0f,
                    yStart + (yEnd - yStart) * (p2.y * yscale + 1.0f) / 2.0f, 0.0f);

                pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
            }
        }

        ComputePolygonProperties(pPolygonProperties, &silhouette[0], silhouette.size());
    }

    // 指先を描画
    for (int handIndex = 0; handIndex < pHandAnalysisSilhouetteState->handCount; handIndex++)
    {
        const nn::irsensor::Hand& hand = pHandAnalysisSilhouetteState->hands[handIndex];

        // 検出物の中央を描画
        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * (hand.palm.center.x * xscale + 1.0f) / 2.0f,
            yStart + (yEnd - yStart) * (hand.palm.center.y * yscale + 1.0f) / 2.0f, 0.0f);
        pPrimitiveRenderer->SetColor(Green);
        pPrimitiveRenderer->DrawSphere(
            pCommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface_Solid,
            nns::gfx::PrimitiveRenderer::Subdiv_Coarse,
            begin,
            8.0f / static_cast<float>(nn::irsensor::IrCameraImageWidth));

        for (int i = 0; i < nn::irsensor::HandFinger_Count; ++i)
        {
            if (hand.fingers[i].isValid)
            {
                nn::util::VectorSet(&begin,
                    xStart + (xEnd - xStart) * (hand.fingers[i].tip.x * xscale + 1.0f) / 2.0f,
                    yStart + (yEnd - yStart) * (hand.fingers[i].tip.y * yscale + 1.0f) / 2.0f, 0.0f);
                pPrimitiveRenderer->SetColor(Red);
                pPrimitiveRenderer->DrawSphere(
                    pCommandBuffer,
                    nns::gfx::PrimitiveRenderer::Surface_Solid,
                    nns::gfx::PrimitiveRenderer::Subdiv_Coarse,
                    begin,
                    8.0f / static_cast<float>(nn::irsensor::IrCameraImageWidth));

                nn::util::VectorSet(&begin,
                    xStart + (xEnd - xStart) * (hand.fingers[i].root.x * xscale + 1.0f) / 2.0f,
                    yStart + (yEnd - yStart) * (hand.fingers[i].root.y * yscale + 1.0f) / 2.0f, 0.0f);
                pPrimitiveRenderer->SetColor(Blue);
                pPrimitiveRenderer->DrawSphere(
                    pCommandBuffer,
                    nns::gfx::PrimitiveRenderer::Surface_Solid,
                    nns::gfx::PrimitiveRenderer::Subdiv_Coarse,
                    begin,
                    8.0f / static_cast<float>(nn::irsensor::IrCameraImageWidth));
            }
        }
    }
}//NOLINT(readability/fn_size)

void WriteAdaptiveClusteringProcessorState(GraphicsSystem* pGraphicsSystem,
    const nn::irsensor::AdaptiveClusteringProcessorState* pAdaptiveClusteringState,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pGraphicsSystem);
    NN_ASSERT_NOT_NULL(pAdaptiveClusteringState);

    const nn::util::Uint8x4 White = { { 255, 255, 255, 255 } };

    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();
    nn::gfx::CommandBuffer* pCommandBuffer = &pGraphicsSystem->GetCommandBuffer();

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // グリッドを描画
    nn::util::Vector3fType begin;
    nn::util::Vector3fType end;
    pPrimitiveRenderer->SetLineWidth(10.f);

    float xStart = ConvertFromPixelToTextureFactorX(offsetX);
    float xEnd = xStart + 0.6f;
    float yEnd = ConvertFromPixelToTextureFactorY(offsetY);
    float yStart = yEnd - 0.8f;

    pPrimitiveRenderer->SetLineWidth(10.f);
    pPrimitiveRenderer->SetColor(White);

    nn::util::VectorSet(&begin, xStart, yStart, 0.0f);
    nn::util::VectorSet(&end, xStart, yEnd, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin, xEnd, yEnd, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&end, xEnd, yStart, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin, xStart, yStart, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);

    // 各点を描画
    for (auto i = 0; i < pAdaptiveClusteringState->objectCount; ++i)
    {
        const nn::irsensor::AdaptiveClusteringData* pObject = &pAdaptiveClusteringState->objects[i];

        nn::util::Unorm8x4 Gray = { {255, 255, 255, 255 } };
        nn::util::Uint8x4 Green = { { 0, 255, 0, 255 } };
        if (pObject->isIncomplete)
        {
            pPrimitiveRenderer->SetColor(Green);
        }
        else
        {
            pPrimitiveRenderer->SetColor(Gray);
        }
        nn::util::Vector3fType center;
        nn::util::VectorSet(&center,
            xStart + (xEnd - xStart) * pObject->centroid.x / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * pObject->centroid.y / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        float diameter = std::sqrtf(pObject->area / static_cast<float>(nn::irsensor::IrCameraImageWidth * nn::irsensor::IrCameraImageHeight)) * 2.0f;
        pPrimitiveRenderer->DrawSphere(
            pCommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface_Solid,
            nns::gfx::PrimitiveRenderer::Subdiv_Normal,
            center,
            diameter);
    }
}

void WriteInitial(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    DeviceState* pDeviceState,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    NN_ASSERT_NOT_NULL(pDeviceState);

    const nn::util::Unorm8x4 White = { { 255, 255, 255, 255 } };
    const nn::util::Unorm8x4 Red = { { 255, 0, 0, 255 } };

    pTextWriter->SetScale(1.0f, 1.0f);
    auto i = 1;
    pTextWriter->SetTextColor((pDeviceState->npadSettingTypes == NpadSettingTypes_JoyRight) ? Red : White);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("Initialize With Npad (%s)", NpadSettingTypeName[NpadSettingTypes_JoyRight]);
    pTextWriter->SetTextColor((pDeviceState->npadSettingTypes == NpadSettingTypes_JoyLeft) ? Red : White);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("Initialize With Npad (%s)", NpadSettingTypeName[NpadSettingTypes_JoyLeft]);
    pTextWriter->SetTextColor((pDeviceState->npadSettingTypes == NpadSettingTypes_FullKey) ? Red : White);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("Initialize With Npad (%s)", NpadSettingTypeName[NpadSettingTypes_FullKey]);
    pTextWriter->SetTextColor((pDeviceState->npadSettingTypes == NpadSettingTypes_JoyDual) ? Red : White);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("Initialize With Npad (%s)", NpadSettingTypeName[NpadSettingTypes_JoyDual]);
    pTextWriter->SetTextColor((pDeviceState->npadSettingTypes == NpadSettingTypes_Handheld) ? Red : White);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("Initialize With Npad (%s)", NpadSettingTypeName[NpadSettingTypes_Handheld]);
}

void WriteInitialized(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    DeviceState* pDeviceState,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    const nn::util::Unorm8x4 White = { { 255, 255, 255, 255 } };
    const nn::util::Unorm8x4 Red = { { 255, 0, 0, 255 } };

    auto i = 1;
    pTextWriter->SetScale(1.0f, 1.0f);
    pTextWriter->SetTextColor((pDeviceState->padInitializedMenuSelect == 0) ? Red : White);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("Get IrCameraStatus");
    pTextWriter->SetTextColor((pDeviceState->padInitializedMenuSelect == 1) ? Red : White);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("Get ImageProcessorStatus");
    pTextWriter->SetTextColor((pDeviceState->padInitializedMenuSelect == 2) ? Red : White);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("Check FirmwareVersion");
}

void WriteOutputFirmwareCheck(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    DeviceState* pDeviceState,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    const nn::util::Unorm8x4 White = { { 255, 255, 255, 255 } };

    auto i = 1;
    pTextWriter->SetScale(1.0f, 1.0f);
    if (pDeviceState->padInitializedMenuSelect == 2)
    {
        pTextWriter->SetTextColor(White);
        pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
        PrintIrsensorError(pTextWriter, pDeviceState->firmwareUpdateStatusResult);
    }
}

void WriteOutput(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    DeviceState* pDeviceState,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    NN_ASSERT_NOT_NULL(pDeviceState);

    IrState irState = pDeviceState->irState;
    if (irState.currentProcessor == IrSensorMode_Moment)
    {
        WriteMomentProcessorState(pTextWriter, irState.momentApiResult, irState.momentApiOutCount, &irState.momentProcessorState[0],
            offsetX, offsetY);
    }
    else if (irState.currentProcessor == IrSensorMode_Clustering)
    {
        WriteClusteringProcessorState(pTextWriter, irState.clusteringApiResult, irState.clusteringApiOutCount, &irState.clusteringProcessorState[0],
            offsetX, offsetY);
    }
    else if (irState.currentProcessor == IrSensorMode_ImageTransfer)
    {
        WriteImageTransferProcessorState(pTextWriter, irState.imageTransferApiResult, &irState.imageTransferProcessorState,
            offsetX, offsetY);
    }
    else if (irState.currentProcessor == IrSensorMode_Pointing)
    {
        WritePointingProcessorState(pTextWriter, irState.pointingApiResult, irState.pointingApiOutCount, &irState.pointingProcessorState[0],
            offsetX, offsetY);
    }
    else if (irState.currentProcessor == IrSensorMode_HandAnalysis)
    {
        WriteHandAnalysisProcessorState(pTextWriter, irState.handAnalysisConfigCurrent.mode,
            irState.handAnalysisApiResult, irState.handAnalysisPolygon.signedArea,
            &irState.handAnalysisImageState, &irState.handAnalysisSilhouetteState,
            offsetX, offsetY);
    }
    else if (irState.currentProcessor == IrSensorMode_IrLed)
    {
        WriteIrLedProcessorState(pTextWriter, irState.irLedApiResult, offsetX, offsetY);
    }
    else if (irState.currentProcessor == IrSensorMode_AdaptiveClustering)
    {
        WriteAdaptiveClusteringProcessorState(pTextWriter, irState.adaptiveClusteringApiResult, &irState.adaptiveClusteringProcessorState[0], offsetX, offsetY);
    }
}

void WritePerformance(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    DeviceState* pDeviceState,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    NN_ASSERT_NOT_NULL(pDeviceState);

    const nn::util::Unorm8x4 Yellow = { { 128, 128, 0, 255 } };

    pTextWriter->SetScale(0.8f, 0.8f);
    pTextWriter->SetTextColor(Yellow);

    auto i = 0;
    IrState irState = pDeviceState->irState;

    if (irState.currentProcessor == IrSensorMode_Moment
        || irState.currentProcessor == IrSensorMode_Clustering
        || irState.currentProcessor == IrSensorMode_Pointing
        || irState.currentProcessor == IrSensorMode_HandAnalysis)
    {
        pTextWriter->SetCursor(offsetX, offsetY + 15 * i++);
        pTextWriter->Print("Data Drop Ratio : %lld [%] Ave[%d] : %0.3f [%] Max : %d [%]", 100 - pDeviceState->latestSampleCount, pDeviceState->updateCounter, pDeviceState->averagePlr, pDeviceState->maxPlr);
        pTextWriter->SetCursor(offsetX, offsetY + 15 * i++);
        pTextWriter->Print("Worst Burst Data Drop : %d [frame] Ave[%d] : %0.3f [frame] Max : %d [frame]", pDeviceState->burstDropCountWorstCurrent, pDeviceState->updateCounter, pDeviceState->burstDropCountWorstAve, pDeviceState->burstDropCountWorstMax);
    }
    else if (irState.currentProcessor == IrSensorMode_ImageTransfer)
    {
        pTextWriter->SetCursor(offsetX, offsetY + 15 * i++);
        pTextWriter->Print("Image Frame Interval : %lld [ms] Ave[%d] : %0.3f [ms]", pDeviceState->frameIntervalMs, pDeviceState->frameIntervalCount, pDeviceState->aveFrameIntervalMs);
    }
}

void WriteParameters(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    DeviceState* pDeviceState,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    NN_ASSERT_NOT_NULL(pDeviceState);

    WriteProcessorState(pTextWriter, pDeviceState, offsetX, offsetY);
    WriteProcessorParameters(pTextWriter, pDeviceState, offsetX, offsetY);
}

void WriteOutputConfig(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    DeviceState* pDeviceState,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    NN_ASSERT_NOT_NULL(pDeviceState);

    const nn::util::Unorm8x4 White = { { 255, 255, 255, 255 } };
    const nn::util::Unorm8x4 Red = { { 255, 0, 0, 255 } };

    pTextWriter->SetScale(1.0f, 1.0f);
    auto i = 1;
    pTextWriter->SetTextColor((pDeviceState->padOutputMenuSelect == 0) ? Red : White);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("GetOutputType : %s", OutputTypeName[pDeviceState->outMode]);
    pTextWriter->SetTextColor((pDeviceState->padOutputMenuSelect == 1) ? Red : White);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("UseApiType : %s", OutputApiTypeName[pDeviceState->outApiTypes]);
    pTextWriter->SetTextColor((pDeviceState->padOutputMenuSelect == 2) ? Red : White);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    pTextWriter->Print("CountMax : %d", pDeviceState->outCountMax);
    pTextWriter->SetTextColor((pDeviceState->padOutputMenuSelect == 3) ? Red : White);
    pTextWriter->SetCursor(offsetX, offsetY + 30 * i++);
    if (pDeviceState->outputLevelNext == OutputLevel_Normal)
    {
        pTextWriter->Print("OutputLevel : Normal");
    }
    else if (pDeviceState->outputLevelNext == OutputLevel_Detail)
    {
        pTextWriter->Print("OutputLevel : Detail");
    }
}

void WriteStatusBar(GraphicsSystem* pGraphicsSystem,
    nn::gfx::util::DebugFontTextWriter* pTextWriter,
    DeviceState* pDeviceState,
    int screenIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pGraphicsSystem);
    NN_ASSERT_NOT_NULL(pTextWriter);
    NN_ASSERT_NOT_NULL(pDeviceState);

    const nn::util::Unorm8x4 White = { { 255, 255, 255, 255 } };
    const nn::util::Unorm8x4 Red = { { 255, 0, 0, 255 } };
    const nn::util::Unorm8x4 Green = { { 0, 32, 0, 255 } };
    const nn::util::Unorm8x4 Cyan = { { 0, 64, 64, 255 } };
    const nn::util::Unorm8x4 Blue = { { 0, 32, 128, 255 } };
    const nn::util::Unorm8x4 Purple = { { 32, 0, 128, 255 } };
    const nn::util::Unorm8x4 Mazenta = { { 64, 0, 64, 255 } };

    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();
    nn::gfx::CommandBuffer* pCommandBuffer = &pGraphicsSystem->GetCommandBuffer();

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // 背景を描画
    const float xStart = -1.0f;
    const float xEnd = 1.0f;
    float yStart = 1.0f;
    float yEnd = 0.93f;
    if (screenIndex == 1)
    {
        yStart = 0.0f;
        yEnd = -0.07f;
    }
    nn::util::Vector3fType corner;
    nn::util::VectorSet(&corner, xStart, yStart, 0.0f);

    nn::util::Vector3fType size;
    nn::util::VectorSet(&size, xEnd - xStart, yEnd - yStart, 0.0f);

    if (g_Page == 0)
    {
        pPrimitiveRenderer->SetColor(Green);
    }
    else if (g_Page == 1)
    {
        pPrimitiveRenderer->SetColor(Cyan);
    }
    else if (g_Page == 2)
    {
        pPrimitiveRenderer->SetColor(Blue);
    }
    else if (g_Page == 3)
    {
        pPrimitiveRenderer->SetColor(Purple);
    }
    else if (g_Page == 4)
    {
        pPrimitiveRenderer->SetColor(Mazenta);
    }
    pPrimitiveRenderer->DrawQuad(
        pCommandBuffer,
        corner,
        size);

    // 文字列を描画
    float offsetX = 0.0f;
    float offsetY = 0.0f;
    if (screenIndex == 1)
    {
        offsetX = 0.0f;
        offsetY = 360.0f;
    }
    pTextWriter->SetScale(1.0f, 1.0f);
    pTextWriter->SetTextColor(White);
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print("Controller:%d  Type:%s", screenIndex + 2 * g_Page, NpadTypeName[pDeviceState->npadTypes]);
    pTextWriter->SetCursor(offsetX + 270.f, offsetY);
    pTextWriter->Print("Camera:%s", pDeviceState->isCameraStatusValid ? DeviceStateName[pDeviceState->irCameraStatus] : "Unknown");
    pTextWriter->SetCursor(offsetX + 480.f, offsetY);
    pTextWriter->Print("Processor:%s", pDeviceState->isProcessorStatusValid ? ProcessorStateName[pDeviceState->irProcessorStatus] : "Unknown");

    offsetX = 690.0f;
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print("Sequence:");
    pTextWriter->SetCursor(offsetX + 90.0f, offsetY);
    pTextWriter->SetTextColor((pDeviceState->irToolState == IrSensorApiTestToolState_Initial) ? Red : White);
    pTextWriter->Print("Initial->");
    pTextWriter->SetCursor(offsetX + 160.0f, offsetY);
    pTextWriter->SetTextColor((pDeviceState->irToolState == IrSensorApiTestToolState_Initialized) ? Red : White);
    pTextWriter->Print("Initialized->");
    pTextWriter->SetCursor(offsetX + 260.0f, offsetY);
    pTextWriter->SetTextColor((pDeviceState->irToolState == IrSensorApiTestToolState_CameraReady) ? Red : White);
    pTextWriter->Print("CameraReady->");
    pTextWriter->SetCursor(offsetX + 400.0f, offsetY);
    pTextWriter->SetTextColor((pDeviceState->irToolState == IrSensorApiTestToolState_Running) ? Red : White);
    pTextWriter->Print("Running");
}

void UpdateNpadConfiguration(DeviceState* pDeviceState, Npad::Id npadIds) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pDeviceState);

    switch(pDeviceState->npadSettingTypes)
    {
    case NpadSettingTypes_JoyRight:
        {
            nn::hid::SetNpadJoyAssignmentModeSingle(npadIds, nn::hid::NpadJoyDeviceType::NpadJoyDeviceType_Right);
        }
        break;
    case NpadSettingTypes_JoyLeft:
        {
            nn::hid::SetNpadJoyAssignmentModeSingle(npadIds, nn::hid::NpadJoyDeviceType::NpadJoyDeviceType_Left);
        }
        break;
    case NpadSettingTypes_JoyDual:
        {
            nn::hid::SetNpadJoyAssignmentModeDual(npadIds);
        }
        break;
    case NpadSettingTypes_FullKey:
        {
            nn::hid::SetNpadJoyAssignmentModeDual(npadIds);
        }
        break;
    case NpadSettingTypes_Handheld:
        {
            nn::hid::SetNpadJoyAssignmentModeDual(npadIds);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void UpdateInitial(DeviceState* pDeviceState, Npad::Id npadIds) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pDeviceState);

    // メニューの移動
    if ((pDeviceState->buttonDownSetJoy & Npad::Button::B::Mask).IsAnyOn()
        || (pDeviceState->buttonDownSetJoy & Npad::Button::Down::Mask).IsAnyOn())
    {
        pDeviceState->npadSettingTypes = static_cast<NpadSettingTypes>(static_cast<int>((pDeviceState->npadSettingTypes) + 1) % NpadSettingTypes_CountMax);
    };

    if ((pDeviceState->buttonDownSetJoy & Npad::Button::X::Mask).IsAnyOn()
        || (pDeviceState->buttonDownSetJoy & Npad::Button::Up::Mask).IsAnyOn())
    {
        pDeviceState->npadSettingTypes = static_cast<NpadSettingTypes>(static_cast<int>((pDeviceState->npadSettingTypes) + NpadSettingTypes_CountMax - 1) % NpadSettingTypes_CountMax);
    };

    // イベントトリガー
    nn::hid::NpadButtonSet buttonDownSetJoy = pDeviceState->buttonDownSetJoy;
    if ((buttonDownSetJoy & Npad::Button::R::Mask).IsAnyOn()
        || (buttonDownSetJoy & Npad::Button::L::Mask).IsAnyOn())
    {
        // Npad スタイルの設定
        UpdateNpadConfiguration(pDeviceState, npadIds);

        //IR カメラのハンドルの取得
        if (pDeviceState->padTypes == PadTypes_Npad)
        {
            pDeviceState->irCameraHandle = nn::irsensor::GetIrCameraHandle(npadIds);
        }
        //IR カメラの初期化
        nn::irsensor::Initialize(pDeviceState->irCameraHandle);

        //初期化完了状態に進む
        pDeviceState->irToolState = IrSensorApiTestToolState_Initialized;
    }
}

void UpdateInitialized(DeviceState* pDeviceState) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pDeviceState);

    // 状態の更新
    pDeviceState->irCameraStatus = nn::irsensor::GetIrCameraStatus(pDeviceState->irCameraHandle);
    pDeviceState->isCameraStatusValid = true;
    pDeviceState->irProcessorStatus = nn::irsensor::GetImageProcessorStatus(pDeviceState->irCameraHandle);
    pDeviceState->isProcessorStatusValid = true;

    nn::hid::NpadButtonSet buttonDownSetJoy = pDeviceState->buttonDownSetJoy;

    // メニューの移動
    int* pPadInitializedMenuSelect = &pDeviceState->padInitializedMenuSelect;
    if ((buttonDownSetJoy & Npad::Button::B::Mask).IsAnyOn()
        || (buttonDownSetJoy & Npad::Button::Down::Mask).IsAnyOn())
    {
        if (++*pPadInitializedMenuSelect == MenuInitializedLength)
        {
            *pPadInitializedMenuSelect = 0;
        }
    };

    if ((buttonDownSetJoy & Npad::Button::X::Mask).IsAnyOn()
        || (buttonDownSetJoy & Npad::Button::Up::Mask).IsAnyOn())
    {
        if (--*pPadInitializedMenuSelect < 0)
        {
            *pPadInitializedMenuSelect = MenuInitializedLength - 1;
        }
    };

    if ((buttonDownSetJoy & Npad::Button::R::Mask).IsAnyOn()
        || (buttonDownSetJoy & Npad::Button::L::Mask).IsAnyOn())
    {
        switch (*pPadInitializedMenuSelect)
        {
        case 0:
            {
                pDeviceState->irCameraStatus = nn::irsensor::GetIrCameraStatus(pDeviceState->irCameraHandle);
                pDeviceState->isCameraStatusValid = true;
                // カメラ設定に進む
                pDeviceState->irToolState = IrSensorApiTestToolState_CameraReady;
            }
            break;
        case 1:
            {
                pDeviceState->irProcessorStatus = nn::irsensor::GetImageProcessorStatus(pDeviceState->irCameraHandle);
                pDeviceState->isProcessorStatusValid = true;
                // カメラ設定に進む
                pDeviceState->irToolState = IrSensorApiTestToolState_CameraReady;
            }
            break;
        case 2:
            {
                // FirmwareUpdate Check を走らせる
                pDeviceState->firmwareUpdateTrialCount++;
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    if (pDeviceState->firmwareUpdateTrialCount > FirmwareUpdateRetryCountMax)
    {
        pDeviceState->firmwareUpdateTrialCount = 0;
        NN_LOG("FirmwareUpdate Timeout\n");
    }

    if (pDeviceState->firmwareUpdateTrialCount > 0)
    {
        bool isUpdateNeeded = false;
        nn::Result result = nn::irsensor::CheckFirmwareUpdateNecessity(&isUpdateNeeded, pDeviceState->irCameraHandle);
        if (result.IsSuccess())
        {
            if (isUpdateNeeded)
            {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
                // 選択 UI なしの強制アップデート版コンサポ呼び出し
                // 成功しても、失敗しても 1回だけ呼び出し。
                nn::hid::ControllerFirmwareUpdateArg arg;
                arg.enableForceUpdate = false;
                nn::hid::ShowControllerFirmwareUpdate(arg);
#elif defined( NN_BUILD_CONFIG_OS_WIN )
                NN_LOG("Invoke ControllerFirmwareUpdate is not suppported on Windows environment.\n");
#else
            #error "unsupported os"
#endif
            }
            pDeviceState->firmwareUpdateTrialCount = 0;
        }
        else if (::nn::irsensor::ResultIrsensorUnavailable::Includes(result))
        {
            pDeviceState->firmwareUpdateTrialCount = 0;
        }
        else if (::nn::irsensor::ResultIrsensorFirmwareCheckIncompleted::Includes(result))
        {
            // リトライする
            pDeviceState->firmwareUpdateTrialCount++;
        }
        else
        {
            NN_ABORT("Invalid Error\n");
        }
        pDeviceState->firmwareUpdateStatusResult = result;
    }
}

void UpdateRunning(DeviceState* pDeviceState, int index) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pDeviceState);
    NN_UNUSED(index);

    const int modeIndex = MenuOutputLength;
    const int configIndex = modeIndex + 4;

    static nn::os::Tick keyPressedTime = nn::os::Tick();
    static int counter = 0;

    nn::hid::NpadButtonSet buttonDownSetJoy = pDeviceState->buttonDownSetJoy;
    nn::hid::NpadButtonSet buttonDownSetJoyRaw = pDeviceState->buttonDownSetJoyRaw;
    int* pPadOutputMenuSelect = &pDeviceState->padOutputMenuSelect;
    IrState* pIrState = &pDeviceState->irState;
    nn::irsensor::IrCameraHandle handle = pDeviceState->irCameraHandle;

    // 状態の更新
    pDeviceState->irCameraStatus = nn::irsensor::GetIrCameraStatus(pDeviceState->irCameraHandle);
    pDeviceState->isCameraStatusValid = true;
    pDeviceState->irProcessorStatus = nn::irsensor::GetImageProcessorStatus(pDeviceState->irCameraHandle);
    pDeviceState->isProcessorStatusValid = true;

    // メニューの移動
    if ((buttonDownSetJoy & Npad::Button::B::Mask).IsAnyOn()
        || (buttonDownSetJoy & Npad::Button::Down::Mask).IsAnyOn())
    {
        if (++*pPadOutputMenuSelect == MenuOutputLength + MenuHeight[pDeviceState->irState.nextProcessor])
        {
            *pPadOutputMenuSelect = 0;
        }
    };

    if ((buttonDownSetJoy & Npad::Button::X::Mask).IsAnyOn()
        || (buttonDownSetJoy & Npad::Button::Up::Mask).IsAnyOn())
    {
        if (--*pPadOutputMenuSelect < 0)
        {
            *pPadOutputMenuSelect = MenuOutputLength + MenuHeight[pDeviceState->irState.nextProcessor] - 1;
        }
    };

    // 値を増加
    if ((buttonDownSetJoyRaw & Npad::Button::Y::Mask).IsAnyOn() || (buttonDownSetJoyRaw & Npad::Button::A::Mask).IsAnyOn()
        || (buttonDownSetJoyRaw & Npad::Button::Left::Mask).IsAnyOn() || (buttonDownSetJoyRaw & Npad::Button::Right::Mask).IsAnyOn())
    {
        int8_t increment = ((buttonDownSetJoyRaw & Npad::Button::A::Mask).IsAnyOn()
            || (buttonDownSetJoyRaw & Npad::Button::Right::Mask).IsAnyOn()) ? 1 : -1;

        nn::irsensor::IrCameraConfig* pConfig;
        int64_t timeMin, timeMax;

        if (pIrState->nextProcessor == IrSensorMode_Moment)
        {
            pConfig = &pIrState->momentProcessorConfig.irCameraConfig;
            timeMin = nn::irsensor::MomentProcessorExposureTimeMin.GetMicroSeconds();
            timeMax = nn::irsensor::MomentProcessorExposureTimeMax.GetMicroSeconds();
        }
        else if (pIrState->nextProcessor == IrSensorMode_Clustering)
        {
            pConfig = &pIrState->clusteringProcessorConfig.irCameraConfig;
            timeMin = nn::irsensor::ClusteringProcessorExposureTimeMin.GetMicroSeconds();
            timeMax = nn::irsensor::ClusteringProcessorExposureTimeMax.GetMicroSeconds();
        }
        else if (pIrState->nextProcessor == IrSensorMode_ImageTransfer)
        {
            pConfig = &pIrState->imageTransferProcessorConfigNext.irCameraConfig;
            timeMin = nn::irsensor::ImageTransferProcessorExposureTimeMin.GetMicroSeconds();
            timeMax = nn::irsensor::ImageTransferProcessorExposureTimeMax.GetMicroSeconds();
        }
        else if (pIrState->nextProcessor == IrSensorMode_Pointing)
        {
            // Pointing では設定しないため適当に初期化しておく
            pConfig = nullptr;
            timeMin = 0;
            timeMax = 0;
        }
        else if (pIrState->nextProcessor == IrSensorMode_HandAnalysis)
        {
            pConfig = nullptr;
        }
        else if (pIrState->nextProcessor == IrSensorMode_IrLed)
        {
            pConfig = nullptr;
        }
        else if (pIrState->nextProcessor == IrSensorMode_AdaptiveClustering)
        {
            pConfig = nullptr;
        }
        else
        {
            return;
        }

        if ((nn::os::GetSystemTick() - keyPressedTime).ToTimeSpan().GetMilliSeconds() > 50)
        {
            counter = 0;
        }

        if (counter > 50)
        {
            if (*pPadOutputMenuSelect == modeIndex + 1)
            {
                // Exposure Time を変更する
                pConfig->exposureTime += nn::TimeSpanType::FromMicroSeconds(increment);
                pConfig->exposureTime = nn::TimeSpanType::FromMicroSeconds(std::min(
                    std::max(pConfig->exposureTime.GetMicroSeconds(), timeMin),
                    timeMax));
            }

            if (pIrState->nextProcessor == IrSensorMode_Moment)
            {
                int indexNum = *pPadOutputMenuSelect - MenuOutputLength;
                switch(indexNum)
                {
                case 6:
                    // Intensity Threshold を変更する
                    pIrState->momentProcessorConfig.preprocessIntensityThreshold += increment;
                    pIrState->momentProcessorConfig.preprocessIntensityThreshold = std::min(
                        std::max(pIrState->momentProcessorConfig.preprocessIntensityThreshold, 0),
                        nn::irsensor::IrCameraIntensityMax);
                    break;
                case 7:
                    // WOI.X を変更する
                    pIrState->momentProcessorConfig.windowOfInterest.x += increment;
                    pIrState->momentProcessorConfig.windowOfInterest.x = static_cast<int16_t>(std::min(
                        std::max(static_cast<int>(pIrState->momentProcessorConfig.windowOfInterest.x), 0),
                        nn::irsensor::IrCameraImageWidth - pIrState->momentProcessorConfig.windowOfInterest.width));
                    break;
                case 8:
                    // WOI.Y を変更する
                    pIrState->momentProcessorConfig.windowOfInterest.y += increment;
                    pIrState->momentProcessorConfig.windowOfInterest.y = static_cast<int16_t>(std::min(
                        std::max(static_cast<int>(pIrState->momentProcessorConfig.windowOfInterest.y), 0),
                        nn::irsensor::IrCameraImageHeight - pIrState->momentProcessorConfig.windowOfInterest.height));
                    break;
                case 9:
                    // WOI.Width を変更する
                    pIrState->momentProcessorConfig.windowOfInterest.width += increment;
                    pIrState->momentProcessorConfig.windowOfInterest.width = static_cast<int16_t>(std::min(
                        std::max(static_cast<int>(pIrState->momentProcessorConfig.windowOfInterest.width), 1),
                        nn::irsensor::IrCameraImageWidth - pIrState->momentProcessorConfig.windowOfInterest.x));
                    break;
                case 10:
                    // WOI.Height を変更する
                    pIrState->momentProcessorConfig.windowOfInterest.height += increment;
                    pIrState->momentProcessorConfig.windowOfInterest.height = static_cast<int16_t>(std::min(
                        std::max(static_cast<int>(pIrState->momentProcessorConfig.windowOfInterest.height), 1),
                        nn::irsensor::IrCameraImageHeight - pIrState->momentProcessorConfig.windowOfInterest.y));
                    break;
                default:
                    break;
                }
            }

            if (pIrState->nextProcessor == IrSensorMode_Clustering)
            {
                int indexNum = *pPadOutputMenuSelect - MenuOutputLength;
                switch (indexNum)
                {
                case 5:
                    {
                        // Object Pixel Count Min を変更する
                        pIrState->clusteringProcessorConfig.objectPixelCountMin += increment * 100;
                        pIrState->clusteringProcessorConfig.objectPixelCountMin = std::min(
                            std::max(pIrState->clusteringProcessorConfig.objectPixelCountMin, 0),
                            pIrState->clusteringProcessorConfig.objectPixelCountMax - 1);
                    }
                    break;
                case 6:
                    {
                        // Object Pixel Count Max を変更する
                        pIrState->clusteringProcessorConfig.objectPixelCountMax += increment * 100;
                        pIrState->clusteringProcessorConfig.objectPixelCountMax = std::max(
                            std::min(pIrState->clusteringProcessorConfig.objectPixelCountMax, nn::irsensor::IrCameraImageWidth * nn::irsensor::IrCameraImageHeight),
                            pIrState->clusteringProcessorConfig.objectPixelCountMin + 1);
                    }
                    break;
                case 7:
                    {
                        // Object Intensity Min を変更する
                        pIrState->clusteringProcessorConfig.objectIntensityMin += increment;
                        pIrState->clusteringProcessorConfig.objectIntensityMin =
                            std::max(std::min(pIrState->clusteringProcessorConfig.objectIntensityMin, nn::irsensor::IrCameraIntensityMax), 0);
                    }
                    break;
                case 9:
                    {
                        // WOI.X を変更する
                        pIrState->clusteringProcessorConfig.windowOfInterest.x += increment;
                        pIrState->clusteringProcessorConfig.windowOfInterest.x = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->clusteringProcessorConfig.windowOfInterest.x), 0),
                            nn::irsensor::IrCameraImageWidth - pIrState->clusteringProcessorConfig.windowOfInterest.width));
                    }
                    break;
                case 10:
                    {
                        // WOI.Y を変更する
                        pIrState->clusteringProcessorConfig.windowOfInterest.y += increment;
                        pIrState->clusteringProcessorConfig.windowOfInterest.y = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->clusteringProcessorConfig.windowOfInterest.y), 0),
                            nn::irsensor::IrCameraImageHeight - pIrState->clusteringProcessorConfig.windowOfInterest.height));
                    }
                    break;
                case 11:
                    {
                        // WOI.Width を変更する
                        pIrState->clusteringProcessorConfig.windowOfInterest.width += increment;
                        pIrState->clusteringProcessorConfig.windowOfInterest.width = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->clusteringProcessorConfig.windowOfInterest.width), 1),
                            nn::irsensor::IrCameraImageWidth - pIrState->clusteringProcessorConfig.windowOfInterest.x));
                        if (pIrState->clusteringProcessorConfig.windowOfInterest.width <= 1
                            && pIrState->clusteringProcessorConfig.windowOfInterest.height <= 20)
                        {
                            // ハードバグの禁則
                            pIrState->clusteringProcessorConfig.windowOfInterest.width -= increment;
                        }
                    }
                    break;
                case 12:
                    {
                        // WOI.Height を変更する
                        pIrState->clusteringProcessorConfig.windowOfInterest.height += increment;
                        pIrState->clusteringProcessorConfig.windowOfInterest.height = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->clusteringProcessorConfig.windowOfInterest.height), 1),
                            nn::irsensor::IrCameraImageHeight - pIrState->clusteringProcessorConfig.windowOfInterest.y));
                        if (pIrState->clusteringProcessorConfig.windowOfInterest.width <= 1
                            && pIrState->clusteringProcessorConfig.windowOfInterest.height <= 20)
                        {
                            // ハードバグの禁則
                            pIrState->clusteringProcessorConfig.windowOfInterest.height -= increment;
                        }
                    }
                    break;
                default:
                    // Donothing
                    break;
                }
            }

            if (pIrState->nextProcessor == IrSensorMode_ImageTransfer)
            {
                int indexNum = *pPadOutputMenuSelect - MenuOutputLength;
                switch (indexNum)
                {
                case 7:
                    {
                        // StartX を変更する
                        pIrState->imageTransferProcessorConfigNext.trimmingStartX += increment;
                        pIrState->imageTransferProcessorConfigNext.trimmingStartX = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->imageTransferProcessorConfigNext.trimmingStartX), 0),
                            (nn::irsensor::IrCameraImageWidth >> pIrState->imageTransferProcessorConfigNext.origFormat)
                            - (nn::irsensor::IrCameraImageWidth >> pIrState->imageTransferProcessorConfigNext.trimmingFormat)));
                    }
                    break;
                case 8:
                    {
                        // StartY を変更する
                        pIrState->imageTransferProcessorConfigNext.trimmingStartY += increment;
                        pIrState->imageTransferProcessorConfigNext.trimmingStartY = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->imageTransferProcessorConfigNext.trimmingStartY), 0),
                            (nn::irsensor::IrCameraImageHeight >> pIrState->imageTransferProcessorConfigNext.origFormat)
                            - (nn::irsensor::IrCameraImageHeight >> pIrState->imageTransferProcessorConfigNext.trimmingFormat)));
                    }
                    break;
                default:
                    // Donothing
                    break;
                }
            }
        }
        keyPressedTime = nn::os::GetSystemTick();
        counter++;
    }

    if ((buttonDownSetJoy & Npad::Button::Y::Mask).IsAnyOn() || (buttonDownSetJoy & Npad::Button::A::Mask).IsAnyOn()
        || (buttonDownSetJoy & Npad::Button::Left::Mask).IsAnyOn() || (buttonDownSetJoy & Npad::Button::Right::Mask).IsAnyOn())
    {

        int8_t increment = ((buttonDownSetJoy & Npad::Button::A::Mask).IsAnyOn()
            || (buttonDownSetJoy & Npad::Button::Right::Mask).IsAnyOn()) ? 1 : -1;

        nn::irsensor::IrCameraConfig* pConfig;
        int64_t timeMin, timeMax;

        if (pIrState->nextProcessor == IrSensorMode_Moment)
        {
            pConfig = &pIrState->momentProcessorConfig.irCameraConfig;
            timeMin = nn::irsensor::MomentProcessorExposureTimeMin.GetMicroSeconds();
            timeMax = nn::irsensor::MomentProcessorExposureTimeMax.GetMicroSeconds();
        }
        else if (pIrState->nextProcessor == IrSensorMode_Clustering)
        {
            pConfig = &pIrState->clusteringProcessorConfig.irCameraConfig;
            timeMin = nn::irsensor::ClusteringProcessorExposureTimeMin.GetMicroSeconds();
            timeMax = nn::irsensor::ClusteringProcessorExposureTimeMax.GetMicroSeconds();
        }
        else if (pIrState->nextProcessor == IrSensorMode_ImageTransfer)
        {
            pConfig = &pIrState->imageTransferProcessorConfigNext.irCameraConfig;
            timeMin = nn::irsensor::ImageTransferProcessorExposureTimeMin.GetMicroSeconds();
            timeMax = nn::irsensor::ImageTransferProcessorExposureTimeMax.GetMicroSeconds();
        }
        else if (pIrState->nextProcessor == IrSensorMode_Pointing)
        {
            // Pointing では設定しないため適当に初期化しておく
            pConfig = nullptr;
            timeMin = 0;
            timeMax = 0;
        }
        else if (pIrState->nextProcessor == IrSensorMode_HandAnalysis)
        {
            // 設定しないため適当に初期化しておく
            pConfig = nullptr;
        }
        else if (pIrState->nextProcessor == IrSensorMode_IrLed)
        {
            // 設定しないため適当に初期化しておく
            pConfig = nullptr;
        }
        else if (pIrState->nextProcessor == IrSensorMode_AdaptiveClustering)
        {
            // 設定しないため適当に初期化しておく
            pConfig = nullptr;
        }
        else
        {
            return;
        }

        // モードの切り替え
        if (*pPadOutputMenuSelect == modeIndex)
        {
            int8_t mode = static_cast<int8_t>(pIrState->nextProcessor) + increment;
            pIrState->nextProcessor = static_cast<IrSensorMode>(std::min(
                std::max(mode, static_cast<int8_t>(IrSensorMode_Moment)),
                static_cast<int8_t>(IrSensorMode_Count - 1)));
        }

        if (*pPadOutputMenuSelect <= configIndex)
        {
            int8_t lightTarget;

            switch (*pPadOutputMenuSelect)
            {
            case 0:
                {
                    // 自動出力 or 1shot
                    pDeviceState->outMode = static_cast<OutputMode>(
                        (static_cast<int>(pDeviceState->outMode) + OutputMode_CountMax + increment) % OutputMode_CountMax);
                }
                break;
            case 1:
                {
                    if (increment > 0)
                    {
                        if ((pDeviceState->irState.currentProcessor != IrSensorMode_ImageTransfer)
                            && (pDeviceState->irState.currentProcessor != IrSensorMode_HandAnalysis)
                            && (pDeviceState->irState.currentProcessor != IrSensorMode_IrLed))
                        {
                            // GetState or GetStates
                            pDeviceState->outApiTypes = OutputApiTypes_States;
                        }
                        else
                        {
                            pDeviceState->outApiTypes = OutputApiTypes_State;
                            pDeviceState->outCountMax = 1;
                        }
                    }
                    else
                    {
                        if (pDeviceState->irState.currentProcessor != IrSensorMode_Pointing
                            && pDeviceState->irState.currentProcessor != IrSensorMode_AdaptiveClustering)
                        {
                            // GetState or GetStates
                            pDeviceState->outApiTypes = OutputApiTypes_State;
                            pDeviceState->outCountMax = 1;
                        }
                        else
                        {
                            pDeviceState->outApiTypes = OutputApiTypes_States;
                        }
                    }
                }
                break;
            case 2:
                {
                    if (pDeviceState->outApiTypes == OutputApiTypes_States)
                    {
                        pDeviceState->outCountMax += increment;
                        pDeviceState->outCountMax = std::max(pDeviceState->outCountMax, 0);
                    }
                }
                break;
            case 3:
                {
                    if ((pDeviceState->irState.currentProcessor == IrSensorMode_Pointing)
                        || (pDeviceState->irState.currentProcessor == IrSensorMode_ImageTransfer))
                    {
                        int outputLevel = static_cast<int>(pDeviceState->outputLevelNext) + increment;
                        outputLevel = std::max(
                            std::min(outputLevel, static_cast<int>(OutputLevel_Detail)),
                            static_cast<int>(OutputLevel_Normal));
                        pDeviceState->outputLevelNext = static_cast<OutputLevel>(outputLevel);
                    }
                    else
                    {
                        pDeviceState->outputLevelNext = OutputLevel_Normal;
                    }
                }
                break;
            case 5:
                {
                    if (pIrState->nextProcessor == IrSensorMode_HandAnalysis)
                    {
                        // HandAnalysis のモードを切り替える
                        int8_t mode = static_cast<int8_t>(pIrState->handAnalysisConfigNext.mode) + increment;
                        mode = std::max(
                            std::min(mode, static_cast<int8_t>(nn::irsensor::HandAnalysisMode_SilhouetteOnly)),
                            static_cast<int8_t>(nn::irsensor::HandAnalysisMode_Silhouette));
                        pIrState->handAnalysisConfigNext.mode = static_cast<nn::irsensor::HandAnalysisMode>(mode);
                    }
                    else if (pIrState->nextProcessor == IrSensorMode_IrLed)
                    {
                        // Light Target を変更する
                        lightTarget = static_cast<int8_t>(pIrState->irLedProcessorConfig.lightTarget) + increment;
                        lightTarget = std::max(
                            std::min(lightTarget, static_cast<int8_t>(nn::irsensor::IrCameraLightTarget_None)),
                            static_cast<int8_t>(nn::irsensor::IrCameraLightTarget_AllObjects));
                        pIrState->irLedProcessorConfig.lightTarget = static_cast<nn::irsensor::IrCameraLightTarget>(lightTarget);
                    }
                    else if (pIrState->nextProcessor == IrSensorMode_AdaptiveClustering)
                    {
                        // AdaptiveClustering のモードを変更する
                        int8_t mode = static_cast<int8_t>(pIrState->adaptiveClusteringProcessorConfig.mode) + increment;
                        mode = std::max(
                            std::min(mode, static_cast<int8_t>(nn::irsensor::AdaptiveClusteringMode_DynamicFov)),
                            static_cast<int8_t>(nn::irsensor::AdaptiveClusteringMode_StaticFov));
                        pIrState->adaptiveClusteringProcessorConfig.mode = static_cast<nn::irsensor::AdaptiveClusteringMode>(mode);
                    }
                    else
                    {
                        pConfig->exposureTime += nn::TimeSpanType::FromMicroSeconds(increment);

                        pConfig->exposureTime = nn::TimeSpanType::FromMicroSeconds(std::min(
                            std::max(pConfig->exposureTime.GetMicroSeconds(), timeMin),
                            timeMax));
                    }
                }
                break;
            case 6:
                // Light Target を変更する
                lightTarget = static_cast<int8_t>(pConfig->lightTarget) + increment;
                lightTarget = std::max(
                    std::min(lightTarget, static_cast<int8_t>(nn::irsensor::IrCameraLightTarget_None)),
                    static_cast<int8_t>(nn::irsensor::IrCameraLightTarget_AllObjects));
                pConfig->lightTarget = static_cast<nn::irsensor::IrCameraLightTarget>(lightTarget);
                break;
            case 7:
                // Digital Gain を変更する
                pConfig->gain += increment;
                pConfig->gain = std::min(
                    std::max(pConfig->gain, nn::irsensor::IrCameraGainMin),
                    nn::irsensor::IrCameraGainMax);
                break;
            case 8:
                // Negative Image を変更する
                pConfig->isNegativeImageUsed = !pConfig->isNegativeImageUsed;
                break;
            default:
                break;
            }
        }
        else
        {
            if (pIrState->nextProcessor == IrSensorMode_Moment)
            {
                int indexNum = *pPadOutputMenuSelect - MenuOutputLength;
                switch (indexNum)
                {
                case 5:
                    // PreProcess を変更する
                    pIrState->momentProcessorConfig.preprocess =
                        (pIrState->momentProcessorConfig.preprocess == nn::irsensor::MomentProcessorPreprocess_Binarize) ?
                        nn::irsensor::MomentProcessorPreprocess_Cutoff : nn::irsensor::MomentProcessorPreprocess_Binarize;
                    break;
                case 6:
                    // Intensity Threshold を変更する
                    pIrState->momentProcessorConfig.preprocessIntensityThreshold += increment;
                    pIrState->momentProcessorConfig.preprocessIntensityThreshold = std::min(
                        std::max(pIrState->momentProcessorConfig.preprocessIntensityThreshold, 0),
                        nn::irsensor::IrCameraIntensityMax);
                    break;
                case 7:
                    // WOI.X を変更する
                    pIrState->momentProcessorConfig.windowOfInterest.x += increment;
                    pIrState->momentProcessorConfig.windowOfInterest.x = static_cast<int16_t>(std::min(
                        std::max(static_cast<int>(pIrState->momentProcessorConfig.windowOfInterest.x), 0),
                        nn::irsensor::IrCameraImageWidth - pIrState->momentProcessorConfig.windowOfInterest.width));
                    break;
                case 8:
                    // WOI.Y を変更する
                    pIrState->momentProcessorConfig.windowOfInterest.y += increment;
                    pIrState->momentProcessorConfig.windowOfInterest.y = static_cast<int16_t>(std::min(
                        std::max(static_cast<int>(pIrState->momentProcessorConfig.windowOfInterest.y), 0),
                        nn::irsensor::IrCameraImageHeight - pIrState->momentProcessorConfig.windowOfInterest.height));
                    break;
                case 9:
                    // WOI.Width を変更する
                    pIrState->momentProcessorConfig.windowOfInterest.width += increment;
                    pIrState->momentProcessorConfig.windowOfInterest.width = static_cast<int16_t>(std::min(
                        std::max(static_cast<int>(pIrState->momentProcessorConfig.windowOfInterest.width), 1),
                        nn::irsensor::IrCameraImageWidth - pIrState->momentProcessorConfig.windowOfInterest.x));
                    break;
                case 10:
                    // WOI.Height を変更する
                    pIrState->momentProcessorConfig.windowOfInterest.height += increment;
                    pIrState->momentProcessorConfig.windowOfInterest.height = static_cast<int16_t>(std::min(
                        std::max(static_cast<int>(pIrState->momentProcessorConfig.windowOfInterest.height), 1),
                        nn::irsensor::IrCameraImageHeight - pIrState->momentProcessorConfig.windowOfInterest.y));
                    break;
                default:
                    break;
                }
            }
            else if (pIrState->nextProcessor == IrSensorMode_Clustering)
            {
                int indexNum = *pPadOutputMenuSelect - MenuOutputLength;
                switch (indexNum)
                {
                case 5:
                    // Object Pixel Count Min を変更する
                    pIrState->clusteringProcessorConfig.objectPixelCountMin += increment;
                    pIrState->clusteringProcessorConfig.objectPixelCountMin = std::min(
                        std::max(pIrState->clusteringProcessorConfig.objectPixelCountMin, 0),
                        pIrState->clusteringProcessorConfig.objectPixelCountMax - 1);
                    break;
                case 6:
                    // Object Pixel Count Max を変更する
                    pIrState->clusteringProcessorConfig.objectPixelCountMax += increment;
                    pIrState->clusteringProcessorConfig.objectPixelCountMax = std::max(
                        std::min(pIrState->clusteringProcessorConfig.objectPixelCountMax, nn::irsensor::IrCameraImageWidth * nn::irsensor::IrCameraImageHeight),
                        pIrState->clusteringProcessorConfig.objectPixelCountMin + 1);
                    break;
                case 7:
                    // Object Intensity Min を変更する
                    pIrState->clusteringProcessorConfig.objectIntensityMin += increment;
                    pIrState->clusteringProcessorConfig.objectIntensityMin =
                        std::max(std::min(pIrState->clusteringProcessorConfig.objectIntensityMin, nn::irsensor::IrCameraIntensityMax), 0);
                    break;
                case 8:
                    // 外部光フィルタ機能の有効無効を切り替える
                    pIrState->clusteringProcessorConfig.isExternalLightFilterEnabled = !pIrState->clusteringProcessorConfig.isExternalLightFilterEnabled;
                    break;
                case 9:
                    // WOI.X を変更する
                    pIrState->clusteringProcessorConfig.windowOfInterest.x += increment;
                    pIrState->clusteringProcessorConfig.windowOfInterest.x = static_cast<int16_t>(std::min(
                        std::max(static_cast<int>(pIrState->clusteringProcessorConfig.windowOfInterest.x), 0),
                        nn::irsensor::IrCameraImageWidth - pIrState->clusteringProcessorConfig.windowOfInterest.width));
                    break;
                case 10:
                    // WOI.Y を変更する
                    pIrState->clusteringProcessorConfig.windowOfInterest.y += increment;
                    pIrState->clusteringProcessorConfig.windowOfInterest.y = static_cast<int16_t>(std::min(
                        std::max(static_cast<int>(pIrState->clusteringProcessorConfig.windowOfInterest.y), 0),
                        nn::irsensor::IrCameraImageHeight - pIrState->clusteringProcessorConfig.windowOfInterest.height));
                    break;
                case 11:
                    // WOI.Width を変更する
                    pIrState->clusteringProcessorConfig.windowOfInterest.width += increment;
                    pIrState->clusteringProcessorConfig.windowOfInterest.width = static_cast<int16_t>(std::min(
                        std::max(static_cast<int>(pIrState->clusteringProcessorConfig.windowOfInterest.width), 1),
                        nn::irsensor::IrCameraImageWidth - pIrState->clusteringProcessorConfig.windowOfInterest.x));
                    if (pIrState->clusteringProcessorConfig.windowOfInterest.width <= 1
                        && pIrState->clusteringProcessorConfig.windowOfInterest.height <= 20)
                    {
                        // ハードバグの禁則
                        pIrState->clusteringProcessorConfig.windowOfInterest.width -= increment;
                    }
                    break;
                case 12:
                    // WOI.Height を変更する
                    pIrState->clusteringProcessorConfig.windowOfInterest.height += increment;
                    pIrState->clusteringProcessorConfig.windowOfInterest.height = static_cast<int16_t>(std::min(
                        std::max(static_cast<int>(pIrState->clusteringProcessorConfig.windowOfInterest.height), 1),
                        nn::irsensor::IrCameraImageHeight - pIrState->clusteringProcessorConfig.windowOfInterest.y));
                    if (pIrState->clusteringProcessorConfig.windowOfInterest.width <= 1
                        && pIrState->clusteringProcessorConfig.windowOfInterest.height <= 20)
                    {
                        // ハードバグの禁則
                        pIrState->clusteringProcessorConfig.windowOfInterest.height -= increment;
                    }
                    break;
                default:
                    break;
                }
            }
            else if (pIrState->nextProcessor == IrSensorMode_ImageTransfer)
            {
                int indexNum = *pPadOutputMenuSelect - MenuOutputLength;
                switch (indexNum)
                {
                case 5:
                    {
                        // 切り取り元の画像フォーマットを変更する
                        int8_t format = static_cast<int8_t>(pIrState->imageTransferProcessorConfigNext.origFormat) + increment;
                        format = std::max(
                            std::min(format, static_cast<int8_t>(nn::irsensor::ImageTransferProcessorFormat_20x15)),
                            static_cast<int8_t>(nn::irsensor::ImageTransferProcessorFormat_320x240));
                        pIrState->imageTransferProcessorConfigNext.origFormat = static_cast<nn::irsensor::ImageTransferProcessorFormat>(format);

                        // 必要に応じて出力画像フォーマットを合わせる
                        pIrState->imageTransferProcessorConfigNext.trimmingFormat = static_cast<nn::irsensor::ImageTransferProcessorFormat>(std::max(
                            std::min(static_cast<int8_t>(pIrState->imageTransferProcessorConfigNext.trimmingFormat),
                                static_cast<int8_t>(nn::irsensor::ImageTransferProcessorFormat_20x15)),
                            static_cast<int8_t>(pIrState->imageTransferProcessorConfigNext.origFormat)));
                        // サイズの更新
                        pIrState->imageTransferProcessorConfigNext.trimmingStartX = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->imageTransferProcessorConfigNext.trimmingStartX), 0),
                            (nn::irsensor::IrCameraImageWidth >> pIrState->imageTransferProcessorConfigNext.origFormat)
                            - (nn::irsensor::IrCameraImageWidth >> pIrState->imageTransferProcessorConfigNext.trimmingFormat)));
                        pIrState->imageTransferProcessorConfigNext.trimmingStartY = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->imageTransferProcessorConfigNext.trimmingStartY), 0),
                            (nn::irsensor::IrCameraImageHeight >> pIrState->imageTransferProcessorConfigNext.origFormat)
                            - (nn::irsensor::IrCameraImageHeight >> pIrState->imageTransferProcessorConfigNext.trimmingFormat)));
                    }
                    break;
                case 6:
                    {
                        // 画像フォーマットを変更する
                        int8_t format = static_cast<int8_t>(pIrState->imageTransferProcessorConfigNext.trimmingFormat) + increment;
                        format = std::max(
                            std::min(format, static_cast<int8_t>(nn::irsensor::ImageTransferProcessorFormat_20x15)),
                            static_cast<int8_t>(nn::irsensor::ImageTransferProcessorFormat_320x240));
                        pIrState->imageTransferProcessorConfigNext.trimmingFormat = static_cast<nn::irsensor::ImageTransferProcessorFormat>(format);

                        // 必要に応じて切り取り画像フォーマットを合わせる
                        pIrState->imageTransferProcessorConfigNext.origFormat = static_cast<nn::irsensor::ImageTransferProcessorFormat>(std::max(
                            std::min(static_cast<int8_t>(pIrState->imageTransferProcessorConfigNext.origFormat),
                                static_cast<int8_t>(pIrState->imageTransferProcessorConfigNext.trimmingFormat)),
                            static_cast<int8_t>(nn::irsensor::ImageTransferProcessorFormat_320x240)));
                        // サイズの更新
                        pIrState->imageTransferProcessorConfigNext.trimmingStartX = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->imageTransferProcessorConfigNext.trimmingStartX), 0),
                            (nn::irsensor::IrCameraImageWidth >> pIrState->imageTransferProcessorConfigNext.origFormat)
                            - (nn::irsensor::IrCameraImageWidth >> pIrState->imageTransferProcessorConfigNext.trimmingFormat)));
                        pIrState->imageTransferProcessorConfigNext.trimmingStartY = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->imageTransferProcessorConfigNext.trimmingStartY), 0),
                            (nn::irsensor::IrCameraImageHeight >> pIrState->imageTransferProcessorConfigNext.origFormat)
                            - (nn::irsensor::IrCameraImageHeight >> pIrState->imageTransferProcessorConfigNext.trimmingFormat)));
                    }
                    break;
                case 7:
                    {
                        // StartX を変更する
                        pIrState->imageTransferProcessorConfigNext.trimmingStartX += increment;
                        pIrState->imageTransferProcessorConfigNext.trimmingStartX = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->imageTransferProcessorConfigNext.trimmingStartX), 0),
                            (nn::irsensor::IrCameraImageWidth >> pIrState->imageTransferProcessorConfigNext.origFormat)
                            - (nn::irsensor::IrCameraImageWidth >> pIrState->imageTransferProcessorConfigNext.trimmingFormat)));
                    }
                    break;
                case 8:
                    {
                        // StartY を変更する
                        pIrState->imageTransferProcessorConfigNext.trimmingStartY += increment;
                        pIrState->imageTransferProcessorConfigNext.trimmingStartY = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->imageTransferProcessorConfigNext.trimmingStartY), 0),
                            (nn::irsensor::IrCameraImageHeight >> pIrState->imageTransferProcessorConfigNext.origFormat)
                            - (nn::irsensor::IrCameraImageHeight >> pIrState->imageTransferProcessorConfigNext.trimmingFormat)));
                    }
                    break;
                case 9:
                    {
                        pIrState->imageTransferProcessorConfigNext.isExternalLightFilterEnabled = !pIrState->imageTransferProcessorConfigNext.isExternalLightFilterEnabled;
                    }
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }
            }
        }

    }

    // one shot 用のフラグ
    bool isClicked = false;

    // 出力モードの設定 or ContinuousRun の発行
    if ((buttonDownSetJoy & Npad::Button::R::Mask).IsAnyOn()
        || (buttonDownSetJoy & Npad::Button::L::Mask).IsAnyOn())
    {
        isClicked = true;
        if (*pPadOutputMenuSelect < modeIndex)
        {
            pDeviceState->outCurrentMode = pDeviceState->outMode;
            pDeviceState->outCurrentApiTypes = pDeviceState->outApiTypes;
            pDeviceState->outCurrentCountMax = pDeviceState->outCountMax;
            pDeviceState->outputLevelCurrent = pDeviceState->outputLevelNext;
        }
        else
        { // ContinuousRun を発行// モードの切り替え (レジスタ更新を含む)
            // Run 前にデータをクリアしておく
            ClearIrStateData(&pDeviceState->irState);

            if (pIrState->nextProcessor == IrSensorMode_Moment)
            {
                nn::irsensor::RunMomentProcessor(handle, pIrState->momentProcessorConfig);
                // WOI の更新
                pIrState->momentProcessorWoi = pIrState->momentProcessorConfig.windowOfInterest;
            }
            else if (pIrState->nextProcessor == IrSensorMode_Clustering)
            {
                nn::irsensor::RunClusteringProcessor(handle, pIrState->clusteringProcessorConfig);
                // WOI の更新
                pIrState->clusteringProcessorWoi = pIrState->clusteringProcessorConfig.windowOfInterest;
            }
            else if (pIrState->nextProcessor == IrSensorMode_ImageTransfer)
            {
                size_t bufferSize = 0;
                if (pIrState->imageTransferProcessorConfigNext.trimmingFormat == nn::irsensor::ImageTransferProcessorFormat_320x240)
                {
                    bufferSize = nn::irsensor::ImageTransferProcessorWorkBufferSize320x240;
                }
                else if (pIrState->imageTransferProcessorConfigNext.trimmingFormat == nn::irsensor::ImageTransferProcessorFormat_160x120)
                {
                    bufferSize = nn::irsensor::ImageTransferProcessorWorkBufferSize160x120;
                }
                else if (pIrState->imageTransferProcessorConfigNext.trimmingFormat == nn::irsensor::ImageTransferProcessorFormat_80x60)
                {
                    bufferSize = nn::irsensor::ImageTransferProcessorWorkBufferSize80x60;
                }
                else if (pIrState->imageTransferProcessorConfigNext.trimmingFormat == nn::irsensor::ImageTransferProcessorFormat_40x30)
                {
                    bufferSize = nn::irsensor::ImageTransferProcessorWorkBufferSize40x30;
                }
                else if (pIrState->imageTransferProcessorConfigNext.trimmingFormat == nn::irsensor::ImageTransferProcessorFormat_20x15)
                {
                    bufferSize = nn::irsensor::ImageTransferProcessorWorkBufferSize20x15;
                }

                // Run 前にバッファをクリアしておく
                uint8_t* destinationPixels = pDeviceState->cameraViewer.copiedImageBuffer.Map< uint8_t >();
                memset(destinationPixels, 0, nn::irsensor::ImageTransferProcessorWorkBufferSize320x240);
                pDeviceState->cameraViewer.copiedImageBuffer.FlushMappedRange(0, nn::irsensor::ImageTransferProcessorWorkBufferSize320x240);
                pDeviceState->cameraViewer.copiedImageBuffer.Unmap();

                nn::irsensor::RunImageTransferProcessor(
                    handle,
                    pIrState->imageTransferProcessorConfigNext,
                    static_cast<void*>(pIrState->imageTransferWorkBuffer),
                    bufferSize);

                pIrState->imageTransferProcessorConfigCurrent = pIrState->imageTransferProcessorConfigNext;
                pIrState->format = pIrState->imageTransferProcessorConfigCurrent.trimmingFormat;
            }
            else if (pIrState->nextProcessor == IrSensorMode_Pointing)
            {
                nn::irsensor::RunPointingProcessor(handle);
            }
            else if (pIrState->nextProcessor == IrSensorMode_HandAnalysis)
            {
                // Run 前にバッファをクリアしておく
                uint8_t* destinationPixels = pDeviceState->cameraViewer.copiedImageBuffer.Map< uint8_t >();
                memset(destinationPixels, 0, nn::irsensor::ImageTransferProcessorWorkBufferSize320x240);
                pDeviceState->cameraViewer.copiedImageBuffer.FlushMappedRange(0, nn::irsensor::ImageTransferProcessorWorkBufferSize320x240);
                pDeviceState->cameraViewer.copiedImageBuffer.Unmap();

                nn::irsensor::RunHandAnalysis(handle, pIrState->handAnalysisConfigNext);
                pIrState->handAnalysisConfigCurrent = pIrState->handAnalysisConfigNext;
            }
            else if (pIrState->nextProcessor == IrSensorMode_IrLed)
            {
                nn::irsensor::RunIrLedProcessor(handle, pIrState->irLedProcessorConfig);
            }
            else if (pIrState->nextProcessor == IrSensorMode_AdaptiveClustering)
            {
                nn::irsensor::RunAdaptiveClusteringProcessor(handle, pIrState->adaptiveClusteringProcessorConfig);
            }

            pIrState->currentProcessor = pIrState->nextProcessor;

            if (pIrState->nextProcessor == IrSensorMode_Pointing
                || pIrState->nextProcessor == IrSensorMode_AdaptiveClustering)
            {
                pDeviceState->outApiTypes = OutputApiTypes_States;
                pDeviceState->outCountMax = 3;
            }
            else if (pIrState->nextProcessor == IrSensorMode_ImageTransfer)
            {
                pDeviceState->outApiTypes = OutputApiTypes_State;
                pDeviceState->outCountMax = 1;
            }
            else
            {
                pDeviceState->outApiTypes = OutputApiTypes_State;
                pDeviceState->outCountMax = 1;
                pDeviceState->outputLevelNext = OutputLevel_Normal;
                pDeviceState->outputLevelCurrent = OutputLevel_Normal;
            }


            // 出力パラメータを更新
            pDeviceState->outCurrentMode = pDeviceState->outMode;
            pDeviceState->outCurrentApiTypes = pDeviceState->outApiTypes;
            pDeviceState->outCurrentCountMax = pDeviceState->outCountMax;
            pDeviceState->outputLevelCurrent = pDeviceState->outputLevelNext;
        }
    }
    // ストップして CameraReady に戻る
    else if ((buttonDownSetJoy & Npad::Button::ZR::Mask).IsAnyOn()
        || (buttonDownSetJoy & Npad::Button::ZL::Mask).IsAnyOn())
    {
        pDeviceState->outCurrentMode = OutputMode_None;
        pDeviceState->outCurrentApiTypes = OutputApiTypes_State;
        pDeviceState->outCurrentCountMax = 1;
        pDeviceState->outputLevelNext = OutputLevel_Normal;
        pDeviceState->padOutputMenuSelect = 0;
        pDeviceState->latestSampleCount = 0;
        pDeviceState->latestSamplingNum = 0;
        pDeviceState->updateCounter = 0;
        pDeviceState->averagePlr = 0.0f;
        pDeviceState->maxPlr = 0;
        pDeviceState->burstDropCount = 0;
        pDeviceState->burstDropCountWorst = 0;
        pDeviceState->burstDropCountWorstAve = 0.0f;
        pDeviceState->burstDropCountWorstCurrent = 0;
        pDeviceState->burstDropCountWorstMax = 0;
        pDeviceState->aveFrameIntervalMs = 0;
        pDeviceState->frameIntervalCount = 0;
        pDeviceState->frameIntervalMs = 0;
        pDeviceState->prevIntervalUpdateTime = 0;
        pDeviceState->irState.imageTransferFrameInterval = 0;
        pDeviceState->irState.imageTransferGetDataTime = 0;
        pDeviceState->irToolState = IrSensorApiTestToolState_CameraReady;
        nn::irsensor::StopImageProcessorAsync(pDeviceState->irCameraHandle);
        return;
    }
    // Finalize
    else if ((buttonDownSetJoy & Npad::Button::Plus::Mask).IsAnyOn()
        || (buttonDownSetJoy & Npad::Button::Minus::Mask).IsAnyOn())
    {
        pDeviceState->outCurrentMode = OutputMode_None;
        pDeviceState->outCurrentApiTypes = OutputApiTypes_State;
        pDeviceState->outCurrentCountMax = 1;
        pDeviceState->outputLevelCurrent = OutputLevel_Normal;
        pDeviceState->outputLevelNext = OutputLevel_Normal;
        pDeviceState->irToolState = IrSensorApiTestToolState_Initial;
        nn::irsensor::StopImageProcessorAsync(pDeviceState->irCameraHandle);
        int timeOutCounter = 0;
        const int MaxCount = 333;
        pDeviceState->irProcessorStatus = nn::irsensor::GetImageProcessorStatus(pDeviceState->irCameraHandle);
        while (pDeviceState->irProcessorStatus != nn::irsensor::ImageProcessorStatus_Stopped)
        {
            if (timeOutCounter++ > MaxCount)
            {
                NN_LOG("StopImageProcessor Timeout\n");
                break;
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));
            pDeviceState->irProcessorStatus = nn::irsensor::GetImageProcessorStatus(pDeviceState->irCameraHandle);
        }
        nn::irsensor::Finalize(pDeviceState->irCameraHandle);

        // データのクリア
        ClearDeviceState(pDeviceState);
        return;
    }

    if (pDeviceState->outCurrentMode == OutputMode_Auto
        || (pDeviceState->outCurrentMode == OutputMode_OneShot && isClicked))
    {
        int count = 0;
        // データの更新
        if (pDeviceState->irState.currentProcessor == IrSensorMode_Moment)
        {
            if (pDeviceState->outCurrentApiTypes == OutputApiTypes_State)
            {
                pDeviceState->irState.momentApiResult = nn::irsensor::GetMomentProcessorState(
                    &pDeviceState->irState.momentProcessorState[0],
                    pDeviceState->irCameraHandle);
                if (pDeviceState->irState.momentApiResult.IsSuccess())
                {
                    count = 1;
                }
                // NN_LOG("GetMoment: result:%02X samplingNum:%d avgInt:%f\n",
                //        pDeviceState->irState.momentApiResult,
                //        pDeviceState->irState.momentProcessorState[0].samplingNumber,
                //        pDeviceState->irState.momentProcessorState[0].blocks[0].averageIntensity);
            }
            else
            {
                pDeviceState->irState.momentApiResult = nn::irsensor::GetMomentProcessorStates(
                    &pDeviceState->irState.momentProcessorState[0],
                    &count,
                    pDeviceState->outCurrentCountMax,
                    pDeviceState->irCameraHandle);
            }
            pDeviceState->irState.momentApiOutCount = count;
        }
        else if (pDeviceState->irState.currentProcessor == IrSensorMode_Clustering)
        {
            if (pDeviceState->outCurrentApiTypes == OutputApiTypes_State)
            {
                pDeviceState->irState.clusteringApiResult = nn::irsensor::GetClusteringProcessorState(
                    &pDeviceState->irState.clusteringProcessorState[0],
                    pDeviceState->irCameraHandle);
                if (pDeviceState->irState.momentApiResult.IsSuccess())
                {
                    count = 1;
                }
            }
            else
            {
                pDeviceState->irState.clusteringApiResult = nn::irsensor::GetClusteringProcessorStates(
                    &pDeviceState->irState.clusteringProcessorState[0],
                    &count,
                    pDeviceState->outCurrentCountMax,
                    pDeviceState->irCameraHandle);
            }
            pDeviceState->irState.clusteringApiOutCount = count;
        }
        else if (pDeviceState->irState.currentProcessor == IrSensorMode_ImageTransfer)
        {
            GetBufferToImage(&pDeviceState->cameraViewer, &pDeviceState->irState, pDeviceState->irCameraHandle);
        }
        else if (pDeviceState->irState.currentProcessor == IrSensorMode_Pointing)
        {
            if (pDeviceState->outCurrentApiTypes == OutputApiTypes_States)
            {
                pDeviceState->irState.pointingApiResult = nn::irsensor::GetPointingProcessorStates(
                    &pDeviceState->irState.pointingProcessorState[0],
                    &count,
                    pDeviceState->outCurrentCountMax,
                    pDeviceState->irCameraHandle);

                nn::irsensor::GetPointingProcessorMarkerStates(
                    &pDeviceState->irState.pointingProcessorMarkerState[0],
                    &count,
                    pDeviceState->outCurrentCountMax,
                    pDeviceState->irCameraHandle);
            }
            pDeviceState->irState.pointingApiOutCount = count;
        }
        else if (pDeviceState->irState.currentProcessor == IrSensorMode_HandAnalysis)
        {
            if ((pDeviceState->irState.handAnalysisConfigCurrent.mode == nn::irsensor::HandAnalysisMode_Silhouette)
                || (pDeviceState->irState.handAnalysisConfigCurrent.mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage)
                || (pDeviceState->irState.handAnalysisConfigCurrent.mode == nn::irsensor::HandAnalysisMode_SilhouetteOnly))
            {
                pDeviceState->irState.handAnalysisApiResult = nn::irsensor::GetHandAnalysisSilhouetteState(
                    &pDeviceState->irState.handAnalysisSilhouetteState,
                    &count,
                    pDeviceState->outCurrentCountMax,
                    0,
                    pDeviceState->irCameraHandle);
            }
            if ((pDeviceState->irState.handAnalysisConfigCurrent.mode == nn::irsensor::HandAnalysisMode_Image)
                || (pDeviceState->irState.handAnalysisConfigCurrent.mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage))
            {
                pDeviceState->irState.handAnalysisApiResult = nn::irsensor::GetHandAnalysisImageState(
                    &pDeviceState->irState.handAnalysisImageState,
                    &count,
                    pDeviceState->outCurrentCountMax,
                    0,
                    pDeviceState->irCameraHandle);

                uint16_t* pSourcePixels = pDeviceState->irState.handAnalysisImageState.image;
                uint16_t* pDestinationPixels = pDeviceState->cameraViewer.copiedImageBuffer.Map< uint16_t >();
                size_t imageSize = nn::irsensor::IrHandAnalysisImageWidth * nn::irsensor::IrHandAnalysisImageHeight * sizeof(uint16_t);
                memcpy(pDestinationPixels, pSourcePixels, imageSize);
                for (size_t i = 0; i < imageSize; i++)
                {
                    const float max = static_cast<float>(0xffff);
                    const float value = static_cast<float>(pDestinationPixels[i]) / max;
                    const float scaledValue = value * 100.0f > 1.0f ? 1.0f : value * 100.0f;

                    pDestinationPixels[i] = static_cast<uint16_t>(scaledValue * max);
                }
                pDeviceState->cameraViewer.copiedImageBuffer.FlushMappedRange(0, imageSize);
                pDeviceState->cameraViewer.copiedImageBuffer.Unmap();
            }
        }
        else if (pDeviceState->irState.currentProcessor == IrSensorMode_IrLed)
        {
            pDeviceState->irState.irLedApiResult = nn::irsensor::GetIrLedProcessorState(
                pDeviceState->irCameraHandle);
        }
        else if (pDeviceState->irState.currentProcessor == IrSensorMode_AdaptiveClustering)
        {
            if (pDeviceState->outCurrentApiTypes == OutputApiTypes_States)
            {
                pDeviceState->irState.adaptiveClusteringApiResult = nn::irsensor::GetAdaptiveClusteringProcessorStates(
                    &pDeviceState->irState.adaptiveClusteringProcessorState[0],
                    &count,
                    pDeviceState->outCurrentCountMax,
                    0,
                    pDeviceState->irCameraHandle);
            }
            pDeviceState->irState.adaptiveClusteringApiOutCount = count;
        }
    }
}//NOLINT(readability/fn_size)

void UpdateCameraReady(DeviceState* pDeviceState) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pDeviceState);

    const int modeIndex = 0;
    const int configIndex = 4;

    static nn::os::Tick keyPressedTime = nn::os::Tick();
    static int counter = 0;

    nn::hid::NpadButtonSet buttonDownSetJoy = pDeviceState->buttonDownSetJoy;
    nn::hid::NpadButtonSet buttonDownSetJoyRaw = pDeviceState->buttonDownSetJoyRaw;
    IrState* pIrState = &pDeviceState->irState;
    int* pPadMenuSelect = &pDeviceState->padMenuSelect;

    // 状態の更新
    pDeviceState->irCameraStatus = nn::irsensor::GetIrCameraStatus(pDeviceState->irCameraHandle);
    pDeviceState->isCameraStatusValid = true;
    pDeviceState->irProcessorStatus = nn::irsensor::GetImageProcessorStatus(pDeviceState->irCameraHandle);
    pDeviceState->isProcessorStatusValid = true;

    // メニューの移動
    if ((buttonDownSetJoy & Npad::Button::B::Mask).IsAnyOn()
        || (buttonDownSetJoy & Npad::Button::Down::Mask).IsAnyOn())
    {
        if (++*pPadMenuSelect == MenuHeight[pIrState->nextProcessor])
        {
            *pPadMenuSelect = 0;
        }
    };

    if ((buttonDownSetJoy & Npad::Button::X::Mask).IsAnyOn()
        || (buttonDownSetJoy & Npad::Button::Up::Mask).IsAnyOn())
    {
        if (--*pPadMenuSelect < 0)
        {
            *pPadMenuSelect = MenuHeight[pIrState->nextProcessor] - 1;
        }
    };

    // 値を増加
    if ((buttonDownSetJoyRaw & Npad::Button::Y::Mask).IsAnyOn() || (buttonDownSetJoyRaw & Npad::Button::A::Mask).IsAnyOn()
        || (buttonDownSetJoyRaw & Npad::Button::Left::Mask).IsAnyOn() || (buttonDownSetJoyRaw & Npad::Button::Right::Mask).IsAnyOn())
    {
        nn::irsensor::IrCameraConfig* pConfig;
        int64_t timeMin, timeMax;

        if (pIrState->nextProcessor == IrSensorMode_Moment)
        {
            pConfig = &pIrState->momentProcessorConfig.irCameraConfig;
            timeMin = nn::irsensor::MomentProcessorExposureTimeMin.GetMicroSeconds();
            timeMax = nn::irsensor::MomentProcessorExposureTimeMax.GetMicroSeconds();
        }
        else if (pIrState->nextProcessor == IrSensorMode_Clustering)
        {
            pConfig = &pIrState->clusteringProcessorConfig.irCameraConfig;
            timeMin = nn::irsensor::ClusteringProcessorExposureTimeMin.GetMicroSeconds();
            timeMax = nn::irsensor::ClusteringProcessorExposureTimeMax.GetMicroSeconds();
        }
        else if (pIrState->nextProcessor == IrSensorMode_ImageTransfer)
        {
            pConfig = &pIrState->imageTransferProcessorConfigNext.irCameraConfig;
            timeMin = nn::irsensor::ImageTransferProcessorExposureTimeMin.GetMicroSeconds();
            timeMax = nn::irsensor::ImageTransferProcessorExposureTimeMax.GetMicroSeconds();
        }
        else if (pIrState->nextProcessor == IrSensorMode_Pointing)
        {
            pConfig = nullptr;
        }
        else if (pIrState->nextProcessor == IrSensorMode_HandAnalysis)
        {
            pConfig = nullptr;
        }
        else if (pIrState->nextProcessor == IrSensorMode_IrLed)
        {
            pConfig = nullptr;
        }
        else if (pIrState->nextProcessor == IrSensorMode_AdaptiveClustering)
        {
            pConfig = nullptr;
        }
        else
        {
            return;
        }

        int8_t increment = ((buttonDownSetJoyRaw & Npad::Button::A::Mask).IsAnyOn()
            || (buttonDownSetJoyRaw & Npad::Button::Right::Mask).IsAnyOn()) ? 1 : -1;

        if ((nn::os::GetSystemTick() - keyPressedTime).ToTimeSpan().GetMilliSeconds() > 50)
        {
            counter = 0;
        }

        if (counter > 50)
        {
            if (*pPadMenuSelect == 1)
            {
                // Exposure Time を変更する
                pConfig->exposureTime += nn::TimeSpanType::FromMicroSeconds(increment);
                pConfig->exposureTime = nn::TimeSpanType::FromMicroSeconds(std::min(
                    std::max(pConfig->exposureTime.GetMicroSeconds(), timeMin),
                    timeMax));
            }

            if (pIrState->nextProcessor == IrSensorMode_Moment)
            {
                switch (*pPadMenuSelect)
                {
                case 6:
                    {
                        // Intensity Threshold を変更する
                        pIrState->momentProcessorConfig.preprocessIntensityThreshold += increment;
                        pIrState->momentProcessorConfig.preprocessIntensityThreshold = std::min(
                            std::max(pIrState->momentProcessorConfig.preprocessIntensityThreshold, 0),
                            nn::irsensor::IrCameraIntensityMax);
                    }
                    break;
                case 7:
                    {
                        // WOI.X を変更する
                        pIrState->momentProcessorConfig.windowOfInterest.x += increment;
                        pIrState->momentProcessorConfig.windowOfInterest.x = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->momentProcessorConfig.windowOfInterest.x), 0),
                            nn::irsensor::IrCameraImageWidth - pIrState->momentProcessorConfig.windowOfInterest.width));
                    }
                    break;
                case 8:
                    {
                        // WOI.Y を変更する
                        pIrState->momentProcessorConfig.windowOfInterest.y += increment;
                        pIrState->momentProcessorConfig.windowOfInterest.y = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->momentProcessorConfig.windowOfInterest.y), 0),
                            nn::irsensor::IrCameraImageHeight - pIrState->momentProcessorConfig.windowOfInterest.height));
                    }
                    break;
                case 9:
                    {
                        // WOI.Width を変更する
                        pIrState->momentProcessorConfig.windowOfInterest.width += increment;
                        pIrState->momentProcessorConfig.windowOfInterest.width = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->momentProcessorConfig.windowOfInterest.width), 1),
                            nn::irsensor::IrCameraImageWidth - pIrState->momentProcessorConfig.windowOfInterest.x));
                    }
                    break;
                case 10:
                    {
                        // WOI.Height を変更する
                        pIrState->momentProcessorConfig.windowOfInterest.height += increment;
                        pIrState->momentProcessorConfig.windowOfInterest.height = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->momentProcessorConfig.windowOfInterest.height), 1),
                            nn::irsensor::IrCameraImageHeight - pIrState->momentProcessorConfig.windowOfInterest.y));
                    }
                    break;
                default:
                    break;
                }
            }

            if (pIrState->nextProcessor == IrSensorMode_Clustering)
            {
                switch (*pPadMenuSelect)
                {
                case 5:
                    {
                        // Object Pixel Count Min を変更する
                        pIrState->clusteringProcessorConfig.objectPixelCountMin += increment * 100;
                        pIrState->clusteringProcessorConfig.objectPixelCountMin = std::min(
                            std::max(pIrState->clusteringProcessorConfig.objectPixelCountMin, 0),
                            pIrState->clusteringProcessorConfig.objectPixelCountMax - 1);
                    }
                    break;
                case 6:
                    {
                        // Object Pixel Count Max を変更する
                        pIrState->clusteringProcessorConfig.objectPixelCountMax += increment * 100;
                        pIrState->clusteringProcessorConfig.objectPixelCountMax = std::max(
                            std::min(pIrState->clusteringProcessorConfig.objectPixelCountMax, nn::irsensor::IrCameraImageWidth * nn::irsensor::IrCameraImageHeight),
                            pIrState->clusteringProcessorConfig.objectPixelCountMin + 1);
                    }
                    break;
                case 7:
                    {
                        // Object Intensity Min を変更する
                        pIrState->clusteringProcessorConfig.objectIntensityMin += increment;
                        pIrState->clusteringProcessorConfig.objectIntensityMin =
                            std::max(std::min(pIrState->clusteringProcessorConfig.objectIntensityMin, nn::irsensor::IrCameraIntensityMax), 0);
                    }
                    break;
                case 9:
                    {
                        // WOI.X を変更する
                        pIrState->clusteringProcessorConfig.windowOfInterest.x += increment;
                        pIrState->clusteringProcessorConfig.windowOfInterest.x = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->clusteringProcessorConfig.windowOfInterest.x), 0),
                            nn::irsensor::IrCameraImageWidth - pIrState->clusteringProcessorConfig.windowOfInterest.width));
                    }
                    break;
                case 10:
                    {
                        // WOI.Y を変更する
                        pIrState->clusteringProcessorConfig.windowOfInterest.y += increment;
                        pIrState->clusteringProcessorConfig.windowOfInterest.y = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->clusteringProcessorConfig.windowOfInterest.y), 0),
                            nn::irsensor::IrCameraImageHeight - pIrState->clusteringProcessorConfig.windowOfInterest.height));
                    }
                    break;
                case 11:
                    {
                        // WOI.Width を変更する
                        pIrState->clusteringProcessorConfig.windowOfInterest.width += increment;
                        pIrState->clusteringProcessorConfig.windowOfInterest.width = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->clusteringProcessorConfig.windowOfInterest.width), 1),
                            nn::irsensor::IrCameraImageWidth - pIrState->clusteringProcessorConfig.windowOfInterest.x));
                        if (pIrState->clusteringProcessorConfig.windowOfInterest.width <= 1
                            && pIrState->clusteringProcessorConfig.windowOfInterest.height <= 20)
                        {
                            // ハードバグの禁則
                            pIrState->clusteringProcessorConfig.windowOfInterest.width -= increment;
                        }
                    }
                    break;
                case 12:
                    {
                        // WOI.Height を変更する
                        pIrState->clusteringProcessorConfig.windowOfInterest.height += increment;
                        pIrState->clusteringProcessorConfig.windowOfInterest.height = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->clusteringProcessorConfig.windowOfInterest.height), 1),
                            nn::irsensor::IrCameraImageHeight - pIrState->clusteringProcessorConfig.windowOfInterest.y));
                        if (pIrState->clusteringProcessorConfig.windowOfInterest.width <= 1
                            && pIrState->clusteringProcessorConfig.windowOfInterest.height <= 20)
                        {
                            // ハードバグの禁則
                            pIrState->clusteringProcessorConfig.windowOfInterest.height -= increment;
                        }
                    }
                    break;
                default:
                    // Donothing
                    break;
                }
            }

            if (pIrState->nextProcessor == IrSensorMode_ImageTransfer)
            {
                switch (*pPadMenuSelect)
                {
                case 7:
                    {
                        // StartX を変更する
                        pIrState->imageTransferProcessorConfigNext.trimmingStartX += increment;
                        pIrState->imageTransferProcessorConfigNext.trimmingStartX = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->imageTransferProcessorConfigNext.trimmingStartX), 0),
                            (nn::irsensor::IrCameraImageWidth >> pIrState->imageTransferProcessorConfigNext.origFormat)
                            - (nn::irsensor::IrCameraImageWidth >> pIrState->imageTransferProcessorConfigNext.trimmingFormat)));
                    }
                    break;
                case 8:
                    {
                        // StartY を変更する
                        pIrState->imageTransferProcessorConfigNext.trimmingStartY += increment;
                        pIrState->imageTransferProcessorConfigNext.trimmingStartY = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->imageTransferProcessorConfigNext.trimmingStartY), 0),
                            (nn::irsensor::IrCameraImageHeight >> pIrState->imageTransferProcessorConfigNext.origFormat)
                            - (nn::irsensor::IrCameraImageHeight >> pIrState->imageTransferProcessorConfigNext.trimmingFormat)));
                    }
                    break;
                default:
                    // Donothing
                    break;
                }
            }
        }
        keyPressedTime = nn::os::GetSystemTick();
        counter++;
    }

    if ((buttonDownSetJoy & Npad::Button::Y::Mask).IsAnyOn() || (buttonDownSetJoy & Npad::Button::A::Mask).IsAnyOn()
        || (buttonDownSetJoy & Npad::Button::Left::Mask).IsAnyOn() || (buttonDownSetJoy & Npad::Button::Right::Mask).IsAnyOn())
    {
        int8_t increment = ((buttonDownSetJoy & Npad::Button::A::Mask).IsAnyOn()
            || (buttonDownSetJoy & Npad::Button::Right::Mask).IsAnyOn()) ? 1 : -1;

        // モードの切り替え
        if (*pPadMenuSelect == modeIndex)
        {
            int8_t mode = static_cast<int8_t>(pIrState->nextProcessor) + increment;
            pIrState->nextProcessor = static_cast<IrSensorMode>(std::min(
                std::max(mode, static_cast<int8_t>(IrSensorMode_Moment)),
                static_cast<int8_t>(IrSensorMode_Count - 1)));
        }

        nn::irsensor::IrCameraConfig* pConfig;
        int64_t timeMin, timeMax;

        if (pIrState->nextProcessor == IrSensorMode_Moment)
        {
            pConfig = &pIrState->momentProcessorConfig.irCameraConfig;
            timeMin = nn::irsensor::MomentProcessorExposureTimeMin.GetMicroSeconds();
            timeMax = nn::irsensor::MomentProcessorExposureTimeMax.GetMicroSeconds();
        }
        else if (pIrState->nextProcessor == IrSensorMode_Clustering)
        {
            pConfig = &pIrState->clusteringProcessorConfig.irCameraConfig;
            timeMin = nn::irsensor::ClusteringProcessorExposureTimeMin.GetMicroSeconds();
            timeMax = nn::irsensor::ClusteringProcessorExposureTimeMax.GetMicroSeconds();
        }
        else if (pIrState->nextProcessor == IrSensorMode_ImageTransfer)
        {
            pConfig = &pIrState->imageTransferProcessorConfigNext.irCameraConfig;
            timeMin = nn::irsensor::ImageTransferProcessorExposureTimeMin.GetMicroSeconds();
            timeMax = nn::irsensor::ImageTransferProcessorExposureTimeMax.GetMicroSeconds();
        }
        else if (pIrState->nextProcessor == IrSensorMode_HandAnalysis)
        {
            pConfig = nullptr;
        }
        else if (pIrState->nextProcessor == IrSensorMode_IrLed)
        {
            pConfig = nullptr;
        }
        else if (pIrState->nextProcessor == IrSensorMode_AdaptiveClustering)
        {
            pConfig = nullptr;
        }
        else
        {
            return;
        }

        if (*pPadMenuSelect <= configIndex)
        {
            int8_t lightTarget;

            switch (*pPadMenuSelect)
            {
            case 1:
                {
                    if (pIrState->nextProcessor == IrSensorMode_HandAnalysis)
                    {
                        // HandAnalysis のモードを切り替える
                        int8_t mode = static_cast<int8_t>(pIrState->handAnalysisConfigNext.mode) + increment;
                        mode = std::max(
                            std::min(mode, static_cast<int8_t>(nn::irsensor::HandAnalysisMode_SilhouetteOnly)),
                            static_cast<int8_t>(nn::irsensor::HandAnalysisMode_Silhouette));
                        pIrState->handAnalysisConfigNext.mode = static_cast<nn::irsensor::HandAnalysisMode>(mode);
                    }
                    else if (pIrState->nextProcessor == IrSensorMode_IrLed)
                    {
                        // Light Target を変更する
                        lightTarget = static_cast<int8_t>(pIrState->irLedProcessorConfig.lightTarget) + increment;
                        lightTarget = std::max(
                            std::min(lightTarget, static_cast<int8_t>(nn::irsensor::IrCameraLightTarget_None)),
                            static_cast<int8_t>(nn::irsensor::IrCameraLightTarget_AllObjects));
                        pIrState->irLedProcessorConfig.lightTarget = static_cast<nn::irsensor::IrCameraLightTarget>(lightTarget);
                    }
                    else if (pIrState->nextProcessor == IrSensorMode_AdaptiveClustering)
                    {
                        // AdaptiveClustering のモードを変更する
                        int8_t mode = static_cast<int8_t>(pIrState->adaptiveClusteringProcessorConfig.mode) + increment;
                        mode = std::max(
                            std::min(mode, static_cast<int8_t>(nn::irsensor::AdaptiveClusteringMode_DynamicFov)),
                            static_cast<int8_t>(nn::irsensor::AdaptiveClusteringMode_StaticFov));
                        pIrState->adaptiveClusteringProcessorConfig.mode = static_cast<nn::irsensor::AdaptiveClusteringMode>(mode);
                    }
                    else
                    {
                        pConfig->exposureTime += nn::TimeSpanType::FromMicroSeconds(increment);

                        pConfig->exposureTime = nn::TimeSpanType::FromMicroSeconds(std::min(
                            std::max(pConfig->exposureTime.GetMicroSeconds(), timeMin),
                            timeMax));
                    }
                }
                break;
            case 2:
                // Light Target を変更する
                lightTarget = static_cast<int8_t>(pConfig->lightTarget) + increment;
                lightTarget = std::max(
                    std::min(lightTarget, static_cast<int8_t>(nn::irsensor::IrCameraLightTarget_None)),
                    static_cast<int8_t>(nn::irsensor::IrCameraLightTarget_AllObjects));
                pConfig->lightTarget = static_cast<nn::irsensor::IrCameraLightTarget>(lightTarget);
                break;
            case 3:
                // Digital Gain を変更する
                pConfig->gain += increment;
                pConfig->gain = std::min(
                    std::max(pConfig->gain, nn::irsensor::IrCameraGainMin),
                    nn::irsensor::IrCameraGainMax);
                break;
            case 4:
                // Negative Image を変更する
                pConfig->isNegativeImageUsed = !pConfig->isNegativeImageUsed;
                break;
            default:
                break;
            }
        }
        else
        {
            if (pIrState->nextProcessor == IrSensorMode_Moment)
            {
                switch (*pPadMenuSelect)
                {
                case 5:
                    // PreProcess を変更する
                    pIrState->momentProcessorConfig.preprocess =
                        (pIrState->momentProcessorConfig.preprocess == nn::irsensor::MomentProcessorPreprocess_Binarize) ?
                        nn::irsensor::MomentProcessorPreprocess_Cutoff : nn::irsensor::MomentProcessorPreprocess_Binarize;
                    break;
                case 6:
                    // Intensity Threshold を変更する
                    pIrState->momentProcessorConfig.preprocessIntensityThreshold += increment;
                    pIrState->momentProcessorConfig.preprocessIntensityThreshold = std::min(
                        std::max(pIrState->momentProcessorConfig.preprocessIntensityThreshold, 0),
                        nn::irsensor::IrCameraIntensityMax);
                    break;
                case 7:
                    {
                        // WOI.X を変更する
                        pIrState->momentProcessorConfig.windowOfInterest.x += increment;
                        pIrState->momentProcessorConfig.windowOfInterest.x = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->momentProcessorConfig.windowOfInterest.x), 0),
                            nn::irsensor::IrCameraImageWidth - pIrState->momentProcessorConfig.windowOfInterest.width));
                    }
                    break;
                case 8:
                    {
                        // WOI.Y を変更する
                        pIrState->momentProcessorConfig.windowOfInterest.y += increment;
                        pIrState->momentProcessorConfig.windowOfInterest.y = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->momentProcessorConfig.windowOfInterest.y), 0),
                            nn::irsensor::IrCameraImageHeight - pIrState->momentProcessorConfig.windowOfInterest.height));
                    }
                    break;
                case 9:
                    {
                        // WOI.Width を変更する
                        pIrState->momentProcessorConfig.windowOfInterest.width += increment;
                        pIrState->momentProcessorConfig.windowOfInterest.width = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->momentProcessorConfig.windowOfInterest.width), 1),
                            nn::irsensor::IrCameraImageWidth - pIrState->momentProcessorConfig.windowOfInterest.x));
                    }
                    break;
                case 10:
                    {
                        // WOI.Height を変更する
                        pIrState->momentProcessorConfig.windowOfInterest.height += increment;
                        pIrState->momentProcessorConfig.windowOfInterest.height = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->momentProcessorConfig.windowOfInterest.height), 1),
                            nn::irsensor::IrCameraImageHeight - pIrState->momentProcessorConfig.windowOfInterest.y));
                    }
                    break;
                default:
                    break;
                }
            }
            else if (pIrState->nextProcessor == IrSensorMode_Clustering)
            {
                switch (*pPadMenuSelect)
                {
                case 5:
                    // Object Pixel Count Min を変更する
                    pIrState->clusteringProcessorConfig.objectPixelCountMin += increment;
                    pIrState->clusteringProcessorConfig.objectPixelCountMin = std::min(
                        std::max(pIrState->clusteringProcessorConfig.objectPixelCountMin, 0),
                        pIrState->clusteringProcessorConfig.objectPixelCountMax - 1);
                    break;
                case 6:
                    // Object Pixel Count Max を変更する
                    pIrState->clusteringProcessorConfig.objectPixelCountMax += increment;
                    pIrState->clusteringProcessorConfig.objectPixelCountMax = std::max(
                        std::min(pIrState->clusteringProcessorConfig.objectPixelCountMax, nn::irsensor::IrCameraImageWidth * nn::irsensor::IrCameraImageHeight),
                        pIrState->clusteringProcessorConfig.objectPixelCountMin + 1);
                    break;
                case 7:
                    // Object Intensity Min を変更する
                    pIrState->clusteringProcessorConfig.objectIntensityMin += increment;
                    pIrState->clusteringProcessorConfig.objectIntensityMin =
                        std::max(std::min(pIrState->clusteringProcessorConfig.objectIntensityMin, nn::irsensor::IrCameraIntensityMax), 0);
                    break;
                case 8:
                    // 外部光フィルタ機能の有効無効を切り替える
                    pIrState->clusteringProcessorConfig.isExternalLightFilterEnabled = !pIrState->clusteringProcessorConfig.isExternalLightFilterEnabled;
                    break;
                case 9:
                    {
                        // WOI.X を変更する
                        pIrState->clusteringProcessorConfig.windowOfInterest.x += increment;
                        pIrState->clusteringProcessorConfig.windowOfInterest.x = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->clusteringProcessorConfig.windowOfInterest.x), 0),
                            nn::irsensor::IrCameraImageWidth - pIrState->clusteringProcessorConfig.windowOfInterest.width));
                    }
                    break;
                case 10:
                    {
                        // WOI.Y を変更する
                        pIrState->clusteringProcessorConfig.windowOfInterest.y += increment;
                        pIrState->clusteringProcessorConfig.windowOfInterest.y = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->clusteringProcessorConfig.windowOfInterest.y), 0),
                            nn::irsensor::IrCameraImageHeight - pIrState->clusteringProcessorConfig.windowOfInterest.height));
                    }
                    break;
                case 11:
                    {
                        // WOI.Width を変更する
                        pIrState->clusteringProcessorConfig.windowOfInterest.width += increment;
                        pIrState->clusteringProcessorConfig.windowOfInterest.width = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->clusteringProcessorConfig.windowOfInterest.width), 1),
                            nn::irsensor::IrCameraImageWidth - pIrState->clusteringProcessorConfig.windowOfInterest.x));
                        if (pIrState->clusteringProcessorConfig.windowOfInterest.width <= 1
                            && pIrState->clusteringProcessorConfig.windowOfInterest.height <= 20)
                        {
                            // ハードバグの禁則
                            pIrState->clusteringProcessorConfig.windowOfInterest.width -= increment;
                        }
                    }
                    break;
                case 12:
                    {
                        // WOI.Height を変更する
                        pIrState->clusteringProcessorConfig.windowOfInterest.height += increment;
                        pIrState->clusteringProcessorConfig.windowOfInterest.height = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->clusteringProcessorConfig.windowOfInterest.height), 1),
                            nn::irsensor::IrCameraImageHeight - pIrState->clusteringProcessorConfig.windowOfInterest.y));
                        if (pIrState->clusteringProcessorConfig.windowOfInterest.width <= 1
                            && pIrState->clusteringProcessorConfig.windowOfInterest.height <= 20)
                        {
                            // ハードバグの禁則
                            pIrState->clusteringProcessorConfig.windowOfInterest.height -= increment;
                        }
                    }
                    break;
                default:
                    break;
                }
            }
            else if (pIrState->nextProcessor == IrSensorMode_ImageTransfer)
            {
                switch (*pPadMenuSelect)
                {
                case 5:
                    {
                        // 切り取り元の画像フォーマットを変更する
                        int8_t format = static_cast<int8_t>(pIrState->imageTransferProcessorConfigNext.origFormat) + increment;
                        format = std::max(
                            std::min(format, static_cast<int8_t>(nn::irsensor::ImageTransferProcessorFormat_20x15)),
                            static_cast<int8_t>(nn::irsensor::ImageTransferProcessorFormat_320x240));
                        pIrState->imageTransferProcessorConfigNext.origFormat = static_cast<nn::irsensor::ImageTransferProcessorFormat>(format);

                        // 必要に応じて出力画像フォーマットを合わせる
                        pIrState->imageTransferProcessorConfigNext.trimmingFormat = static_cast<nn::irsensor::ImageTransferProcessorFormat>(std::max(
                            std::min(static_cast<int8_t>(pIrState->imageTransferProcessorConfigNext.trimmingFormat),
                                static_cast<int8_t>(nn::irsensor::ImageTransferProcessorFormat_20x15)),
                            static_cast<int8_t>(pIrState->imageTransferProcessorConfigNext.origFormat)));
                        // サイズの更新
                        pIrState->imageTransferProcessorConfigNext.trimmingStartX = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->imageTransferProcessorConfigNext.trimmingStartX), 0),
                            (nn::irsensor::IrCameraImageWidth >> pIrState->imageTransferProcessorConfigNext.origFormat)
                            - (nn::irsensor::IrCameraImageWidth >> pIrState->imageTransferProcessorConfigNext.trimmingFormat)));
                        pIrState->imageTransferProcessorConfigNext.trimmingStartY = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->imageTransferProcessorConfigNext.trimmingStartY), 0),
                            (nn::irsensor::IrCameraImageHeight >> pIrState->imageTransferProcessorConfigNext.origFormat)
                            - (nn::irsensor::IrCameraImageHeight >> pIrState->imageTransferProcessorConfigNext.trimmingFormat)));
                    }
                    break;
                case 6:
                    {
                        // 画像フォーマットを変更する
                        int8_t format = static_cast<int8_t>(pIrState->imageTransferProcessorConfigNext.trimmingFormat) + increment;
                        format = std::max(
                            std::min(format, static_cast<int8_t>(nn::irsensor::ImageTransferProcessorFormat_20x15)),
                            static_cast<int8_t>(nn::irsensor::ImageTransferProcessorFormat_320x240));
                        pIrState->imageTransferProcessorConfigNext.trimmingFormat = static_cast<nn::irsensor::ImageTransferProcessorFormat>(format);

                        // 必要に応じて切り取り画像フォーマットを合わせる
                        pIrState->imageTransferProcessorConfigNext.origFormat = static_cast<nn::irsensor::ImageTransferProcessorFormat>(std::max(
                            std::min(static_cast<int8_t>(pIrState->imageTransferProcessorConfigNext.origFormat),
                                static_cast<int8_t>(pIrState->imageTransferProcessorConfigNext.trimmingFormat)),
                            static_cast<int8_t>(nn::irsensor::ImageTransferProcessorFormat_320x240)));
                        // サイズの更新
                        pIrState->imageTransferProcessorConfigNext.trimmingStartX = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->imageTransferProcessorConfigNext.trimmingStartX), 0),
                            (nn::irsensor::IrCameraImageWidth >> pIrState->imageTransferProcessorConfigNext.origFormat)
                            - (nn::irsensor::IrCameraImageWidth >> pIrState->imageTransferProcessorConfigNext.trimmingFormat)));
                        pIrState->imageTransferProcessorConfigNext.trimmingStartY = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->imageTransferProcessorConfigNext.trimmingStartY), 0),
                            (nn::irsensor::IrCameraImageHeight >> pIrState->imageTransferProcessorConfigNext.origFormat)
                            - (nn::irsensor::IrCameraImageHeight >> pIrState->imageTransferProcessorConfigNext.trimmingFormat)));
                    }
                    break;
                case 7:
                    {
                        // StartX を変更する
                        pIrState->imageTransferProcessorConfigNext.trimmingStartX += increment;
                        pIrState->imageTransferProcessorConfigNext.trimmingStartX = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->imageTransferProcessorConfigNext.trimmingStartX), 0),
                            (nn::irsensor::IrCameraImageWidth >> pIrState->imageTransferProcessorConfigNext.origFormat)
                            - (nn::irsensor::IrCameraImageWidth >> pIrState->imageTransferProcessorConfigNext.trimmingFormat)));
                    }
                    break;
                case 8:
                    {
                        // StartY を変更する
                        pIrState->imageTransferProcessorConfigNext.trimmingStartY += increment;
                        pIrState->imageTransferProcessorConfigNext.trimmingStartY = static_cast<int16_t>(std::min(
                            std::max(static_cast<int>(pIrState->imageTransferProcessorConfigNext.trimmingStartY), 0),
                            (nn::irsensor::IrCameraImageHeight >> pIrState->imageTransferProcessorConfigNext.origFormat)
                            - (nn::irsensor::IrCameraImageHeight >> pIrState->imageTransferProcessorConfigNext.trimmingFormat)));
                    }
                    break;
                case 9:
                    {
                        pIrState->imageTransferProcessorConfigNext.isExternalLightFilterEnabled = !pIrState->imageTransferProcessorConfigNext.isExternalLightFilterEnabled;
                    }
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }
            }
        }
    }

    // Running ステートへ遷移
    nn::irsensor::IrCameraHandle handle = pDeviceState->irCameraHandle;
    if ((buttonDownSetJoy & Npad::Button::R::Mask).IsAnyOn()
        || (buttonDownSetJoy & Npad::Button::L::Mask).IsAnyOn())
    {
        // Run 前にデータをクリアしておく
        ClearIrStateData(&pDeviceState->irState);

        // モードの切り替え (レジスタ更新を含む)
        if (pIrState->nextProcessor == IrSensorMode_Moment)
        {
            nn::irsensor::RunMomentProcessor(handle, pIrState->momentProcessorConfig);
            // WOI の更新
            pIrState->momentProcessorWoi = pIrState->momentProcessorConfig.windowOfInterest;
        }
        else if (pIrState->nextProcessor == IrSensorMode_Clustering)
        {
            nn::irsensor::RunClusteringProcessor(handle, pIrState->clusteringProcessorConfig);
            // WOI の更新
            pIrState->clusteringProcessorWoi = pIrState->clusteringProcessorConfig.windowOfInterest;
        }
        else if (pIrState->nextProcessor == IrSensorMode_ImageTransfer)
        {
            size_t bufferSize = 0;
            if (pIrState->imageTransferProcessorConfigNext.trimmingFormat == nn::irsensor::ImageTransferProcessorFormat_320x240)
            {
                bufferSize = nn::irsensor::ImageTransferProcessorWorkBufferSize320x240;
            }
            else if (pIrState->imageTransferProcessorConfigNext.trimmingFormat == nn::irsensor::ImageTransferProcessorFormat_160x120)
            {
                bufferSize = nn::irsensor::ImageTransferProcessorWorkBufferSize160x120;
            }
            else if (pIrState->imageTransferProcessorConfigNext.trimmingFormat == nn::irsensor::ImageTransferProcessorFormat_80x60)
            {
                bufferSize = nn::irsensor::ImageTransferProcessorWorkBufferSize80x60;
            }
            else if (pIrState->imageTransferProcessorConfigNext.trimmingFormat == nn::irsensor::ImageTransferProcessorFormat_40x30)
            {
                bufferSize = nn::irsensor::ImageTransferProcessorWorkBufferSize40x30;
            }
            else if (pIrState->imageTransferProcessorConfigNext.trimmingFormat == nn::irsensor::ImageTransferProcessorFormat_20x15)
            {
                bufferSize = nn::irsensor::ImageTransferProcessorWorkBufferSize20x15;
            }

            // Run 前にバッファをクリアしておく
            uint8_t* destinationPixels = pDeviceState->cameraViewer.copiedImageBuffer.Map< uint8_t >();
            memset(destinationPixels, 0, nn::irsensor::ImageTransferProcessorWorkBufferSize320x240);
            pDeviceState->cameraViewer.copiedImageBuffer.FlushMappedRange(0, nn::irsensor::ImageTransferProcessorWorkBufferSize320x240);
            pDeviceState->cameraViewer.copiedImageBuffer.Unmap();

            nn::irsensor::RunImageTransferProcessor(
                handle,
                pIrState->imageTransferProcessorConfigNext,
                static_cast<void*>(pIrState->imageTransferWorkBuffer),
                bufferSize);
            pIrState->imageTransferProcessorConfigCurrent = pIrState->imageTransferProcessorConfigNext;
            pIrState->format = pIrState->imageTransferProcessorConfigCurrent.trimmingFormat;
        }
        else if (pIrState->nextProcessor == IrSensorMode_Pointing)
        {
            nn::irsensor::RunPointingProcessor(handle);
        }
        else if (pIrState->nextProcessor == IrSensorMode_HandAnalysis)
        {
            // Run 前にバッファをクリアしておく
            uint8_t* destinationPixels = pDeviceState->cameraViewer.copiedImageBuffer.Map< uint8_t >();
            memset(destinationPixels, 0, nn::irsensor::ImageTransferProcessorWorkBufferSize320x240);
            pDeviceState->cameraViewer.copiedImageBuffer.FlushMappedRange(0, nn::irsensor::ImageTransferProcessorWorkBufferSize320x240);
            pDeviceState->cameraViewer.copiedImageBuffer.Unmap();

            nn::irsensor::RunHandAnalysis(handle, pIrState->handAnalysisConfigNext);
            pIrState->handAnalysisConfigCurrent = pIrState->handAnalysisConfigNext;
        }
        else if (pIrState->nextProcessor == IrSensorMode_IrLed)
        {
            nn::irsensor::RunIrLedProcessor(handle, pIrState->irLedProcessorConfig);
        }
        else if (pIrState->nextProcessor == IrSensorMode_AdaptiveClustering)
        {
            nn::irsensor::RunAdaptiveClusteringProcessor(handle, pIrState->adaptiveClusteringProcessorConfig);
        }

        pIrState->currentProcessor = pIrState->nextProcessor;

        if (pIrState->nextProcessor == IrSensorMode_Pointing
            || pIrState->nextProcessor == IrSensorMode_AdaptiveClustering)
        {
            pDeviceState->outApiTypes = OutputApiTypes_States;
            pDeviceState->outCountMax = 3;
        }
        else
        {
            pDeviceState->outApiTypes = OutputApiTypes_State;
            pDeviceState->outCountMax = 1;
        }
        pDeviceState->irToolState = IrSensorApiTestToolState_Running;
    }
    else if ((buttonDownSetJoy & Npad::Button::ZR::Mask).IsAnyOn()
        || (buttonDownSetJoy & Npad::Button::ZL::Mask).IsAnyOn())
    {
        // Set default
        if (pIrState->nextProcessor == IrSensorMode_Moment)
        {
            nn::irsensor::GetMomentProcessorDefaultConfig(&pDeviceState->irState.momentProcessorConfig);
        }
        else if (pIrState->nextProcessor == IrSensorMode_Clustering)
        {
            nn::irsensor::GetClusteringProcessorDefaultConfig(&pDeviceState->irState.clusteringProcessorConfig);
        }
        else if (pIrState->nextProcessor == IrSensorMode_ImageTransfer)
        {
            nn::irsensor::GetImageTransferProcessorDefaultConfig(&pDeviceState->irState.imageTransferProcessorConfigNext);
            nn::irsensor::GetImageTransferProcessorDefaultConfig(&pDeviceState->irState.imageTransferProcessorConfigCurrent);
        }
        else if (pIrState->nextProcessor == IrSensorMode_IrLed)
        {
            nn::irsensor::GetIrLedProcessorDefaultConfig(&pDeviceState->irState.irLedProcessorConfig);
        }
    }
    // Finalize
    else if ((buttonDownSetJoy & Npad::Button::Plus::Mask).IsAnyOn()
        || (buttonDownSetJoy & Npad::Button::Minus::Mask).IsAnyOn())
    {
        pDeviceState->irToolState = IrSensorApiTestToolState_Initial;
        int timeOutCounter = 0;
        const int MaxCount = 333;
        nn::irsensor::ImageProcessorStatus processorStatus = nn::irsensor::GetImageProcessorStatus(pDeviceState->irCameraHandle);
        while (processorStatus != nn::irsensor::ImageProcessorStatus_Stopped)
        {
            if (timeOutCounter++ > MaxCount)
            {
                NN_LOG("StopImageProcessor Timeout\n");
                break;
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));
            processorStatus = nn::irsensor::GetImageProcessorStatus(pDeviceState->irCameraHandle);
        }
        nn::irsensor::Finalize(pDeviceState->irCameraHandle);
        // データのクリア
        ClearDeviceState(pDeviceState);
    }

}//NOLINT(readability/fn_size)

void Update(DeviceState deviceState[], Npad::Id npadIds[])
{
    for (int i = 0; i < NpadIdCountMax; ++i)
    {
        switch (deviceState[i].irToolState)
        {
        case IrSensorApiTestToolState_Initial:
            {
                UpdateInitial(&deviceState[i], npadIds[i]);
            }
            break;
        case IrSensorApiTestToolState_Initialized:
            {
                UpdateInitialized(&deviceState[i]);
            }
            break;
        case IrSensorApiTestToolState_CameraReady:
            {
                UpdateCameraReady(&deviceState[i]);
            }
            break;
        case IrSensorApiTestToolState_Running:
            {
                UpdateRunning(&deviceState[i], i);
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

void Draw(GraphicsSystem* pGraphicsSystem,
    DeviceState deviceState[])
{
    NN_ASSERT_NOT_NULL(pGraphicsSystem);

    nn::gfx::util::DebugFontTextWriter* pTextWriter = &pGraphicsSystem->GetDebugFont();

    pGraphicsSystem->BeginDraw();

    // Draw ToolVersion
    WriteToolVersion(pTextWriter, MenuToolVersionX, MenuToolVersionY);

    // Draw Fps
    WriteFps(pTextWriter, MenuFpsX, MenuFpsY);

    for (int i = 0; i < NpadIdCountMax; ++i)
    {
        if (g_Page == i / 2)
        {
            // Draw Input State
            switch (deviceState[i].irToolState)
            {
            case IrSensorApiTestToolState_Initial:
                {
                    WriteInitial(pTextWriter, &deviceState[i], MenuParameterOffsetX, MenuOutputOffsetY + DisplayHeight / 2.0f * (i % 2));
                }
                break;
            case IrSensorApiTestToolState_Initialized:
                {
                    WriteInitialized(pTextWriter, &deviceState[i], MenuParameterOffsetX, MenuOutputOffsetY + DisplayHeight / 2.0f * (i % 2));
                    WriteOutputFirmwareCheck(pTextWriter, &deviceState[i], MenuResultOffsetX, MenuOutputOffsetY + DisplayHeight / 2.0f * (i % 2));
                }
                break;
            case IrSensorApiTestToolState_CameraReady:
                {
                    WriteParameters(pTextWriter, &deviceState[i], MenuParameterOffsetX, MenuOutputOffsetY + DisplayHeight / 2.0f * (i % 2));
                }
                break;
            case IrSensorApiTestToolState_Running:
                {
                    WriteParameters(pTextWriter, &deviceState[i], MenuParameterOffsetX, MenuOutputOffsetY + DisplayHeight / 2.0f * (i % 2));
                    WriteOutputConfig(pTextWriter, &deviceState[i], MenuOutputConfigOffsetX, MenuOutputOffsetY + DisplayHeight / 2.0f * (i % 2));
                    WriteOutput(pTextWriter, &deviceState[i], MenuResultOffsetX, MenuOutputOffsetY + DisplayHeight / 2.0f * (i % 2));
                    WritePerformance(pTextWriter, &deviceState[i], MenuResultOffsetX, MenuPerformanceOffsetY + DisplayHeight / 2.0f * (i % 2));
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }

            // Draw Output State
            // Draw Status Bar
            WriteStatusBar(pGraphicsSystem, pTextWriter, &deviceState[i], i % 2);

            // Draw Help
            bool isUseContinuousRunCommand = false;
            if (deviceState[i].padOutputMenuSelect >= MenuOutputLength)
            {
                isUseContinuousRunCommand = true;
            }
            WriteHelp(pGraphicsSystem, pTextWriter, deviceState[i].irToolState, deviceState[i].npadTypes, MenuHelpOffsetX, MenuHelpOffsetY + DisplayHeight / 2.0f * (i % 2), isUseContinuousRunCommand);

            // Draw Image
            if (deviceState[i].irToolState == IrSensorApiTestToolState_CameraReady || deviceState[i].irToolState == IrSensorApiTestToolState_Running)
            {
                IrState irState = deviceState[i].irState;
                if (irState.currentProcessor == IrSensorMode_Moment)
                {
                    WriteMomentProcessorState(
                        pGraphicsSystem,
                        irState.momentProcessorWoi,
                        &irState.momentProcessorState[0],
                        MenuImageOffsetX,
                        MenuOutputOffsetY + DisplayHeight / 2.0f * (i % 2));
                }
                else if (irState.currentProcessor == IrSensorMode_Clustering)
                {
                    WriteClusteringProcessorState(
                        pGraphicsSystem,
                        irState.clusteringProcessorWoi,
                        &irState.clusteringProcessorState[0],
                        MenuImageOffsetX,
                        MenuOutputOffsetY + DisplayHeight / 2.0f * (i % 2));
                }
                else if (irState.currentProcessor == IrSensorMode_ImageTransfer)
                {
                    // 読み込んだカメラ画像 nn::gfx::buffer を、テクスチャにタイリングします
                    CopyImageBuffer(pGraphicsSystem, &deviceState[i].cameraViewer, &irState);
                    WriteImageTransferProcessorState(
                        pGraphicsSystem,
                        &deviceState[i].cameraViewer,
                        &irState,
                        deviceState[i].outputLevelCurrent,
                        MenuImageOffsetX,
                        MenuOutputOffsetY + DisplayHeight / 2.0f * (i % 2));
                }
                else if (irState.currentProcessor == IrSensorMode_Pointing)
                {
                    WritePointingProcessorState(
                        pGraphicsSystem,
                        &irState.pointingProcessorState[0],
                        &irState.pointingProcessorMarkerState[0],
                        deviceState[i].outputLevelCurrent,
                        irState.pointingApiOutCount,
                        MenuImageOffsetX,
                        MenuOutputOffsetY + DisplayHeight / 2.0f * (i % 2));
                }
                else if (irState.currentProcessor == IrSensorMode_HandAnalysis)
                {
                    // 画像の表示
                    if ((irState.handAnalysisConfigCurrent.mode == nn::irsensor::HandAnalysisMode_Image)
                        || (irState.handAnalysisConfigCurrent.mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage))

                    {
                        CopyImageBuffer(pGraphicsSystem, &deviceState[i].cameraViewer, &irState);
                        WriteHandAnalysisImageProcessorState(
                            pGraphicsSystem,
                            &deviceState[i].cameraViewer,
                            &irState,
                            MenuImageOffsetX,
                            MenuOutputOffsetY + DisplayHeight / 2.0f * (i % 2));
                    }
                    // シルエットの表示
                    if ((irState.handAnalysisConfigCurrent.mode == nn::irsensor::HandAnalysisMode_Silhouette)
                        || (irState.handAnalysisConfigCurrent.mode == nn::irsensor::HandAnalysisMode_SilhouetteOnly)
                        || (irState.handAnalysisConfigCurrent.mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage))

                    {
                        PolygonProperties properties = {};
                        WriteHandAnalysisSilhouetteProcessorState(
                            pGraphicsSystem,
                            &irState.handAnalysisSilhouetteState,
                            &properties,
                            MenuImageOffsetX,
                            MenuOutputOffsetY + DisplayHeight / 2.0f * (i % 2));

                        // ここでは、Draw用の deviceState にデータ保存しているので、後で書き戻す
                        deviceState[i].irState.handAnalysisPolygon = properties;
                    }
                }
                else if (irState.currentProcessor == IrSensorMode_AdaptiveClustering)
                {
                    WriteAdaptiveClusteringProcessorState(
                        pGraphicsSystem,
                        &irState.adaptiveClusteringProcessorState[0],
                        MenuImageOffsetX,
                        MenuOutputOffsetY + DisplayHeight / 2.0f * (i % 2));
                }
            }
        }
    }

    pTextWriter->Draw(&pGraphicsSystem->GetCommandBuffer());

    pGraphicsSystem->EndDraw();
}//NOLINT(readability/fn_size)

void LibraryAppletThread(void* pArg) NN_NOEXCEPT
{
    NN_UNUSED(pArg);

    while (g_IsAppletThreadRunning)
    {
        nn::err::ShowError(nn::err::MakeErrorCode(2001, 1));
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
    }
    return;
}

void StartLibraryApplet() NN_NOEXCEPT
{
    if (!g_IsAppletThreadRunning)
    {
        NN_LOG("Create LibraryApplet Thread\n");
        g_IsAppletThreadRunning = true;
        nn::os::CreateThread(
            &g_AppletThread, LibraryAppletThread, nullptr, g_AppletThreadStack,
            sizeof(g_AppletThreadStack), nn::os::DefaultThreadPriority);
        nn::os::StartThread(&g_AppletThread);
    }
    return;
}

void StopLibraryApplet() NN_NOEXCEPT
{
    if (g_IsAppletThreadRunning)
    {
        g_IsAppletThreadRunning = false;
        nn::os::WaitThread(&g_AppletThread);
        nn::os::DestroyThread(&g_AppletThread);
        NN_LOG("Destroy LibraryApplet Thread\n");
    }
    return;
}

void MeasureThread(void* pArg) NN_NOEXCEPT
{
    DeviceState* pDeviceStates = reinterpret_cast<DeviceState*>(pArg);

    static int counter = 0;
    while (g_IsMeasureThreadRunning)
    {
        nn::os::WaitTimerEvent(&g_TimerEvent);
        nn::os::ClearTimerEvent(&g_TimerEvent);

        for (int i = 0; i < NpadIdCountMax; ++i)
        {
            DeviceState* pState = pDeviceStates + i;
            // 実行中のみ測定
            if (pState->irToolState == IrSensorApiTestToolState_Running)
            {
                nn::Result result;
                // 動作モードのチェック
                switch (pState->irState.currentProcessor)
                {
                case IrSensorMode_Moment:
                    {
                        result = pState->irState.momentApiResult;
                        if (result.IsSuccess())
                        {
                            if (pState->latestSamplingNum == pState->irState.momentProcessorState[0].samplingNumber)
                            {
                                pState->burstDropCount++;
                                if (pState->burstDropCountWorst < pState->burstDropCount)
                                {
                                    pState->burstDropCountWorst = pState->burstDropCount;
                                }
                            }
                            else
                            {
                                pState->burstDropCount = 0;
                            }
                            pState->latestSamplingNum = pState->irState.momentProcessorState[0].samplingNumber;
                        }
                    }
                    break;
                case IrSensorMode_Clustering:
                    {
                        result = pState->irState.clusteringApiResult;
                        if (result.IsSuccess())
                        {
                            if (pState->latestSamplingNum == pState->irState.clusteringProcessorState[0].samplingNumber)
                            {
                                pState->burstDropCount++;
                                if (pState->burstDropCountWorst < pState->burstDropCount)
                                {
                                    pState->burstDropCountWorst = pState->burstDropCount;
                                }
                            }
                            else
                            {
                                pState->burstDropCount = 0;
                            }
                            pState->latestSamplingNum = pState->irState.clusteringProcessorState[0].samplingNumber;
                        }
                    }
                    break;
                case IrSensorMode_ImageTransfer:
                    {
                        // 画像転送の場合は更新時間を表示する
                        if ((pState->prevIntervalUpdateTime < pState->irState.imageTransferGetDataTime)
                            && (pState->irState.imageTransferFrameInterval != 0))
                        {
                            pState->prevIntervalUpdateTime = pState->irState.imageTransferGetDataTime;
                            pState->frameIntervalCount++;
                            pState->frameIntervalMs = pState->irState.imageTransferFrameInterval;
                            pState->aveFrameIntervalMs
                                = (pState->aveFrameIntervalMs * (pState->frameIntervalCount - 1)
                                    + static_cast<float>(pState->frameIntervalMs)) / pState->frameIntervalCount;
                        }
                    }
                    break;
                case IrSensorMode_Pointing:
                    {
                        result = pState->irState.pointingApiResult;
                        if (result.IsSuccess())
                        {
                            if (pState->latestSamplingNum == pState->irState.pointingProcessorState[0].samplingNumber)
                            {
                                pState->burstDropCount++;
                                if (pState->burstDropCountWorst < pState->burstDropCount)
                                {
                                    pState->burstDropCountWorst = pState->burstDropCount;
                                }
                            }
                            else
                            {
                                pState->burstDropCount = 0;
                            }
                            pState->latestSamplingNum = pState->irState.pointingProcessorState[0].samplingNumber;
                        }
                    }
                    break;
                case IrSensorMode_HandAnalysis:
                    {
                        result = pState->irState.handAnalysisApiResult;
                        if (result.IsSuccess())
                        {
                            int64_t samplingNum = 0;
                            if (pState->irState.handAnalysisConfigCurrent.mode == nn::irsensor::HandAnalysisMode_Silhouette
                                || pState->irState.handAnalysisConfigCurrent.mode == nn::irsensor::HandAnalysisMode_SilhouetteOnly)
                            {
                                samplingNum = pState->irState.handAnalysisSilhouetteState.samplingNumber;
                            }
                            else if (pState->irState.handAnalysisConfigCurrent.mode == nn::irsensor::HandAnalysisMode_Image)
                            {
                                samplingNum = pState->irState.handAnalysisImageState.samplingNumber;
                            }
                            else if (pState->irState.handAnalysisConfigCurrent.mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage)
                            {
                                samplingNum = std::max(pState->irState.handAnalysisSilhouetteState.samplingNumber
                                    , pState->irState.handAnalysisImageState.samplingNumber);
                            }

                            if (pState->latestSamplingNum == samplingNum)
                            {
                                pState->burstDropCount++;
                                if (pState->burstDropCountWorst < pState->burstDropCount)
                                {
                                    pState->burstDropCountWorst = pState->burstDropCount;
                                }
                            }
                            else
                            {
                                pState->burstDropCount = 0;
                            }
                            pState->latestSamplingNum = samplingNum;
                        }
                    }
                    break;
                case IrSensorMode_IrLed:
                    {
                        // 何もしない
                    }
                    break;
                case IrSensorMode_AdaptiveClustering:
                    {
                        result = pState->irState.adaptiveClusteringApiResult;
                        if (result.IsSuccess())
                        {
                            if (pState->latestSamplingNum == pState->irState.adaptiveClusteringProcessorState[0].samplingNumber)
                            {
                                pState->burstDropCount++;
                                if (pState->burstDropCountWorst < pState->burstDropCount)
                                {
                                    pState->burstDropCountWorst = pState->burstDropCount;
                                }
                            }
                            else
                            {
                                pState->burstDropCount = 0;
                            }
                            pState->latestSamplingNum = pState->irState.adaptiveClusteringProcessorState[0].samplingNumber;
                        }
                    }
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }

                if (result.IsFailure())
                {
                    // NN_LOG("Get Error on MeasureThread Npad:%d\n", i);
                    continue;
                }


                if (pState->updateCounter == 0 && counter > 100)
                {
                    counter = 0;
                }
                // 1.5秒ごとに結果を更新
                if (counter % 100 == 0 && counter != 0)
                {
                    int64_t latestSamplingNum = pState->latestSamplingNum;
                    if (pState->irState.currentProcessor == IrSensorMode_Pointing)
                    {
                        latestSamplingNum /= 3;
                    }
                    if (latestSamplingNum - pState->prevSamplingNum <= 100)
                    {
                        pState->burstDropCountWorstCurrent = pState->burstDropCountWorst;
                        pState->latestSampleCount = std::max(std::min(static_cast<int>(latestSamplingNum - pState->prevSamplingNum), 100), 0);
                        if (pState->burstDropCountWorstMax < pState->burstDropCountWorst)
                        {
                            pState->burstDropCountWorstMax = pState->burstDropCountWorst;
                        }
                        pState->burstDropCountWorstAve
                            = ((pState->burstDropCountWorstAve * pState->updateCounter) + pState->burstDropCountWorstCurrent) / static_cast<float>(pState->updateCounter + 1);

                        if (pState->maxPlr < 100 - pState->latestSampleCount) pState->maxPlr = static_cast<int>(100 - pState->latestSampleCount);
                        pState->averagePlr = ((pState->averagePlr * pState->updateCounter) + (100 - pState->latestSampleCount)) / static_cast<float>(pState->updateCounter + 1);
                        pState->burstDropCountWorst = 0;
                        pState->updateCounter++;
                    }
                    pState->prevSamplingNum = latestSamplingNum;
                }
            }
        }
        counter++;
    }
    return;
}//NOLINT(readability/fn_size)

void StartMeasureThread(DeviceState* pDeviceState) NN_NOEXCEPT
{
    if (!g_IsMeasureThreadRunning)
    {
        NN_LOG("Create Measure Thread\n");
        // 計測用タイマを起動(1.5秒ごと)
        nn::os::InitializeTimerEvent(&g_TimerEvent, nn::os::EventClearMode_ManualClear);
        nn::os::StartPeriodicTimerEvent(&g_TimerEvent, nn::TimeSpan::FromMilliSeconds(1000), nn::TimeSpan::FromMilliSeconds(15));

        g_IsMeasureThreadRunning = true;
        nn::os::CreateThread(
            &g_MeasureThread, MeasureThread, (void*)pDeviceState, g_MeasureThreadStack,
            sizeof(g_MeasureThreadStack), nn::os::DefaultThreadPriority);
        nn::os::StartThread(&g_MeasureThread);
    }
    return;
}

void StopMeasureThread() NN_NOEXCEPT
{
    if (g_IsMeasureThreadRunning)
    {
        g_IsMeasureThreadRunning = false;
        nn::os::WaitThread(&g_MeasureThread);
        nn::os::DestroyThread(&g_MeasureThread);
        // タイマを停止
        nn::os::StopTimerEvent(&g_TimerEvent);
        nn::os::FinalizeTimerEvent(&g_TimerEvent);
        NN_LOG("Destroy Measure Thread\n");
    }
    return;
}

void UpdateThread(void* pArg) NN_NOEXCEPT
{
    DeviceState* pDeviceStates = reinterpret_cast<DeviceState*>(pArg);

    while (g_IsUpdateThreadRunning)
    {
        nn::os::WaitTimerEvent(&g_UpdateTimerEvent);
        Update(pDeviceStates, g_NpadIds);
        // 更新が完了したら、メインスレッドに通知
        nn::os::SignalEvent(&g_UpdateEvent);
    }
    return;
}

void StartUpdateThread(DeviceState* pDeviceState) NN_NOEXCEPT
{
    if (!g_IsUpdateThreadRunning)
    {
        NN_LOG("Create Update Thread\n");
        nn::os::InitializeEvent(&g_UpdateEvent, false, nn::os::EventClearMode_AutoClear);
        nn::os::InitializeTimerEvent(&g_UpdateTimerEvent, nn::os::EventClearMode_AutoClear);
        nn::os::StartPeriodicTimerEvent(&g_UpdateTimerEvent, nn::TimeSpan::FromMilliSeconds(0), nn::TimeSpan::FromMicroSeconds(16666));

        g_IsUpdateThreadRunning = true;
        nn::os::CreateThread(
            &g_UpdateThread, UpdateThread, (void*)pDeviceState, g_UpdateThreadStack,
            sizeof(g_UpdateThreadStack), nn::os::DefaultThreadPriority);
        nn::os::StartThread(&g_UpdateThread);
    }
    return;
}

void StopUpdateThread() NN_NOEXCEPT
{
    if (g_IsUpdateThreadRunning)
    {
        g_IsUpdateThreadRunning = false;
        nn::os::WaitThread(&g_UpdateThread);
        nn::os::DestroyThread(&g_UpdateThread);
        // タイマを停止
        nn::os::StopTimerEvent(&g_UpdateTimerEvent);
        nn::os::FinalizeTimerEvent(&g_UpdateTimerEvent);
        nn::os::FinalizeEvent(&g_UpdateEvent);
        NN_LOG("Destroy Update Thread\n");
    }
    return;
}

}

extern "C" void nnMain()
{
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    nv::SetGraphicsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
    nv::SetGraphicsDevtoolsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
    nv::InitializeGraphics(std::malloc(GraphicsMemorySize), GraphicsMemorySize);
#endif

    ApplicationHeap applicationHeap;
    applicationHeap.Initialize(ApplicationHeapSize);

    GraphicsSystem* pGraphicsSystem = new ::GraphicsSystem();
    pGraphicsSystem->SetApplicationHeap(&applicationHeap);
    pGraphicsSystem->Initialize();

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    HWND hwnd = reinterpret_cast<HWND>(
        pGraphicsSystem->GetNativeWindowHandle());

    g_DefaultWndProc = reinterpret_cast<WNDPROC>(
        ::GetWindowLongPtr(hwnd, GWLP_WNDPROC));

    ::SetWindowLongPtr(hwnd, GWLP_WNDPROC, reinterpret_cast<LONG_PTR>(WndProc));

    ::RegisterTouchWindow(hwnd, 0);
#endif

    DeviceState deviceState[NpadIdCountMax] = {};
    DeviceState deviceStateForDraw[NpadIdCountMax] = {};

    nn::hid::InitializeNpad();

    // Npad の持ち方のスタイルを設定
    nn::hid::SetSupportedNpadStyleSet(
        nn::hid::NpadStyleJoyRight::Mask
        | nn::hid::NpadStyleJoyLeft::Mask
        | nn::hid::NpadStyleJoyDual::Mask
        | nn::hid::NpadStyleHandheld::Mask
        | nn::hid::NpadStyleFullKey::Mask);

    // 使用する Npad を設定
    nn::hid::SetSupportedNpadIdType(g_NpadIds, NpadIdCountMax);

    for (int i = 0; i < NpadIdCountMax; ++i)
    {
        NN_LOG("NpadPlayerNumber(%d)\n", g_NpadIds[i]);

        // Camera Viewer を設定
        InitializeCameraViewer(pGraphicsSystem, &deviceState[i].cameraViewer);
        deviceState[i].cameraViewer.cameraImageBuffer =
            applicationHeap.Allocate(nn::irsensor::ImageTransferProcessorImageSize320x240, nn::os::MemoryPageSize);
        deviceState[i].irState.imageTransferWorkBuffer =
            applicationHeap.Allocate(nn::irsensor::ImageTransferProcessorWorkBufferSize320x240, nn::os::MemoryPageSize);

        // Tool を Initial 状態に設定
        deviceState[i].irToolState = IrSensorApiTestToolState_Initial;
        // 初期状態は状態不明
        deviceState[i].irCameraStatus = nn::irsensor::IrCameraStatus_Unconnected;
        deviceState[i].isCameraStatusValid = false;
        deviceState[i].irProcessorStatus = nn::irsensor::ImageProcessorStatus_Stopped;
        deviceState[i].isProcessorStatusValid = false;
        deviceState[i].outMode = OutputMode_Auto;
        // ファームチェック関連の初期化
        deviceState[i].firmwareUpdateStatusResult = nn::ResultSuccess();
        deviceState[i].firmwareUpdateTrialCount = 0;

        // IrSensor を Moment として初期化
        deviceState[i].irState.currentProcessor = deviceState[i].irState.nextProcessor = IrSensorMode_Moment;
        nn::irsensor::GetMomentProcessorDefaultConfig(&deviceState[i].irState.momentProcessorConfig);
        nn::irsensor::GetClusteringProcessorDefaultConfig(&deviceState[i].irState.clusteringProcessorConfig);
        nn::irsensor::GetImageTransferProcessorDefaultConfig(&deviceState[i].irState.imageTransferProcessorConfigNext);
        nn::irsensor::GetImageTransferProcessorDefaultConfig(&deviceState[i].irState.imageTransferProcessorConfigCurrent);
        deviceState[i].irState.handAnalysisConfigCurrent.mode = nn::irsensor::HandAnalysisMode_SilhouetteAndImage;
        deviceState[i].irState.handAnalysisConfigNext.mode = nn::irsensor::HandAnalysisMode_SilhouetteAndImage;
        nn::irsensor::GetIrLedProcessorDefaultConfig(&deviceState[i].irState.irLedProcessorConfig);
        deviceState[i].irState.adaptiveClusteringProcessorConfig.mode = nn::irsensor::AdaptiveClusteringMode_StaticFov;

        // WOI 設定を初期化
        nn::irsensor::Rect initialRect = { 0, 0, ::nn::irsensor::IrCameraImageWidth, ::nn::irsensor::IrCameraImageHeight };
        deviceState[i].irState.momentProcessorWoi = initialRect;
        deviceState[i].irState.clusteringProcessorWoi = initialRect;
    }

    // IRセンサーのデータ更新用のスレッドを起動
    StartUpdateThread(&deviceState[0]);

    // PLR 測定用のスレッドを起動
    StartMeasureThread(&deviceState[0]);

    bool runs = true;
    while (NN_STATIC_CONDITION(runs))
    {
        //Pad Statusの取得
        for (int i = 0; i < NpadIdCountMax; ++i)
        {
            // 前回の Draw 内部で書き変えたデータを書き戻し
            deviceState[i].irState.handAnalysisPolygon = deviceStateForDraw[i].irState.handAnalysisPolygon;

            //現在有効な操作形態(NpadStyleSet)を取得
            nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(g_NpadIds[i]);

            if (style.Test<nn::hid::NpadStyleJoyRight>() == true)
            {
                static Npad::RightState prevState[NpadIdCountMax];
                Npad::RightState currentState;

                // 押されたボタンを検出
                nn::hid::GetNpadState(&currentState, g_NpadIds[i]);

                // アプレット遷移などで、期待する設定になっていなかった場合は、再設定を行う
                if (deviceState[i].npadTypes != NpadTypes_JoyRight)
                {
                    deviceState[i].npadTypes = NpadTypes_JoyRight;
                    style |= nn::hid::NpadStyleJoyRight::Mask;
                    deviceState[i].buttonDownSetJoyRaw.Reset();
                    deviceState[i].buttonDownSetJoy.Reset();
                }
                else
                {
                    deviceState[i].buttonDownSetJoyRaw = currentState.buttons;
                    deviceState[i].buttonDownSetJoy = currentState.buttons & ~prevState[i].buttons;
                }
                prevState[i] = currentState;
            }
            else if (style.Test<nn::hid::NpadStyleJoyLeft>() == true)
            {
                static Npad::LeftState prevState[NpadIdCountMax];
                Npad::LeftState currentState;

                // 押されたボタンを検出
                nn::hid::GetNpadState(&currentState, g_NpadIds[i]);

                // アプレット遷移などで、期待する設定になっていなかった場合は、再設定を行う
                if (deviceState[i].npadTypes != NpadTypes_JoyLeft)
                {
                    deviceState[i].npadTypes = NpadTypes_JoyLeft;
                    style |= nn::hid::NpadStyleJoyLeft::Mask;
                    deviceState[i].buttonDownSetJoyRaw.Reset();
                    deviceState[i].buttonDownSetJoy.Reset();
                }
                else
                {
                    deviceState[i].buttonDownSetJoyRaw = currentState.buttons;
                    deviceState[i].buttonDownSetJoy = currentState.buttons & ~prevState[i].buttons;
                }
                prevState[i] = currentState;

            }
            else if (style.Test<nn::hid::NpadStyleHandheld>() == true)
            {
                static Npad::HandheldState prevState[NpadIdCountMax];
                Npad::HandheldState currentState;

                // 押されたボタンを検出
                nn::hid::GetNpadState(&currentState, g_NpadIds[i]);

                // アプレット遷移などで、期待する設定になっていなかった場合は、再設定を行う
                if (deviceState[i].npadTypes != NpadTypes_Handheld)
                {
                    deviceState[i].npadTypes = NpadTypes_Handheld;
                    style |= nn::hid::NpadStyleHandheld::Mask;
                    deviceState[i].buttonDownSetJoyRaw.Reset();
                    deviceState[i].buttonDownSetJoy.Reset();
                }
                else
                {
                    deviceState[i].buttonDownSetJoyRaw = currentState.buttons;
                    deviceState[i].buttonDownSetJoy = currentState.buttons & ~prevState[i].buttons;
                }
                prevState[i] = currentState;
            }
            else if (style.Test<nn::hid::NpadStyleFullKey>() == true)
            {
                static Npad::FullkeyState prevState[NpadIdCountMax];
                Npad::FullkeyState currentState;

                // 押されたボタンを検出
                nn::hid::GetNpadState(&currentState, g_NpadIds[i]);

                // アプレット遷移などで、期待する設定になっていなかった場合は、再設定を行う
                if (deviceState[i].npadTypes != NpadTypes_FullKey)
                {
                    deviceState[i].npadTypes = NpadTypes_FullKey;
                    style |= nn::hid::NpadStyleFullKey::Mask;
                    deviceState[i].buttonDownSetJoyRaw.Reset();
                    deviceState[i].buttonDownSetJoy.Reset();
                }
                else
                {
                    deviceState[i].buttonDownSetJoyRaw = currentState.buttons;
                    deviceState[i].buttonDownSetJoy = currentState.buttons & ~prevState[i].buttons;
                }
                prevState[i] = currentState;
            }
            else if (style.Test<nn::hid::NpadStyleJoyDual>() == true)
            {
                static Npad::JoyDualState prevState[NpadIdCountMax];
                Npad::JoyDualState currentState;

                // 押されたボタンを検出
                nn::hid::GetNpadState(&currentState, g_NpadIds[i]);

                if (deviceState[i].npadTypes != NpadTypes_JoyDual)
                {
                    deviceState[i].npadTypes = NpadTypes_JoyDual;
                    style |= nn::hid::NpadStyleJoyDual::Mask;
                    deviceState[i].buttonDownSetJoyRaw.Reset();
                    deviceState[i].buttonDownSetJoy.Reset();
                }
                else
                {
                    deviceState[i].buttonDownSetJoyRaw = currentState.buttons;
                    deviceState[i].buttonDownSetJoy = currentState.buttons & ~prevState[i].buttons;
                }
                prevState[i] = currentState;
            }
            else
            {
                // 切断時はボタンをリセットしておく
                deviceState[i].buttonDownSetJoyRaw.Reset();
                deviceState[i].buttonDownSetJoy.Reset();
            }

            if ((deviceState[i].buttonDownSetJoy & Npad::Button::StickL::Mask).IsAnyOn())
            {
                g_Page = (g_Page + PageCountMax - 1) % PageCountMax;
            }
            if ((deviceState[i].buttonDownSetJoy & Npad::Button::StickR::Mask).IsAnyOn())
            {
                g_Page = (g_Page + 1) % PageCountMax;
            }

            // ZR,Y 同時押しでアプリのFPSを変更
            if ((deviceState[i].buttonDownSetJoy & Npad::Button::ZR::Mask).IsAnyOn()
                && (deviceState[i].buttonDownSetJoy & Npad::Button::Y::Mask).IsAnyOn())
            {
                g_FrameRateIndex = (g_FrameRateIndex + 1) % FpsTableLength;
                g_FrameRate = FpsTable[g_FrameRateIndex];
            }

            // A,X,R 同時押しでアプリ終了
            if ((deviceState[i].buttonDownSetJoyRaw & Npad::Button::A::Mask).IsAnyOn()
                && (deviceState[i].buttonDownSetJoyRaw & Npad::Button::X::Mask).IsAnyOn()
                && (deviceState[i].buttonDownSetJoyRaw & Npad::Button::R::Mask).IsAnyOn())
            {
                runs = false;
            }
            // 左,上,L 同時押しでアプリ終了
            if ((deviceState[i].buttonDownSetJoyRaw & Npad::Button::Left::Mask).IsAnyOn()
                && (deviceState[i].buttonDownSetJoyRaw & Npad::Button::Up::Mask).IsAnyOn()
                && (deviceState[i].buttonDownSetJoyRaw & Npad::Button::L::Mask).IsAnyOn())
            {
                runs = false;
            }

            // A,Y 同時押しでライブラリアプレットの起動
            if ((deviceState[i].buttonDownSetJoy & Npad::Button::A::Mask).IsAnyOn()
                && (deviceState[i].buttonDownSetJoy & Npad::Button::Y::Mask).IsAnyOn())
            {
                if (g_IsAppletThreadRunning)
                {
                    StopLibraryApplet();
                }
                else
                {
                    StartLibraryApplet();
                }
            }
        }

        // 状態の更新
        if (nn::os::TryWaitEvent(&g_UpdateEvent))
        {
            std::memcpy(deviceStateForDraw, deviceState, sizeof(deviceState));
        }

        if (g_FrameRate == 30)
        {
            // 30fps にするためスリープ
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(33));
        }

        // 描画
        Draw(pGraphicsSystem, deviceStateForDraw);

        pGraphicsSystem->Synchronize(
            nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / g_FrameRate));
    }

    NN_LOG("Finished\n");

    // 描画用のバッファをクリア
    std::memset(deviceStateForDraw, 0, sizeof(deviceStateForDraw));

    // PLR 測定用のスレッドを停止
    StopMeasureThread();

    StopLibraryApplet();

    StopUpdateThread();

    for (int i = 0; i < NpadIdCountMax; ++i)
    {
        if ((deviceState[i].irToolState != IrSensorApiTestToolState_Initial))
        {
            nn::irsensor::Finalize(deviceState[i].irCameraHandle);
        }
        FinalizeCameraViewer(pGraphicsSystem, &deviceState[i].cameraViewer);
    }

    pGraphicsSystem->Finalize();
    delete pGraphicsSystem;

    for (int i = 0; i < NpadIdCountMax; ++i)
    {
        applicationHeap.Deallocate(deviceState[i].cameraViewer.cameraImageBuffer);
        applicationHeap.Deallocate(deviceState[i].irState.imageTransferWorkBuffer);
    }
    applicationHeap.Finalize();

}//NOLINT(readability/fn_size)
