﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/irsensor.h>
#include <nnt.h>
#include "../Common/testIrsensor_Util.h"

using namespace ::nn::irsensor;

namespace nnt { namespace irsensor {
    IrCameraHandle IrSensorTest::s_Handles[NpadIdCountMax];
    int IrSensorTest::s_HandleCount;
}}

namespace
{
const int NotReadyWaitLoopCountMax = 100;
const nn::TimeSpan PollingInterval = nn::TimeSpan::FromMilliSeconds(15);

class MomentProcessorTest : public ::nnt::irsensor::IrSensorTest
{
protected:
    void DoTestWoiBoundary(int x, int y, int width, int height) NN_NOEXCEPT;

    void DoTestPreprocessBoundary(MomentProcessorPreprocess preprocess) NN_NOEXCEPT;

    void DoTestPreprocessThresholdBoundary(int threshold) NN_NOEXCEPT;

    void DoTestMomentStatistic(const MomentStatistic& block, int index, const MomentProcessorConfig& config) NN_NOEXCEPT;

    void DoTestPreprocessThresholdIntensityMax(MomentProcessorPreprocess preprocess) NN_NOEXCEPT;
};

TEST_F(MomentProcessorTest, DefaultMomentProcessorConfig)
{
    MomentProcessorConfig config;
    GetMomentProcessorDefaultConfig(&config);

    EXPECT_LE(MomentProcessorExposureTimeMin, config.irCameraConfig.exposureTime);
    EXPECT_LE(config.irCameraConfig.exposureTime, MomentProcessorExposureTimeMax);
    EXPECT_LE(IrCameraGainMin, config.irCameraConfig.gain);
    EXPECT_LE(config.irCameraConfig.gain, IrCameraGainMax);
    EXPECT_EQ(false, config.irCameraConfig.isNegativeImageUsed);
    EXPECT_EQ(IrCameraLightTarget_AllObjects, config.irCameraConfig.lightTarget);

    EXPECT_EQ(0, config.windowOfInterest.x);
    EXPECT_EQ(0, config.windowOfInterest.y);
    EXPECT_EQ(IrCameraImageWidth, config.windowOfInterest.width);
    EXPECT_EQ(IrCameraImageHeight, config.windowOfInterest.height);
    EXPECT_EQ(MomentProcessorPreprocess_Cutoff, config.preprocess);
    EXPECT_LE(0, config.preprocessIntensityThreshold);
    EXPECT_LE(config.preprocessIntensityThreshold, IrCameraIntensityMax);
}

void MomentProcessorTest::DoTestWoiBoundary(int x, int y, int width, int height) NN_NOEXCEPT
{
    EXPECT_LT(0, s_HandleCount);

    for (auto i = 0; i < s_HandleCount; ++i)
    {
        const auto& handle = s_Handles[i];

        MomentProcessorConfig config;
        GetMomentProcessorDefaultConfig(&config);

        config.windowOfInterest = MakeRect(x, y, width, height);
        RunMomentProcessor(handle, config);
        StopImageProcessor(handle);
    }

    NNT_IRSENSOR_EXIT_SUCCESS;
}

TEST_F(MomentProcessorTest, WoiXBoundary)
{
    NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(0, 0, IrCameraImageWidth, IrCameraImageHeight), NNT_IRSENSOR_EXIT_0, "");
    NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(1, 0, IrCameraImageWidth - 1, IrCameraImageHeight), NNT_IRSENSOR_EXIT_0, "");
    NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(IrCameraImageWidth - MomentProcessorBlockColumnCount, 0, MomentProcessorBlockColumnCount, IrCameraImageHeight), NNT_IRSENSOR_EXIT_0, "");

    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(1, 0, IrCameraImageWidth, IrCameraImageHeight), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(-1, 0, IrCameraImageWidth, IrCameraImageHeight), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(IrCameraImageWidth - MomentProcessorBlockColumnCount + 1, 0, MomentProcessorBlockColumnCount - 1, IrCameraImageHeight), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(IrCameraImageWidth - MomentProcessorBlockColumnCount + 1, 0, MomentProcessorBlockColumnCount, IrCameraImageHeight), "");
}

TEST_F(MomentProcessorTest, WoiYBoundary)
{
    NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(0, 1, IrCameraImageWidth, IrCameraImageHeight - 1), NNT_IRSENSOR_EXIT_0, "");
    NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(0, IrCameraImageHeight - MomentProcessorBlockRowCount, IrCameraImageWidth, MomentProcessorBlockRowCount), NNT_IRSENSOR_EXIT_0, "");

    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(0, 1, IrCameraImageWidth, IrCameraImageHeight), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(0, -1, IrCameraImageWidth, IrCameraImageHeight), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(0, IrCameraImageHeight - MomentProcessorBlockRowCount + 1, IrCameraImageWidth, MomentProcessorBlockRowCount - 1), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(0, IrCameraImageHeight - MomentProcessorBlockRowCount + 1, IrCameraImageWidth, MomentProcessorBlockRowCount), "");
}

TEST_F(MomentProcessorTest, WoiWidthBoundary)
{
    NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(0, 0, MomentProcessorBlockColumnCount, IrCameraImageHeight), NNT_IRSENSOR_EXIT_0, "");

    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(0, 0, MomentProcessorBlockColumnCount - 1, IrCameraImageHeight), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(1, 0, MomentProcessorBlockColumnCount - 1, IrCameraImageHeight), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(0, 0, IrCameraImageWidth + 1, IrCameraImageHeight), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(1, 0, IrCameraImageWidth + 1, IrCameraImageHeight), "");
}

TEST_F(MomentProcessorTest, WoiHeightBoundary)
{
    NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(0, 0, IrCameraImageWidth, MomentProcessorBlockRowCount), NNT_IRSENSOR_EXIT_0, "");

    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(0, 0, IrCameraImageWidth, MomentProcessorBlockRowCount - 1), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(1, 0, IrCameraImageWidth, MomentProcessorBlockRowCount - 1), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(0, 0, IrCameraImageWidth, IrCameraImageHeight + 1), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(1, 0, IrCameraImageWidth, IrCameraImageHeight + 1), "");
}

void MomentProcessorTest::DoTestPreprocessBoundary(MomentProcessorPreprocess preprocess) NN_NOEXCEPT
{
    EXPECT_LT(0, s_HandleCount);

    for (auto i = 0; i < s_HandleCount; ++i)
    {
        const auto& handle = s_Handles[i];

        MomentProcessorConfig config;
        GetMomentProcessorDefaultConfig(&config);

        config.preprocess = preprocess;
        RunMomentProcessor(handle, config);
        StopImageProcessor(handle);
    }

    NNT_IRSENSOR_EXIT_SUCCESS;
}

TEST_F(MomentProcessorTest, PreprocessBoundary)
{
    NNT_IRSENSOR_EXPECT_EXIT(DoTestPreprocessBoundary(MomentProcessorPreprocess_Binarize), NNT_IRSENSOR_EXIT_0, "");
    NNT_IRSENSOR_EXPECT_EXIT(DoTestPreprocessBoundary(MomentProcessorPreprocess_Cutoff), NNT_IRSENSOR_EXIT_0, "");

    EXPECT_DEATH_IF_SUPPORTED(DoTestPreprocessBoundary(static_cast<MomentProcessorPreprocess>(-1)), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestPreprocessBoundary(static_cast<MomentProcessorPreprocess>(2)), "");
}

void MomentProcessorTest::DoTestPreprocessThresholdBoundary(int threshold) NN_NOEXCEPT
{
    EXPECT_LT(0, s_HandleCount);

    for (auto i = 0; i < s_HandleCount; ++i)
    {
        const auto& handle = s_Handles[i];

        MomentProcessorConfig config;
        GetMomentProcessorDefaultConfig(&config);

        config.preprocessIntensityThreshold = threshold;
        RunMomentProcessor(handle, config);
        StopImageProcessor(handle);
    }

    NNT_IRSENSOR_EXIT_SUCCESS;
}

TEST_F(MomentProcessorTest, PreprocessThresholdBoundary)
{
    NNT_IRSENSOR_EXPECT_EXIT(DoTestPreprocessThresholdBoundary(0), NNT_IRSENSOR_EXIT_0, "");
    NNT_IRSENSOR_EXPECT_EXIT(DoTestPreprocessThresholdBoundary(IrCameraIntensityMax), NNT_IRSENSOR_EXIT_0, "");

    EXPECT_DEATH_IF_SUPPORTED(DoTestPreprocessThresholdBoundary(-1), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestPreprocessThresholdBoundary(IrCameraIntensityMax + 1), "");
}

void MomentProcessorTest::DoTestMomentStatistic(const MomentStatistic& block, int index, const MomentProcessorConfig& config) NN_NOEXCEPT
{
    auto width = config.windowOfInterest.width / MomentProcessorBlockColumnCount;
    auto height = config.windowOfInterest.height / MomentProcessorBlockRowCount;
    auto x = config.windowOfInterest.x + (index % MomentProcessorBlockColumnCount) * width;
    auto y = config.windowOfInterest.y + (index / MomentProcessorBlockColumnCount) * height;

    EXPECT_LE(0, block.averageIntensity);
    EXPECT_LE(block.averageIntensity, IrCameraIntensityMax);
    if (block.averageIntensity == 0)
    {
        EXPECT_EQ(0, block.centroid.x);
        EXPECT_EQ(0, block.centroid.y);
    }
    else
    {
        EXPECT_LE(x, block.centroid.x);
        EXPECT_LE(block.centroid.x, x + width - 1);
        EXPECT_LE(y, block.centroid.y);
        EXPECT_LE(block.centroid.y, y + height - 1);
    }
}

TEST_F(MomentProcessorTest, GetMomentProcessorState)
{
    EXPECT_LT(0, s_HandleCount);

    for (auto i = 0; i < s_HandleCount; ++i)
    {
        const auto& handle = s_Handles[i];

        MomentProcessorConfig config;
        GetMomentProcessorDefaultConfig(&config);

        // ASSERT_EQ(IrCameraStatus_Available, GetIrCameraStatus(handle));
        RunMomentProcessor(handle, config);

        MomentProcessorState state;
        // NNT_ASSERT_RESULT_SUCCESS(GetMomentProcessorState(&state, handle));
        nn::Result result = nn::ResultSuccess();
        int counter = 0;
        while (NN_STATIC_CONDITION(true))
        {
            result = GetMomentProcessorState(&state, handle);
            if (nn::irsensor::ResultIrsensorNotReady::Includes(result))
            {
                counter++;
                nn::os::SleepThread(PollingInterval);
                ASSERT_LT(counter, NotReadyWaitLoopCountMax);
            }
            else
            {
                break;
            }
        }
        EXPECT_LE(0, state.samplingNumber);

        for (auto j = 0; j < MomentProcessorBlockCount; ++j)
        {
            DoTestMomentStatistic(state.blocks[j], j, config);
        }

        StopImageProcessor(handle);
    }
}

TEST_F(MomentProcessorTest, GetMomentProcessorStates)
{
    EXPECT_LT(0, s_HandleCount);

    for (auto i = 0; i < s_HandleCount; ++i)
    {
        const auto& handle = s_Handles[i];

        MomentProcessorConfig config;
        GetMomentProcessorDefaultConfig(&config);

        // ASSERT_EQ(IrCameraStatus_Available, GetIrCameraStatus(handle));
        RunMomentProcessor(handle, config);

        MomentProcessorState states[MomentProcessorStateCountMax];
        int count;
        // NNT_ASSERT_RESULT_SUCCESS(GetMomentProcessorStates(states, &count, ::nnt::irsensor::GetArrayLength(states), handle));
        nn::Result result = nn::ResultSuccess();
        int counter = 0;
        while (NN_STATIC_CONDITION(true))
        {
            result = GetMomentProcessorStates(states, &count, ::nnt::irsensor::GetArrayLength(states), handle);
            if (nn::irsensor::ResultIrsensorNotReady::Includes(result))
            {
                counter++;
                nn::os::SleepThread(PollingInterval);
                ASSERT_LT(counter, NotReadyWaitLoopCountMax);
            }
            else
            {
                break;
            }
        }
        EXPECT_LE(0, count);

        for (auto j = 0; j < count; ++j)
        {
            auto& state = states[j];

            EXPECT_LE(0, state.samplingNumber);
            if (j > 0)
            {
                EXPECT_LT(states[j - 1].samplingNumber, state.samplingNumber);
            }

            for (auto k = 0; k < MomentProcessorBlockCount; ++k)
            {
                DoTestMomentStatistic(state.blocks[k], k, config);
            }
        }

        StopImageProcessor(handle);
    }
}

void MomentProcessorTest::DoTestPreprocessThresholdIntensityMax(MomentProcessorPreprocess preprocess) NN_NOEXCEPT
{
    EXPECT_LT(0, s_HandleCount);

    for (auto i = 0; i < s_HandleCount; ++i)
    {
        const auto& handle = s_Handles[i];

        MomentProcessorConfig config;
        GetMomentProcessorDefaultConfig(&config);
        config.preprocess = preprocess;
        config.preprocessIntensityThreshold = IrCameraIntensityMax;

        // ASSERT_EQ(IrCameraStatus_Available, GetIrCameraStatus(handle));
        RunMomentProcessor(handle, config);

        MomentProcessorState state;
        // NNT_ASSERT_RESULT_SUCCESS(GetMomentProcessorState(&state, handle));
        nn::Result result = nn::ResultSuccess();
        int counter = 0;
        while (NN_STATIC_CONDITION(true))
        {
            result = GetMomentProcessorState(&state, handle);
            if (nn::irsensor::ResultIrsensorNotReady::Includes(result))
            {
                counter++;
                nn::os::SleepThread(PollingInterval);
                ASSERT_LT(counter, NotReadyWaitLoopCountMax);
            }
            else
            {
                break;
            }
        }
        EXPECT_LE(0, state.samplingNumber);

        for (auto j = 0; j < MomentProcessorBlockCount; ++j)
        {
            auto& block = state.blocks[j];

            EXPECT_EQ(0, block.averageIntensity);
            EXPECT_EQ(0, block.centroid.x);
            EXPECT_EQ(0, block.centroid.y);
        }

        StopImageProcessor(handle);
    }
}

TEST_F(MomentProcessorTest, PreprocessThresholdIntensityMaxWithBinarize)
{
    DoTestPreprocessThresholdIntensityMax(MomentProcessorPreprocess_Binarize);
}

TEST_F(MomentProcessorTest, PreprocessThresholdIntensityMaxWithCutoff)
{
    DoTestPreprocessThresholdIntensityMax(MomentProcessorPreprocess_Cutoff);
}

void DoTestMomentRegionStatistic(
    const MomentProcessorState* pState,
    const Rect& windowOfInterest,
    int startRow,
    int startColumn,
    int rowCount,
    int columnCount,
    float expectedIntensity,
    float expectedX,
    float expectedY) NN_NOEXCEPT
{
    auto statistic = CalculateMomentRegionStatistic(
        pState,
        windowOfInterest,
        startRow,
        startColumn,
        rowCount,
        columnCount);

    EXPECT_FLOAT_EQ(statistic.averageIntensity, expectedIntensity);
    EXPECT_FLOAT_EQ(statistic.centroid.x, expectedX);
    EXPECT_FLOAT_EQ(statistic.centroid.y, expectedY);
}

TEST(MomentTest, CalculateMomentRegionStatistic)
{
    MomentProcessorState state = {};
    Rect woi = { 0, 0, IrCameraImageWidth, IrCameraImageHeight };

    // 全体が暗い状態
    for (auto& block : state.blocks)
    {
        block.averageIntensity = 0;
        block.centroid.x = 0;
        block.centroid.y = 0;
    }

    DoTestMomentRegionStatistic(
        &state,
        woi,
        0,
        0,
        MomentProcessorBlockRowCount,
        MomentProcessorBlockColumnCount,
        0,
        0,
        0);

    auto blockWidth = IrCameraImageWidth / MomentProcessorBlockColumnCount;
    auto blockHeight = IrCameraImageHeight / MomentProcessorBlockRowCount;

    // 全体が明るい状態
    for (auto i = 0; i < MomentProcessorBlockRowCount; ++i)
    {
        for (auto j = 0; j < MomentProcessorBlockColumnCount; ++j)
        {
            auto& block = state.blocks[j + i * MomentProcessorBlockColumnCount];
            block.averageIntensity = IrCameraIntensityMax;
            block.centroid.x = blockWidth * j + (blockWidth - 1) / 2.f;
            block.centroid.y = blockHeight * i + (blockHeight - 1) / 2.f;
        }
    }

    DoTestMomentRegionStatistic(
        &state,
        woi,
        0,
        0,
        MomentProcessorBlockRowCount,
        MomentProcessorBlockColumnCount,
        static_cast<float>(IrCameraIntensityMax),
        (IrCameraImageWidth - 1) / 2.f,
        (IrCameraImageHeight - 1) / 2.f);

    // 各ブロックの右下隅ピクセルのみ明るい状態
    for (auto i = 0; i < MomentProcessorBlockRowCount; ++i)
    {
        for (auto j = 0; j < MomentProcessorBlockColumnCount; ++j)
        {
            auto& block = state.blocks[j + i * MomentProcessorBlockColumnCount];
            block.averageIntensity = static_cast<float>(IrCameraIntensityMax) / (blockWidth * blockHeight);
            block.centroid.x = blockWidth * j + blockWidth - 1.f;
            block.centroid.y = blockHeight * i + blockHeight - 1.f;
        }
    }

    DoTestMomentRegionStatistic(
        &state,
        woi,
        0,
        0,
        MomentProcessorBlockRowCount,
        MomentProcessorBlockColumnCount,
        static_cast<float>(IrCameraIntensityMax) / (blockWidth * blockHeight),
        (IrCameraImageWidth + blockWidth) / 2.f - 1.f,
        (IrCameraImageHeight + blockHeight) / 2.f - 1.f);

    // 1 列のみ明るい状態
    for (auto& block : state.blocks)
    {
        block.averageIntensity = 0;
        block.centroid.x = 0;
        block.centroid.y = 0;
    }
    for (auto i = 0; i < MomentProcessorBlockRowCount; ++i)
    {
        auto& block = state.blocks[i * MomentProcessorBlockColumnCount];
        block.averageIntensity = IrCameraIntensityMax;
        block.centroid.x = (blockWidth - 1) / 2.f;
        block.centroid.y = blockHeight * i + (blockHeight - 1) / 2.f;
    }

    DoTestMomentRegionStatistic(
        &state,
        woi,
        0,
        0,
        MomentProcessorBlockRowCount,
        1,
        static_cast<float>(IrCameraIntensityMax),
        (blockWidth - 1) / 2.f,
        (IrCameraImageHeight - 1) / 2.f);

    // 1 行のみ明るい状態
    for (auto& block : state.blocks)
    {
        block.averageIntensity = 0;
        block.centroid.x = 0;
        block.centroid.y = 0;
    }
    for (auto i = 0; i < MomentProcessorBlockColumnCount; ++i)
    {
        auto& block = state.blocks[i];
        block.averageIntensity = IrCameraIntensityMax;
        block.centroid.x = blockWidth * i + (blockWidth - 1) / 2.f;
        block.centroid.y = (blockHeight - 1) / 2.f;
    }

    DoTestMomentRegionStatistic(
        &state,
        woi,
        0,
        0,
        1,
        MomentProcessorBlockColumnCount,
        static_cast<float>(IrCameraIntensityMax),
        (IrCameraImageWidth - 1) / 2.f,
        (blockHeight - 1) / 2.f);

    // 1 ブロックのみ明るい状態
    for (auto& block : state.blocks)
    {
        block.averageIntensity = 0;
        block.centroid.x = 0;
        block.centroid.y = 0;
    }
    {
        auto& block = state.blocks[MomentProcessorBlockCount - 1];
        block.averageIntensity = static_cast<float>(IrCameraIntensityMax) / (blockWidth * blockHeight);
        block.centroid.x = IrCameraImageWidth - 1;
        block.centroid.y = IrCameraImageHeight - 1;
    }

    DoTestMomentRegionStatistic(
        &state,
        woi,
        MomentProcessorBlockRowCount - 1,
        MomentProcessorBlockColumnCount - 1,
        1,
        1,
        static_cast<float>(IrCameraIntensityMax) / (blockWidth * blockHeight),
        IrCameraImageWidth - 1,
        IrCameraImageHeight - 1);


    // 四隅のブロックのみが明るい状態
    for (auto i = 0; i < MomentProcessorBlockRowCount; ++i)
    {
        for (auto j = 0; j < MomentProcessorBlockColumnCount; ++j)
        {
            if ( (i == 0 && j ==0)
                || (i == 0 && j == MomentProcessorBlockColumnCount - 1)
                || (i == MomentProcessorBlockRowCount - 1 && j == 0)
                || (i == MomentProcessorBlockRowCount - 1 && j == MomentProcessorBlockColumnCount - 1))
            {
                auto& block = state.blocks[j + i * MomentProcessorBlockColumnCount];
                block.averageIntensity = 255.f;
                block.centroid.x = blockWidth * j + (blockWidth - 1) / 2.f;
                block.centroid.y = blockHeight * i + (blockHeight - 1) / 2.f;
            }
            else
            {
                auto& block = state.blocks[j + i * MomentProcessorBlockColumnCount];
                block.averageIntensity = 0.f;
                block.centroid.x = 0.f;
                block.centroid.y = 0.f;
            }
        }
    }

    DoTestMomentRegionStatistic(
        &state,
        woi,
        0,
        0,
        MomentProcessorBlockRowCount,
        MomentProcessorBlockColumnCount,
        255.f * 4 / MomentProcessorBlockCount,
        (IrCameraImageWidth - 1) / 2.f,
        (IrCameraImageHeight - 1) / 2.f);
} // NOLINT(readability/fn_size)

} // namespace
