﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/irsensor.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nnt.h>
#include "../Common/testIrsensor_Util.h"

using namespace ::nn::irsensor;

namespace nnt { namespace irsensor {
    IrCameraHandle IrSensorTest::s_Handles[NpadIdCountMax];
    int IrSensorTest::s_HandleCount;
}}

namespace
{

// 最大サイズのバッファを用意しておく
NN_ALIGNAS(0x1000)
uint8_t s_WorkBuffer[ImageTransferProcessorQvgaWorkBufferSize];

template <typename T>
class IrCameraTest : public ::nnt::irsensor::IrSensorTest
{
};

typedef ::testing::Types<MomentProcessorConfig, ClusteringProcessorConfig, ImageTransferProcessorConfig> ProcessorConfigTypes;
TYPED_TEST_CASE(IrCameraTest, ProcessorConfigTypes);

template <typename T>
struct ImageProcessorState;

template <>
struct ImageProcessorState<MomentProcessorConfig>
{
    typedef MomentProcessorState type;
};

template <>
struct ImageProcessorState<ClusteringProcessorConfig>
{
    typedef ClusteringProcessorState type;
};

template <>
struct ImageProcessorState<ImageTransferProcessorConfig>
{
    typedef ImageTransferProcessorState type;
};

template <typename T>
struct ImageProcessorSet
{
    T config;
    ::nn::TimeSpanType exposureTimeMin;
    ::nn::TimeSpanType exposureTimeMax;
    typename ImageProcessorState<T>::type state;
};

template <typename T>
void GetImageProcessorSet(ImageProcessorSet<T>* pSet) NN_NOEXCEPT;

template <>
void GetImageProcessorSet(ImageProcessorSet<MomentProcessorConfig>* pSet) NN_NOEXCEPT
{
    GetMomentProcessorDefaultConfig(&pSet->config);
    pSet->exposureTimeMin = MomentProcessorExposureTimeMin;
    pSet->exposureTimeMax = MomentProcessorExposureTimeMax;
}

template <>
void GetImageProcessorSet(ImageProcessorSet<ClusteringProcessorConfig>* pSet) NN_NOEXCEPT
{
    GetClusteringProcessorDefaultConfig(&pSet->config);
    pSet->exposureTimeMin = ClusteringProcessorExposureTimeMin;
    pSet->exposureTimeMax = ClusteringProcessorExposureTimeMax;
}

template <>
void GetImageProcessorSet(ImageProcessorSet<ImageTransferProcessorConfig>* pSet) NN_NOEXCEPT
{
    GetImageTransferProcessorDefaultConfig(&pSet->config);

    pSet->config.format = ImageTransferProcessorFormat_Qvga;

    pSet->exposureTimeMin = ImageTransferProcessorExposureTimeMin;
    pSet->exposureTimeMax = ImageTransferProcessorExposureTimeMax;
}

void RunImageProcessor(const IrCameraHandle& handle, const MomentProcessorConfig& config) NN_NOEXCEPT
{
    RunMomentProcessor(handle, config);
}

void RunImageProcessor(const IrCameraHandle& handle, const ClusteringProcessorConfig& config) NN_NOEXCEPT
{
    RunClusteringProcessor(handle, config);
}

void RunImageProcessor(const IrCameraHandle& handle, const ImageTransferProcessorConfig& config) NN_NOEXCEPT
{
    auto buffer = s_WorkBuffer;

    RunImageTransferProcessor(handle, config, buffer, ImageTransferProcessorQvgaWorkBufferSize);
}

template <typename T>
void DoTestExposureTimeBoundary(const IrCameraHandle& handle, ::nn::TimeSpanType time) NN_NOEXCEPT
{
    ImageProcessorSet<T> processorSet;
    GetImageProcessorSet(&processorSet);

    auto config = processorSet.config;
    config.irCameraConfig.exposureTime = time;

    RunImageProcessor(handle, config);
    StopImageProcessor(handle);

    NNT_IRSENSOR_EXIT_SUCCESS;
}

TYPED_TEST(IrCameraTest, ExposureTimeBoundary)
{
    EXPECT_LT(0, TestFixture::s_HandleCount);

    for (auto i = 0; i < TestFixture::s_HandleCount; ++i)
    {
        const auto& handle = TestFixture::s_Handles[i];

        ImageProcessorSet<TypeParam> processorSet;
        GetImageProcessorSet(&processorSet);

        auto min = processorSet.exposureTimeMin;
        auto max = processorSet.exposureTimeMax;

        NNT_IRSENSOR_EXPECT_EXIT(DoTestExposureTimeBoundary<TypeParam>(handle, min), NNT_IRSENSOR_EXIT_0, "");
        NNT_IRSENSOR_EXPECT_EXIT(DoTestExposureTimeBoundary<TypeParam>(handle, max), NNT_IRSENSOR_EXIT_0, "");

        EXPECT_DEATH_IF_SUPPORTED(DoTestExposureTimeBoundary<TypeParam>(handle, min - ::nn::TimeSpanType::FromNanoSeconds(1)), "");
        EXPECT_DEATH_IF_SUPPORTED(DoTestExposureTimeBoundary<TypeParam>(handle, max + ::nn::TimeSpanType::FromNanoSeconds(1)), "");
    }
}

template <typename T>
void DoTestGainBoundary(const IrCameraHandle& handle, int gain) NN_NOEXCEPT
{
    ImageProcessorSet<T> processorSet;
    GetImageProcessorSet(&processorSet);

    auto config = processorSet.config;
    config.irCameraConfig.gain = gain;

    RunImageProcessor(handle, config);
    StopImageProcessor(handle);

    NNT_IRSENSOR_EXIT_SUCCESS;
}

TYPED_TEST(IrCameraTest, GainBoundary)
{
    EXPECT_LT(0, TestFixture::s_HandleCount);

    for (auto i = 0; i < TestFixture::s_HandleCount; ++i)
    {
        const auto& handle = TestFixture::s_Handles[i];

        auto min = IrCameraGainMin;
        auto max = IrCameraGainMax;

        NNT_IRSENSOR_EXPECT_EXIT(DoTestGainBoundary<TypeParam>(handle, min), NNT_IRSENSOR_EXIT_0, "");
        NNT_IRSENSOR_EXPECT_EXIT(DoTestGainBoundary<TypeParam>(handle, max), NNT_IRSENSOR_EXIT_0, "");

        EXPECT_DEATH_IF_SUPPORTED(DoTestGainBoundary<TypeParam>(handle, min - 1), "");
        EXPECT_DEATH_IF_SUPPORTED(DoTestGainBoundary<TypeParam>(handle, max + 1), "");
    }
}

template <typename T>
void DoTestNegativeImageUsedBoundary(const IrCameraHandle& handle, bool isNegativeImageUsed) NN_NOEXCEPT
{
    ImageProcessorSet<T> processorSet;
    GetImageProcessorSet(&processorSet);

    auto config = processorSet.config;
    config.irCameraConfig.isNegativeImageUsed = isNegativeImageUsed;

    RunImageProcessor(handle, config);
    StopImageProcessor(handle);

    NNT_IRSENSOR_EXIT_SUCCESS;
}

TYPED_TEST(IrCameraTest, NegativeImageUsedBoundary)
{
    EXPECT_LT(0, TestFixture::s_HandleCount);

    for (auto i = 0; i < TestFixture::s_HandleCount; ++i)
    {
        const auto& handle = TestFixture::s_Handles[i];

        NNT_IRSENSOR_EXPECT_EXIT(DoTestNegativeImageUsedBoundary<TypeParam>(handle, false), NNT_IRSENSOR_EXIT_0, "");
        NNT_IRSENSOR_EXPECT_EXIT(DoTestNegativeImageUsedBoundary<TypeParam>(handle, true), NNT_IRSENSOR_EXIT_0, "");
    }
}

template <typename T>
void DoTestLightTargetBoundary(const IrCameraHandle& handle, IrCameraLightTarget target) NN_NOEXCEPT
{
    ImageProcessorSet<T> processorSet;
    GetImageProcessorSet(&processorSet);

    auto config = processorSet.config;
    config.irCameraConfig.lightTarget = target;

    RunImageProcessor(handle, config);
    StopImageProcessor(handle);

    NNT_IRSENSOR_EXIT_SUCCESS;
}

TYPED_TEST(IrCameraTest, LightTargetBoundary)
{
    EXPECT_LT(0, TestFixture::s_HandleCount);

    for (auto i = 0; i < TestFixture::s_HandleCount; ++i)
    {
        const auto& handle = TestFixture::s_Handles[i];

        NNT_IRSENSOR_EXPECT_EXIT(DoTestLightTargetBoundary<TypeParam>(handle, IrCameraLightTarget_AllObjects), NNT_IRSENSOR_EXIT_0, "");
        NNT_IRSENSOR_EXPECT_EXIT(DoTestLightTargetBoundary<TypeParam>(handle, IrCameraLightTarget_NearObjects), NNT_IRSENSOR_EXIT_0, "");
        NNT_IRSENSOR_EXPECT_EXIT(DoTestLightTargetBoundary<TypeParam>(handle, IrCameraLightTarget_FarObjects), NNT_IRSENSOR_EXIT_0, "");
        NNT_IRSENSOR_EXPECT_EXIT(DoTestLightTargetBoundary<TypeParam>(handle, IrCameraLightTarget_None), NNT_IRSENSOR_EXIT_0, "");

        EXPECT_DEATH_IF_SUPPORTED(DoTestLightTargetBoundary<TypeParam>(handle, static_cast<IrCameraLightTarget>(-1)), "");
        EXPECT_DEATH_IF_SUPPORTED(DoTestLightTargetBoundary<TypeParam>(handle, static_cast<IrCameraLightTarget>(4)), "");
    }
}

} // namespace
