﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/irsensor.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nnt.h>
#include "../Common/testIrsensor_Util.h"

using namespace ::nn::irsensor;

namespace nnt { namespace irsensor {
    IrCameraHandle IrSensorTest::s_Handles[NpadIdCountMax];
    int IrSensorTest::s_HandleCount;
}}

namespace
{
const int NotReadyWaitLoopCountMax = 100;
const nn::TimeSpan PollingInterval = nn::TimeSpan::FromMilliSeconds(15);

// 最大サイズのバッファを用意しておく
NN_ALIGNAS(0x1000)
uint8_t s_WorkBuffer[ImageTransferProcessorWorkBufferSize320x240];

class ImageTransferProcessorTest : public ::nnt::irsensor::IrSensorTest
{
protected:
    void DoTestFormatBoundary(ImageTransferProcessorFormat format, void* buffer, size_t size) NN_NOEXCEPT;
};

TEST_F(ImageTransferProcessorTest, DefaultImageTransferProcessorConfig)
{
    ImageTransferProcessorConfig config;
    GetImageTransferProcessorDefaultConfig(&config);

    EXPECT_LE(ImageTransferProcessorExposureTimeMin, config.irCameraConfig.exposureTime);
    EXPECT_LE(config.irCameraConfig.exposureTime, ImageTransferProcessorExposureTimeMax);
    EXPECT_LE(IrCameraGainMin, config.irCameraConfig.gain);
    EXPECT_LE(config.irCameraConfig.gain, IrCameraGainMax);
    EXPECT_EQ(false, config.irCameraConfig.isNegativeImageUsed);
    EXPECT_EQ(IrCameraLightTarget_AllObjects, config.irCameraConfig.lightTarget);
}

void ImageTransferProcessorTest::DoTestFormatBoundary(ImageTransferProcessorFormat format, void* buffer, size_t size) NN_NOEXCEPT
{
    EXPECT_LT(0, s_HandleCount);

    for (auto i = 0; i < s_HandleCount; ++i)
    {
        const auto& handle = s_Handles[i];

        ImageTransferProcessorConfig config;
        GetImageTransferProcessorDefaultConfig(&config);

        config.format = format;
        RunImageTransferProcessor(handle, config, buffer, size);
        StopImageProcessor(handle);
    }

    NNT_IRSENSOR_EXIT_SUCCESS;
}

TEST_F(ImageTransferProcessorTest, FormatBoundary)
{
    auto buffer = s_WorkBuffer;

    NNT_IRSENSOR_EXPECT_EXIT(DoTestFormatBoundary(ImageTransferProcessorFormat_320x240, buffer, ImageTransferProcessorWorkBufferSize320x240), NNT_IRSENSOR_EXIT_0, "");
    NNT_IRSENSOR_EXPECT_EXIT(DoTestFormatBoundary(ImageTransferProcessorFormat_160x120, buffer, ImageTransferProcessorWorkBufferSize160x120), NNT_IRSENSOR_EXIT_0, "");
    NNT_IRSENSOR_EXPECT_EXIT(DoTestFormatBoundary(ImageTransferProcessorFormat_80x60, buffer, ImageTransferProcessorWorkBufferSize80x60), NNT_IRSENSOR_EXIT_0, "");
    NNT_IRSENSOR_EXPECT_EXIT(DoTestFormatBoundary(ImageTransferProcessorFormat_40x30, buffer, ImageTransferProcessorWorkBufferSize40x30), NNT_IRSENSOR_EXIT_0, "");
    NNT_IRSENSOR_EXPECT_EXIT(DoTestFormatBoundary(ImageTransferProcessorFormat_20x15, buffer, ImageTransferProcessorWorkBufferSize20x15), NNT_IRSENSOR_EXIT_0, "");

    EXPECT_DEATH_IF_SUPPORTED(DoTestFormatBoundary(static_cast<ImageTransferProcessorFormat>(-1), buffer, ImageTransferProcessorWorkBufferSize320x240), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestFormatBoundary(static_cast<ImageTransferProcessorFormat>(5), buffer, ImageTransferProcessorWorkBufferSize320x240), "");

    EXPECT_DEATH_IF_SUPPORTED(DoTestFormatBoundary(ImageTransferProcessorFormat_320x240, buffer, ImageTransferProcessorWorkBufferSize320x240 - 1), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestFormatBoundary(ImageTransferProcessorFormat_160x120, buffer, ImageTransferProcessorWorkBufferSize160x120 - 1), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestFormatBoundary(ImageTransferProcessorFormat_80x60, buffer, ImageTransferProcessorWorkBufferSize80x60 - 1), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestFormatBoundary(ImageTransferProcessorFormat_40x30, buffer, ImageTransferProcessorWorkBufferSize40x30 - 1), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestFormatBoundary(ImageTransferProcessorFormat_20x15, buffer, ImageTransferProcessorWorkBufferSize20x15 - 1), "");

    EXPECT_DEATH_IF_SUPPORTED(DoTestFormatBoundary(ImageTransferProcessorFormat_320x240, nullptr, ImageTransferProcessorWorkBufferSize320x240), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestFormatBoundary(ImageTransferProcessorFormat_160x120, nullptr, ImageTransferProcessorWorkBufferSize160x120), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestFormatBoundary(ImageTransferProcessorFormat_80x60, nullptr, ImageTransferProcessorWorkBufferSize80x60), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestFormatBoundary(ImageTransferProcessorFormat_40x30, nullptr, ImageTransferProcessorWorkBufferSize40x30), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestFormatBoundary(ImageTransferProcessorFormat_20x15, nullptr, ImageTransferProcessorWorkBufferSize20x15), "");
}

TEST_F(ImageTransferProcessorTest, GetImageTransferProcessorState)
{
    EXPECT_LT(0, s_HandleCount);

    for (auto i = 0; i < s_HandleCount; ++i)
    {
        const auto& handle = s_Handles[i];
        auto workBuffer = s_WorkBuffer;

        ImageTransferProcessorConfig config;
        GetImageTransferProcessorDefaultConfig(&config);
        config.format = ImageTransferProcessorFormat_320x240;

        // ASSERT_EQ(IrCameraStatus_Available, GetIrCameraStatus(handle));
        RunImageTransferProcessor(handle, config, workBuffer, ImageTransferProcessorWorkBufferSize320x240);

        ImageTransferProcessorState state;
        uint8_t buffer[ImageTransferProcessorImageSize320x240];
        // NNT_ASSERT_RESULT_SUCCESS(GetImageTransferProcessorState(&state, buffer, ImageTransferProcessorQvgaImageSize, handle));
        nn::Result result = nn::ResultSuccess();
        int counter = 0;
        while (NN_STATIC_CONDITION(true))
        {
            result = GetImageTransferProcessorState(&state, buffer, ImageTransferProcessorImageSize320x240, handle);
            if (nn::irsensor::ResultIrsensorNotReady::Includes(result))
            {
                counter++;
                nn::os::SleepThread(PollingInterval);
                ASSERT_LT(counter, NotReadyWaitLoopCountMax);
            }
            else
            {
                break;
            }
        }
        EXPECT_LE(0, state.samplingNumber);

        StopImageProcessor(handle);
    }
}

} // namespace
