﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/mem.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Tick.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/irsensor.h>
#include <nnt.h>

#include "../Common/testIrsensor_Util.h"
#include "testIrSensor_ApiResponseMeasure.h"

using namespace ::nn::irsensor;

namespace
{
nn::hid::NpadIdType g_NpadIds[] = {
    nn::hid::NpadId::No1,
};

const int NpadIdCountMax = sizeof(g_NpadIds) / sizeof(g_NpadIds[0]);

enum ProcessorMode
{
    Moment,
    Clustering,
    ImageTransfer,
    Pointing,
    HandAnalysis,
    ProcessorModeCountMax,
};
ProcessorMode g_Mode = ProcessorMode::Moment;

static ::nn::irsensor::IrCameraHandle s_Handles[NpadIdCountMax];
static int s_HandleCount;

// 計測ループ数
const auto RepeatCountMax = 3;
// データ取得ループ数
const auto SamplingCountMax = 100;
// コマンド呼び出しの定常インターバル
const auto CommandInterval = nn::TimeSpan::FromMilliSeconds(15);
// 起動直後のインターバル
const auto InitialInterval = nn::TimeSpan::FromSeconds(2);

// 各API の期待される処理時間
//
const auto InitializeResponseTime = nn::TimeSpan::FromMilliSeconds(10);
const auto FinalizeResponseTime = nn::TimeSpan::FromMilliSeconds(3);
const auto GetIrCameraHandleResponseTime = nn::TimeSpan::FromMicroSeconds(150);
const auto GetIrCameraStatusResponseTime = nn::TimeSpan::FromMicroSeconds(5);
const auto StopImageProcessorResponseTime = nn::TimeSpan::FromMilliSeconds(300);
const auto GetMomentDefaultResponseTime = nn::TimeSpan::FromMicroSeconds(1);
const auto RunMomentResponseTime = nn::TimeSpan::FromMicroSeconds(30);
const auto GetMomentResponseTime = nn::TimeSpan::FromMicroSeconds(5);
const auto GetClusteringDefaultResponseTime = nn::TimeSpan::FromMicroSeconds(10);
const auto RunClusteringResponseTime = nn::TimeSpan::FromMicroSeconds(30);
const auto GetClusteringResponseTime = nn::TimeSpan::FromMicroSeconds(5);
const auto GetImageTransferDefaultResponseTime = nn::TimeSpan::FromMicroSeconds(1);
const auto RunImageTransferResponseTime = nn::TimeSpan::FromMicroSeconds(30);
const auto GetImageTransferResponseTime = nn::TimeSpan::FromMilliSeconds(5);
// const auto GetPointingDefaultResponseTime = nn::TimeSpan::FromMicroSeconds(1);
const auto RunPointingResponseTime = nn::TimeSpan::FromMicroSeconds(30);
const auto GetPointingResponseTime = nn::TimeSpan::FromMicroSeconds(5);
const auto RunHandAnalysisResponseTime = nn::TimeSpan::FromMicroSeconds(30);
const auto GetHandAnalysisResponseTime = nn::TimeSpan::FromMicroSeconds(30);

TEST(ApiResponseTimeTest, ResponseTest)
{
    ::nnt::irsensor::ApiResponseMeasure apiResponseInitialize;
    ::nnt::irsensor::ApiResponseMeasure apiResponseFinalize;
    ::nnt::irsensor::ApiResponseMeasure apiResponseGetIrCameraHandle;
    ::nnt::irsensor::ApiResponseMeasure apiResponseGetIrCameraStatus;
    ::nnt::irsensor::ApiResponseMeasure apiResponseStopImageProcessor;
    ::nnt::irsensor::ApiResponseMeasure apiResponseGetMomentDefault;
    ::nnt::irsensor::ApiResponseMeasure apiResponseRunMoment;
    ::nnt::irsensor::ApiResponseMeasure apiResponseGetMoment;
    ::nnt::irsensor::ApiResponseMeasure apiResponseGetClusteringDefault;
    ::nnt::irsensor::ApiResponseMeasure apiResponseRunClustering;
    ::nnt::irsensor::ApiResponseMeasure apiResponseGetClustering;
    ::nnt::irsensor::ApiResponseMeasure apiResponseGetImageTransferDefault;
    ::nnt::irsensor::ApiResponseMeasure apiResponseRunImageTransfer;
    ::nnt::irsensor::ApiResponseMeasure apiResponseGetImageTransfer;
    // ::nnt::irsensor::ApiResponseMeasure apiResponseGetPointingDefault;
    ::nnt::irsensor::ApiResponseMeasure apiResponseRunPointing;
    ::nnt::irsensor::ApiResponseMeasure apiResponseGetPointing;
    ::nnt::irsensor::ApiResponseMeasure apiResponseRunHandAnalysis;
    ::nnt::irsensor::ApiResponseMeasure apiResponseGetHandAnalysis;

    nn::hid::InitializeNpad();

    s_HandleCount = NpadIdCountMax;
    // RightJoy のみ使用するモードに設定
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleJoyRight::Mask);

    nn::hid::SetSupportedNpadIdType(g_NpadIds, NpadIdCountMax);
    for (auto i = 0; i < s_HandleCount; ++i)
    {
        auto id = g_NpadIds[i];
        apiResponseGetIrCameraHandle.MeasureFunction([&] {
            s_Handles[i] = ::nn::irsensor::GetIrCameraHandle(id);
        });
    }

    nn::os::SleepThread(InitialInterval);

    for (auto loop = 0; loop < RepeatCountMax; loop++)
    {
        NN_LOG("TestLoop:%d Start\n", loop);
        for (auto i = 0; i < s_HandleCount; ++i)
        {
            apiResponseInitialize.MeasureFunction([&]{
                ::nn::irsensor::Initialize(s_Handles[i]);
            });
        }
        // 少なくとも1台1以上のIR センサー使用可能なデバイスが接続されたかどうか。
        bool isNeverConnected = true;

        for (auto j = 0; j < s_HandleCount; ++j)
        {
            const auto& handle = s_Handles[j];

            auto counter = 0;
            bool isConnected = false;
            while (NN_STATIC_CONDITION(true))
            {
                const auto StatusChangeCountMax = 5;

                apiResponseGetIrCameraStatus.MeasureFunction([&] {
                    GetIrCameraStatus(handle);
                });
                auto status = GetIrCameraStatus(handle);
                if (status == IrCameraStatus_Available)
                {
                    isConnected = true;
                    isNeverConnected = false;
                    break;
                }
                // 15ms のインターバル
                nn::os::SleepThread(CommandInterval);
                counter++;
                if (counter > StatusChangeCountMax)
                {
                    break;
                }
            }

            if (isConnected)
            {
                for (auto i = 0; i < ProcessorModeCountMax; i++)
                {
                    switch (g_Mode)
                    {
                    case Moment:
                        {
                            MomentProcessorConfig config;
                            apiResponseGetMomentDefault.MeasureFunction([&] {
                                GetMomentProcessorDefaultConfig(&config);
                            });
                            apiResponseRunMoment.MeasureFunction([&] {
                                RunMomentProcessor(handle, config);
                            });
                            for (auto k = 0; k < SamplingCountMax; ++k)
                            {
                                // 15ms のインターバル
                                nn::os::SleepThread(CommandInterval);

                                MomentProcessorState state;
                                apiResponseGetMoment.MeasureFunction([&] {
                                    GetMomentProcessorState(&state, handle);
                                });
                                nn::Result result = GetMomentProcessorState(&state, handle);
                                if (nn::irsensor::ResultIrsensorNotReady::Includes(result))
                                {
                                    // 少なくとも GetStateCount 回目のデータは取得できていることを確認する
                                    //EXPECT_NE(SamplingCountMax - 1, k);
                                    //continue;
                                }
                            }
                        }
                        break;
                    case Clustering:
                        {
                            ClusteringProcessorConfig config;
                            apiResponseGetClusteringDefault.MeasureFunction([&] {
                                GetClusteringProcessorDefaultConfig(&config);
                            });

                            apiResponseRunClustering.MeasureFunction([&] {
                                RunClusteringProcessor(handle, config);
                            });
                            for (auto k = 0; k < SamplingCountMax; ++k)
                            {
                                // 15ms のインターバル
                                nn::os::SleepThread(CommandInterval);

                                ClusteringProcessorState state;
                                apiResponseGetClustering.MeasureFunction([&] {
                                    GetClusteringProcessorState(&state, handle);
                                });
                                nn::Result result = GetClusteringProcessorState(&state, handle);
                                if (nn::irsensor::ResultIrsensorNotReady::Includes(result))
                                {
                                    // 少なくとも GetStateCount 回目のデータは取得できていることを確認する
                                    //EXPECT_NE(SamplingCountMax - 1, k);
                                    //continue;
                                }
                            }
                        }
                        break;
                    case ImageTransfer:
                        {
                            ImageTransferProcessorConfig config;
                            apiResponseGetImageTransferDefault.MeasureFunction([&] {
                                GetImageTransferProcessorDefaultConfig(&config);
                            });

                            const size_t ApplicationHeapSize =
                                (nn::irsensor::ImageTransferProcessorQvgaWorkBufferSize + nn::irsensor::ImageTransferProcessorQvgaImageSize) * 2;

                            void* budgetBuffer = new nn::Bit8[ApplicationHeapSize];
                            nn::mem::StandardAllocator applicationHeap;
                            applicationHeap.Initialize(budgetBuffer, ApplicationHeapSize);
                            void* workBuffer = applicationHeap.Allocate(nn::irsensor::ImageTransferProcessorQvgaWorkBufferSize, nn::os::MemoryPageSize);

                            apiResponseRunImageTransfer.MeasureFunction([&] {
                                RunImageTransferProcessor(handle, config, workBuffer, nn::irsensor::ImageTransferProcessorQvgaWorkBufferSize);
                            });

                            void* imageBuffer = applicationHeap.Allocate(nn::irsensor::ImageTransferProcessorQvgaImageSize, nn::os::MemoryPageSize);
                            for (auto k = 0; k < SamplingCountMax; ++k)
                            {
                                // 15ms のインターバル
                                nn::os::SleepThread(CommandInterval);

                                ImageTransferProcessorState state;
                                apiResponseGetImageTransfer.MeasureFunction([&] {
                                    GetImageTransferProcessorState(&state, imageBuffer, nn::irsensor::ImageTransferProcessorQvgaImageSize, handle);
                                });
                            }
                            applicationHeap.Free(imageBuffer);
                            applicationHeap.Free(workBuffer);
                            applicationHeap.Finalize();
                            delete[] reinterpret_cast<nn::Bit8*>(budgetBuffer);
                        }
                        break;
                    case Pointing:
                        {
                            // PointingProcessorConfig config;
                            // apiResponseGetPointingDefault.MeasureFunction([&] {
                            //     GetPointingProcessorDefaultConfig(&config);
                            // });

                            apiResponseRunPointing.MeasureFunction([&] {
                                RunPointingProcessor(handle);
                            });
                            for (auto k = 0; k < SamplingCountMax; ++k)
                            {
                                // 15ms のインターバル
                                nn::os::SleepThread(CommandInterval);

                                PointingProcessorState state;
                                int count = 3;
                                int outCount = 0;
                                apiResponseGetPointing.MeasureFunction([&] {
                                    GetPointingProcessorStates(&state, &outCount, count, handle);
                                });
                                nn::Result result = GetPointingProcessorStates(&state, &outCount, count, handle);
                                if (nn::irsensor::ResultIrsensorNotReady::Includes(result))
                                {
                                    // 少なくとも GetStateCount 回目のデータは取得できていることを確認する
                                    //EXPECT_NE(SamplingCountMax - 1, k);
                                    //continue;
                                }
                            }
                        }
                        break;
                    case HandAnalysis:
                        {
                            HandAnalysisConfig config;
                            config.mode = HandAnalysisMode_SilhouetteAndImage;
                            apiResponseRunHandAnalysis.MeasureFunction([&] {
                                RunHandAnalysis(handle, config);
                            });

                            for (auto k = 0; k < SamplingCountMax; ++k)
                            {
                                // 15ms のインターバル
                                nn::os::SleepThread(CommandInterval);

                                HandAnalysisSilhouetteState state;
                                int count = 1;
                                int outCount = 0;
                                int64_t samplingNum = 0;
                                apiResponseGetHandAnalysis.MeasureFunction([&] {
                                    GetHandAnalysisSilhouetteState(&state, &outCount, count, samplingNum, handle);
                                });
                            }
                        }
                        break;
                    default:
                        NN_UNEXPECTED_DEFAULT;
                    }
                    apiResponseStopImageProcessor.MeasureFunction([&] {
                        StopImageProcessor(handle);
                    });
                    g_Mode = static_cast<ProcessorMode>((g_Mode + 1) % ProcessorModeCountMax);
                }
            }
            nn::os::SleepThread(CommandInterval);
        }

        for (auto i = 0; i < s_HandleCount; ++i)
        {
            apiResponseFinalize.MeasureFunction([&] {
                ::nn::irsensor::Finalize(s_Handles[i]);
            });
        }
        // 1台も接続がなかった場合は失敗
        EXPECT_FALSE(isNeverConnected);
    }

    // 結果の表示
    NN_LOG("Initialize: \t%lld[ns]\n", apiResponseInitialize.GetAverageResponseTime().GetNanoSeconds());
    NN_LOG("Finalize: \t%lld[ns]\n", apiResponseFinalize.GetAverageResponseTime().GetNanoSeconds());
    NN_LOG("GetIrCameraHandle: \t%lld[ns]\n", apiResponseGetIrCameraHandle.GetAverageResponseTime().GetNanoSeconds());
    NN_LOG("GetIrCameraStatus: \t%lld[ns]\n", apiResponseGetIrCameraStatus.GetAverageResponseTime().GetNanoSeconds());
    NN_LOG("StopImageProcessor: \t%lld[ns]\n", apiResponseStopImageProcessor.GetAverageResponseTime().GetNanoSeconds());
    NN_LOG("GetMomentDefault: \t%lld[ns]\n", apiResponseGetMomentDefault.GetAverageResponseTime().GetNanoSeconds());
    NN_LOG("RunMoment: \t%lld[ns]\n", apiResponseRunMoment.GetAverageResponseTime().GetNanoSeconds());
    NN_LOG("GetMoment: \t%lld[ns]\n", apiResponseGetMoment.GetAverageResponseTime().GetNanoSeconds());
    NN_LOG("GetClusteringDefault: \t%lld[ns]\n", apiResponseGetClusteringDefault.GetAverageResponseTime().GetNanoSeconds());
    NN_LOG("RunClustering: \t%lld[ns]\n", apiResponseRunClustering.GetAverageResponseTime().GetNanoSeconds());
    NN_LOG("GetClustering: \t%lld[ns]\n", apiResponseGetClustering.GetAverageResponseTime().GetNanoSeconds());
    NN_LOG("GetImageTransferDefault: \t%lld[ns]\n", apiResponseGetImageTransferDefault.GetAverageResponseTime().GetNanoSeconds());
    NN_LOG("RunImageTransfer: \t%lld[ns]\n", apiResponseRunImageTransfer.GetAverageResponseTime().GetNanoSeconds());
    NN_LOG("GetImageTransfer: \t%lld[ns]\n", apiResponseGetImageTransfer.GetAverageResponseTime().GetNanoSeconds());
    // NN_LOG("GetPointingDefault: \t%lld[ns]\n", apiResponseGetPointingDefault.GetAverageResponseTime().GetNanoSeconds());
    NN_LOG("RunPointing: \t%lld[ns]\n", apiResponseRunPointing.GetAverageResponseTime().GetNanoSeconds());
    NN_LOG("GetPointing: \t%lld[ns]\n", apiResponseGetPointing.GetAverageResponseTime().GetNanoSeconds());
    NN_LOG("RunHandAnalysis: \t%lld[ns]\n", apiResponseRunHandAnalysis.GetAverageResponseTime().GetNanoSeconds());
    NN_LOG("GetHandAnalysis: \t%lld[ns]\n", apiResponseGetHandAnalysis.GetAverageResponseTime().GetNanoSeconds());

    // 結果のチェック
    EXPECT_LT(apiResponseInitialize.GetAverageResponseTime().GetNanoSeconds(), InitializeResponseTime.GetNanoSeconds());
    EXPECT_LT(apiResponseFinalize.GetAverageResponseTime().GetNanoSeconds(), FinalizeResponseTime.GetNanoSeconds());
    EXPECT_LT(apiResponseGetIrCameraHandle.GetAverageResponseTime().GetNanoSeconds(), GetIrCameraHandleResponseTime.GetNanoSeconds());
    EXPECT_LT(apiResponseGetIrCameraStatus.GetAverageResponseTime().GetNanoSeconds(), GetIrCameraStatusResponseTime.GetNanoSeconds());
    EXPECT_LT(apiResponseStopImageProcessor.GetAverageResponseTime().GetNanoSeconds(), StopImageProcessorResponseTime.GetNanoSeconds());
    EXPECT_LT(apiResponseGetMomentDefault.GetAverageResponseTime().GetNanoSeconds(), GetMomentDefaultResponseTime.GetNanoSeconds());
    EXPECT_LT(apiResponseRunMoment.GetAverageResponseTime().GetNanoSeconds(), RunMomentResponseTime.GetNanoSeconds());
    EXPECT_LT(apiResponseGetMoment.GetAverageResponseTime().GetNanoSeconds(), GetMomentResponseTime.GetNanoSeconds());
    EXPECT_LT(apiResponseGetClusteringDefault.GetAverageResponseTime().GetNanoSeconds(), GetClusteringDefaultResponseTime.GetNanoSeconds());
    EXPECT_LT(apiResponseRunClustering.GetAverageResponseTime().GetNanoSeconds(), RunClusteringResponseTime.GetNanoSeconds());
    EXPECT_LT(apiResponseGetClustering.GetAverageResponseTime().GetNanoSeconds(), GetClusteringResponseTime.GetNanoSeconds());
    EXPECT_LT(apiResponseGetImageTransferDefault.GetAverageResponseTime().GetNanoSeconds(), GetImageTransferDefaultResponseTime.GetNanoSeconds());
    EXPECT_LT(apiResponseRunImageTransfer.GetAverageResponseTime().GetNanoSeconds(), RunImageTransferResponseTime.GetNanoSeconds());
    EXPECT_LT(apiResponseGetImageTransfer.GetAverageResponseTime().GetNanoSeconds(), GetImageTransferResponseTime.GetNanoSeconds());
    // EXPECT_LT(apiResponseGetPointingDefault.GetAverageResponseTime().GetNanoSeconds(), GetPointingDefaultResponseTime.GetNanoSeconds());
    EXPECT_LT(apiResponseRunPointing.GetAverageResponseTime().GetNanoSeconds(), RunPointingResponseTime.GetNanoSeconds());
    EXPECT_LT(apiResponseGetPointing.GetAverageResponseTime().GetNanoSeconds(), GetPointingResponseTime.GetNanoSeconds());
    EXPECT_LT(apiResponseRunHandAnalysis.GetAverageResponseTime().GetNanoSeconds(), RunHandAnalysisResponseTime.GetNanoSeconds());
    EXPECT_LT(apiResponseGetHandAnalysis.GetAverageResponseTime().GetNanoSeconds(), GetHandAnalysisResponseTime.GetNanoSeconds());

    NNT_IRSENSOR_EXIT_SUCCESS;
}//NOLINT(readability/fn_size)

} // namespace
