﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//-----------------------------------------------------------------------------
//  nninitSetMallocRegion 動作確認テスト
//-----------------------------------------------------------------------------

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_SdkText.h>
#include <nn/TargetConfigs/build_Compiler.h>

#if defined(NN_BUILD_CONFIG_COMPILER_VC)
    // 日本語以外の環境で表示される文字コードエンコーディングに関する警告の抑制
    #pragma warning( disable : 4566 )
#endif

#include <cstdlib>
#include <cstring>
#include <cerrno>
#include <malloc.h>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/os/os_SdkMemoryAllocatorForThreadLocal.h>

#include <nnt/nntest.h>

#if defined(NN_BUILD_CONFIG_COMPILER_GCC)
extern "C" void* aligned_alloc(size_t alignment, size_t size);
#endif

namespace nn { namespace os {
    uintptr_t   GetMemoryHeapAddress();
    size_t      GetMemoryHeapSize();
}}  // namespace nn::os


namespace nnt { namespace init { namespace startup {

namespace {
    void* g_MallocBegin;
    void* g_MallocEnd;
}

#define NNT_ABORT_IF_NOT_ALIGNED32(value)  \
        NN_ABORT_UNLESS((static_cast<uintptr_t>(value) & (sizeof(uint32_t) - 1)) == 0)

#if defined(NN_BUILD_CONFIG_COMPILER_GCC) || \
    defined(NN_BUILD_CONFIG_COMPILER_CLANG)
//-----------------------------------------------------------------------------

void QueryMallocRegion()
{
    uintptr_t address = nn::os::GetMemoryHeapAddress();
    size_t    size    = nn::os::GetMemoryHeapSize();

    g_MallocBegin = reinterpret_cast<void*>( address );
    g_MallocEnd   = reinterpret_cast<void*>( address + size );
}

//-----------------------------------------------------------------------------
//  start ～ size 分の領域が data8 の値で埋まっていることをチェックする。
//  start および size は 32bit にアライメントされたものを渡すべし。
//
bool CheckMemoryRegion32(void* start, size_t size, uint8_t data8)
{
    NNT_ABORT_IF_NOT_ALIGNED32(reinterpret_cast<uintptr_t>(start));
    NNT_ABORT_IF_NOT_ALIGNED32(size);

    uint32_t data32 = (static_cast<uint32_t>(data8) << 24) |
                      (static_cast<uint32_t>(data8) << 16) |
                      (static_cast<uint32_t>(data8) <<  8) |
                      (static_cast<uint32_t>(data8) <<  0);

    uint32_t* address    = reinterpret_cast<uint32_t*>(start);
    uint32_t* endAddress = address + (size / sizeof(uint32_t));

    while (address < endAddress)
    {
        if (*address != data32)
        {
            return false;
        }
        ++address;
    }
    return true;
}

//-----------------------------------------------------------------------------
// malloc の呼出し、確保した領域のチェック、free の呼出し
//
TEST(nninitSetMallocRegionTest, testMalloc)
{
    QueryMallocRegion();

    const size_t mallocSize = 5000;

    NN_LOG(NN_TEXT("malloc の呼出し\n"));
    void* address = malloc( mallocSize );
    EXPECT_TRUE( address != NULL );

    NN_LOG(NN_TEXT("malloc の獲得アドレスが正しいか address=0x%p\n"), address);
    EXPECT_TRUE( g_MallocBegin <= address && address < g_MallocEnd );

    NN_LOG(NN_TEXT("malloc の獲得領域を 0x55 で埋めておく\n"));
    std::memset(address, 0x55, mallocSize);

    NN_LOG(NN_TEXT("free の呼出し\n"));
    free( address );
    EXPECT_TRUE( true );
}

//-----------------------------------------------------------------------------
// aligned_alloc の呼出し、確保した領域のチェック、free の呼出し
//
TEST(nninitSetMallocRegionTest, testAlignedAlloc)
{
    const size_t alignedAllocSize = 5000;
    const size_t alignedAllocAlignment = 128;

    NN_LOG(NN_TEXT("aligned_alloc の呼出し\n"));
    void* address = aligned_alloc( alignedAllocAlignment, alignedAllocSize );
    EXPECT_TRUE( address != NULL );
    EXPECT_TRUE( (reinterpret_cast<uintptr_t>(address) % alignedAllocAlignment) == 0 );

    NN_LOG(NN_TEXT("aligned_alloc の獲得アドレスが正しいか address=0x%p\n"), address);
    EXPECT_TRUE( g_MallocBegin <= address && address < g_MallocEnd );

    NN_LOG(NN_TEXT("aligned_alloc の獲得領域を 0xAA で埋めておく\n"));
    std::memset(address, 0xAA, alignedAllocSize);

    NN_LOG(NN_TEXT("free の呼出し\n"));
    free( address );
    EXPECT_TRUE( true );
}

//-----------------------------------------------------------------------------
// calloc の呼出し、確保した領域のチェック、free の呼出し
//
TEST(nninitSetMallocRegionTest, testCalloc)
{
    QueryMallocRegion();

    const size_t callocSize = 5000;

    NN_LOG(NN_TEXT("calloc の呼出し\n"));
    void* address = calloc( callocSize / 4, 4 );
    EXPECT_TRUE( address != NULL );

    NN_LOG(NN_TEXT("calloc の獲得アドレスが正しいか address=0x%p\n"), address);
    EXPECT_TRUE( g_MallocBegin <= address && address < g_MallocEnd );

    NN_LOG(NN_TEXT("calloc の獲得領域が 0 初期化されているか\n"));
    EXPECT_TRUE( CheckMemoryRegion32(address, callocSize, 0) );

    NN_LOG(NN_TEXT("free の呼出し\n"));
    free( address );
    EXPECT_TRUE( true );
}

//-----------------------------------------------------------------------------
// realloc の呼出し、確保した領域のチェック、free の呼出し
//
TEST(nninitSetMallocRegionTest, testRealloc)
{
    QueryMallocRegion();

    const size_t reallocSize = 5000;

    NN_LOG(NN_TEXT("malloc の呼出し\n"));
    void* address = malloc( reallocSize );
    EXPECT_TRUE( address != NULL );

    NN_LOG(NN_TEXT("malloc の獲得アドレスが正しいか address=0x%p\n"), address);
    EXPECT_TRUE( g_MallocBegin <= address && address < g_MallocEnd );

    NN_LOG(NN_TEXT("malloc の獲得領域を 0x33 で埋める\n"));
    memset(address, 0x33, reallocSize);


    NN_LOG(NN_TEXT("realloc の呼出し（サイズを小さくする）\n"));
    address = realloc( address, reallocSize / 2 );
    EXPECT_TRUE( address != NULL );

    NN_LOG(NN_TEXT("realloc の獲得アドレスが正しいか address=0x%p\n"), address);
    EXPECT_TRUE( g_MallocBegin <= address && address < g_MallocEnd );

    NN_LOG(NN_TEXT("realloc の新しい領域が 0x33 で埋まっているか\n"));
    NN_LOG("address[0]=0x%08x\n", *reinterpret_cast<uint32_t*>(address));
    EXPECT_TRUE( CheckMemoryRegion32(address, reallocSize / 2, 0x33) );


    NN_LOG(NN_TEXT("realloc の呼出し（サイズを大きくする）\n"));
    address = realloc( address, reallocSize * 2 );
    EXPECT_TRUE( address != NULL );

    NN_LOG(NN_TEXT("realloc の獲得アドレスが正しいか address=0x%p\n"), address);
    EXPECT_TRUE( g_MallocBegin <= address && address < g_MallocEnd );

    NN_LOG(NN_TEXT("realloc の少なくとも前半部分が 0x33 で埋まっているか\n"));
    NN_LOG("address[0]=0x%08x\n", *reinterpret_cast<uint32_t*>(address));
    EXPECT_TRUE( CheckMemoryRegion32(address, reallocSize / 2, 0x33) );


    NN_LOG(NN_TEXT("free の呼出し\n"));
    free( address );
    EXPECT_TRUE( true );
}

//-----------------------------------------------------------------------------
// malloc_usable_size の呼出し、確保した領域のチェック、free の呼出し
//
TEST(nninitSetMallocRegionTest, testMallocUsableSize)
{
    const size_t mallocSize = 5000;

    NN_LOG(NN_TEXT("malloc の呼出し\n"));
    void* address = malloc( mallocSize );
    EXPECT_TRUE( address != NULL );

    NN_LOG(NN_TEXT("malloc_usable_size の呼出し\n"));
    size_t usableSize = malloc_usable_size( address );

    NN_LOG(NN_TEXT("malloc_usable_size の獲得サイズが正しいか usableSize=%zu\n"), usableSize);
    EXPECT_TRUE( usableSize >= mallocSize );


    NN_LOG(NN_TEXT("realloc の呼出し（サイズを大きくする）\n"));
    address = realloc( address, mallocSize * 2 );
    EXPECT_TRUE( address != NULL );

    NN_LOG(NN_TEXT("malloc_usable_size の呼出し\n"));
    usableSize = malloc_usable_size( address );

    NN_LOG(NN_TEXT("malloc_usable_size の獲得サイズが正しいか usableSize=%zu\n"), usableSize);
    EXPECT_TRUE( usableSize >= mallocSize * 2 );


    NN_LOG(NN_TEXT("realloc の呼出し（サイズを小さくする）\n"));
    address = realloc( address, mallocSize / 2 );
    EXPECT_TRUE( address != NULL );

    NN_LOG(NN_TEXT("malloc_usable_size の呼出し\n"));
    usableSize = malloc_usable_size( address );

    NN_LOG(NN_TEXT("malloc_usable_size の獲得サイズが正しいか usableSize=%zu\n"), usableSize);
    EXPECT_TRUE( usableSize >= mallocSize / 2 );


    NN_LOG(NN_TEXT("free の呼出し\n"));
    free( address );
    EXPECT_TRUE( true );
}


//-----------------------------------------------------------------------------
// new の呼出し
//
TEST(nninitSetMallocRegionTest, testNew)
{
    QueryMallocRegion();

    NN_LOG(NN_TEXT("new の呼出し\n"));
    uint32_t* p = new uint32_t[1024 * 1024 / sizeof(uint32_t)];

    NN_LOG(NN_TEXT("new による獲得アドレスが正しいか p=0x%p\n"), p);
    EXPECT_TRUE( g_MallocBegin <= p && p < g_MallocEnd );

    delete[] p;
}

//-----------------------------------------------------------------------------
#endif  // GCC or CLANG

//-----------------------------------------------------------------------------
// nnosAllocateMemoryForThreadLocal() の呼出し、確保した領域のチェック、
// nnosFreeMemoryForThreadLocal() の呼出し
//
TEST(nnosAllocatorForThreadLocal, testAllocatorForThreadLocal)
{
    QueryMallocRegion();

    const size_t size = 5000;

    NN_LOG(NN_TEXT("nnosAllocateMemoryForThreadLocal() の呼出し\n"));
    void* address = nnosAllocateMemoryForThreadLocal( size );
    EXPECT_TRUE( address != NULL );

    NN_LOG(NN_TEXT("nnosAllocateMemoryForThreadLocal() の獲得アドレスが正しいか address=0x%p\n"), address);
    EXPECT_TRUE( g_MallocBegin <= address && address < g_MallocEnd );

    NN_LOG(NN_TEXT("nnosFreeMemoryForThreadLocal() の呼出し\n"));
    nnosFreeMemoryForThreadLocal( address, size );

    EXPECT_TRUE( true );
}

// 失敗したとき、errno が設定されているかをチェック
TEST(defaultMallocErrnoTest, testDefaultMallocErrno)
{
    // malloc
    {
        errno = 0;

        void* p = malloc( SIZE_MAX );
        int saveErrno = errno;

        NN_LOG(NN_TEXT("malloc の呼び出し : addr=0x%p\n"), p);
        ASSERT_TRUE( p == NULL );

        NN_LOG(NN_TEXT("malloc で失敗した時、errno が設定されているか : errno=%d\n"), saveErrno);
        EXPECT_TRUE( saveErrno == ENOMEM );
    }

    // calloc
    {
        errno = 0;

        void* p = calloc( 1, SIZE_MAX );
        int saveErrno = errno;

        NN_LOG(NN_TEXT("calloc の呼び出し : addr=0x%p\n"), p);
        ASSERT_TRUE( p == NULL );

        NN_LOG(NN_TEXT("calloc で失敗した時、errno が設定されているか : errno=%d\n"), saveErrno);
        EXPECT_TRUE( saveErrno == ENOMEM );
    }

    // realloc
    {
        errno = 0;

        const size_t size = 5000;
        void* p = malloc( size );

        NN_LOG(NN_TEXT("malloc の呼び出し : addr=0x%p\n"), p);
        ASSERT_TRUE( p != NULL );

        void* r = realloc( p, SIZE_MAX );
        int saveErrno = errno;

        NN_LOG(NN_TEXT("realloc の呼び出し : addr=0x%p\n"), r);
        ASSERT_TRUE( r == NULL );

        NN_LOG(NN_TEXT("realloc で失敗した時、errno が設定されているか : errno=%d\n"), saveErrno);
        EXPECT_TRUE( saveErrno == ENOMEM );

        free( p );
    }

    // aligned_alloc
    {
        errno = 0;

        void* p = aligned_alloc( SIZE_MAX, SIZE_MAX );
        int saveErrno = errno;

        NN_LOG(NN_TEXT("aligned_alloc の呼び出し : addr=0x%p\n"), p);
        ASSERT_TRUE( p == NULL );

        NN_LOG(NN_TEXT("aligned_alloc で失敗した時、errno が設定されているか : errno=%d\n"), saveErrno);
        EXPECT_TRUE( saveErrno == ENOMEM );
    }
}


}}} // namespace nnt::init::startup

