﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{MultiThreadedAssetFileLoadingHelper.h,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines a helper class to load asset
 *  files in parallel on multiple threads.
 */

#pragma once

#include <nn/os/os_Mutex.h>
#include <nn/nn_Common.h>
#include <nvn/nvn.h>

#include <vector>
#include "AssetFileDataHolder.h"

#include "TextureIDManager.h"

/*
 * AssetLoaderArg
 * --------------
 * Class that is used to pass nvn data and file
 * data needed by the loading helper class.
 */

class AssetLoaderArg
{
    public:
        AssetLoaderArg(NVNdevice* pDevice,
                       const char* pConfigFileName,
                       nn::os::MutexType* pMemoryPoolMutex,
                       TextureIDManager* pTextureIDManager,
                       bool useMipmaps = false,
                       bool useCubemap = false);

        NVNdevice*              GetDevice();
        TextureIDManager*       GetTextureIDManager();
        const char*             GetConfigFileName();
        bool                    GetUseMipmaps();
        bool                    GetUseCubemap();

        void                    LockMemoryPoolMutex();
        void                    UnlockMemoryPoolMutex();

    private:
        NVNdevice*              m_pDevice;
        nn::os::MutexType*      m_pMemoryPoolMutex;
        const char*             m_pConfigFileName;
        bool                    m_useMipmap;
        bool                    m_useCubeMap;

        TextureIDManager*       m_pTextureIDManager;
};

/*
 * MultiThreadedAssetFileLoadingHelper
 * -----------------------------------
 * Helper class that is used to load asset files
 * in parallel.
 */
class MultiThreadedAssetFileLoadingHelper
{
    public:
        NN_IMPLICIT MultiThreadedAssetFileLoadingHelper(AssetLoaderArg* pArg);
        ~MultiThreadedAssetFileLoadingHelper();

        void LoadAssetFile();
        AssetFileDataHolder* GetAssetFileDataHolder();

    private:
        void LoadShaders(const char* pShaderBlockHead);
        NVNProgramData* CreateShaderProgram(const char* pShaderProgramHead);

        void LoadModels(const char* pModelBlockHead);
        NVNModelData* LoadModelData(const char* pModelHead);

        void LoadTextures(const char* pTextureBlockHead);
        NVNTextureData* LoadTextureData(const char* pTextureDataHead);

        AssetFileDataHolder*    m_pDataHolder;

        AssetLoaderArg*         m_pArg;
        NVNbufferBuilder        m_BufferBuilder;
        NVNtextureBuilder       m_TextureBuilder;
        NVNsamplerBuilder       m_SamplerBuilder;

        char*                   m_pFileHead;
        int64_t                 m_FileSize;
};
