﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/lcs.h>


namespace nnt { namespace lcs
{
    /**
     * @brief       テストの操作シナリオです。
     */
    enum Scenario
    {
        Scenario_Normal,
        Scenario_LeaveAtSuspendedReasonEulaRequired,
        Scenario_LeaveAtSuspendedReasonRebootRequired,
        Scenario_ResumeWithoutReboot,
        Scenario_StorageSpaceNotEnoughAfterStartShare,
    };

    /**
     * @brief       事前条件の生成に使用するパラメータです。
     */
    struct NodePreconditionSource
    {
        //! アプリケーションに関する情報です。
        struct
        {
            //! バージョンです。
            uint32_t version;

            //! 起動必須バージョンです。
            uint32_t requiredVersion;

            //! システム要求バージョンです。
            uint32_t requiredSystemVersion;
        } application;

        //! システムに関する情報です。
        struct
        {
            //! バージョンです。
            uint32_t version;

            //! 要求バージョンです。
            uint32_t requiredSystemVersion;

            //! SystemDeliveryInfo のバージョンです。
            uint32_t systemDeliveryProtocolVersion;

            //! ApplicationDeliveryInfo のバージョンです。
            uint32_t applicationDeliveryProtocolVersion;

            //! exFAT の有無です。
            bool hasExFatDriver;

            //! System Delivery Info の後方互換性の有無です。
            bool hasSystemDeliveryProtocolBackwardCompatibility;

            //! Application Delivery Info の後方互換性の有無です。
            bool hasApplicationDeliveryProtocolBackwardCompatibility;

            //! EULA の有無です。
            bool hasEula;
        } system;

        //! ストレージに十分な空き領域があるか否かのフラグです。
        bool isBuiltInUserStorageNotEnough;
    };

    /**
     * @brief       事前条件の生成に使用するパラメータです。
     */
    struct PreconditionSource
    {
        //! 各端末についての事前条件です。
        NodePreconditionSource nodes[nn::lcs::NodeCountMax];

        //! アプリケーションに関する情報です。
        struct
        {
            //! アプリケーションの識別子です。
            nn::ncm::ApplicationId id;
        } application;

        //! ストレージ空き領域不足の時に整理されるアプリケーションに関する情報です。
        struct
        {
            //! アプリケーションの識別子です。
            nn::ncm::ApplicationId id;
        } applicationForClean;

        Scenario scenario;

        void SetApplicationVersion(int version) NN_NOEXCEPT;
        void SetApplicationVersion(int host, int client) NN_NOEXCEPT;
        void SetApplicationVersion(int host, int first, int others) NN_NOEXCEPT;

        void SetApplicationRequiredVersion(int version) NN_NOEXCEPT;
        void SetApplicationRequiredVersion(int host, int client) NN_NOEXCEPT;
        void SetApplicationRequiredVersion(int host, int first, int others) NN_NOEXCEPT;

        void SetApplicationRequiredSystemVersion(int version) NN_NOEXCEPT;
        void SetApplicationRequiredSystemVersion(int host, int client) NN_NOEXCEPT;
        void SetApplicationRequiredSystemVersion(int host, int first, int others) NN_NOEXCEPT;

        void SetSystemVersion(int version) NN_NOEXCEPT;
        void SetSystemVersion(int host, int client) NN_NOEXCEPT;
        void SetSystemVersion(int host, int first, int others) NN_NOEXCEPT;

        void SetSystemRequiredSystemVersion(int version) NN_NOEXCEPT;
        void SetSystemRequiredSystemVersion(int host, int client) NN_NOEXCEPT;
        void SetSystemRequiredSystemVersion(int host, int first, int others) NN_NOEXCEPT;

        void SetSystemDeliveryProtocolVersion(int version) NN_NOEXCEPT;
        void SetSystemDeliveryProtocolVersion(int host, int client) NN_NOEXCEPT;
        void SetSystemDeliveryProtocolVersion(int host, int first, int others) NN_NOEXCEPT;

        void SetApplicationDeliveryProtocolVersion(
            int version, bool hasCompatibility) NN_NOEXCEPT;
        void SetApplicationDeliveryProtocolVersion(
            int host, int client, bool hasCompatibility) NN_NOEXCEPT;
        void SetApplicationDeliveryProtocolVersion(
            int host, int first, int others, bool hasCompatibility) NN_NOEXCEPT;

        void SetExFatDriverEnabled(bool enabled) NN_NOEXCEPT;
        void SetExFatDriverEnabled(bool host, bool client) NN_NOEXCEPT;
        void SetExFatDriverEnabled(bool host, bool first, bool others) NN_NOEXCEPT;

        void SetEulaContained(bool hasEula) NN_NOEXCEPT;
        void SetEulaContained(bool host, bool client) NN_NOEXCEPT;
        void SetEulaContained(bool host, bool first, bool others) NN_NOEXCEPT;

        void SetBuiltInUserStorageNotEnough(bool isFilled) NN_NOEXCEPT;
        void SetBuiltInUserStorageNotEnough(bool host, bool client) NN_NOEXCEPT;
        void SetBuiltInUserStorageNotEnough(bool host, bool first, bool others) NN_NOEXCEPT;
    };

    /**
     * @brief       ある端末についてのテストの事前条件です。
     */
    struct NodePrecondition
    {
        //! アプリケーションに関する情報です。
        struct
        {
            //! ディスプレイバージョンです。Calculate で自動設定します。
            char displayVersion[16];

            //! バージョンです。
            uint32_t version;

            //! 起動必須バージョンです。
            uint32_t requiredVersion;

            //! システム要求バージョンです。
            uint32_t requiredSystemVersion;

        } application;

        //! システムに関する情報です。
        struct
        {
            //! バージョンです。
            uint32_t version;

            //! 要求バージョンです。
            uint32_t requiredSystemVersion;

            //! SystemDeliveryInfo のバージョンです。
            uint32_t systemDeliveryProtocolVersion;

            //! ApplicationDeliveryInfo のバージョンです。
            uint32_t applicationDeliveryProtocolVersion;

            //! exFAT の有無です。
            bool hasExFatDriver;

            //! System Delivery Info の後方互換性の有無です。
            bool hasSystemDeliveryProtocolBackwardCompatibility;

            //! Application Delivery Info の後方互換性の有無です。
            bool hasApplicationDeliveryProtocolBackwardCompatibility;

            //! EULA の有無です。
            bool hasEula;

            //! システム配信が必要か否かのフラグです。
            bool isLupRequired;

            //! システム配信が可能か否かのフラグです。
            bool isLupEnabled;

        } system;

        //! ストレージに十分な空き領域があるか否かのフラグです。
        bool isBuiltInUserStorageNotEnough;
    };

    /**
     * @brief       全端末についてのテストの事前条件です。
     */
    struct Precondition
    {
        //! 各端末についての事前条件です。
        NodePrecondition nodes[nn::lcs::NodeCountMax];

        //! アプリケーションに関する情報です。
        struct
        {
            //! アプリケーションの識別子です。
            nn::ncm::ApplicationId id;

            //! 最新のディスプレイバージョンです。Calculate で自動設定します。
            char latestDisplayVersion[16];

            //! 最新のバージョンです。Calculate で自動設定します。
            uint32_t latestVersion;

            //! 最低の Required System Version です。Calculate で自動設定します。
            uint32_t minimumRequiredSystemVersion;

            //! セッションで合わせられるバージョンです。Calculate で自動設定します。
            uint32_t targetVersion;
        } application;

        //! システムに関する情報です。
        struct
        {
            //! 最新のバージョンです。Calculate で自動設定します。
            uint32_t latestVersion;

            //! exFAT ドライバを所有する中での最新バージョンです。Calculate で自動設定します。
            uint32_t latestVersionExFat;

            //! Application Delivery Protocol の互換性をもつ最小のシステムバージョンです。
            uint32_t protocolRequiredSystemVersion;

            //! 様々な要因を考慮した上でパッチを受信するために必須のバージョンです。
            uint32_t requiredSystemVersion;

            //! System Delivery Info の後方互換性の有無です。Calculate で自動設定します。
            bool hasSystemDeliveryProtocolBackwardCompatibility;

            //! Application Delivery Info の後方互換性の有無です。Calculate で自動設定します。
            bool hasApplicationDeliveryProtocolBackwardCompatibility;
        } system;

        //! マスターのインデックスです。
        int8_t masterIndex;

        //! exFAT ドライバを所有する端末に対してシステムを送信する端末のインデックスです。
        int8_t exFatSenderIndex;

        //! 根本的にパッチ配信が可能な組み合わせか否かのフラグです。
        bool isPatchShareEnabled;

        //! テストの操作シナリオです。
        Scenario scenario;

        //! ストレージ空き領域不足の時に整理されるアプリケーションに関する情報です。
        struct
        {
            //! アプリケーションの識別子です。
            nn::ncm::ApplicationId id;
        } applicationForClean;
    };

    /**
     * @brief       デフォルトの事前条件を生成します。
     * @return      生成された事前条件です。
     */
    PreconditionSource CreateDefaultPreconditionSource() NN_NOEXCEPT;

    /**
     * @brief       与えられた事前条件から導出される事前条件を計算します。
     * @param[in]   source                  元となる事前条件です。
     * @return      生成された事前条件です。
     */
    Precondition Calculate(const PreconditionSource& source) NN_NOEXCEPT;

    /**
     * @brief       バージョン番号から displayVersion を取得します。
     * @param[out]  buffer                  displayVersion の出力先です。
     * @param[in]   bufferSize              buffer のバッファサイズです。
     * @param[in]   version                 変換元のバージョン番号です。
     */
    void ConvertToDisplayVersion(char* buffer, size_t bufferSize, int version) NN_NOEXCEPT;

}} // namespace nnt::lcs
