﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/os/os_Mutex.h>
#include <nn/fs/fs_Result.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_IStorage.h>

namespace nn {
    class MemoryResource;
}

namespace nnt { namespace fs { namespace util {

/**
 * @brief データを書き込んだ位置のメモリのみを確保するメモリストレージです。
 */
class VirtualMemoryStorage : public nn::fs::IStorage
{
    NN_DISALLOW_COPY(VirtualMemoryStorage);
    NN_DISALLOW_MOVE(VirtualMemoryStorage);

public:
    static const size_t BlockSize = 256 * 1024; //!< ブロックの管理サイズ

public:
    /**
     *  @brief  コンストラクタです。
     */
    VirtualMemoryStorage() NN_NOEXCEPT;

    /**
     * @brief   コンストラクタです。
     *
     * @param[in]   storageSize ストレージのサイズ
     */
    explicit VirtualMemoryStorage(int64_t storageSize) NN_NOEXCEPT;

    /**
     * @brief   デストラクタです。
     */
    virtual ~VirtualMemoryStorage() NN_NOEXCEPT NN_OVERRIDE
    {
        Finalize();
    }

    /**
     * @brief   初期化をします。
     *
     * @param[in]   storageSize ストレージのサイズ
     */
    void Initialize(int64_t storageSize) NN_NOEXCEPT;

    /**
     * @brief   終了処理をします。
     */
    void Finalize() NN_NOEXCEPT;

    /**
     * @brief   ストレージの内容をバッファに読み込みます。
     *
     * @param[in]   offset  読み込み開始位置
     * @param[out]  buffer  読み込んだ内容をコピーするバッファ
     * @param[in]   size    読み込むデータサイズ
     *
     * @return  関数の処理結果を返します。
     */
    virtual nn::Result Read(
        int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   バッファの内容をストレージに書き込みます。
     *
     * @param[in]   offset  書き込み開始位置
     * @param[in]   buffer  書き込むデータ
     * @param[in]   size    書き込むデータサイズ
     *
     * @return  関数の処理結果を返します。
     */
    virtual nn::Result Write(
        int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   フラッシュします。
     *
     * @return  関数の処理結果を返します。
     * @retval  ResultSuccess   必ず成功します。
     */
    virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_SUCCESS;
    }

    /**
     * @brief   特定のメモリ領域に既定の操作を行います。
     *
     * @param[out]  outBuffer       範囲指定処理の結果を格納するバッファ
     * @param[in]   outBufferSize   範囲指定処理の結果を格納するバッファのサイズ
     * @param[in]   operationId     実行する操作
     * @param[in]   offset          メモリ領域の開始位置
     * @param[in]   size            メモリ領域のサイズ
     * @param[in]   inBuffer        範囲指定処理に渡すバッファ
     * @param[in]   inBufferSize    範囲指定処理に渡すバッファのサイズ
     *
     * @return  関数の処理結果を返します。
     */
    virtual nn::Result OperateRange(
        void* outBuffer,
        size_t outBufferSize,
        nn::fs::OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE;

    using IStorage::OperateRange;

    /**
     * @brief   ストレージのサイズを設定します。
     *
     * @param[in]   size    新しいサイズ
     *
     * @return  関数の処理結果を返します。
     */
    virtual nn::Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   ストレージのサイズを取得します。
     *
     * @param[out]  outValue    取得したサイズ格納先
     *
     * @return  関数の処理結果を返します。
     * @retval  ResultSuccess   必ず成功します。
     */
    virtual nn::Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        *outValue = m_Size;
        NN_RESULT_SUCCESS;
    }

    /**
     * @brief   ストレージのサイズを取得します。
     *
     * @return  ストレージのサイズ
     */
    int64_t GetSize() const NN_NOEXCEPT
    {
        return m_Size;
    }

    /**
     *  @brief  管理しているブロックの数を取得します。
     *
     *  @return  ブロックの数
     */
    size_t GetBlockCount() const NN_NOEXCEPT
    {
        return m_List.size();
    }

    static void SetAllocator(nn::MemoryResource* pAllocator) NN_NOEXCEPT;

private:
    //! ブロックのバッファです。
    static void DeleteBuffer(void* buffer) NN_NOEXCEPT;
    typedef std::unique_ptr<char[], decltype(&DeleteBuffer)> BlockBuffer;

    //! ストレージメモリを管理する領域（ブロック）です。
    class Block : public nn::util::IntrusiveListBaseNode<Block>
    {
        NN_DISALLOW_COPY(Block);

    public:
        /**
         * @brief   コンストラクタです。
         *
         * @param[in]   offset  ブロックが管理するメモリの開始位置
         * @param[in]   buffer  ブロックが管理するメモリのバッファ
         */
        Block(int64_t offset, BlockBuffer&& buffer) NN_NOEXCEPT
            : m_Offset(offset)
            , m_Buffer(std::move(buffer))
        {
        }

        /**
         * @brief   ブロックの開始位置を取得します。
         *
         * @return  ブロックの開始位置
         */
        int64_t GetOffset() const NN_NOEXCEPT
        {
            return m_Offset;
        }

        /**
         * @brief   ブロックのバッファを取得します。
         *
         * @return  ブロックのバッファ
         */
        const BlockBuffer& GetBuffer() const NN_NOEXCEPT
        {
            return m_Buffer;
        }

    private:
        int64_t     m_Offset; //!< メモリのオフセット
        BlockBuffer m_Buffer; //!< メモリのバッファ
    };

    //! ブロックのリストです。
    typedef nn::util::IntrusiveList<
        Block, nn::util::IntrusiveListBaseNodeTraits<Block> > BlockList;

    //! ブロック検索クラスです。
    class BlockFinder
    {
    public:
        //! コンストラクタです。
        explicit BlockFinder(int64_t blockOffset) NN_NOEXCEPT
            : m_blockOffset(blockOffset)
        {
        }

        //! 関数演算子です。
        bool operator()(const Block& block) const NN_NOEXCEPT
        {
            return block.GetOffset() == m_blockOffset;
        }

    private:
        int64_t m_blockOffset;
    };

private:
    /**
     * @brief   指定したバッファをクリア（0 埋め）します。
     *
     * @param[in]   buffer      クリアするバッファ
     * @param[in]   offset      ストレージの先頭からの開始位置
     * @param[in]   size        クリアするサイズ
     * @param[in]   blockOffset 指定した領域を含むブロックのオフセット
     *
     *  @return
     */
    static void ClearBuffer(
        void* buffer, int64_t offset, size_t size, int64_t blockOffset) NN_NOEXCEPT;

    /**
     * @brief   指定したバッファにストレージからデータをコピーします。
     *
     * @param[in]   buffer  コピー先のバッファ
     * @param[in]   offset  ストレージの先頭からの開始位置
     * @param[in]   size    コピーするサイズ
     * @param[in]   block   指定した領域を含むブロック
     */
    static void ReadBuffer(
        void* buffer, int64_t offset, size_t size, const Block& block) NN_NOEXCEPT;

    /**
     * @brief   指定したバッファからストレージにデータをコピーします。
     *
     * @param[in]   buffer  コピー先元のバッファ
     * @param[in]   offset  ストレージの先頭からの開始位置
     * @param[in]   size    コピーするサイズ
     * @param[in]   block   指定した領域を含むブロック
     */
    static void WriteBuffer(
        const void* buffer, int64_t offset, size_t size, const Block& block) NN_NOEXCEPT;

    /**
     * @brief   ブロックを作成します。
     *
     * @param[in]   offset  ストレージの先頭からの開始位置
     *
     * @return  ブロックを表すイテレータ
     */
    BlockList::iterator CreateBlock(int64_t offset) NN_NOEXCEPT;

private:
    int64_t         m_Size;             //!< ストレージのサイズ
    nn::os::Mutex   m_Mutex;            //!< ミューテックス
    BlockList       m_List;             //!< ブロックリスト
    bool            m_IsInitialized;    //!< 初期化したかどうか

    static nn::MemoryResource* g_pAllocator; //!< メモリアロケータ
};

}}}
