﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>

#include <nn/os/os_Mutex.h>
#include <nn/nn_Common.h>
#include <nn/fs/fs_PriorityPrivate.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fsa/fs_IFile.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util_function_stl.h>
#include <nnt/fsUtil/testFs_util_GoogleTestUtility.h>

namespace nnt { namespace fs { namespace util {

class AccessRangeRecord : public nn::fs::detail::Newable
{
public:
    AccessRangeRecord() NN_NOEXCEPT
        : m_IsWriteAccess(false),
          m_Position(0),
          m_Size(0)
    {
    }

    void Initialize(bool isWriteAccess, int64_t position, size_t size) NN_NOEXCEPT
    {
        m_IsWriteAccess = isWriteAccess;
        m_Position = position;
        m_Size = size;
    }

    bool IsWriteAccess() const NN_NOEXCEPT
    {
        return m_IsWriteAccess;
    }

    int64_t GetPosition() const NN_NOEXCEPT
    {
        return m_Position;
    }

    size_t GetSize() const NN_NOEXCEPT
    {
        return m_Size;
    }

private:
    bool m_IsWriteAccess;
    int64_t m_Position;
    size_t m_Size;
};

class AccessRangeRecorder
{
public:
    AccessRangeRecorder() NN_NOEXCEPT
        : m_Locker(false)
    {
    }

public:
    void Record(bool isWriteAccess, int64_t offset, size_t size) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> locker(m_Locker);
        AccessRangeRecord record;
        record.Initialize(isWriteAccess, offset, size);
        m_AccessRangeData.push_back(record);
    }

    void Clear() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> locker(m_Locker);
        m_AccessRangeData.clear();
    }

    template<typename Checker>
    void Check(Checker checker) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> locker(m_Locker);

        std::sort(
            m_AccessRangeData.begin(),
            m_AccessRangeData.end(),
            [](const AccessRangeRecord& entryLeft, const AccessRangeRecord& entryRight) NN_NOEXCEPT
            {
                return entryLeft.GetPosition() < entryRight.GetPosition();
            });

        for( auto& entry : m_AccessRangeData )
        {
            NNT_FS_ASSERT_NO_FATAL_FAILURE(checker(entry));
        }
    }

private:
    Vector<AccessRangeRecord> m_AccessRangeData;
    nn::os::Mutex m_Locker;
};

class RecordingAccessRangeStorage : public nn::fs::IStorage
{
public:
    explicit RecordingAccessRangeStorage(nn::fs::IStorage* pStorage) NN_NOEXCEPT
        : m_pStorage(pStorage)
    {
        NN_SDK_ASSERT_NOT_NULL(pStorage);
    }

    virtual nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pStorage);
        m_Recorder.Record(false, offset, size);
        NN_RESULT_DO(m_pStorage->Read(offset, buffer, size));
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pStorage);
        m_Recorder.Record(true, offset, size);
        NN_RESULT_DO(m_pStorage->Write(offset, buffer, size));
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pStorage);
        NN_RESULT_DO(m_pStorage->Flush());
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pStorage);
        NN_RESULT_DO(m_pStorage->GetSize(outValue));
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pStorage);
        NN_RESULT_DO(m_pStorage->SetSize(size));
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result OperateRange(
        void* outBuffer,
        size_t outBufferSize,
        nn::fs::OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize
    ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pStorage);
        NN_RESULT_DO(m_pStorage->OperateRange(
            outBuffer,
            outBufferSize,
            operationId,
            offset,
            size,
            inBuffer,
            inBufferSize));
        NN_RESULT_SUCCESS;
    }

public:
    AccessRangeRecorder * GetRecorder() NN_NOEXCEPT
    {
        return &m_Recorder;
    }

private:
    nn::fs::IStorage* const m_pStorage;
    AccessRangeRecorder m_Recorder;
};

class RecordingAccessRangeFile : public nn::fs::fsa::IFile, public nn::fs::detail::Newable
{
public:
    explicit RecordingAccessRangeFile(std::unique_ptr<nn::fs::fsa::IFile>&& pFile) NN_NOEXCEPT
        : m_pFile(std::move(pFile))
    {
    }

    virtual ~RecordingAccessRangeFile() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    virtual nn::Result DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFile);
        m_Recorder.Record(false, offset, size);
        NN_RESULT_DO(m_pFile->Read(outValue, offset, buffer, size, option));
        NN_RESULT_SUCCESS;
    }
    virtual nn::Result DoWrite(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFile);
        m_Recorder.Record(true, offset, size);
        NN_RESULT_DO(m_pFile->Write(offset, buffer, size, option));
        NN_RESULT_SUCCESS;
    }
    virtual nn::Result DoFlush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFile);
        NN_RESULT_DO(m_pFile->Flush());
        NN_RESULT_SUCCESS;
    }
    virtual nn::Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFile);
        NN_RESULT_DO(m_pFile->SetSize(size));
        NN_RESULT_SUCCESS;
    }
    virtual nn::Result DoGetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFile);
        NN_RESULT_DO(m_pFile->GetSize(outValue));
        NN_RESULT_SUCCESS;
    }
    virtual nn::Result DoOperateRange(void* outBuffer, size_t outBufferSize, nn::fs::OperationId operationId, int64_t offset, int64_t size, const void* inBuffer, size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFile);
        NN_RESULT_DO(m_pFile->OperateRange(outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize));
        NN_RESULT_SUCCESS;
    }

public:
    AccessRangeRecorder* GetRecorder() NN_NOEXCEPT
    {
        return &m_Recorder;
    }

private:
    std::unique_ptr<nn::fs::fsa::IFile> m_pFile;
    AccessRangeRecorder m_Recorder;
};

class AsynchronousAccessAlignmentAndSplitSizeChecker
{
public:
    AsynchronousAccessAlignmentAndSplitSizeChecker(
        nn::fs::PriorityRaw priority,
        AccessRangeRecorder* pRecorder,
        nn::fs::IStorage* pStorage
    ) NN_NOEXCEPT
        : m_Priority(priority),
          m_pRecorder(pRecorder),
          m_pStorage(pStorage),
          m_pFile(nullptr)
    {
        NN_SDK_ASSERT_NOT_NULL(pRecorder);
        NN_SDK_ASSERT_NOT_NULL(pStorage);
    }

    AsynchronousAccessAlignmentAndSplitSizeChecker(
        nn::fs::PriorityRaw priority,
        AccessRangeRecorder* pRecorder,
        nn::fs::fsa::IFile* pFile
    ) NN_NOEXCEPT
        : m_Priority(priority),
        m_pRecorder(pRecorder),
        m_pStorage(nullptr),
        m_pFile(pFile)
    {
        NN_SDK_ASSERT_NOT_NULL(pRecorder);
        NN_SDK_ASSERT_NOT_NULL(pFile);
    }

public:
    void CheckForReading() NN_NOEXCEPT;
    void CheckForWriting() NN_NOEXCEPT;

private:
    bool IsLowerPriority() const NN_NOEXCEPT
    {
        return (m_Priority == nn::fs::PriorityRaw_Low
            || m_Priority == nn::fs::PriorityRaw_Background);
    }

    nn::Result Write(int64_t offset, const char* buffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_pStorage != nullptr || m_pFile != nullptr);

        if( m_pStorage != nullptr )
        {
            NN_RESULT_DO(m_pStorage->Write(offset, buffer, size));
        }

        if( m_pFile != nullptr )
        {
            NN_RESULT_DO(m_pFile->Write(offset, buffer, size, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result Read(int64_t offset, char* buffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_pStorage != nullptr || m_pFile != nullptr);

        if( m_pStorage != nullptr )
        {
            NN_RESULT_DO(m_pStorage->Read(offset, buffer, size));
        }

        if( m_pFile != nullptr )
        {
            size_t readSize;
            NN_RESULT_DO(m_pFile->Read(&readSize, offset, buffer, size, nn::fs::ReadOption::MakeValue(0)));
        }

        NN_RESULT_SUCCESS;
    }

private:
    nn::fs::PriorityRaw m_Priority;
    AccessRangeRecorder* m_pRecorder;
    nn::fs::IStorage* m_pStorage;
    nn::fs::fsa::IFile* m_pFile;
};

}}} // namespace nnt::fs::util
