﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cinttypes> // int32_t
#include <cstddef> // nullptr, std::size_t
#include <cstdlib> // std::malloc, std::free

#include <nnt/codecUtil/testCodec_TestBase.h>
#include <nnt/codecUtil/testCodec_ObjectBase.h>

namespace nnt {
namespace codec {

template <class T>
T* CreateOpusDecoder(int32_t option) NN_NOEXCEPT;

template <>
NN_FORCEINLINE nn::codec::OpusDecoder* CreateOpusDecoder<nn::codec::OpusDecoder>(int32_t option) NN_NOEXCEPT
{
    NN_UNUSED(option);
    return new nn::codec::OpusDecoder;
}

template <>
NN_FORCEINLINE nn::codec::HardwareOpusDecoder* CreateOpusDecoder<nn::codec::HardwareOpusDecoder>(int32_t option) NN_NOEXCEPT
{
    return new nn::codec::HardwareOpusDecoder(option);
}

template <>
NN_FORCEINLINE nn::codec::OpusMultiStreamDecoder* CreateOpusDecoder<nn::codec::OpusMultiStreamDecoder>(int32_t option) NN_NOEXCEPT
{
    NN_UNUSED(option);
    return new nn::codec::OpusMultiStreamDecoder;
}

template <>
NN_FORCEINLINE nn::codec::HardwareOpusMultiStreamDecoder* CreateOpusDecoder<nn::codec::HardwareOpusMultiStreamDecoder>(int32_t option) NN_NOEXCEPT
{
    return new nn::codec::HardwareOpusMultiStreamDecoder(option);
}

template <class T>
NN_FORCEINLINE void DeleteOpusDecoder(T* pDecoder) NN_NOEXCEPT
{
    if (pDecoder != nullptr)
    {
        delete pDecoder;
    }
}

template <typename DecoderType>
class OpusDecoderBase
    : protected ObjectBase
{

protected:
    OpusDecoderBase() NN_NOEXCEPT
        : decoder()
        , decoderWorkBufferAddress(nullptr)
        , decoderWorkBufferSize(0)
    {}

    virtual ~OpusDecoderBase() NN_NOEXCEPT
    {
        if (nullptr != decoderWorkBufferAddress)
        {
            std::free(decoderWorkBufferAddress);
        }
        decoderWorkBufferAddress = nullptr;
    }

    virtual void Initialize(const TestParameterSet& parameterSet) NN_NOEXCEPT
    {
        // Define some aliases
        const int& sampleRate = parameterSet.sampleRate;
        const int& channelCount = parameterSet.channelCount;
        NN_ASSERT(sampleRate != 0);
        NN_ASSERT(channelCount != 0);
        // Initializing
        decoderWorkBufferSize = decoder.GetWorkBufferSize(sampleRate, channelCount);
        decoderWorkBufferAddress = std::malloc(decoderWorkBufferSize);
        ASSERT_NE(nullptr, decoderWorkBufferAddress);
        // decoder.Initialize() を呼ぶかどうかはテスト次第なので、ここではしない。
        m_IsInitialized = true;
    }

    virtual void Finalize() NN_NOEXCEPT
    {
        m_IsInitialized = false;
        if (decoder.IsInitialized())
        {
            decoder.Finalize();
        }
        if (nullptr != decoderWorkBufferAddress)
        {
            std::free(decoderWorkBufferAddress);
        }
        decoderWorkBufferAddress = nullptr;
        decoderWorkBufferSize = 0;
    }

    DecoderType decoder;
    void* decoderWorkBufferAddress;
    std::size_t decoderWorkBufferSize;
};

}} // nnt::codec
