﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/image/image_JpegEncoder.h>
#include <nn/image/image_JpegDecoder.h>
#include <nn/image/image_ExifBuilder.h>
#include <nn/image/image_ExifExtractor.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/teamcity/testTeamcity_Logger.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nnt/nnt_Argument.h>
#include <nn/os/os_MemoryHeap.h>
#include <nn/init/init_Malloc.h>

#include "testImageJpeg_Io.h"
#include "testImageJpeg_Path.h"
#include "testImageJpeg_TestInvoker.h"
#include "testImageJpeg_Tga.h"

#include <nn/TargetConfigs/build_Base.h>

using nnt::image::jpeg::io::Buffer;
using nnt::image::jpeg::Tga;

namespace
{

enum Condition
{
    Condition_Basic = 0,
    Condition_Rgb24 = 1,
};

struct Rgba32
{
    static const nn::image::PixelFormat value = nn::image::PixelFormat_Rgba32;
};
struct Rgb24
{
    static const nn::image::PixelFormat value = nn::image::PixelFormat_Rgb24;
};

enum TestSet
{
    TestSet_EncoderAnalyze = 0,
    TestSet_EncoderEncode,
    TestSet_DecoderAnalyze,
    TestSet_DecoderDecode,
};

inline const char* GetConditionName(Condition e)
{
    switch (e)
    {
    case Condition_Basic:
        return "Basic";
    case Condition_Rgb24:
        return "Rgb24";
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

inline const char* GetTestSetName(TestSet s)
{
    switch (s)
    {
    case TestSet_EncoderAnalyze:
        return "JpegEncoder::Analyze";
    case TestSet_EncoderEncode:
        return "JpegEncoder::Encode";
    case TestSet_DecoderAnalyze:
        return "JpegDecoder::Analyze";
    case TestSet_DecoderDecode:
        return "JpegDecoder::Decode";
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

#define NNT_TESTSF_LOG_HEADER() \
    "<?xml version='1.0' encoding='UTF-8'?>\n" \
    "<TestResults>\n"

#define NNT_TESTSF_LOG_ENTRY(id, alt, duration) \
    "    <TestResult id=\"%s\" alt=\"%s\">\n" \
    "        <Duration unit=\"nsec\" value=\"%d\" />\n"\
    "    </TestResult>\n", \
    (id), (alt), (duration)

#define NNT_TESTSF_LOG_FOOTER() \
    "</TestResults>\n"

void LogResult(nnt::image::jpeg::TestResult *results, const size_t ResultCount)
{
    char testIdStr[128];

    NN_LOG(NNT_TESTSF_LOG_HEADER());
    for (auto i = 0; i < static_cast<int>(ResultCount); ++ i)
    {
        results[i].testId.ToString(testIdStr,sizeof(testIdStr));
        NN_LOG(
            NNT_TESTSF_LOG_ENTRY(
                testIdStr, results[i].testId.GetAlternativeString(),
                results[i].stat.average));
    }
    NN_LOG(NNT_TESTSF_LOG_FOOTER());
}

} // ~namespace <anonymous>

namespace encoderTest
{

const char TestFilePathVga[] = "Noise_VGA.tga";
const char TestFilePath720p[] = "Black_720P.tga";
const char TestFilePath1080p[] = "Noise_1080P.tga";

template <typename PixelFormat>
size_t RunTests(nnt::image::jpeg::TestResult *results, const size_t ResultCount, const Condition Cond, const int LoopCount)
{
    NN_UNUSED(ResultCount);
    const char *condStr = GetConditionName(Cond);

    // TGA データの読み込み
    Buffer tgaDataVga;
    NN_ABORT_UNLESS(nnt::image::jpeg::io::ReadAssetWithAllocation(&tgaDataVga, TestFilePathVga));
    Buffer tgaData720p;
    NN_ABORT_UNLESS(nnt::image::jpeg::io::ReadAssetWithAllocation(&tgaData720p, TestFilePath720p));
    Buffer tgaData1080p;
    NN_ABORT_UNLESS(nnt::image::jpeg::io::ReadAssetWithAllocation(&tgaData1080p, TestFilePath1080p));

    //
    nn::image::JpegEncoder encoder;
    Buffer workBuf;
    Buffer pxData(10 * 1024 * 1024);
    Buffer jpegData(10 * 1024 * 1024);

    // エンコード時に得られる情報
    size_t codedSize;
    nn::image::JpegStatus status;

    // エンコード機能の計測の、事前準備に使う関数オブジェクト
    auto analyzePre = [&] (const Buffer &tgaData, int pxAlign, int quality, nn::image::JpegSamplingRatio ratio)
    {
        // TGA から画像サイズを取得
        nnt::image::jpeg::Dimension pxDim;
        NN_ABORT_UNLESS(Tga::GetDimension(&pxDim, tgaData.GetPointer(), tgaData.GetSize()));

        // TGA -> ピクセルデータ 変換
        auto alignedDim = pxDim.GetAlignedDimension(pxAlign);
        auto pxSize = alignedDim.width * alignedDim.height * 4;
        if (pxSize > static_cast<decltype(pxSize)>(pxData.GetSize()))
        {
            pxData.SetSize(pxSize);
        }
        Tga::GetPixels(
            reinterpret_cast<uint32_t*>(pxData.GetPointer()), pxSize, pxAlign,
            tgaData.GetPointer(), tgaData.GetSize());

        // Encoder 設定
        nn::image::Dimension dim = {pxDim.width, pxDim.height};
        encoder.SetPixelData(pxData.GetPointer(), PixelFormat::value, dim, pxAlign);
        encoder.SetQuality(quality);
        encoder.SetSamplingRatio(ratio);
    };
    auto encodePre = [&] (const Buffer &tgaData, int pxAlign, int quality, nn::image::JpegSamplingRatio ratio)
    {
        analyzePre(tgaData, pxAlign, quality, ratio);

        // Encoder 事前処理
        NN_ABORT_UNLESS(encoder.Analyze() == nn::image::JpegStatus_Ok);
        auto workBufSize = encoder.GetAnalyzedWorkBufferSize();
        if (workBufSize > workBuf.GetSize())
        {
            workBuf.SetSize(workBufSize);
        }
    };
    auto analyzeTest = [&] { status = encoder.Analyze(); };
    auto encodeTest = [&] { status = encoder.Encode(&codedSize, jpegData.GetPointer(), jpegData.GetSize(), workBuf.GetPointer(), workBuf.GetSize()); };
    auto post = [&] { NN_ASSERT(status == nn::image::JpegStatus_Ok); };

    // テストケース
    const char *setStr = nullptr;
    setStr = GetTestSetName(TestSet_EncoderAnalyze);
    nnt::image::jpeg::TestEntity analyzeTests[] = {
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 0, "Vga_1_100_420",
            [&] { analyzePre(tgaDataVga, 1, 100, nn::image::JpegSamplingRatio_420); },
            analyzeTest, post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 1, "Vga_1_50_420",
            [&] { analyzePre(tgaDataVga, 1, 50, nn::image::JpegSamplingRatio_420); },
            analyzeTest, post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 2, "Vga_1_1_420",
            [&] { analyzePre(tgaDataVga, 1, 1, nn::image::JpegSamplingRatio_420); },
            analyzeTest, post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 3, "Vga_4_100_420",
            [&] { analyzePre(tgaDataVga, 4, 100, nn::image::JpegSamplingRatio_420); },
            analyzeTest, post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 4, "Vga_13_100_420",
            [&] { analyzePre(tgaDataVga, 13, 100, nn::image::JpegSamplingRatio_420); },
            analyzeTest, post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 5, "720P_1_100_420",
            [&] { analyzePre(tgaData720p, 1, 100, nn::image::JpegSamplingRatio_420); },
            analyzeTest, post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 6, "1080P_1_100_420",
            [&] { analyzePre(tgaData1080p, 1, 100, nn::image::JpegSamplingRatio_420); },
            analyzeTest, post),
    };

    setStr = GetTestSetName(TestSet_EncoderEncode);
    nnt::image::jpeg::TestEntity encodeTests[] = {
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 0, "Vga_1_100_420",
            [&] { encodePre(tgaDataVga, 1, 100, nn::image::JpegSamplingRatio_420); },
            encodeTest, post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 1, "Vga_1_100_422",
            [&] { encodePre(tgaDataVga, 1, 100, nn::image::JpegSamplingRatio_422); },
            encodeTest, post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 2, "Vga_1_100_444",
            [&] { encodePre(tgaDataVga, 1, 100, nn::image::JpegSamplingRatio_444); },
            encodeTest, post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 3, "Vga_1_50_420",
            [&] { encodePre(tgaDataVga, 1, 50, nn::image::JpegSamplingRatio_420); },
            encodeTest, post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 4, "Vga_1_1_420",
            [&] { encodePre(tgaDataVga, 1, 1, nn::image::JpegSamplingRatio_420); },
            encodeTest, post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 5, "Vga_4_100_420",
            [&] { encodePre(tgaDataVga, 4, 100, nn::image::JpegSamplingRatio_420); },
            encodeTest, post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 6, "Vga_13_100_420",
            [&] { encodePre(tgaDataVga, 13, 100, nn::image::JpegSamplingRatio_420); },
            encodeTest, post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 7, "720P_1_100_420",
            [&] { encodePre(tgaData720p, 1, 100, nn::image::JpegSamplingRatio_420); },
            encodeTest, post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 8, "1080P_1_100_420",
            [&] { encodePre(tgaData1080p, 1, 100, nn::image::JpegSamplingRatio_420); },
            encodeTest, post),
    };

    auto analyzeTestCount = sizeof(analyzeTests) / sizeof(analyzeTests[0]);
    auto encodeTestCount = sizeof(encodeTests) / sizeof(encodeTests[0]);
    NN_ASSERT(ResultCount >= analyzeTestCount + encodeTestCount);

    uint32_t counter = 0;
    for (int i = 0; i < static_cast<int>(analyzeTestCount); ++ i)
    {
        results[counter] = analyzeTests[i].Invoke(LoopCount);
        ++ counter;
    }
    for (int i = 0; i < static_cast<int>(encodeTestCount); ++ i)
    {
        results[counter] = encodeTests[i].Invoke(LoopCount);
        ++ counter;
    }
    NN_ASSERT(counter == analyzeTestCount + encodeTestCount);
    return analyzeTestCount + encodeTestCount;
} // NOLINT(readability/fn_size)

} // ~namespace encoderTest

namespace decoderTest
{

const char TestFilePathVga_100_420[]    = "Noise_VGA_100_420.jpg";
const char TestFilePathVga_100_422[]    = "Noise_VGA_100_422.jpg";
const char TestFilePathVga_100_444[]    = "Noise_VGA_100_444.jpg";
const char TestFilePathVga_50_420[]     = "Noise_VGA_50_420.jpg";
const char TestFilePathVga_1_420[]      = "Noise_VGA_1_420.jpg";
const char TestFilePath720p_100_420[]   = "Noise_720P_100_420.jpg";
const char TestFilePath1080p_100_420[]  = "Noise_1080P_100_420.jpg";

template <typename PixelFormat>
size_t RunTests(nnt::image::jpeg::TestResult *results, const size_t ResultCount, const Condition Cond, const int LoopCount)
{
    NN_UNUSED(ResultCount);
    const char *condStr = GetConditionName(Cond);

    // TGA データの読み込み
    Buffer jpegDataVga_100_420;
    NN_ABORT_UNLESS(nnt::image::jpeg::io::ReadAssetWithAllocation(&jpegDataVga_100_420, TestFilePathVga_100_420));
    Buffer jpegDataVga_100_422;
    NN_ABORT_UNLESS(nnt::image::jpeg::io::ReadAssetWithAllocation(&jpegDataVga_100_422, TestFilePathVga_100_422));
    Buffer jpegDataVga_100_444;
    NN_ABORT_UNLESS(nnt::image::jpeg::io::ReadAssetWithAllocation(&jpegDataVga_100_444, TestFilePathVga_100_444));
    Buffer jpegDataVga_50_420;
    NN_ABORT_UNLESS(nnt::image::jpeg::io::ReadAssetWithAllocation(&jpegDataVga_50_420, TestFilePathVga_50_420));
    Buffer jpegDataVga_1_420;
    NN_ABORT_UNLESS(nnt::image::jpeg::io::ReadAssetWithAllocation(&jpegDataVga_1_420, TestFilePathVga_1_420));
    Buffer jpegData720p_100_420;
    NN_ABORT_UNLESS(nnt::image::jpeg::io::ReadAssetWithAllocation(&jpegData720p_100_420, TestFilePath720p_100_420));
    Buffer jpegData1080p_100_420;
    NN_ABORT_UNLESS(nnt::image::jpeg::io::ReadAssetWithAllocation(&jpegData1080p_100_420, TestFilePath1080p_100_420));

    //
    nn::image::JpegDecoder decoder;
    Buffer workBuf;
    Buffer pxData(10 * 1024 * 1024);
    nn::image::Dimension pxDim;

    // エンコード時に得られる情報
    nn::image::JpegStatus status;

    // エンコード機能の計測の、事前準備に使う関数オブジェクト
    auto analyzePre = [&] (const Buffer &jpegData, int resolution)
    {
        decoder.SetImageData(jpegData.GetPointer(), jpegData.GetSize());
        decoder.SetResolutionDenominator(resolution);
        decoder.SetPixelFormat(PixelFormat::value);
    };
    auto decodePre = [&] (const Buffer &jpegData, int resolution, int pxAlign)
    {
        analyzePre(jpegData, resolution);

        // Decoder 事前処理
        NN_ABORT_UNLESS(decoder.Analyze() == nn::image::JpegStatus_Ok);

        pxDim = decoder.GetAnalyzedDimension();
        size_t alignedWidth = (pxDim.width % pxAlign == 0)?
                pxDim.width:
                pxDim.width - pxDim.width % pxAlign + pxAlign;
        size_t alignedHeight = (pxDim.height % pxAlign == 0)?
                pxDim.height:
                pxDim.height - pxDim.height % pxAlign + pxAlign;
        size_t pxSize = alignedWidth * alignedHeight * 4;
        if (pxSize > pxData.GetSize())
        {
            pxData.SetSize(pxSize);
        }

        auto workBufSize = decoder.GetAnalyzedWorkBufferSize();
        if (workBufSize > workBuf.GetSize())
        {
            workBuf.SetSize(workBufSize);
        }
    };
    auto analyzeTest = [&] { status = decoder.Analyze(); };
    auto decodeTest = [&] (int pxAlign) {
        status = decoder.Decode(pxData.GetPointer(), pxData.GetSize(), pxAlign, workBuf.GetPointer(), workBuf.GetSize());
    };
    auto post = [&] { NN_ASSERT(status == nn::image::JpegStatus_Ok); };

    // テストケース
    const char *setStr = nullptr;
    setStr = GetTestSetName(TestSet_DecoderAnalyze);
    nnt::image::jpeg::TestEntity analyzeTests[] = {
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 0, "Vga_100_420_1",
            [&] { analyzePre(jpegDataVga_100_420, 1); },
            analyzeTest, post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 1, "Vga_100_422_1",
            [&] { analyzePre(jpegDataVga_100_422, 1); },
            analyzeTest, post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 2, "Vga_100_444_1",
            [&] { analyzePre(jpegDataVga_100_444, 1); },
            analyzeTest, post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 3, "Vga_50_420_1",
            [&] { analyzePre(jpegDataVga_50_420, 1); },
            analyzeTest, post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 4, "Vga_1_420_1",
            [&] { analyzePre(jpegDataVga_1_420, 1); },
            analyzeTest, post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 5, "720P_100_420_1",
            [&] { analyzePre(jpegData720p_100_420, 1); },
            analyzeTest, post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 6, "1080P_100_420_1",
            [&] { analyzePre(jpegData1080p_100_420, 1); },
            analyzeTest, post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 7, "Vga_100_420_2",
            [&] { analyzePre(jpegDataVga_100_420, 2); },
            analyzeTest, post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 8, "Vga_100_420_4",
            [&] { analyzePre(jpegDataVga_100_420, 4); },
            analyzeTest, post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 9, "Vga_100_420_8",
            [&] { analyzePre(jpegDataVga_100_420, 8); },
            analyzeTest, post),
    };

    setStr = GetTestSetName(TestSet_DecoderDecode);
    nnt::image::jpeg::TestEntity decodeTests[] = {
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 0, "Vga_1_100_420_1",
            [&] { decodePre(jpegDataVga_100_420, 1, 1); },
            [&] { decodeTest(1); },
            post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 1, "Vga_1_100_422_1",
            [&] { decodePre(jpegDataVga_100_422, 1, 1); },
            [&] { decodeTest(1); },
            post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 2, "Vga_1_100_444_1",
            [&] { decodePre(jpegDataVga_100_444, 1, 1); },
            [&] { decodeTest(1); },
            post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 3, "Vga_1_50_420_1",
            [&] { decodePre(jpegDataVga_100_420, 1, 1); },
            [&] { decodeTest(1); },
            post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 4, "Vga_1_1_420_1",
            [&] { decodePre(jpegDataVga_100_420, 1, 1); },
            [&] { decodeTest(1); },
            post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 5, "Vga_4_100_420_1",
            [&] { decodePre(jpegDataVga_100_420, 1, 4); },
            [&] { decodeTest(4); },
            post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 6, "Vga_13_100_420_1",
            [&] { decodePre(jpegDataVga_100_420, 1, 13); },
            [&] { decodeTest(13); },
            post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 7, "720P_1_100_420_1",
            [&] { decodePre(jpegData720p_100_420, 1, 1); },
            [&] { decodeTest(1); },
            post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 8, "1080P_1_100_420_1",
            [&] { decodePre(jpegData1080p_100_420, 1, 1); },
            [&] { decodeTest(1); },
            post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 9, "Vga_1_100_420_2",
            [&] { decodePre(jpegDataVga_100_420, 2, 1); },
            [&] { decodeTest(1); },
            post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 10, "Vga_1_100_420_4",
            [&] { decodePre(jpegDataVga_100_420, 4, 1); },
            [&] { decodeTest(1); },
            post),
        NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(
            condStr, setStr, 11, "Vga_1_100_420_8",
            [&] { decodePre(jpegDataVga_100_420, 8, 1); },
            [&] { decodeTest(1); },
            post),
    };

    auto analyzeTestCount = sizeof(analyzeTests) / sizeof(analyzeTests[0]);
    auto encodeTestCount = sizeof(decodeTests) / sizeof(decodeTests[0]);
    NN_ASSERT(ResultCount >= analyzeTestCount + encodeTestCount);

    uint32_t counter = 0;
    for (int i = 0; i < static_cast<int>(analyzeTestCount); ++ i)
    {
        results[counter] = analyzeTests[i].Invoke(LoopCount);
        ++ counter;
    }
    for (int i = 0; i < static_cast<int>(encodeTestCount); ++ i)
    {
        results[counter] = decodeTests[i].Invoke(LoopCount);
        ++ counter;
    }
    NN_ASSERT(counter == analyzeTestCount + encodeTestCount);
    return analyzeTestCount + encodeTestCount;
} // NOLINT(readability/fn_size)

} // ~namespace encoderTest

TEST(ImageJpeg_Performance, Basic)
{
    static nnt::image::jpeg::TestResult results[1024] = {};
    static const size_t ResultCount = sizeof(results) / sizeof(results[0]);

    auto cond = Condition_Basic;
    const int TestLoopCount = 200;

    size_t s = 0;
    s += encoderTest::RunTests<Rgba32>(results + s, ResultCount - s, cond, TestLoopCount);
    s += decoderTest::RunTests<Rgba32>(results + s, ResultCount - s, cond, TestLoopCount);

    LogResult(results, s);
}

TEST(ImageJpeg_Performance, Rgb24)
{
    static nnt::image::jpeg::TestResult results[1024] = {};
    static const size_t ResultCount = sizeof(results) / sizeof(results[0]);

    auto cond = Condition_Rgb24;
    const int TestLoopCount = 200;

    size_t s = 0;
    s += encoderTest::RunTests<Rgb24>(results + s, ResultCount - s, cond, TestLoopCount);
    s += decoderTest::RunTests<Rgb24>(results + s, ResultCount - s, cond, TestLoopCount);

    LogResult(results, s);
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    static auto alloc = [](size_t size) -> void* {
        return std::malloc(size);
    };
    static auto dealloc = [](void* p, size_t size) {
        NN_UNUSED(size);
        std::free(p);
    };

    ::testing::InitGoogleTest(&argc, argv);

    // TeamCity の表示を適切にするため、イベントリスナの登録を一旦すべて解除し、
    // ServiceMessageLogger -> デフォルトのイベントリスナ の順で登録し直す。
    ::testing::TestEventListeners& listeners = ::testing::UnitTest::GetInstance()->listeners();
    ::testing::TestEventListener* defaultResultPrinter = listeners.Release(listeners.default_result_printer());
    listeners.Append(new nnt::teamcity::ServiceMessageLogger());
    listeners.Append(defaultResultPrinter);

    nn::fs::SetAllocator(alloc, dealloc);
    nn::fs::MountHostRoot();

    if (nnt::image::jpeg::io::Initialize())
    {
        int testResult = RUN_ALL_TESTS();
        nnt::image::jpeg::io::Cleanup();
        nnt::Exit(testResult);
    }

    nnt::Exit(-1);
}
