﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nnt/nntest.h>

#include <nn/nn_Common.h>

#include <nn/image/image_JpegDecoder.h>
#include <image_LibjpegHelper.h>

#include "testImageJpeg_Io.h"
#include "testImageJpeg_LibjpegApi.h"

namespace t = nnt::image::jpeg;

namespace
{
bool ErrorOk(int errorCode, int reason)
{
    NN_UNUSED(reason);
    return errorCode == 0;
}
}

#define NNT_IMAGE_ENABLE_JPEG_DECODER_ASSERTION
#define NNT_IMAGE_ENABLE_JPEG_DECODER_DECODE
#define NNT_IMAGE_ENABLE_JPEG_DECODER_BAD
#define NNT_IMAGE_ENABLE_JPEG_DECODER_RGB24
#define NNT_IMAGE_ENABLE_JPEG_DECODER_ANOTHER

#if defined NNT_IMAGE_ENABLE_JPEG_DECODER_ASSERTION

TEST(ImageJpegDecoder, AssertionTest)
{
    NN_LOG("----------------------------------------------------\n");
    NN_LOG("Testing nn::image::JpegDecoder pre-condition check\n");

    const char *assetPath = "./B-E-01_Noise_VGA_100_422.jpg";
    size_t jpegSize;
    nnt::image::jpeg::io::GetAssetSize(&jpegSize, assetPath);
    void *jpegData = malloc(jpegSize);
    ASSERT_TRUE(nnt::image::jpeg::io::ReadAsset(jpegData, jpegSize, assetPath));

    nn::image::JpegDecoder decoder;
    size_t exifSize;
    void *workBuf = malloc(10 * 1024 * 1024);
    void *pxData = malloc(10 * 1024 * 1024);

    // Invalid data specification
    NN_LOG("  - Test ID: %s\n", "Initialization");
    EXPECT_DEATH_IF_SUPPORTED({decoder.SetImageData(nullptr, jpegSize);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({decoder.SetImageData(jpegData, 0);}, ".*");

    // Unregistered
    NN_LOG("  - Test ID: %s\n", "Registration check");
    EXPECT_DEATH_IF_SUPPORTED({decoder.Analyze();}, ".*");

    // Bad parameter
    NN_LOG("  - Test ID: %s\n", "Decoding params");
    EXPECT_DEATH_IF_SUPPORTED({decoder.SetResolutionDenominator(0);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({decoder.SetResolutionDenominator(3);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({decoder.SetResolutionDenominator(17);}, ".*");

    decoder.SetImageData(jpegData, jpegSize);

    // Unanalyzed
    NN_LOG("  - Test ID: %s\n", "Analyzation check");
    EXPECT_DEATH_IF_SUPPORTED({decoder.GetAnalyzedDimension();}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({decoder.GetAnalyzedWorkBufferSize();}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({decoder.GetAnalyzedExifData(&exifSize);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({decoder.Decode(pxData, 10 * 1024 * 1024, 1, workBuf, 10 * 1024 * 1024);}, ".*");

    EXPECT_EQ(nn::image::JpegStatus_Ok, decoder.Analyze());

    decoder.SetImageData(jpegData, jpegSize);

    // Unanalyzed (again)
    NN_LOG("  - Test ID: %s\n", "Analyzation check (again)");
    EXPECT_DEATH_IF_SUPPORTED({decoder.GetAnalyzedDimension();}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({decoder.GetAnalyzedWorkBufferSize();}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({decoder.GetAnalyzedExifData(&exifSize);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({decoder.Decode(pxData, 10 * 1024 * 1024, 1, workBuf, 10 * 1024 * 1024);}, ".*");

    EXPECT_EQ(nn::image::JpegStatus_Ok, decoder.Analyze());

    nn::image::Dimension dim = decoder.GetAnalyzedDimension();
    size_t workBufSize = decoder.GetAnalyzedWorkBufferSize();
    // const void *exifData = decoder.GetAnalyzedExifData(&exifSize);
    NN_ASSERT(dim.width == 640);
    NN_ASSERT(dim.height == 480);
    NN_ASSERT(dim.width * dim.height * 4 <= 10 * 1024 * 1024);
    NN_ASSERT(workBufSize <= 10 * 1024 * 1024);

    // Bad argument for Decode method.
    NN_LOG("  - Test ID: %s\n", "Arguments for 'Decode' method");
    size_t pxSize = dim.width * dim.height * 4;
    EXPECT_DEATH_IF_SUPPORTED({decoder.Decode(nullptr, pxSize, 1, workBuf, workBufSize);}, ".*"); // Null outbuf
    EXPECT_DEATH_IF_SUPPORTED({decoder.Decode(pxData, pxSize - 1, 1, workBuf, workBufSize);}, ".*"); // small output
    EXPECT_DEATH_IF_SUPPORTED({decoder.Decode(
        pxData, 644 * dim.height * 4 - 1, 7,
        workBuf, workBufSize);}, ".*"); // small output(2)
    EXPECT_DEATH_IF_SUPPORTED({decoder.Decode(pxData, pxSize, 0, workBuf, workBufSize);}, ".*"); // Bad alignment
    EXPECT_DEATH_IF_SUPPORTED({decoder.Decode(pxData, pxSize, 1, nullptr, workBufSize);}, ".*"); // Bad workbuf
    EXPECT_DEATH_IF_SUPPORTED({decoder.Decode(pxData, pxSize, 1, workBuf, 0);}, ".*"); // Bad workbufSize
    EXPECT_DEATH_IF_SUPPORTED({decoder.Decode(pxData, pxSize, 1, workBuf, workBufSize - 1);}, ".*"); // Bad workbufSize

    free(jpegData);
    free(workBuf);
    free(pxData);
} // NOLINT(readability/fn_size)

#endif // NNT_IMAGE_ENABLE_JPEG_DECODER_ASSERTION

struct TestCaseJpegDecoder
{
    const char *file;
    int scalingParam;
    int alignment;
    bool passAnalyze;
    nn::image::JpegStatus status;
};

#if defined(NNT_IMAGE_ENABLE_JPEG_DECODER_DECODE)

TEST(ImageJpegDecoder, DecodeTest)
{
    NN_LOG("----------------------------------------------------\n");
    NN_LOG("Testing helper API for JPEG decoder\n");

    TestCaseJpegDecoder normalCases[] = {
        {"./U-D-01_Progressive_420_VGA_i1.jpg", 0, 1, true, nn::image::JpegStatus_Ok},
        {"./U-D-02_Baseline_444_VGA_8_maxExif.jpg", 8, 7, true, nn::image::JpegStatus_Ok},
    };

    for (int i = 0; i < static_cast<int>(sizeof(normalCases) / sizeof(TestCaseJpegDecoder)); i++)
    {
        TestCaseJpegDecoder &test = normalCases[i];

        NN_LOG("- Image[%d]: %s\n", i, test.file);

        size_t jpegSize;
        nnt::image::jpeg::io::GetAssetSize(&jpegSize, test.file);
        t::io::Buffer inputJpeg(jpegSize);
        ASSERT_TRUE(nnt::image::jpeg::io::ReadAsset(inputJpeg.GetPointer(), inputJpeg.GetSize(), test.file));

        // Get true properties
        t::io::Buffer pxTruth;
        nn::image::detail::JpegHeader header = {};
        {
            ASSERT_EQ(
                nn::image::JpegStatus_Ok,
                nn::image::detail::ExtractJpegHeader(
                    &header,
                    inputJpeg.GetPointer(), inputJpeg.GetSize(),
                    test.scalingParam > 0? test.scalingParam: 1));

            pxTruth.SetSize(header.dimension.width * header.dimension.height * 4);
            t::io::Buffer workBuf(header.bufSizeForDecomp);

            nnt::image::jpeg::TestArgDecode arg = {
                ErrorOk,
                test.scalingParam > 0? test.scalingParam: 1,
                {static_cast<uint16_t>(header.dimension.width), static_cast<uint16_t>(header.dimension.height)}};
            ASSERT_TRUE(
                nnt::image::jpeg::DecompressBinary(
                    pxTruth.GetPointerAs<uint32_t>(), pxTruth.GetSize(),
                    inputJpeg.GetPointer(), inputJpeg.GetSize(), arg,
                    workBuf.GetPointer(), workBuf.GetSize()));
        }

        // Get test properties
        t::io::Buffer pxTest;
        nn::image::Dimension outDim;
        {
            nn::image::JpegDecoder decoder;
            decoder.SetImageData(inputJpeg.GetPointer(), inputJpeg.GetSize());
            if (test.scalingParam > 0)
            {
                decoder.SetResolutionDenominator(test.scalingParam);
            }
            EXPECT_EQ(nn::image::JpegStatus_Ok, decoder.Analyze());

            auto dim = decoder.GetAnalyzedDimension();
            NN_LOG("  - Estimated dimension: {%d, %d}\n", dim.width, dim.height);
            EXPECT_EQ(header.dimension.width, dim.width);
            EXPECT_EQ(header.dimension.height, dim.height);

            size_t exifSize;
            const void *exifData = decoder.GetAnalyzedExifData(&exifSize);
            EXPECT_EQ(header.hasExif, exifData != nullptr);
            if (header.hasExif)
            {
                NN_LOG("  - Extracted Exif: %08lx (%d bytes)\n", exifData, exifSize);
                EXPECT_EQ(header.exifData, exifData);
                EXPECT_EQ(header.exifSize, exifSize);

                // static 版のテスト
                EXPECT_EQ(
                    nn::image::JpegStatus_Ok,
                    nn::image::JpegDecoder::GetExifData(&exifData, &exifSize, inputJpeg.GetPointer(), inputJpeg.GetSize()));
                EXPECT_EQ(header.exifData, exifData);
                EXPECT_EQ(header.exifSize, exifSize);
            }

            t::io::Buffer workBuf(decoder.GetAnalyzedWorkBufferSize());
            NN_LOG("  - Estimated work buffer size: %d\n", workBuf.GetSize());
            EXPECT_EQ(
                header.bufSizeForDecomp + dim.width * nn::image::detail::LibjpegProperty_1LoopLineNum * 3,
                workBuf.GetSize());

            outDim.width = (dim.width % test.alignment == 0?
                    dim.width:
                    dim.width - dim.width % test.alignment + test.alignment);
            outDim.height = dim.height;
            pxTest.SetSize(outDim.width * outDim.height * sizeof(uint32_t));

            EXPECT_EQ(
                nn::image::JpegStatus_Ok,
                decoder.Decode(
                    pxTest.GetPointer(), pxTest.GetSize(), test.alignment,
                    workBuf.GetPointer(), workBuf.GetSize()));
        }

        uint64_t error = 0x00ull;
        for (int j = 0; j < header.dimension.height; j++)
        {
            for (int k = 0; k < header.dimension.width; k++)
            {
                // JpegDecoder は R,G,B,A の順に格納されるが、テスト側の実装はエンディアン依存で ABGR にもなり得る。
                auto pTestPx = reinterpret_cast<uint8_t*>(&pxTest.GetPointerAs<uint32_t>()[j * outDim.width + k]);
                auto truePx = pxTruth.GetPointerAs<uint32_t>()[j * header.dimension.width + k];
                error += std::abs((int)pTestPx[0] - (int)((truePx & 0xFF000000) >> 24));
                error += std::abs((int)pTestPx[1] - (int)((truePx & 0x00FF0000) >> 16));
                error += std::abs((int)pTestPx[2] - (int)((truePx & 0x0000FF00) >>  8));
            }
        }
        EXPECT_EQ(0ull, error);
    }
}  // NOLINT(readability/fn_size)

#endif // NNT_IMAGE_ENABLE_JPEG_DECODER_DECODE

#if defined(NNT_IMAGE_ENABLE_JPEG_DECODER_BAD)

TEST(ImageJpegDecoder, DecodeTest_BadJpeg)
{
    NN_LOG("----------------------------------------------------\n");
    NN_LOG("Testing helper API for JPEG decoder with bad JPEG\n");

    TestCaseJpegDecoder badCases[] = {
        {"./z05_BadSof_4color(A-D-02).jpg", 0, 1, false, nn::image::JpegStatus_WrongFormat},
        {"./z08_BadDht_NoDC0(A-D-02).jpg", 0, 3, true, nn::image::JpegStatus_WrongFormat},
        {"./z11_BadDqt_Unknown(A-D-02).jpg", 1, 7, true, nn::image::JpegStatus_Ok},
        {"./z13_Terminated_NoPix(A-D-02).jpg", 0, 2, false, nn::image::JpegStatus_WrongFormat},
        {"./z14_Terminated_WithEoi(A-D-02).jpg", 4, 13, true, nn::image::JpegStatus_Ok},
    };

    for (int i = 0; i < static_cast<int>(sizeof(badCases) / sizeof(TestCaseJpegDecoder)); i++)
    {
        TestCaseJpegDecoder &test = badCases[i];

        NN_LOG("- Image[%d]: %s\n", i, test.file);

        size_t jpegSize;
        nnt::image::jpeg::io::GetAssetSize(&jpegSize, test.file);
        t::io::Buffer inputJpeg(jpegSize);
        ASSERT_TRUE(nnt::image::jpeg::io::ReadAsset(inputJpeg.GetPointer(), inputJpeg.GetSize(), test.file));

        // Get test properties
        nn::image::JpegDecoder decoder;
        decoder.SetImageData(inputJpeg.GetPointer(), inputJpeg.GetSize());
        if (test.scalingParam > 0)
        {
            decoder.SetResolutionDenominator(test.scalingParam);
        }

        if (!test.passAnalyze)
        {
            EXPECT_EQ(test.status, decoder.Analyze());
        }
        else
        {
            ASSERT_EQ(nn::image::JpegStatus_Ok, decoder.Analyze());

            auto dim = decoder.GetAnalyzedDimension();
            t::io::Buffer workBuf(decoder.GetAnalyzedWorkBufferSize());

            nn::image::Dimension outDim;
            t::io::Buffer pxTest;
            outDim.width = (dim.width % test.alignment == 0?
                    dim.width:
                    dim.width - dim.width % test.alignment + test.alignment);
            outDim.height = dim.height;
            pxTest.SetSize(outDim.width * outDim.height * sizeof(uint32_t));

            EXPECT_EQ(
                test.status,
                decoder.Decode(
                    pxTest.GetPointer(), pxTest.GetSize(), test.alignment,
                    workBuf.GetPointer(), workBuf.GetSize()));
        }
    }
}

#endif // NNT_IMAGE_ENABLE_JPEG_DECODER_BAD

#if defined(NNT_IMAGE_ENABLE_JPEG_DECODER_RGB24)

TEST(ImageJpegDecoder, DecodeTestRgb24)
{
    NN_LOG("----------------------------------------------------\n");
    NN_LOG("Testing helper API for JPEG decoder\n");

    TestCaseJpegDecoder normalCases[] = {
        {"./U-D-01_Progressive_420_VGA_i1.jpg", 0, 1, true, nn::image::JpegStatus_Ok},
        {"./U-D-02_Baseline_444_VGA_8_maxExif.jpg", 8, 7, true, nn::image::JpegStatus_Ok},
    };

    for (int i = 0; i < static_cast<int>(sizeof(normalCases) / sizeof(TestCaseJpegDecoder)); i++)
    {
        TestCaseJpegDecoder &test = normalCases[i];

        NN_LOG("- Image[%d]: %s\n", i, test.file);

        size_t jpegSize;
        nnt::image::jpeg::io::GetAssetSize(&jpegSize, test.file);
        t::io::Buffer inputJpeg(jpegSize);
        ASSERT_TRUE(nnt::image::jpeg::io::ReadAsset(inputJpeg.GetPointer(), inputJpeg.GetSize(), test.file));

        // Get true properties
        t::io::Buffer pxTruth;
        nn::image::detail::JpegHeader header = {};
        {
            ASSERT_EQ(
                nn::image::JpegStatus_Ok,
                nn::image::detail::ExtractJpegHeader(
                    &header,
                    inputJpeg.GetPointer(), inputJpeg.GetSize(),
                    test.scalingParam > 0? test.scalingParam: 1));

            pxTruth.SetSize(header.dimension.width * header.dimension.height * 4);
            t::io::Buffer workBuf(header.bufSizeForDecomp);

            nnt::image::jpeg::TestArgDecode arg = {
                ErrorOk,
                test.scalingParam > 0? test.scalingParam: 1,
                {static_cast<uint16_t>(header.dimension.width), static_cast<uint16_t>(header.dimension.height)}};
            ASSERT_TRUE(
                nnt::image::jpeg::DecompressBinary(
                    pxTruth.GetPointerAs<uint32_t>(), pxTruth.GetSize(),
                    inputJpeg.GetPointer(), inputJpeg.GetSize(), arg,
                    workBuf.GetPointer(), workBuf.GetSize()));
        }

        // Get test properties
        t::io::Buffer pxTest;
        nn::image::Dimension outDim;
        {
            nn::image::JpegDecoder decoder;
            decoder.SetImageData(inputJpeg.GetPointer(), inputJpeg.GetSize());
            if (test.scalingParam > 0)
            {
                decoder.SetResolutionDenominator(test.scalingParam);
            }
            decoder.SetPixelFormat(nn::image::PixelFormat_Rgb24);
            EXPECT_EQ(nn::image::JpegStatus_Ok, decoder.Analyze());

            auto dim = decoder.GetAnalyzedDimension();
            NN_LOG("  - Estimated dimension: {%d, %d}\n", dim.width, dim.height);
            EXPECT_EQ(header.dimension.width, dim.width);
            EXPECT_EQ(header.dimension.height, dim.height);

            size_t exifSize;
            const void *exifData = decoder.GetAnalyzedExifData(&exifSize);
            EXPECT_EQ(header.hasExif, exifData != nullptr);
            if (header.hasExif)
            {
                NN_LOG("  - Extracted Exif: %08lx (%d bytes)\n", exifData, exifSize);
                EXPECT_EQ(header.exifData, exifData);
                EXPECT_EQ(header.exifSize, exifSize);

                // static 版のテスト
                EXPECT_EQ(
                    nn::image::JpegStatus_Ok,
                    nn::image::JpegDecoder::GetExifData(&exifData, &exifSize, inputJpeg.GetPointer(), inputJpeg.GetSize()));
                EXPECT_EQ(header.exifData, exifData);
                EXPECT_EQ(header.exifSize, exifSize);
            }

            t::io::Buffer workBuf(decoder.GetAnalyzedWorkBufferSize());
            NN_LOG("  - Estimated work buffer size: %d\n", workBuf.GetSize());
            EXPECT_EQ(header.bufSizeForDecomp, workBuf.GetSize());

            outDim.width = (dim.width % test.alignment == 0?
                    dim.width:
                    dim.width - dim.width % test.alignment + test.alignment);
            outDim.height = dim.height;
            pxTest.SetSize(outDim.width * outDim.height * sizeof(uint32_t));

            EXPECT_EQ(
                nn::image::JpegStatus_Ok,
                decoder.Decode(
                    pxTest.GetPointer(), pxTest.GetSize(), test.alignment,
                    workBuf.GetPointer(), workBuf.GetSize()));
        }

        uint64_t error = 0x00ull;
        for (int j = 0; j < header.dimension.height; j++)
        {
            for (int k = 0; k < header.dimension.width; k++)
            {
                // JpegDecoder は R,G,B,A の順に格納されるが、テスト側の実装はエンディアン依存で ABGR にもなり得る。
                auto pTestPx = &pxTest.GetPointerAs<uint8_t>()[(j * outDim.width + k) * 3];
                auto truePx = pxTruth.GetPointerAs<uint32_t>()[j * header.dimension.width + k];
                error += std::abs((int)pTestPx[0] - (int)((truePx & 0xFF000000) >> 24));
                error += std::abs((int)pTestPx[1] - (int)((truePx & 0x00FF0000) >> 16));
                error += std::abs((int)pTestPx[2] - (int)((truePx & 0x0000FF00) >>  8));
            }
        }
        EXPECT_EQ(0ull, error);
    }
}  // NOLINT(readability/fn_size)

#endif // NNT_IMAGE_ENABLE_JPEG_DECODER_DECODE

#if defined(NNT_IMAGE_ENABLE_JPEG_DECODER_ANOTHER)
namespace
{

}
TEST(ImageJpegDecoder, DecodeTestAnother)
{


}
#endif
