﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>

#include <nn/nn_Log.h>

#include "testImageJpeg_LibjpegApi.h"
#include "testImageJpeg_Tga.h"

namespace nnt { namespace image { namespace jpeg {

namespace
{

static const int HeaderSize = 18;

bool GetTgaProfile(
    uint16_t *pOutWidth, uint16_t *pOutHeight, uint8_t *pOutBits,
    const void *tga, const size_t size)
{
    // このTGAフォーマットにしか対応しない。
    uint8_t headerFix[12] = {0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
    if (size < HeaderSize || std::memcmp(tga, headerFix, sizeof(headerFix)) != 0)
    {
        return false;
    }

    const uint8_t *tgaBytes = reinterpret_cast<const uint8_t*>(tga);

    // 幅、高さ、ビット数を取得
    uint16_t width = tgaBytes[12] + (tgaBytes[13] << 8);
    uint16_t height = tgaBytes[14] + (tgaBytes[15] << 8);
    uint8_t bits = tgaBytes[16];

    // 対応していないビット数を検出
    if (!(bits == 24 || bits == 32))
    {
        return false;
    }

    *pOutWidth = width;
    *pOutHeight = height;
    *pOutBits = bits;
    return true;
}

#define ROUND_UP(v, d) (((v) % (d)) != 0? (v) - ((v) % (d)) + (d): (v))

}

bool Tga::GetDimension(Dimension *pOutDim, const void *tga, const size_t size)
{
    uint16_t width = 0;
    uint16_t height = 0;
    uint8_t bits = 0;
    if (!GetTgaProfile(&width, &height, &bits, tga, size))
    {
        return false;
    }

    pOutDim->width = width;
    pOutDim->height = height;
    return true;
}

void Tga::GetPixels(
    uint32_t *pixel,
    const size_t pixelSize,
    const int lineAlignment,
    const void *tga,
    const size_t tgaSize)
{
    uint16_t width = 0;
    uint16_t height = 0;
    uint8_t bits = 0;
    if (!GetTgaProfile(&width, &height, &bits, tga, tgaSize) ||
        pixelSize < static_cast<size_t>(ROUND_UP(width, lineAlignment) * height))
    {
        return;
    }

    uint8_t bytes = bits / 8;
    const uint8_t *tgaBytes = reinterpret_cast<const uint8_t*>(tga) + HeaderSize;
    for (int h = 0; h < height; h++)
    {
        uint32_t *pixLine = pixel + h * ROUND_UP(width, lineAlignment);
        const uint8_t *tgaLine = tgaBytes + (height - h - 1) * width * bytes; //< TGAは上下逆に格納される。(3DCGあるある)
        for (int w = 0; w < width; w++)
        {
            // B, G, R, (a) データを上から保存。
            pixLine[w] =
                (tgaLine[w * bytes] << 8) +
                (tgaLine[w * bytes + 1] << 16) +
                (tgaLine[w * bytes + 2] << 24) + 0xFF;
        }
    }
}

}}}
