﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <nn/nn_Abort.h>

#undef NN_TEST_JPEG_CONFIG_SAVE_ENABLED

/**
    @file IO関連APIを宣言します。
 */

namespace nnt { namespace image { namespace jpeg { namespace io {

class Buffer
{
    NN_DISALLOW_COPY(Buffer);
    NN_DISALLOW_MOVE(Buffer);
private:
    char* m_Buffer;
    size_t m_Size;

    void Free() NN_NOEXCEPT
    {
        if (m_Buffer)
        {
            std::free(m_Buffer);
            m_Buffer = nullptr;
            m_Size = 0;
        }
    }

public:
    Buffer() NN_NOEXCEPT
        : m_Buffer(nullptr)
        , m_Size(0)
    {
    }
    explicit Buffer(size_t size) NN_NOEXCEPT
        : m_Buffer(nullptr)
        , m_Size(0)
    {
        SetSize(size);
    }
    ~Buffer() NN_NOEXCEPT
    {
        Free();
    }
    NN_IMPLICIT operator bool() const NN_NOEXCEPT
    {
        return m_Buffer != nullptr;
    }

    char* SetSize(size_t size) NN_NOEXCEPT
    {
        Free();

        if (size == 0)
        {
            return nullptr;
        }

        m_Buffer = reinterpret_cast<char*>(std::malloc(size));
        NN_ABORT_UNLESS(m_Buffer);
        m_Size = size;
        return m_Buffer;
    }
    char* GetPointer() const NN_NOEXCEPT
    {
        return GetPointerAs<char>();
    }
    template <typename T>
    T* GetPointerAs() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_Buffer);
        return reinterpret_cast<T*>(m_Buffer);
    }
    size_t GetSize() const NN_NOEXCEPT
    {
        return m_Size;
    }
};

/**
    @brief  JPEGライブラリテスト用の入出力系を初期化します。
    @detail Externals/Binaries/TestData/nn_image_jpeg を入力元としてマウントし、
            C:\Windows\Temp\SIGLO_JPEG_TEST_hoeghoge を出力先としてマウントします。
 */
bool Initialize();

/**
    @brief  JPEGライブラリテスト用の入出力系をクリーンアップます。
    @detail 全てのマウントポイントをアンマウントしたうえで、
            前回のInitialize(.)の呼び出し時に作成された C:\Windows\Temp\SIGLO_JPEG_TEST_hoeghoge を削除します。
 */
bool Cleanup();

/**
    @brief  入力元ボリューム上の指定されたパスのファイルサイズを取得します。
    @param[out] pSize 取得されたファイルサイズが格納されます。
    @param[in]  path サイズを取得したいファイルの、入力元ボリューム内でのパスを指定します。
    @return pSizeに有効な値が格納された場合は true が返ります。
 */
bool GetAssetSize(size_t *pSize, const char *path);

/**
    @brief  入力元ボリューム上の指定されたパスを読み込みます。
    @param[out] data 読み込まれたデータが格納されます。
    @param[in]  kDataSize 読み込み対象のファイルのサイズを指定します。
    @param[in]  path 読み込みたいファイルの、入力元ボリューム内でのパスを指定します。
    @return dataに有効なデータが格納された場合は true が返ります。
 */
bool ReadAsset(
    void *data,
    const size_t kDataSize,
    const char *path);

inline bool ReadAsset(const Buffer& buffer, const char *path)
{
    return ReadAsset(buffer.GetPointer(), buffer.GetSize(), path);
}

inline bool ReadAssetWithAllocation(Buffer *pOut, const char* path)
{
    size_t tgaSize;
    if (!nnt::image::jpeg::io::GetAssetSize(&tgaSize, path))
    {
        return false;
    }
    pOut->SetSize(tgaSize);
    return nnt::image::jpeg::io::ReadAsset(*pOut, path);
}

#ifdef NN_TEST_JPEG_CONFIG_SAVE_ENABLED
/**
    @brief  出力先ボリューム上の指定されたパスのファイルサイズを取得します。
    @param[out] pSize 取得されたファイルサイズが格納されます。
    @param[in]  path サイズを取得したいファイルの、出力先ボリューム内でのパスを指定します。
    @return pSizeに有効な値が格納された場合は true が返ります。
 */
bool GetSavedataSize(size_t *pSize, const char *path);

/**
    @brief  出力先ボリューム上の指定されたパスを読み込みます。
    @param[out] data 読み込まれたデータが格納されます。
    @param[in]  kDataSize 読み込み対象のファイルのサイズを指定します。
    @param[in]  path 読み込みたいファイルの、出力先ボリューム内でのパスを指定します。
    @return dataに有効なデータが格納された場合は true が返ります。
 */
bool ReadSavedata(
    void *data,
    const size_t kDataSize,
    const char *path);

inline bool ReadSavedata(const Buffer& buffer, const char *path)
{
    return ReadSaveData(buffer.GetPointer(), buffer.GetSize(), path);
}

/**
    @brief  出力先ボリューム上の指定されたパスにデータを書き込みます。
    @param[in]  path 書き込みたいファイルの、出力先ボリューム内でのパスを指定します。
    @param[in]  data 書き込みたいデータを指定します。
    @param[in]  kDataSize 書き込みたいデータのサイズを指定します。
    @return dataに有効なデータが書き込まれた場合は true が返ります。
 */
bool WriteSavedata(
    const char *path,
    const void *data,
    const size_t kDataSize);

inline bool WriteSavedata(const char *path, const Buffer& buffer)
{
    return WriteSavedata(path, buffer.GetPointer(), buffer.GetSize());
}
#endif

}}}}
