﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  ユーティリティ関数
//---------------------------------------------------------------------------

#include <cstring>
#include <cstdlib>

#include <nn/os.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nnt/nntest.h>

#include "test_Util.h"

namespace nnt { namespace i2c { namespace util {

// ポートに設定する送受信バッファ
const size_t BufferSize = 1 * nn::os::MemoryPageSize;
NN_ALIGNAS(nn::os::MemoryPageSize) char s_SendBuffer[BufferSize];
NN_ALIGNAS(nn::os::MemoryPageSize) char s_ReceiveBuffer[BufferSize];

const nn::i2c::TransactionOption InOption = static_cast<nn::i2c::TransactionOption>( nn::i2c::TransactionOption_StartCondition | nn::i2c::TransactionOption_StopCondition );

void ClearBuffer(char* data, int dataBytes)
{
    std::memset(data, 0x00, dataBytes);
}
bool CompareTestData(const char* expectedData, const char* actualData, int dataBytes)
{
    for (int i = 0; i < dataBytes; i++)
    {
        if (expectedData[i] != actualData[i])
        {
            NN_LOG("\n** Data did not match at index %d **\n\n", i);
            NN_LOG("Expected | Actual\n");
            for (int dump = 0; dump < dataBytes; dump++)
            {
                 NN_LOG("  %02x | %02x\n", expectedData[dump], actualData[dump]);
            }
            return false;
        }
    }
    return true;
}

void TestReadRegister(nn::i2c::I2cSession& i2c, const char* pReg, size_t regBytes, const char* pExpectedData, size_t dataBytes )
{
    nn::Result result;
    char* bufferToReceive = s_ReceiveBuffer;

    // 念のため
    ASSERT_TRUE(BufferSize >= dataBytes);

    ClearBuffer(bufferToReceive, BufferSize);

    NN_LOG("Sending reg address: %d bytes\n", regBytes);
    result = nn::i2c::Send(i2c, pReg, regBytes, InOption);
    NN_LOG("Send result:%d\n", result.GetDescription());
    ASSERT_TRUE(result.IsSuccess());

    NN_LOG("Receiving %d bytes\n", dataBytes);
    result = nn::i2c::Receive(bufferToReceive, i2c, dataBytes, InOption);
    NN_LOG("Receive result:%d\n", result.GetDescription());
    ASSERT_TRUE(result.IsSuccess());
    if (pExpectedData)
    {
        ASSERT_TRUE(CompareTestData(pExpectedData, bufferToReceive, dataBytes));
    }
    else // 期待値を指定していなければ、ダンプのみ行う
    {
        NN_LOG("Received data\n");
        for (size_t dump = 0; dump < dataBytes; dump++)
        {
             NN_LOG("  0x%02x\n", bufferToReceive[dump]);
        }
    }

    NN_LOG("Verify done. Success!\n");
}

}}}  // namespace nnt::i2c::util
