﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  CommandListFormatter 関連のテスト
//---------------------------------------------------------------------------

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/i2c/i2c.h>

#include <nnt/nntest.h>

namespace {

// 期待されるコマンドの値 ( Ref. i2c_CommandListFormat.h )
const uint8_t CommandSend      = 0x00; //!< Send
const uint8_t CommandReceive   = 0x01; //!< Receive
const uint8_t CommandExtension = 0x02; //!< Extension
const uint8_t CommandSleep     = 0x00; //!< Extension-Sub Sleep

// 期待されるオプションの値
const uint8_t OptionStartCondition = 1 << 6; //!< StartCondition
const uint8_t OptionStopCondition  = 1 << 7; //!< StopCondition

} // namespace


/**
 *   CommandListFormatter が生成するコマンドリストのバイト列が、
 *   想定した仕様通りのものであるかをテストします。
 */
TEST( CommandListFormatter, test_CommandListFormatter )
{
    // 送信データ
    const uint8_t                  sendBuf[2] = {0x0A, 0x0B};

    // コマンドリスト
    uint8_t                        commandList[nn::i2c::CommandListLengthCountMax];
    nn::i2c::CommandListFormatter  commandListFormatter(commandList, sizeof(commandList));

    // Step1-6 でコマンドリストを生成 ------------------------------------------------------------

    // 1. Send 1 byte w/ StartCondition
    commandListFormatter.EnqueueSendCommand(
        nn::i2c::TransactionOption::TransactionOption_StartCondition,
        sendBuf,
        1 );

    // 2. Send 2 bytes w/ StartCondition + StopCondition
    commandListFormatter.EnqueueSendCommand(
        static_cast<nn::i2c::TransactionOption>( nn::i2c::TransactionOption::TransactionOption_StartCondition | nn::i2c::TransactionOption::TransactionOption_StopCondition),
        sendBuf,
        2 );

    // 3. Receive 1 byte w/ StartCondition
    commandListFormatter.EnqueueReceiveCommand(
        nn::i2c::TransactionOption::TransactionOption_StartCondition,
        1 );

    // 4. Receive 2 bytes w/ StartCondition + StopCondition
    commandListFormatter.EnqueueReceiveCommand(
        static_cast<nn::i2c::TransactionOption>( nn::i2c::TransactionOption::TransactionOption_StartCondition | nn::i2c::TransactionOption::TransactionOption_StopCondition),
        2 );

    // 5. Sleep 1 microsecond
    commandListFormatter.EnqueueSleepCommand( 0x01 );

    // 6. Sleep 0x02 microseconds
    commandListFormatter.EnqueueSleepCommand( 0x02 );

    // ------------------------------------------------------------

    // Step1-6 で CommandListFormatter が生成するバイト列の期待値
    const uint8_t ExpectedByteSequence[] = {
        CommandSend | OptionStartCondition, 1, sendBuf[0],                                    // 1
        CommandSend | OptionStartCondition | OptionStopCondition, 2, sendBuf[0], sendBuf[1],  // 2
        CommandReceive | OptionStartCondition, 1,                                             // 3
        CommandReceive | OptionStartCondition | OptionStopCondition, 2,                       // 4
        CommandExtension | CommandSleep, 1,                                                   // 5
        CommandExtension | CommandSleep, 2,                                                   // 6
    };

    // 生成されたコマンド列の期待される長さ
    const size_t ExpectedCommandListLength = sizeof(ExpectedByteSequence) / sizeof(ExpectedByteSequence[0]);

    // 検証
    // NN_LOG("CommandList Length = %d.\n", ExpectedCommandListLength);

    // 生成したコマンド列の長さが等しいか
    NN_ABORT_UNLESS( ExpectedCommandListLength == commandListFormatter.GetCurrentLength() );

    // 生成したコマンド列の値が想定通りか
    for(size_t i=0; i < ExpectedCommandListLength; i++ )
    {
        // NN_LOG("[%d] Expected=0x%02x, Generated=0x%02x\n", i, ExpectedByteSequence[i], commandList[i]);
        NN_ABORT_UNLESS( ExpectedByteSequence[i] == commandList[i] );
    }
}
