﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/nn_Windows.h>
#endif

#include <cstdlib>

#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Debug.h>
#include <nn/os/os_Mutex.h>
#include <nn/time/time_ApiForMenu.h>
#include <nn/time/time_StandardNetworkSystemClock.h>
#include <nn/time/time_TimeZoneApi.h>

#if !defined(__LINE__Var)
#define __LINE__Var 0 // NOLINT(preprocessor/const)
#endif

namespace nnt { namespace http {

struct Mutex
{
    nn::os::MutexType data;

    void lock() NN_NOEXCEPT
    {
        nn::os::LockMutex(&data);
    }
    void unlock() NN_NOEXCEPT
    {
        nn::os::UnlockMutex(&data);
    }
};
#define NNT_HTTP_MUTEX_INITIALIZER(recursive) {NN_OS_MUTEX_INITIALIZER(recursive)}

struct Buffer
{
    NN_DISALLOW_COPY(Buffer);

private:
    void* _address;
    size_t _size;

    void swap(Buffer& rhs) NN_NOEXCEPT
    {
        std::swap(_address, rhs._address);
        std::swap(_size, rhs._size);
    }
public:
    Buffer() NN_NOEXCEPT
        : Buffer(0)
    {
    }
    explicit Buffer(size_t size) NN_NOEXCEPT
        : _address(size > 0? malloc(size): nullptr)
        , _size(size)
    {
        if (size > 0 && !_address)
        {
            nn::os::MemoryInfo mi;
            nn::os::QueryMemoryInfo(&mi);
            NN_LOG("OS Memory Info for Debug *:;:*:;:*:;:*:;:*:;:*:;:*:;:*:;:*:;:*:;:\n");
            NN_LOG(" - MemorySize:  %zu of %llu\n", mi.totalUsedMemorySize, mi.totalAvailableMemorySize);
            NN_LOG(" - HeapSize:    %zu of %zu\n", mi.allocatedMemoryHeapSize, mi.totalMemoryHeapSize);
            NN_LOG(" - Program:     %zu\n", mi.programSize);
            NN_LOG(" - ThreadStack: %zu [count=%d]\n", mi.totalThreadStackSize, mi.threadCount);
            NN_ABORT_UNLESS(_address, "Requested size of memory cannot be allocated: %zu\n", size);
        }
    }
    Buffer(Buffer&& rhs) NN_NOEXCEPT
        : _address(rhs._address)
        , _size(rhs._size)
    {
        rhs._address = nullptr;
        rhs._size = 0u;
    }
    ~Buffer() NN_NOEXCEPT
    {
        if (_address)
        {
            free(_address);
        }
    }
    Buffer& operator =(Buffer&& rhs) NN_NOEXCEPT
    {
        Buffer tmp(std::move(rhs));
        swap(tmp);
        return *this;
    }
    NN_IMPLICIT operator bool() const NN_NOEXCEPT
    {
        return _address != nullptr && _size != 0;
    }
    void* GetAddress() const NN_NOEXCEPT
    {
        return _address;
    }
    template <typename T>
    T* Get() const NN_NOEXCEPT
    {
        return reinterpret_cast<T*>(_address);
    }
    size_t GetSize() const NN_NOEXCEPT
    {
        return _size;
    }
};

inline void TryLogNetworkTime() NN_NOEXCEPT
{
    nn::time::PosixTime posix;
    auto r = nn::time::StandardNetworkSystemClock::GetCurrentTime(&posix);
    NN_ABORT_UNLESS(r.IsSuccess() || nn::time::ResultClockInvalid::Includes(r));
    if (!r.IsSuccess())
    {
        return;
    }

    nn::time::CalendarTime calendar;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::ToCalendarTime(&calendar, nullptr, posix));

    NN_LOG("[nnt::http]-------------------------------------------------------------------\n");
    NN_LOG("[nnt::http] Event time stamp : %04d-%02d-%02d %02d:%02d:%02d (UTC+0)\n",
        calendar.year, calendar.month, calendar.day,
        calendar.hour, calendar.minute, calendar.second);
}

#define NNT_HTTP_LOG(...) \
    do { ::nnt::http::TryLogNetworkTime(); NN_LOG(__VA_ARGS__); } while (NN_STATIC_CONDITION(false))

// bool を返す場合の結果の検査
#define NNT_HTTP_RETURN_FALSE_UNLESS(cond) \
    do { if (!(cond)) { ::nnt::http::TryLogNetworkTime(); NN_LOG("%s(%d): " #cond " => false\n", __FILE__, __LINE__); return false;} } while (NN_STATIC_CONDITION(false))

#define NNT_HTTP_RETURN_FALSE_UNLESS_NOT(cond) \
    NNT_HTTP_RETURN_FALSE_UNLESS(!(cond))

#define NNT_HTTP_RETURN_FALSE_UNLESS_EQ(lhs, rhs) \
    NNT_HTTP_RETURN_FALSE_UNLESS((lhs) == (rhs))

#define NNT_HTTP_RETURN_FALSE_UNLESS_NE(lhs, rhs) \
    NNT_HTTP_RETURN_FALSE_UNLESS((lhs) != (rhs))

// bool を返す場合の結果の検査
#define NNT_HTTP_RETURN_FALSE_UNLESS_RESULT_SUCCESS(result) \
    do { nn::Result _r = (result); if (!_r.IsSuccess()) { ::nnt::http::TryLogNetworkTime(); NN_LOG("%s(%d): Result(%d / %d)\n", __FILE__, __LINE__, _r.GetModule(), _r.GetDescription()); return false;} } while (NN_STATIC_CONDITION(false))

#define NNT_HTTP_RETURN_FALSE_UNLESS_RESULT_INCLUDED(exp, result) \
    do { nn::Result _r = (result); if (!exp::Includes(_r)) { ::nnt::http::TryLogNetworkTime(); NN_LOG("%s(%d): Result(%d / %d)\n", __FILE__, __LINE__, _r.GetModule(), _r.GetDescription()); return false;} } while (NN_STATIC_CONDITION(false))

// Result を返す場合の結果の検査
#define NNT_HTTP_RESULT_DO(result) \
    do { nn::Result _r = (result); if (!_r.IsSuccess()) { ::nnt::http::TryLogNetworkTime(); NN_LOG("%s(%d): Result(%d / %d)\n", __FILE__, __LINE__, _r.GetModule(), _r.GetDescription()); return _r;} } while (NN_STATIC_CONDITION(false))

// Result に対する ASSERT_TRUE 的なもの
#define NNT_HTTP_ASSERT_RESULT_SUCCESS(result) \
    do { nn::Result _r = (result); if (!_r.IsSuccess()) { ::nnt::http::TryLogNetworkTime(); NN_LOG("%s(%d): Result(%d / %d)\n", __FILE__, __LINE__, _r.GetModule(), _r.GetDescription()); ASSERT_TRUE(_r.IsSuccess());} } while (NN_STATIC_CONDITION(false))

#define NNT_HTTP_ASSERT_RESULT_INCLUDED(exp, result) \
    do { nn::Result _r = (result); if (!exp::Includes(_r)) { ::nnt::http::TryLogNetworkTime(); NN_LOG("%s(%d): Result(%d / %d)\n", __FILE__, __LINE__, _r.GetModule(), _r.GetDescription()); ASSERT_TRUE(exp::Includes(_r));} } while (NN_STATIC_CONDITION(false))

// Result に対する EXPECT_TRUE 的なもの
#define NNT_HTTP_EXPECT_RESULT_SUCCESS(result) \
    do { nn::Result _r = (result); if (!_r.IsSuccess()) { ::nnt::http::TryLogNetworkTime(); NN_LOG("%s(%d): Result(%d / %d)\n", __FILE__, __LINE__, _r.GetModule(), _r.GetDescription()); EXPECT_TRUE(_r.IsSuccess()); } } while (NN_STATIC_CONDITION(false))

#define NNT_HTTP_EXPECT_RESULT_INCLUDED(exp, result) \
    do { nn::Result _r = (result); if (!exp::Includes(_r)) { ::nnt::http::TryLogNetworkTime(); NN_LOG("%s(%d): Result(%d / %d)\n", __FILE__, __LINE__, _r.GetModule(), _r.GetDescription()); EXPECT_TRUE(exp::Includes(_r)); } } while (NN_STATIC_CONDITION(false))

}} // ~namespace nnt::http
