﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/http/json/http_JsonErrorMap.h>
#include <nn/http/json/http_RapidJsonApi.h>
#include <nn/http/json/http_JsonPath.h>
#include "testHttp_JsonAdaptor.h"
#include "testHttp_JsonCancellation.h"
#include "testHttp_RapidJsonInputStream.h"

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/util/util_ScopeExit.h>

#include <nnt/nntest.h>
#include "testHttp_Util.h"

namespace a = nn::http;
namespace t = nnt::http;

#define NNT_HTTP_ENABLE_RAPID_JSON_DEPTH
#define NNT_HTTP_ENABLE_RAPID_JSON_DEPTH_WITH_OBJECT_PARSE
#define NNT_HTTP_ENABLE_RAPID_JSON_LENGTH
#define NNT_HTTP_ENABLE_RAPID_JSON_LENGTH_WITH_OBJECT_PARSE
#define NNT_HTTP_ENABLE_RAPID_JSON_VALUE
#define NNT_HTTP_ENABLE_RAPID_JSON_OBJECT_END_HANDLE

namespace {

CancellableForTest g_Cancellable;

template<typename AdaptorType>
void TestBasic(
    AdaptorType& adaptorBase,
    const char* inputJson,
    bool print = false) NN_NOEXCEPT
{
    t::MemoryInputStreamForRapidJson stream;
    stream.Set(inputJson, strnlen(inputJson, 2048));

    t::JsonPrintAdaptor<AdaptorType> printer(adaptorBase);
    auto r = print
        ? a::json::ImportJsonByRapidJson<a::json::DefaultJsonErrorMap>(printer, stream, &g_Cancellable)
        : a::json::ImportJsonByRapidJson<a::json::DefaultJsonErrorMap>(adaptorBase, stream, &g_Cancellable);
    NNT_HTTP_EXPECT_RESULT_SUCCESS(r);
}

template<typename AdaptorType, typename ResultType>
void TestBasicError(
    AdaptorType& adaptorBase,
    const char* inputJson,
    bool print = false) NN_NOEXCEPT
{
    t::MemoryInputStreamForRapidJson stream;
    stream.Set(inputJson, strnlen(inputJson, 2048));

    t::JsonPrintAdaptor<AdaptorType> printer(adaptorBase);
    auto r = print
        ? a::json::ImportJsonByRapidJson<a::json::DefaultJsonErrorMap>(printer, stream, &g_Cancellable)
        : a::json::ImportJsonByRapidJson<a::json::DefaultJsonErrorMap>(adaptorBase, stream, &g_Cancellable);
    NNT_HTTP_EXPECT_RESULT_INCLUDED(ResultType, r);
}

} // ~namespace <anonymous>

#if defined(NNT_HTTP_ENABLE_RAPID_JSON_DEPTH)

namespace depth
{
char VariousDepthJsons[][1024] = {
    "{}",
    "{ \"2\": 2 }",
    "{ \"2\": [ 3 ] }",
    "{ \"2\": { \"4\": 4.0 } }",
    "{ \"2\": { \"4\": [ -5 ]} }",
    "{ \"2\": { \"4\": { \"6\": -6.0 } } }",
    "{ \"2\": { \"4\": { \"6\": [ 7.000000 ] } } }",
    "{ \"2\": { \"4\": { \"6\": { \"8\": 8 } } } }",
    "{ \"2\": { \"4\": { \"6\": { \"8\": [ 9e+0 ] } } } }",
    "{ \"2\": { \"4\": { \"6\": { \"8\": { \"10\" : 10 } } } } }",
};
const Expect Expects[] = {
    Expect("$.2",            2),
    Expect("$.2[0]",         3),
    Expect("$.2.4",          3.99,   4.01),
    Expect("$.2.4[0]",      -5),
    Expect("$.2.4.6",       -6.01,  -5.99),
    Expect("$.2.4.6[0]",     6.99,   7.01),
    Expect("$.2.4.6.8",      8),
    Expect("$.2.4.6.8[0]",   8.99,   9.01),
    Expect("$.2.4.6.8.10",  10),
};

template <int Depth, bool FailOnDepthExcess>
class Adaptor
    : public AdaptorForTest<Depth, 128, FailOnDepthExcess, true>
{
public:
    typedef AdaptorForTest<Depth, 128, FailOnDepthExcess, true> Base;
    typedef typename Base::JsonPathType JsonPathType;

protected:
    virtual const Expect* Get(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ABORT_UNLESS(index < GetExpectCount());
        return &Expects[index];
    }
    virtual int GetExpectCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return Depth - 1;
    }
public:
    Adaptor() NN_NOEXCEPT
    {
    }
};

template <int Depth>
void RunTest() NN_NOEXCEPT
{
    Adaptor<Depth, false> adaptorBase;
    for (auto i = 0; i < sizeof(VariousDepthJsons) / sizeof(VariousDepthJsons[0]); ++ i)
    {
        TestBasic(adaptorBase, VariousDepthJsons[i]);
    }
    EXPECT_TRUE(adaptorBase);
}

template <int Depth>
void RunErrorTest(int threshold) NN_NOEXCEPT
{
    Adaptor<Depth, true> adaptorBase;
    for (auto i = 0; i < sizeof(VariousDepthJsons) / sizeof(VariousDepthJsons[0]); ++ i)
    {
        if (i >= threshold)
        {
            TestBasicError<Adaptor<Depth, true>, a::json::ResultJsonParsingTerminated>(adaptorBase, VariousDepthJsons[i]);
        }
        else
        {
            TestBasic(adaptorBase, VariousDepthJsons[i]);
        }
    }
    EXPECT_TRUE(adaptorBase);
}
} // ~namespace depth

TEST(HttpJson, JsonPath_Depth)
{
    depth::RunTest<10>();
    depth::RunTest<9>();
    depth::RunTest<8>();
    depth::RunTest<7>();
    depth::RunTest<6>();
    depth::RunTest<5>();
    depth::RunTest<4>();
    depth::RunTest<3>();
    depth::RunTest<2>();
    depth::RunTest<1>();
}


TEST(HttpJson, JsonPath_DepthExcess)
{
    depth::RunErrorTest<10>(10);
    depth::RunErrorTest<9>(9);
    depth::RunErrorTest<8>(8);
    depth::RunErrorTest<7>(7);
    depth::RunErrorTest<6>(6);
    depth::RunErrorTest<5>(5);
    depth::RunErrorTest<4>(4);
    depth::RunErrorTest<3>(3);
    depth::RunErrorTest<2>(2);
    depth::RunErrorTest<1>(1);
}

#endif // NNT_HTTP_ENABLE_RAPID_JSON_DEPTH

#if defined(NNT_HTTP_ENABLE_RAPID_JSON_DEPTH_WITH_OBJECT_PARSE)

namespace depth_with_object_parse
{
    char JsonDepthIs4[] = {
        "{ \"2\": { \"4\": 4 } }",
    };
    const Expect ExpectsForDepthIs4OrMore[] = {
        Expect("$",              Expect::ObjectBegin),
        Expect("$.2",            Expect::ObjectBegin),
        Expect("$.2.4",          4),
        Expect("$.2",            Expect::ObjectEnd),
        Expect("$",              Expect::ObjectEnd),
    };
    const int ExpectsForDepthIs4OrMoreCount = sizeof(ExpectsForDepthIs4OrMore) / sizeof(ExpectsForDepthIs4OrMore[0]);

    const Expect ExpectsForDepthIs3[] = {
        Expect("$",              Expect::ObjectBegin),
        Expect("$.2",            Expect::ObjectBegin),
        Expect("$.2",            Expect::ObjectEnd),
        Expect("$",              Expect::ObjectEnd),
    };
    const int ExpectsForDepthIs3Count = sizeof(ExpectsForDepthIs3) / sizeof(ExpectsForDepthIs3[0]);

    const Expect ExpectsForDepthIs2OrLess[] = {
        Expect("$",              Expect::ObjectBegin),
        Expect("$",              Expect::ObjectEnd),
    };
    const int ExpectsForDepthIs2OrLessCount = sizeof(ExpectsForDepthIs2OrLess) / sizeof(ExpectsForDepthIs2OrLess[0]);

    const Expect ExpectsForDepthIs3Error[] = {
        Expect("$",              Expect::ObjectBegin),
        Expect("$.2",            Expect::ObjectBegin),
    };
    const int ExpectsForDepthIs3ErrorCount = sizeof(ExpectsForDepthIs3Error) / sizeof(ExpectsForDepthIs3Error[0]);

    const Expect ExpectsForDepthIs2OrLessError[] = {
        Expect("$",              Expect::ObjectBegin),
    };
    const int ExpectsForDepthIs2OrLessErrorCount = sizeof(ExpectsForDepthIs2OrLessError) / sizeof(ExpectsForDepthIs2OrLessError[0]);

    template <int Depth, bool FailOnDepthExcess>
    class Adaptor
        : public AdaptorForTest<Depth, 128, FailOnDepthExcess, false>
    {
    public:
        typedef AdaptorForTest<Depth, 128, FailOnDepthExcess, false> Base;
        typedef typename Base::JsonPathType JsonPathType;
    protected:
        virtual const Expect* Get(int index) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS(index < GetExpectCount());
            return &m_Expects[index];
        }
        virtual int GetExpectCount() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_ExpectCount;
        }
    public:
        Adaptor(const Expect* expects, int expectCount) NN_NOEXCEPT : m_Expects(expects), m_ExpectCount(expectCount)
        {
        }
    private:
        const Expect* m_Expects;
        int m_ExpectCount;
    };

    template <int Depth>
    void RunTest(const Expect expects[], int expectsCount) NN_NOEXCEPT
    {
        Adaptor<Depth, false> adaptorBase(expects, expectsCount);
        TestBasic(adaptorBase, JsonDepthIs4);
        EXPECT_TRUE(adaptorBase);
    }

    template <int Depth>
    void RunErrorTest(const Expect expects[], int expectsCount, bool isError) NN_NOEXCEPT
    {
        Adaptor<Depth, true> adaptorBase(expects, expectsCount);
        if (isError)
        {
            TestBasicError<Adaptor<Depth, true>, a::json::ResultJsonParsingTerminated>(adaptorBase, JsonDepthIs4);
        }
        else
        {
            TestBasic(adaptorBase, JsonDepthIs4);
        }
        EXPECT_TRUE(adaptorBase);
    }
} // ~namespace depth

TEST(HttpJson, JsonPath_DepthWithObjectParse)
{
    using namespace depth_with_object_parse;

    RunTest<5>(ExpectsForDepthIs4OrMore, ExpectsForDepthIs4OrMoreCount);
    RunTest<4>(ExpectsForDepthIs4OrMore, ExpectsForDepthIs4OrMoreCount);
    RunTest<3>(ExpectsForDepthIs3, ExpectsForDepthIs3Count);
    RunTest<2>(ExpectsForDepthIs2OrLess, ExpectsForDepthIs2OrLessCount);
    RunTest<1>(ExpectsForDepthIs2OrLess, ExpectsForDepthIs2OrLessCount);
}

TEST(HttpJson, JsonPath_DepthExcessWithObjectParse)
{
    using namespace depth_with_object_parse;

    RunErrorTest<5>(ExpectsForDepthIs4OrMore, ExpectsForDepthIs4OrMoreCount, false);
    RunErrorTest<4>(ExpectsForDepthIs4OrMore, ExpectsForDepthIs4OrMoreCount, false);
    RunErrorTest<3>(ExpectsForDepthIs3Error, ExpectsForDepthIs3ErrorCount, true);
    RunErrorTest<2>(ExpectsForDepthIs2OrLessError, ExpectsForDepthIs2OrLessErrorCount, true);
    RunErrorTest<1>(ExpectsForDepthIs2OrLessError, ExpectsForDepthIs2OrLessErrorCount, true);
}

#endif // NNT_HTTP_ENABLE_RAPID_JSON_DEPTH_WITH_OBJECT_PARSE

#if defined(NNT_HTTP_ENABLE_RAPID_JSON_LENGTH)

namespace length
{
const char VariousLengthJson[][1024] = {
    "{"
        "\"0123\" :"
            "{"
                "\"4567\" : "
                    "["
                        "{"
                            "\"A\" : 0,"
                            "\"89ab\" : 1,"
                            "\"B\" : 2"
                        "},"
                        "3"
                    "],"
                "\"89ab\" : 4,"
                "\"d\" : 19"
            "},"
        "\"012_\" :"
            "{"
                "\"c\" : [[5, 6], [7], [8], [9], [10], [11], [12], [13], [14], [15], [16], [17], [18]]"
            "},"
        "\"efgh\" : 20,"
        "\"ijkl\" : [21, 22]"
    "}",
    "{"
        "\"m\" : [[23, 24, [25, 26], 27], [[[[28]]]]],"
        "\"n\" : [29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41],"
        "\"o\" : 42"
    "}"
};

struct ExpectedWithLevel
{
    Expect expect;
    int invalidAt;
};
const ExpectedWithLevel Expects[] = {
    {Expect("$.0123.4567[0].A",      0), 15},
    {Expect("$.0123.4567[0].89ab",   1), 18},
    {Expect("$.0123.4567[0].B",      2), 18},
    {Expect("$.0123.4567[1]",        3), 13},
    {Expect("$.0123.89ab",           4), 10},
    {Expect("$.0123.d",              19), 10},
    {Expect("$.012_.c[0][0]",        5), 13},
    {Expect("$.012_.c[0][1]",        6), 13},
    {Expect("$.012_.c[1][0]",        7), 13},
    {Expect("$.012_.c[2][0]",        8), 13},
    {Expect("$.012_.c[3][0]",        9), 13},
    {Expect("$.012_.c[4][0]",        10), 13},
    {Expect("$.012_.c[5][0]",        11), 13},
    {Expect("$.012_.c[6][0]",        12), 13},
    {Expect("$.012_.c[7][0]",        13), 13},
    {Expect("$.012_.c[8][0]",        14), 13},
    {Expect("$.012_.c[9][0]",        15), 13},
    {Expect("$.012_.c[10][0]",       16), 14},
    {Expect("$.012_.c[11][0]",       17), 14},
    {Expect("$.012_.c[12][0]",       18), 14},
    {Expect("$.efgh",                20), 5},
    {Expect("$.ijkl[0]",             21), 8},
    {Expect("$.ijkl[1]",             22), 8},
    {Expect("$.m[0][0]",             23), 8},
    {Expect("$.m[0][1]",             24), 8},
    {Expect("$.m[0][2][0]",          25), 11},
    {Expect("$.m[0][2][1]",          26), 11},
    {Expect("$.m[0][3]",             27), 8},
    {Expect("$.m[1][0][0][0][0]",    28), 17},
    {Expect("$.n[0]",                29), 5},
    {Expect("$.n[1]",                30), 5},
    {Expect("$.n[2]",                31), 5},
    {Expect("$.n[3]",                32), 5},
    {Expect("$.n[4]",                33), 5},
    {Expect("$.n[5]",                34), 5},
    {Expect("$.n[6]",                35), 5},
    {Expect("$.n[7]",                36), 5},
    {Expect("$.n[8]",                37), 5},
    {Expect("$.n[9]",                38), 5},
    {Expect("$.n[10]",               39), 6},
    {Expect("$.n[11]",               40), 6},
    {Expect("$.n[12]",               41), 6},
    {Expect("$.o",                   42), 2},
};

template <int Length>
class Adaptor
    : public AdaptorForTest<32, Length, true, true>
{
public:
    typedef AdaptorForTest<32, Length, true, true> Base;
    typedef typename Base::JsonPathType JsonPathType;
protected:
    virtual const Expect* Get(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        int ct = 0;
        for (auto& e: Expects)
        {
            if (Length > e.invalidAt)
            {
                if (ct == index)
                {
                    return &e.expect;
                }
                ++ ct;
            }
        }
        NN_ABORT("Overflow");
    }
    virtual int GetExpectCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        int ct = 0;
        for (auto& e: Expects)
        {
            if (Length > e.invalidAt)
            {
                ++ ct;
            }
        }
        return ct;
    }
public:
    Adaptor() NN_NOEXCEPT
    {
    }
};

template <int Length>
void RunTest() NN_NOEXCEPT
{
    Adaptor<Length> adaptorBase;
    for (auto i = 0; i < sizeof(VariousLengthJson) / sizeof(VariousLengthJson[0]); ++ i)
    {
        TestBasic(adaptorBase, VariousLengthJson[i]);
    }
    EXPECT_TRUE(adaptorBase);
}
} // ~namespace length

TEST(HttpJson, JsonPath_Length)
{
    length::RunTest<20>();
    length::RunTest<19>();
    length::RunTest<18>();
    length::RunTest<17>();
    length::RunTest<16>();
    length::RunTest<15>();
    length::RunTest<14>();
    length::RunTest<13>();
    length::RunTest<12>();
    length::RunTest<11>();
    length::RunTest<10>();
    length::RunTest<9>();
    length::RunTest<8>();
    length::RunTest<7>();
    length::RunTest<6>();
    length::RunTest<5>();
    length::RunTest<4>();
    length::RunTest<3>();
    length::RunTest<2>();
    length::RunTest<1>();
}

#endif // NNT_HTTP_ENABLE_RAPID_JSON_LENGTH

#if defined(NNT_HTTP_ENABLE_RAPID_JSON_LENGTH_WITH_OBJECT_PARSE)

namespace length_with_object_parse
{
    // JSON パスの長さが 9
    // オブジェクトが 1 と 8 の位置に来る
    const char JsonLengthIs9[] =
        "{"
            "\"34567\": { \"9\": 9}"
        "}";

    const Expect ExpectsForLengthIs9OrMore[] = {
        Expect("$",                 Expect::ObjectBegin),
        Expect("$.34567",           Expect::ObjectBegin),
        Expect("$.34567.9",         9),
        Expect("$.34567",           Expect::ObjectEnd),
        Expect("$",                 Expect::ObjectEnd),
    };
    const int ExpectsForLengthIs9OrMoreCount = sizeof(ExpectsForLengthIs9OrMore) / sizeof(ExpectsForLengthIs9OrMore[0]);

    const Expect ExpectsForLengthIs8Or7[] = {
        Expect("$",                 Expect::ObjectBegin),
        Expect("$.34567",           Expect::ObjectBegin),
        Expect("$.34567",           Expect::ObjectEnd),
        Expect("$",                 Expect::ObjectEnd),
    };
    const int ExpectsForLengthIs8Or7Count = sizeof(ExpectsForLengthIs8Or7) / sizeof(ExpectsForLengthIs8Or7[0]);

    const Expect ExpectsForLengthIs6OrLess[] = {
        Expect("$",                 Expect::ObjectBegin),
        Expect("$",                 Expect::ObjectEnd),
    };
    const int ExpectsForLengthIs6OrLessCount = sizeof(ExpectsForLengthIs6OrLess) / sizeof(ExpectsForLengthIs6OrLess[0]);

    template <int Length>
    class Adaptor
        : public AdaptorForTest<32, Length, true, false>
    {
    public:
        typedef AdaptorForTest<32, Length, true, false> Base;
        typedef typename Base::JsonPathType JsonPathType;
    protected:
        virtual const Expect* Get(int index) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS(index < GetExpectCount());
            return &m_Expects[index];
        }
        virtual int GetExpectCount() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_ExpectCount;
        }
    public:
        Adaptor(const Expect* expects, int expectCount) NN_NOEXCEPT : m_Expects(expects), m_ExpectCount(expectCount)
        {
        }
    private:
        const Expect* m_Expects;
        int m_ExpectCount;
    };

    template <int Length>
    void RunTest(const Expect* expects, int expectCount) NN_NOEXCEPT
    {
        Adaptor<Length> adaptorBase(expects, expectCount);
        TestBasic(adaptorBase, JsonLengthIs9);
        EXPECT_TRUE(adaptorBase);
    }
} // ~namespace length_with_object_parse

TEST(HttpJson, JsonPath_LengthWithObjectParse)
{
    using namespace length_with_object_parse;

    RunTest<10>(ExpectsForLengthIs9OrMore, ExpectsForLengthIs9OrMoreCount);
    RunTest<9>(ExpectsForLengthIs9OrMore, ExpectsForLengthIs9OrMoreCount);
    RunTest<8>(ExpectsForLengthIs8Or7, ExpectsForLengthIs8Or7Count);
    RunTest<7>(ExpectsForLengthIs8Or7, ExpectsForLengthIs8Or7Count);
    RunTest<6>(ExpectsForLengthIs6OrLess, ExpectsForLengthIs6OrLessCount);
    RunTest<5>(ExpectsForLengthIs6OrLess, ExpectsForLengthIs6OrLessCount);
    RunTest<4>(ExpectsForLengthIs6OrLess, ExpectsForLengthIs6OrLessCount);
    RunTest<3>(ExpectsForLengthIs6OrLess, ExpectsForLengthIs6OrLessCount);
    RunTest<2>(ExpectsForLengthIs6OrLess, ExpectsForLengthIs6OrLessCount);
    RunTest<1>(ExpectsForLengthIs6OrLess, ExpectsForLengthIs6OrLessCount);
}

#endif // NNT_HTTP_ENABLE_RAPID_JSON_LENGTH


#if defined(NNT_HTTP_ENABLE_RAPID_JSON_VALUE)
namespace value
{
const char VariousValueJsons[][1024] = {
    "{"
        "\"a0\": null,"
        "\"a1\": true,"
        "\"a2\": {"
            "\"b0\": false,"
            "\"b1\": [0, 1, 2, 3, 4, 5],"
            "\"b2\": {"
                "\"c0\": -65536,"
                "\"c1\": [0.00, 1.01, 2.02, 3.03, 4.04],"
                "\"c2\": {\"d0\": -3.141592},"
                "\"c3\": [-0, -1, -2, -3, -4, -5],"
                "\"c4\": \"asdf\""
            "},"
            "\"b3\": \"\","
            "\"b4\": [\"1\", \"22\", \"333\", \"4444\", \"55555\"]"
        "},"
        "\"a3\": [null],"
        "\"a4\": [{ \"e0\": 0}, { \"e1\": 1}]"
    "}",
};
const Expect Expects[] = {
    Expect("$",              Expect::ObjectBegin),
    Expect("$.a0",           nullptr),
    Expect("$.a1",           true),
    Expect("$.a2",           Expect::ObjectBegin),
    Expect("$.a2.b0",        false),
    Expect("$.a2.b1[0]",     0),
    Expect("$.a2.b1[1]",     1),
    Expect("$.a2.b1[2]",     2),
    Expect("$.a2.b1[3]",     3),
    Expect("$.a2.b1[4]",     4),
    Expect("$.a2.b1[5]",     5),
    Expect("$.a2.b2",        Expect::ObjectBegin),
    Expect("$.a2.b2.c0",     -65536),
    Expect("$.a2.b2.c1[0]",  -0.005, 0.005),
    Expect("$.a2.b2.c1[1]",  1.005, 1.015),
    Expect("$.a2.b2.c1[2]",  2.015, 2.025),
    Expect("$.a2.b2.c1[3]",  3.025, 3.035),
    Expect("$.a2.b2.c1[4]",  4.035, 4.045),
    Expect("$.a2.b2.c2"   ,  Expect::ObjectBegin),
    Expect("$.a2.b2.c2.d0",  -3.141642, -3.141542),
    Expect("$.a2.b2.c2"   ,  Expect::ObjectEnd),
    Expect("$.a2.b2.c3[0]",  0),
    Expect("$.a2.b2.c3[1]",  -1),
    Expect("$.a2.b2.c3[2]",  -2),
    Expect("$.a2.b2.c3[3]",  -3),
    Expect("$.a2.b2.c3[4]",  -4),
    Expect("$.a2.b2.c3[5]",  -5),
    Expect("$.a2.b2.c4",     "asdf"),
    Expect("$.a2.b2",        Expect::ObjectEnd),
    Expect("$.a2.b3",        ""),
    Expect("$.a2.b4[0]",     "1"),
    Expect("$.a2.b4[1]",     "22"),
    Expect("$.a2.b4[2]",     "333"),
    Expect("$.a2.b4[3]",     "4444"),
    Expect("$.a2.b4[4]",     "55555"),
    Expect("$.a2",           Expect::ObjectEnd),
    Expect("$.a3[0]",        nullptr),
    Expect("$.a4[0]",        Expect::ObjectBegin),
    Expect("$.a4[0].e0",     0),
    Expect("$.a4[0]",        Expect::ObjectEnd),
    Expect("$.a4[1]",        Expect::ObjectBegin),
    Expect("$.a4[1].e1",     1),
    Expect("$.a4[1]",        Expect::ObjectEnd),
    Expect("$",              Expect::ObjectEnd),
};

class Adaptor
    : public AdaptorForTest<32, 128, true, false>
{
public:
    typedef AdaptorForTest<32, 128, true, false> Base;
    typedef Base::JsonPathType JsonPathType;
protected:
    virtual const Expect* Get(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ABORT_UNLESS(index < GetExpectCount());
        return &Expects[index];
    }
    virtual int GetExpectCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return sizeof(Expects) / sizeof(Expects[0]);
    }
};

void RunTest() NN_NOEXCEPT
{
    Adaptor adaptorBase;
    for (auto i = 0; i < sizeof(VariousValueJsons) / sizeof(VariousValueJsons[0]); ++ i)
    {
        TestBasic(adaptorBase, VariousValueJsons[i]);
    }
    EXPECT_TRUE(static_cast<bool>(adaptorBase));
}
} // ~namespace value

TEST(HttpJson, JsonPath_Value)
{
    value::RunTest();
}
#endif // NNT_HTTP_ENABLE_RAPID_JSON_VALUE

#if defined(NNT_HTTP_ENABLE_RAPID_JSON_OBJECT_END_HANDLE)
namespace object_end
{
    // Depth:4, Length:5 の Json
    const char Json[] =
        "{"
            "\"a\": {"
                "\"b\": 0"
            "}"
        "}";

    const Expect ValidPattern[] = {
        Expect("$",              Expect::ObjectBegin),
        Expect("$.a",            Expect::ObjectBegin),
        Expect("$.a.b",          0),
        Expect("$.a",            Expect::ObjectEnd),
        Expect("$",              Expect::ObjectEnd),
    };
    const int ValidPatternCount = sizeof(ValidPattern) / sizeof(ValidPattern[0]);

    const Expect InvalidToValidPattern[] = {
        Expect("$",              Expect::ObjectBegin),
        Expect("$.a",            Expect::ObjectBegin),
        Expect("$.a",            Expect::ObjectEnd),
        Expect("$",              Expect::ObjectEnd),
    };
    const int InvalidToValidPatternCount = sizeof(InvalidToValidPattern) / sizeof(InvalidToValidPattern[0]);

    const Expect InvalidPattern[] = {
        Expect("$",              Expect::ObjectBegin),
        Expect("$",              Expect::ObjectEnd),
    };
    const int InvalidPatternCount = sizeof(InvalidPattern) / sizeof(InvalidPattern[0]);

    template<int Depth, int Length>
    class Adaptor
        : public AdaptorForTest<Depth, Length, false, false>
    {
    public:
        typedef AdaptorForTest<Depth, Length, false, false> Base;
        typedef typename Base::JsonPathType JsonPathType;

        Adaptor(const Expect* expects, int expectsCount) NN_NOEXCEPT : m_Expects(expects), m_ExpectsCount(expectsCount)
        {
        }
    protected:
        virtual const Expect* Get(int index) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS(index < GetExpectCount());
            return &m_Expects[index];
        }
        virtual int GetExpectCount() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_ExpectsCount;
        }
    private:
        const Expect* m_Expects;
        int m_ExpectsCount;
    };

    template<int Depth, int Length>
    void RunTest(const Expect* expects, int expectsCount) NN_NOEXCEPT
    {
        Adaptor<Depth, Length> adaptorBase(expects, expectsCount);
        TestBasic(adaptorBase, Json);
        EXPECT_TRUE(static_cast<bool>(adaptorBase));
    }
} // ~namespace object_end

TEST(HttpJson, JsonPath_ObjectEndHandle)
{
    // ObjectEnd 時に Valid なパターン
    object_end::RunTest<4, 32>(object_end::ValidPattern, object_end::ValidPatternCount);
    object_end::RunTest<4, 5>(object_end::ValidPattern, object_end::ValidPatternCount);

    // ObjectEnd 時に Invalid だが、処理中に Valid になるパターン
    object_end::RunTest<3, 32>(object_end::InvalidToValidPattern, object_end::InvalidToValidPatternCount);
    object_end::RunTest<4, 4>(object_end::InvalidToValidPattern, object_end::InvalidToValidPatternCount);
    object_end::RunTest<4, 3>(object_end::InvalidToValidPattern, object_end::InvalidToValidPatternCount);

    // ObjectEnd 時に Invalid で、処理を通しても Invalid なパターン
    object_end::RunTest<2, 32>(object_end::InvalidPattern, object_end::InvalidPatternCount);
    object_end::RunTest<4, 2>(object_end::InvalidPattern, object_end::InvalidPatternCount);
    object_end::RunTest<4, 1>(object_end::InvalidPattern, object_end::InvalidPatternCount);
}
#endif // NNT_HTTP_ENABLE_RAPID_JSON_VALUE

