﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Net.Sockets;
using System.Threading;
using System.IO;

using Nintendo.Htcs;

namespace SleepTestTool
{
    internal class Program
    {
        private static HtcsCommunicator m_HtcsCommunicator = new HtcsCommunicator();

        private struct TestParams
        {
            public string keyString;
            public string testName;
            public string targetIp;
            public string targetSerial;
            public ManualResetEvent finishedEvent;
        };

        private static void TestManager(SleepTestToolParams args)
        {
            var tcpListener = new TcpListener(System.Net.IPAddress.Loopback, 0);
            tcpListener.Start();

            var htcsPortDesc = new HtcsPortDescriptor(HtcsPeerName.Any, "HtcsSleepTestManager");
            var portMapItem = new PortMapItem(htcsPortDesc, ((System.Net.IPEndPoint)tcpListener.LocalEndpoint));
            while (m_HtcsCommunicator.RegisterPort(portMapItem) != 0)
            {
                Console.WriteLine("Registration error.");
                Thread.Sleep(1000);
            }

            WaitCallback waitCallback = new WaitCallback(TestBody);

            var finishedEvents = new ManualResetEvent[args.Count];
            for (int i = 0; i < args.Count; i++)
            {
                var client = tcpListener.AcceptTcpClient();

                TestParams param;

                using (var reader = new StreamReader(client.GetStream(), Encoding.ASCII, true))
                {
                    var data = reader.ReadToEnd().Trim('\0').Split(',');
                    param.keyString = data[0];
                    param.testName = data[1];
                    param.targetIp = args.Ip;
                    param.targetSerial = args.Serial;

                    finishedEvents[i] = new ManualResetEvent(false);
                    param.finishedEvent = finishedEvents[i];
                }

                client.Close();

                ThreadPool.QueueUserWorkItem(waitCallback, param);
            }

            m_HtcsCommunicator.UnregisterPort(htcsPortDesc);

            tcpListener.Stop();

            Console.WriteLine("Waiting all tests end.");
            WaitHandle.WaitAll(finishedEvents);
            Console.WriteLine("All tests finished.");
        }

        private static void TestBody(object obj)
        {
            TestParams testParams = (TestParams)obj;
            Console.WriteLine("Start " + testParams.testName);
            var test = new SleepTest(testParams.keyString, m_HtcsCommunicator, testParams.targetIp, testParams.targetSerial);
            switch (testParams.testName)
            {
                case "Socket":
                    {
                        test.SleepAfterCreateSocket(testParams.testName);
                    }
                    break;
                case "Connected":
                    {
                        test.SleepAfterConnected(testParams.testName);
                    }
                    break;
                case "BeforeTrans1":
                    {
                        test.SleepBeforeTransmission(testParams.testName);
                    }
                    break;
                case "BeforeTrans2":
                    {
                        test.SleepBeforeTransmission_HostContinue(testParams.testName);
                    }
                    break;
                case "Transmission":
                case "TransmissionGrad":
                    {
                        test.SleepWhileTransmission(testParams.testName);
                    }
                    break;
                case "Listen1":
                case "Accept1":
                    {
                        test.SleepAfterPortOpened(testParams.testName);
                    }
                    break;
                case "Listen2":
                case "Accept2":
                    {
                        test.SleepAfterPortOpened_HostContinue(testParams.testName);
                    }
                    break;
                case "ShutdownRd":
                    {
                        test.ShutdownRd(testParams.testName);
                    }
                    break;
                case "ShutdownWr":
                    {
                        test.ShutdownWr(testParams.testName);
                    }
                    break;
                case "ShutdownRw":
                    {
                        test.ShutdownRw(testParams.testName);
                    }
                    break;
                case "ConnectedFail":
                case "HostServerClose":
                    {
                        test.SleepAndCloseAfterTargetClientConnected(testParams.testName);
                    }
                    break;
                case "HostClientClose":
                    {
                        test.SleepAndCloseAfterTargetServerConnected(testParams.testName);
                    }
                    break;
                case "RecvHostClose":
                    {
                        test.SleepWhileRecv_HostClose(testParams.testName);
                    }
                    break;
                case "HostRecconect":
                    {
                        test.SleepAfterConnected_HostReconnect(testParams.testName);
                    }
                    break;
                case "NonBlocking":
                    {
                        test.NonBlocking(testParams.testName);
                    }
                    break;
                default:
                    Console.WriteLine("No such test case: {0}", testParams.testName);
                    break;
            }
            Console.WriteLine("End " + testParams.testName);

            testParams.finishedEvent.Set();
        }

        private static void PrintHtcsInfo(object sender, HtcsInfoUpdatedEventArgs e)
        {
            Console.WriteLine("-  TargetList -");
            foreach (Target item in e.TargetList)
            {
                Console.WriteLine(item.ToString());
            }
            Console.WriteLine("---------------");

            Console.WriteLine("--- PortMap ---");
            foreach (PortMapItem item in e.PortMap)
            {
                Console.WriteLine(item.ToString());
            }
            Console.WriteLine("---------------");
        }

        private static void Main(string[] args)
        {
            SleepTestToolArgs opt = new SleepTestToolArgs();
            try
            {
                if (!opt.ParseArgs(args))
                {
                    return;
                }
            }
            catch
            {
                Environment.ExitCode = 1;
                return;
            }

            // HtcsInfo が更新されたときに呼ばれる関数を登録
            //   HtcsInfo リスト表示関数の登録
            m_HtcsCommunicator.HtcsInfoUpdated += PrintHtcsInfo;

            // TargetManager への接続が確立されるまでループ
            while (m_HtcsCommunicator.Start() == false)
            {
                Console.WriteLine("Failed connecting to TargetManager.");
                Thread.Sleep(1000);
            }

            TestManager(opt.Params);

            m_HtcsCommunicator.Stop();
        }
    }
}
