﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Net;
using System.IO;
using System.Diagnostics;
using System.Reflection;
using System.Drawing;

namespace SleepTestTool
{
    internal class ControlTargetUtil
    {
        private const string NintendoSdkRootMarkFileName = "NintendoSdkRootMark";

        public static string FindSigloRoot()
        {
            string startPath = Assembly.GetExecutingAssembly().Location;
            string currentDirectory = startPath;
            string rootDirectory = Path.GetPathRoot(currentDirectory);

            while (rootDirectory != currentDirectory)
            {
                string sigloRootMarkFilePath = Path.Combine(currentDirectory, NintendoSdkRootMarkFileName);
                if (File.Exists(sigloRootMarkFilePath))
                {
                    return currentDirectory;
                }

                currentDirectory = Path.GetFullPath(System.IO.Path.Combine(currentDirectory, ".."));
            }

            throw new InvalidDataException(string.Format("{0} を見つけられませんでした。\nPath=\n{1}", NintendoSdkRootMarkFileName, startPath));
        }

        public static void DoControlTarget(string arg)
        {
            var process = new Process();
            process.StartInfo.FileName = Path.Combine(FindSigloRoot(), @"Tools\CommandLineTools\ControlTargetPrivate.exe");
            process.StartInfo.Arguments = arg;
            process.StartInfo.RedirectStandardOutput = true;
            process.StartInfo.UseShellExecute = false;

            process.Start();
            process.WaitForExit();
        }

        public static void DoHidShell(string arg)
        {
            var process = new Process();
            process.StartInfo.FileName = Path.Combine(FindSigloRoot(), @"Tools\CommandLineTools\Hid\HidShell.exe");
            process.StartInfo.Arguments = arg;
            process.StartInfo.RedirectStandardOutput = true;
            process.StartInfo.UseShellExecute = false;

            process.Start();
            process.WaitForExit();
        }

        public static bool IsBlackScreen(string targetAddress)
        {
            string targetUrl = "http://" + targetAddress;
            // Create a request using a URL that can receive a post.
            HttpWebRequest request;
            request = (HttpWebRequest)WebRequest.Create(targetUrl + "/cgi-bin/lcd/portrait/bmp");

            // Get the response.
            HttpWebResponse response;
            response = (HttpWebResponse)request.GetResponse();
            response.Close();

            request = (HttpWebRequest)WebRequest.Create(targetUrl + "/capture/image.bmp");
            response = (HttpWebResponse)request.GetResponse();
            Stream dataStream = response.GetResponseStream();

            Bitmap bitmap = new Bitmap(dataStream);

            dataStream.Close();
            response.Close();

            for (int x = 0; x < bitmap.Width; ++x)
            {
                for(int y = 0; y < bitmap.Height; ++y)
                {
                    if (!(bitmap.GetPixel(x, y).R == Color.Black.R
                        && bitmap.GetPixel(x, y).G == Color.Black.G
                        && bitmap.GetPixel(x, y).B == Color.Black.B))
                    {
                        return false;
                    }
                }
            }

            return true;
        }

        public static void Sleep(string targetAddress)
        {
            if(String.IsNullOrEmpty(targetAddress))
            {
                Console.WriteLine("Skip trigger sleep");
                return;
            }
            Console.WriteLine("Trigger sleep");
            DoControlTarget("press-power-button-for-sc7 --ip-addr " + targetAddress);
        }

        public static void Wake(string targetAddress, string targetSerial)
        {
            if (String.IsNullOrEmpty(targetAddress))
            {
                Console.WriteLine("Skip trigger wake");
                return;
            }
            Console.WriteLine("Trigger wake");
            DoControlTarget("press-power-button-for-sc7 --ip-addr " + targetAddress);

            System.Threading.Thread.Sleep(20 * 1000);
            // DevMenu が表示されていることを簡易的に何か表示されているかで確認
            if (IsBlackScreen(targetAddress))
            {
                Console.WriteLine("Target did not wake up. Retry press-power-button.");
                DoControlTarget("press-power-button-for-sc7 --ip-addr " + targetAddress);
                System.Threading.Thread.Sleep(20 * 1000);
                if (IsBlackScreen(targetAddress))
                {
                    throw new Exception("Target did not wake up");
                }
            }

            // DevMenu 上の Resume ボタンが表示される位置を何度か押してみる
            for (int i = 0; i < 5; i++)
            {
                DoHidShell("send-event TouchScreen 0 TouchBegan 0 1200 700 -t " + targetSerial);
                DoHidShell("send-event TouchScreen 0 TouchEnded 0 -t " + targetSerial);
                System.Threading.Thread.Sleep(1000);
            }
        }

        public static void SleepAndWake(string targetAddress, string targetSerial)
        {
            Sleep(targetAddress);
            System.Threading.Thread.Sleep(15 * 1000);
            Wake(targetAddress, targetSerial);
        }
    }
}
