﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Net.Sockets;
using System.Threading;
using System.IO;

using Nintendo.Htcs;

namespace BulkClient
{
    internal class Program
    {
        private static HtcsCommunicator m_HtcsCommunicator = new HtcsCommunicator();
        private static TargetList m_TargetList;
        private static PortMap m_PortMap;
        private static AutoResetEvent m_FindServerEvent = new AutoResetEvent(false);
        private const string TargetServerPortName = "ServerInTarget";
        private static bool m_EndFlag = false;

        private static void ClientSample()
        {
            PortMapItem portMapItem;
            while (true)
            {
                // HTCS ポート名が TargetServerPortName で表されるサーバが開くまで待機
                m_FindServerEvent.WaitOne();

                try
                {
                    // HTCS ポート名が TargetServerPortName のもののうち、最初に見つかったものを取得
                    //   複数のターゲットが同名の HTCS ポートを開いている場合は、
                    //   PortMapItem.HtcsPortDescriptor.HtcsPortName で取得できるターゲット名で識別してください
                    portMapItem = m_PortMap.First(x => x.HtcsPortDescriptor.HtcsPortName == TargetServerPortName);
                    break;
                }
                catch (InvalidOperationException)
                {
                    // タイミングによって古いポートを見つけてしまい、閉じられて失敗することがあるようなので、リトライする
                }
            }

            // TCP ソケットを作成、EndPoint 情報を基に Connect
            TcpClient client = new TcpClient();
            try
            {
                client.Connect(portMapItem.EndPoint);
            }
            catch (SocketException e)
            {
                Console.WriteLine("Connect failed : error code" + e.ErrorCode);
                return;
            }
            Console.WriteLine("Connected to server on target.");

            EchoBack(client);

            // ソケットをクローズ
            client.Close();
            Console.WriteLine("Closed.");
        }

        // 送られてきたデータをそのまま相手に送り返す
        private static void EchoBack(TcpClient client)
        {
            using (var reader = new BinaryReader(client.GetStream(), Encoding.ASCII, true))
            using (var writer = new BinaryWriter(client.GetStream(), Encoding.ASCII, true))
            {
                while (true)
                {
                    int recvByteCount;
                    int sendByteCount;

                    // 受信サイズ (4 byte) と送信サイズ (4 byte) をそれぞれ取得
                    try
                    {
                        recvByteCount = reader.ReadInt32();
                        sendByteCount = reader.ReadInt32();
                    }
                    catch (EndOfStreamException)
                    {
                        break;
                    }

                    // 送信サイズ・受信サイズが 0 byte ならテスト終了
                    if (recvByteCount == 0 && sendByteCount == 0)
                    {
                        m_EndFlag = true;
                        break;
                    }

                    // 送信データの用意
                    byte[] data = new byte[sendByteCount];

                    // Send Ack (1 byte)
                    byte ack = 0;
                    writer.Write(ack);

                    // Bulk Recv
                    reader.ReadBytes(recvByteCount);

                    // Bulk Send
                    writer.Write(data);
                }
            }
        }

        private static void CheckHtcsInfo(object sender, HtcsInfoUpdatedEventArgs e)
        {
            // HtcsInfo のキャッシュを更新
            m_TargetList = e.TargetList;
            m_PortMap = e.PortMap;

            // ポート名が TargetServerPortName のポートが開かれていたら通知を投げる
            if (e.PortMap.Any(x => x.HtcsPortDescriptor.HtcsPortName == TargetServerPortName))
            {
                m_FindServerEvent.Set();
            }
        }

        private static void PrintHtcsInfo(object sender, HtcsInfoUpdatedEventArgs e)
        {
            Console.WriteLine("-  TargetList -");
            foreach (Target item in e.TargetList)
            {
                Console.WriteLine(item.ToString());
            }
            Console.WriteLine("---------------");
            Console.WriteLine("--- PortMap ---");
            foreach (PortMapItem item in e.PortMap)
            {
                Console.WriteLine(item.ToString());
            }
            Console.WriteLine("---------------");
        }

        private static void Main(string[] args)
        {
            // HtcsInfo が更新されたときに呼ばれる関数を登録
            //   HtcsInfo リスト表示関数の登録
            m_HtcsCommunicator.HtcsInfoUpdated += PrintHtcsInfo;
            //   HtcsInfo 内のポートマッピングに、接続予定のポートが追加されたかをチェックする関数の登録
            m_HtcsCommunicator.HtcsInfoUpdated += CheckHtcsInfo;

            // TargetManager への接続が確立されるまでループ
            while (m_HtcsCommunicator.Start() == false)
            {
                Console.WriteLine("Failed connecting to TargetManager.");
                Console.WriteLine("Press Enter to retry.");
                Console.ReadLine();
            }

            while (!m_EndFlag)
            {
                ClientSample();
            }

            m_HtcsCommunicator.Stop();

            Console.WriteLine("Sample end");
            Console.ReadLine();
        }
    }
}
