﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <nn/nn_Log.h>
#include <nnt/nntest.h>
#include <nnt/htcsUtil/testHtcs_util.h>
#include <nnt/htcsUtil/testHtcs_utilMemoryLeakDetector.h>

/*
*   切断関係のテストです。
*   事前に Target Manager を立ち上げておく必要があります。
*   テスト時に Tests/Htcs/Sources/Tools/StressTestTool を実行する必要があります。
*/

namespace
{
    const size_t ThreadStackSize = 16 * 1024;
    NN_ALIGNAS(4096) uint8_t g_ThreadStacks[ThreadStackSize];

    static const size_t KeyStringSize = 11;
    static char g_KeyString[KeyStringSize];

    void* Allocate(size_t size)
    {
        return malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        free(p);
    }

    class DisconnectTest : public ::testing::Test
    {
    protected:
        static void SetUpTestCase() {
            for (int i = 0; i < (KeyStringSize - 1); i++)
            {
                int seed = nn::os::GetSystemTick().GetInt64Value() % 26;
                g_KeyString[i] = static_cast<char>('a' + seed);
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            }
            g_KeyString[KeyStringSize - 1] = '\0';
        }

        virtual void SetUp()
        {
            nn::htcs::Initialize(Allocate, Deallocate);
        }
        virtual void TearDown()
        {
            nn::htcs::Finalize();
        }
    };

    void SuspendSend(void* arg)
    {
        int socket = *static_cast<int*>(arg);
        const size_t dataSize = 512 * 1024 * 1024;
        uint8_t* sendBuffer = reinterpret_cast<uint8_t*>(std::malloc(dataSize));
        ASSERT_NE(nullptr, sendBuffer);

        nnt::htcs::util::FillBuffer(sendBuffer, dataSize);

        auto sentByte = nn::htcs::Send(socket, sendBuffer, dataSize, 0);
        while (sentByte >= 0)
        {
            sentByte = nn::htcs::Send(socket, sendBuffer, dataSize, 0);
        }
        // TODO: 正しいエラー値のチェック
        NN_LOG("LastError=%d\n", nn::htcs::GetLastError());

        std::free(sendBuffer);
    }

    void SuspendReceive(void* arg)
    {
        int socket = *static_cast<int*>(arg);
        const size_t dataSize = 512 * 1024 * 1024;
        uint8_t* recvBuffer = reinterpret_cast<uint8_t*>(std::malloc(dataSize));
        ASSERT_NE(nullptr, recvBuffer);
        std::memset(recvBuffer, 0, dataSize);

        auto readByte = nn::htcs::Recv(socket, recvBuffer, dataSize, nn::htcs::HTCS_MSG_WAITALL);
        while (readByte >= 0)
        {
            readByte = nn::htcs::Recv(socket, recvBuffer, dataSize, nn::htcs::HTCS_MSG_WAITALL);
        }
        // TODO: 正しいエラー値のチェック
        NN_LOG("LastError=%d\n", nn::htcs::GetLastError());

        std::free(recvBuffer);
    }
}

TEST_F(DisconnectTest, Start)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    int socket;
    while (NN_STATIC_CONDITION(true))
    {
        socket = nn::htcs::Socket();
        if (socket >= 0)
        {
            break;
        }
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1000) );
    }

    nnt::htcs::util::ConnectToTestServer(socket);

    const char* testName = "DisconnectTest";

    size_t bufferSize = std::strlen(g_KeyString) + 1 + std::strlen(testName) + 1;
    char* buffer = reinterpret_cast<char*>(std::malloc(bufferSize));
    ASSERT_NE(nullptr, buffer);
    buffer[0] = '\0';

    std::strcat(buffer, g_KeyString);
    std::strcat(buffer, ",");
    std::strcat(buffer, testName);

    ASSERT_EQ(nn::htcs::Send(socket, buffer, bufferSize, 0), bufferSize);

    std::free(buffer);
}

TEST_F(DisconnectTest, CloseSend)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    nn::os::ThreadType thread;
    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "TargetToHost_");
    std::strcat(address.portName.name, g_KeyString);

    int listenSocket = nn::htcs::Socket();
    ASSERT_LE(0, listenSocket);

    EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &address));
    EXPECT_EQ(0, nn::htcs::Listen(listenSocket, 1));

    NN_LOG("Waiting for connection from host.\n");
    int socket = nn::htcs::Accept(listenSocket, nullptr);
    EXPECT_LE(0, socket);

    ASSERT_EQ(true, nn::os::CreateThread(&thread, SuspendSend, &socket, g_ThreadStacks, ThreadStackSize, nn::os::DefaultThreadPriority).IsSuccess());
    nn::os::StartThread(&thread);

    // 通信が始まるのを待機
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // 通信中に Close で切断
    EXPECT_EQ(0, nn::htcs::Close(socket));

    nn::os::WaitThread(&thread);
    nn::os::DestroyThread(&thread);

    EXPECT_EQ(0, nn::htcs::Close(listenSocket));
}

TEST_F(DisconnectTest, CloseRecv)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    nn::os::ThreadType thread;
    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "HostToTarget_");
    std::strcat(address.portName.name, g_KeyString);

    int listenSocket = nn::htcs::Socket();
    ASSERT_LE(0, listenSocket);

    EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &address));
    EXPECT_EQ(0, nn::htcs::Listen(listenSocket, 1));

    NN_LOG("Waiting for connection from host.\n");
    int socket = nn::htcs::Accept(listenSocket, nullptr);
    EXPECT_LE(0, socket);

    ASSERT_EQ(true, nn::os::CreateThread(&thread, SuspendReceive, &socket, g_ThreadStacks, ThreadStackSize, nn::os::DefaultThreadPriority).IsSuccess());
    nn::os::StartThread(&thread);

    // 通信が始まるのを待機
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // 通信中に Close で切断
    EXPECT_EQ(0, nn::htcs::Close(socket));

    nn::os::WaitThread(&thread);
    nn::os::DestroyThread(&thread);

    EXPECT_EQ(0, nn::htcs::Close(listenSocket));
}

TEST_F(DisconnectTest, ShutdownReadServer)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "ShutdownRead_");
    std::strcat(address.portName.name, g_KeyString);

    int listenSocket = nn::htcs::Socket();
    ASSERT_LE(0, listenSocket);

    EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &address));
    EXPECT_EQ(0, nn::htcs::Listen(listenSocket, 1));

    NN_LOG("Waiting for connection from host.\n");
    int socket = nn::htcs::Accept(listenSocket, nullptr);
    EXPECT_LE(0, socket);

    EXPECT_EQ(0, nn::htcs::Shutdown(socket, nn::htcs::HTCS_SHUT_RD));

    const size_t dataSize = 32 * 1024 * 1024;
    uint8_t* sendBuffer = reinterpret_cast<uint8_t*>(std::malloc(dataSize));
    uint8_t* readBuffer = reinterpret_cast<uint8_t*>(std::malloc(dataSize));
    ASSERT_NE(nullptr, sendBuffer);
    ASSERT_NE(nullptr, readBuffer);

    nnt::htcs::util::FillBuffer(sendBuffer, dataSize);

    EXPECT_EQ(dataSize, nn::htcs::Send(socket, sendBuffer, dataSize, 0));

    EXPECT_EQ(-1, nn::htcs::Recv(socket, readBuffer, dataSize, 0));
    NN_LOG("LastError = %d\n", nn::htcs::GetLastError());

    std::free(sendBuffer);
    std::free(readBuffer);

    // 対向のエラー確認前に Close しないように一定時間待つ
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

    EXPECT_EQ(0, nn::htcs::Close(socket));
    EXPECT_EQ(0, nn::htcs::Close(listenSocket));
}

TEST_F(DisconnectTest, ShutdownWriteServer)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "ShutdownWrite_");
    std::strcat(address.portName.name, g_KeyString);

    int listenSocket = nn::htcs::Socket();
    ASSERT_LE(0, listenSocket);
    EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &address));
    EXPECT_EQ(0, nn::htcs::Listen(listenSocket, 1));

    NN_LOG("Waiting for connection from host.\n");
    int socket = nn::htcs::Accept(listenSocket, nullptr);
    EXPECT_LE(0, socket);

    EXPECT_EQ(0, nn::htcs::Shutdown(socket, nn::htcs::HTCS_SHUT_WR));

    const size_t dataSize = 32 * 1024 * 1024;
    uint8_t* buffer = reinterpret_cast<uint8_t*>(std::malloc(dataSize));
    ASSERT_NE(nullptr, buffer);

    EXPECT_EQ(dataSize, nn::htcs::Recv(socket, buffer, dataSize, nn::htcs::HTCS_MSG_WAITALL));

    EXPECT_EQ(-1, nn::htcs::Send(socket, buffer, dataSize, 0));
    NN_LOG("LastError = %d\n", nn::htcs::GetLastError());

    std::free(buffer);

    // 対向のエラー確認前に Close しないように一定時間待つ
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

    EXPECT_EQ(0, nn::htcs::Close(socket));
    EXPECT_EQ(0, nn::htcs::Close(listenSocket));
}

TEST_F(DisconnectTest, ShutdownReadWriteServer)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "ShutdownReadWrite_");
    std::strcat(address.portName.name, g_KeyString);

    int listenSocket = nn::htcs::Socket();
    ASSERT_LE(0, listenSocket);
    EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &address));
    EXPECT_EQ(0, nn::htcs::Listen(listenSocket, 1));

    NN_LOG("Waiting for connection from host.\n");
    int socket = nn::htcs::Accept(listenSocket, nullptr);
    EXPECT_LE(0, socket);

    EXPECT_EQ(0, nn::htcs::Shutdown(socket, nn::htcs::HTCS_SHUT_RDWR));

    const size_t dataSize = 32 * 1024 * 1024;
    uint8_t* buffer = reinterpret_cast<uint8_t*>(std::malloc(dataSize));
    ASSERT_NE(nullptr, buffer);

    EXPECT_EQ(-1, nn::htcs::Recv(socket, buffer, dataSize, 0));
    NN_LOG("LastError = %d\n", nn::htcs::GetLastError());

    EXPECT_EQ(-1, nn::htcs::Send(socket, buffer, dataSize, 0));
    NN_LOG("LastError = %d\n", nn::htcs::GetLastError());

    std::free(buffer);

    // 対向のエラー確認前に Close しないように一定時間待つ
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

    EXPECT_EQ(0, nn::htcs::Close(socket));
    EXPECT_EQ(0, nn::htcs::Close(listenSocket));
}

#if 0
TEST_F(DisconnectTest, ShutdownReadClient)
{
}

TEST_F(DisconnectTest, ShutdownWriteClient)
{
}

TEST_F(DisconnectTest, ShutdownReadWriteClient)
{
}
#endif
