﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <nn/nn_Log.h>
#include <nnt/nntest.h>
#include <nnt/htcsUtil/testHtcs_util.h>
#include <nnt/htcsUtil/testHtcs_utilMemoryLeakDetector.h>

/*
*   接続周りのテストです。
*   事前に Target Manager を立ち上げておく必要があります。
*   テスト時に Tests/Htcs/Sources/Tools/StressTestTool を実行する必要があります。
*/

namespace
{
    const size_t ThreadStackSize = 16 * 1024;
    NN_ALIGNAS(4096) uint8_t g_ThreadStacks[ThreadStackSize];

    static const size_t KeyStringSize = 11;
    static char g_KeyString[KeyStringSize];

    void* Allocate(size_t size)
    {
        return malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        free(p);
    }

    class ConnectTest : public ::testing::Test
    {
    protected:
        static void SetUpTestCase() {
            for (int i = 0; i < (KeyStringSize - 1); i++)
            {
                int seed = nn::os::GetSystemTick().GetInt64Value() % 26;
                g_KeyString[i] = static_cast<char>('a' + seed);
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            }
            g_KeyString[KeyStringSize - 1] = '\0';
        }

        virtual void SetUp()
        {
            nn::htcs::Initialize(Allocate, Deallocate);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        }
        virtual void TearDown()
        {
            nn::htcs::Finalize();
        }
    };

    void ProcessEchoMode(int socket, nnt::htcs::util::ThreadArgs* pArgs)
    {
        ASSERT_GE(pArgs->size, (size_t)0);
        ASSERT_GE(pArgs->count, 0);

        NN_LOG("Start %s (size = %lld, count = %lld)\n", __FUNCTION__, pArgs->size, pArgs->count);

        // 準備
        uint8_t* sendBuffer = reinterpret_cast<uint8_t*>(std::malloc(pArgs->size));
        uint8_t* recvBuffer = reinterpret_cast<uint8_t*>(std::malloc(pArgs->size));

        ASSERT_NE(nullptr, sendBuffer);
        ASSERT_NE(nullptr, recvBuffer);

        nnt::htcs::util::FillBuffer(sendBuffer, pArgs->size);
        std::memset(recvBuffer, 0, pArgs->size);

        // 待機
        if (pArgs->pBarrier != nullptr)
        {
            pArgs->pBarrier->Await();
        }

        for (int i = 0; i < pArgs->count; i++)
        {
            nnt::htcs::util::Echo(socket, pArgs->size, sendBuffer, recvBuffer);
        }

        std::free(sendBuffer);
        std::free(recvBuffer);

        NN_LOG("End \n", __FUNCTION__);
    }
}

TEST_F(ConnectTest, Start)
{
    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    nnt::htcs::util::ConnectToTestServer(socket);

    const char* testName = "ConnectTest";

    size_t bufferSize = std::strlen(g_KeyString) + 1 + std::strlen(testName) + 1;
    char* buffer = reinterpret_cast<char*>(std::malloc(bufferSize));
    ASSERT_NE(nullptr, buffer);
    buffer[0] = '\0';

    std::strcat(buffer, g_KeyString);
    std::strcat(buffer, ",");
    std::strcat(buffer, testName);

    ASSERT_EQ(nn::htcs::Send(socket, buffer, bufferSize, 0), bufferSize);
}

TEST_F(ConnectTest, SockAddrNamesMaxServer)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    nn::os::ThreadType thread;
    nnt::htcs::util::ThreadArgs arg;

    std::strcpy(arg.address.portName.name, "LongPortName12345678_");
    std::strcat(arg.address.portName.name, g_KeyString);
    std::strcpy(arg.address.peerName.name, "LongPeerName1234567890123456789");
    ASSERT_EQ(31, std::strlen(arg.address.portName.name));
    ASSERT_EQ(31, std::strlen(arg.address.peerName.name));

    arg.mode = nnt::htcs::util::ThreadMode_Echo;
    arg.pBarrier = nullptr;
    arg.size = 1024;
    arg.count = 10;
    arg.verbose = false;

    nn::os::CreateThread(&thread, nnt::htcs::util::ServerThread, &arg, g_ThreadStacks, ThreadStackSize, nn::os::DefaultThreadPriority);

    nn::os::StartThread(&thread);

    nn::os::WaitThread(&thread);

    nn::os::DestroyThread(&thread);
}

TEST_F(ConnectTest, SockAddrNamesMaxClient)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    nn::os::ThreadType thread;
    nnt::htcs::util::ThreadArgs arg;

    std::strcpy(arg.address.portName.name, "LongPortName12345678_");
    std::strcat(arg.address.portName.name, g_KeyString);
    std::strcpy(arg.address.peerName.name, "LongPeerName1234567890123456789");
    ASSERT_EQ(31, std::strlen(arg.address.portName.name));
    ASSERT_EQ(31, std::strlen(arg.address.peerName.name));

    arg.mode = nnt::htcs::util::ThreadMode_Echo;
    arg.pBarrier = nullptr;
    arg.size = 1024;
    arg.count = 10;
    arg.verbose = false;

    nn::os::CreateThread(&thread, nnt::htcs::util::ClientThread, &arg, g_ThreadStacks, ThreadStackSize, nn::os::DefaultThreadPriority);

    nn::os::StartThread(&thread);

    nn::os::WaitThread(&thread);

    nn::os::DestroyThread(&thread);
}

TEST_F(ConnectTest, SockAddrNamesOverServer)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    nn::os::ThreadType thread;
    nnt::htcs::util::ThreadArgs arg;

    std::strcpy(arg.address.portName.name, "LongPortName12345678_");
    std::strcat(arg.address.portName.name, g_KeyString);
    std::strcpy(arg.address.peerName.name, "LongPeerName1234567890123456789");
    ASSERT_EQ(31, std::strlen(arg.address.portName.name));
    ASSERT_EQ(31, std::strlen(arg.address.peerName.name));
    arg.address.portName.name[31] = 'X';
    arg.address.peerName.name[31] = 'X';

    arg.mode = nnt::htcs::util::ThreadMode_Echo;
    arg.pBarrier = nullptr;
    arg.size = 1024;
    arg.count = 10;
    arg.verbose = false;

    nn::os::CreateThread(&thread, nnt::htcs::util::ServerThread, &arg, g_ThreadStacks, ThreadStackSize, nn::os::DefaultThreadPriority);

    nn::os::StartThread(&thread);

    nn::os::WaitThread(&thread);

    nn::os::DestroyThread(&thread);
}

TEST_F(ConnectTest, SockAddrNamesOverClient)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    nnt::htcs::util::ThreadArgs arg;

    std::strcpy(arg.address.portName.name, "LongPortName12345678_");
    std::strcat(arg.address.portName.name, g_KeyString);
    std::strcpy(arg.address.peerName.name, "LongPeerName1234567890123456789");
    ASSERT_EQ(31, std::strlen(arg.address.portName.name));
    ASSERT_EQ(31, std::strlen(arg.address.peerName.name));
    arg.address.portName.name[31] = 'X';
    arg.address.peerName.name[31] = 'X';

    arg.mode = nnt::htcs::util::ThreadMode_Echo;
    arg.pBarrier = nullptr;
    arg.size = 1024;
    arg.count = 10;
    arg.verbose = false;

    // TORIAEZU: NULL 終端されていない文字列が出力されてしまうので nnt::htcs::util を使わずに実装

    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    NN_LOG("Trying to connect...\n");
    while (NN_STATIC_CONDITION(true))
    {
        if (nn::htcs::Connect(socket, &arg.address) == 0)
        {
            break;
        }
    }
    NN_LOG("Connected.\n");

    ProcessEchoMode(socket, &arg);

    EXPECT_EQ(0, nn::htcs::Close(socket));
}

TEST_F(ConnectTest, Backlog)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    const int backlogCount = 10; // 確認したい backlog の数

    nnt::htcs::util::ThreadArgs arg;

    arg.address.family = nn::htcs::HTCS_AF_HTCS;
    arg.address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(arg.address.portName.name, "Echo_");
    std::strcat(arg.address.portName.name, g_KeyString);

    arg.mode = nnt::htcs::util::ThreadMode_Echo;
    arg.pBarrier = nullptr;
    arg.size = 1024;
    arg.count = 1;
    arg.verbose = false;

    int listenSocket = nn::htcs::Socket();
    ASSERT_LE(0, listenSocket);

    EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &arg.address));
    EXPECT_EQ(0, nn::htcs::Listen(listenSocket, backlogCount));

    for( int i = 0; i < backlogCount + 1; i++ )
    {
        NN_LOG("Waiting for connection from host.\n");
        int socket = nn::htcs::Accept(listenSocket, nullptr);
        EXPECT_LE(0, socket);

        ProcessEchoMode(socket, &arg);

        EXPECT_EQ(0, nn::htcs::Close(socket));
    }

    NN_LOG("Test after backlog count is 0\n");
    {
        NN_LOG("Waiting for connection from host.\n");
        int socket = nn::htcs::Accept(listenSocket, nullptr);
        EXPECT_LE(0, socket);

        ProcessEchoMode(socket, &arg);

        EXPECT_EQ(0, nn::htcs::Close(socket));
    }

    EXPECT_EQ(0, nn::htcs::Close(listenSocket));
}
