﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <nn/nn_Log.h>
#include <nnt/nntest.h>
#include <nnt/htcsUtil/testHtcs_util.h>
#include <nnt/htcsUtil/testHtcs_utilMemoryLeakDetector.h>

/*
*   サイズの大きいデータを扱う Htcs のストレステスト
*   NX32 では必要なメモリが確保できないため、NX64 でのみ動作します。
*   事前に Target Manager を立ち上げておく必要があります。
*   テスト時に Tests/Htcs/Sources/Tools/StressTestTool を実行する必要があります。
*/

namespace
{
    // テストデータのサイズ
    const size_t EchoDataSize = 512 * 1024 * 1024;

    // スレッドスタック
    const size_t ThreadStackSize = 16 * 1024;
    NN_ALIGNAS(4096) uint8_t g_ThreadStack[ThreadStackSize];

    static const size_t KeyStringSize = 11;
    static char g_KeyString[KeyStringSize];

    void* Allocate(size_t size)
    {
        return malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        free(p);
    }

    class BigData : public ::testing::Test
    {
    protected:
        static void SetUpTestCase() {
            for (int i = 0; i < (KeyStringSize - 1); i++)
            {
                int seed = nn::os::GetSystemTick().GetInt64Value() % 26;
                g_KeyString[i] = static_cast<char>('a' + seed);
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            }
            g_KeyString[KeyStringSize - 1] = '\0';
        }

        virtual void SetUp()
        {
            nn::htcs::Initialize(Allocate, Deallocate);
        }
        virtual void TearDown()
        {
            nn::htcs::Finalize();
        }
    };
}

TEST_F(BigData, Start)
{
    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    nnt::htcs::util::ConnectToTestServer(socket);

    const char* testName = "BigData";

    size_t bufferSize = std::strlen(g_KeyString) + 1 + std::strlen(testName) + 1;
    char* buffer = reinterpret_cast<char*>(std::malloc(bufferSize));
    ASSERT_NE(nullptr, buffer);
    buffer[0] = '\0';

    std::strcat(buffer, g_KeyString);
    std::strcat(buffer, ",");
    std::strcat(buffer, testName);

    ASSERT_EQ(nn::htcs::Send(socket, buffer, bufferSize, 0), bufferSize);

    std::free(buffer);
}

TEST_F(BigData, EchoClient)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    nn::os::ThreadType thread;
    nnt::htcs::util::ThreadArgs arg;
    std::strcpy(arg.address.portName.name, "Echo0_");
    std::strcat(arg.address.portName.name, g_KeyString);
    arg.mode = nnt::htcs::util::ThreadMode_Echo;
    arg.pBarrier = nullptr;
    arg.size = EchoDataSize;
    arg.count = 1;
    arg.verbose = false;

    nn::os::CreateThread(&thread, nnt::htcs::util::ClientThread, &arg, g_ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority);

    nn::os::StartThread(&thread);

    nn::os::WaitThread(&thread);

    nn::os::DestroyThread(&thread);
}

TEST_F(BigData, EchoServer)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    nn::os::ThreadType thread;
    nnt::htcs::util::ThreadArgs arg;

    std::strcpy(arg.address.portName.name, "Echo1_");
    std::strcat(arg.address.portName.name, g_KeyString);
    arg.mode = nnt::htcs::util::ThreadMode_Echo;
    arg.pBarrier = nullptr;
    arg.size = EchoDataSize;
    arg.count = 1;
    arg.verbose = false;

    nn::os::CreateThread(&thread, nnt::htcs::util::ServerThread, &arg, g_ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority);

    nn::os::StartThread(&thread);

    nn::os::WaitThread(&thread);

    nn::os::DestroyThread(&thread);
}
