﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <nn/nn_Log.h>
#include <nn/htcs.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/htcsUtil/testHtcs_utilMemoryLeakDetector.h>

/*
*   Client のテスト
*   事前に Target Manager を立ち上げておく必要があります。
*   テスト時に Tests/Htcs/Sources/Tools/EchoServer を実行する必要があります。
*/

namespace
{
    void* Allocate(size_t size)
    {
        return malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        free(p);
    }

    class BasicClient : public ::testing::Test
    {
    protected:
        virtual void SetUp()
        {
            nn::htcs::Initialize(Allocate, Deallocate);
        }
        virtual void TearDown()
        {
            nn::htcs::Finalize();
        }
    };

    // チェック用文字列を送信し、返ってきたデータが送信したものと同じであることを確認する
    void CheckEcho(int sock, const char *checkString)
    {
        nn::htcs::ssize_t sentBytes = nn::htcs::Send(sock, checkString, strlen(checkString), 0);
        EXPECT_LE(0, sentBytes);
        if (sentBytes < 0)
        {
            NN_LOG("send error %d\n", nn::htcs::GetLastError());
            return;
        }
        // 受信側が ReadLine で読み込むため改行文字を送信
        nn::htcs::Send(sock, "\n", 1, 0);
        EXPECT_LE(0, sentBytes);
        if (sentBytes < 0)
        {
            NN_LOG("send error %d\n", nn::htcs::GetLastError());
            return;
        }

        char buf[128];
        nn::htcs::ssize_t receivedBytes = nn::htcs::Recv(sock, buf, sizeof(buf) - 1, 0);
        EXPECT_LT(0, receivedBytes);
        if (receivedBytes < 0)
        {
            NN_LOG("Recv error %d\n", nn::htcs::GetLastError());
            return;
        }
        else if (receivedBytes == 0)
        {
            NN_LOG("gracefully closed\n");
            return;
        }

        buf[receivedBytes] = '\0';
        EXPECT_STREQ(checkString, buf);
    }

    void SendEnd(int sock)
    {
        const char closeMessage[] = "End\n";
        nn::htcs::ssize_t sentBytes = nn::htcs::Send(sock, closeMessage, strlen(closeMessage), 0);
        EXPECT_LE(0, sentBytes);
        if (sentBytes < 0)
        {
            NN_LOG("send error %d\n", nn::htcs::GetLastError());
        }
    }

}

TEST_F(BasicClient, Simple)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    // 接続したいホスト側サーバのアドレス情報構築
    nn::htcs::SockAddrHtcs addr;
    addr.family = nn::htcs::HTCS_AF_HTCS;
    addr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(addr.portName.name, "ServerInHost");

    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    NN_LOG("Trying to connect...\n");
    while (NN_STATIC_CONDITION(true))
    {
        if (nn::htcs::Connect(socket, &addr) == 0)
        {
            NN_LOG("Connected to server on host.\n");
            break;
        }
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1000) );
    }
    NN_LOG("Connected!\n");

    EXPECT_EQ(-1, nn::htcs::Connect(socket, &addr));
    EXPECT_EQ(nn::htcs::HTCS_EISCONN, nn::htcs::GetLastError());

    CheckEcho( socket, "Sending from Target to Host." );

    EXPECT_EQ(0, nn::htcs::Close(socket));
}

TEST_F(BasicClient, RepeatSendAndReceive)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    // 接続したいホスト側サーバのアドレス情報構築
    nn::htcs::SockAddrHtcs addr;
    addr.family = nn::htcs::HTCS_AF_HTCS;
    addr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(addr.portName.name, "ServerInHost");

    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    NN_LOG("Trying to connect...\n");
    while (NN_STATIC_CONDITION(true))
    {
        if (nn::htcs::Connect(socket, &addr) == 0)
        {
            NN_LOG("Connected to server on host.\n");
            break;
        }
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1000) );
    }
    NN_LOG("Connected!\n");

    for( int i = 0; i < 10; ++i )
    {
        char buffer[8];
        sprintf( buffer, "Echo %d", i );
        CheckEcho( socket, buffer );
    }

    EXPECT_EQ(0, nn::htcs::Close(socket));
}

TEST_F(BasicClient, MultipleSocket)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    // 接続したいホスト側サーバのアドレス情報構築
    nn::htcs::SockAddrHtcs addr;
    addr.family = nn::htcs::HTCS_AF_HTCS;
    addr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(addr.portName.name, "ServerInHost");

    const int socketCount = 11; // EchoServer 側の listen の backlog 数 + 1
    int socket[socketCount];
    for( int i = 0; i < socketCount; ++i )
    {
        socket[i] = nn::htcs::Socket();
        EXPECT_LE(0, socket[i]);
    }

    NN_LOG("Trying to connect...\n");
    while (NN_STATIC_CONDITION(true))
    {
        if (nn::htcs::Connect(socket[0], &addr) == 0)
        {
            NN_LOG("Connected to server on host.\n");
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
    }
    NN_LOG("Connected!\n");

    for( int i = 1; i < socketCount; ++i )
    {
        EXPECT_EQ(0, nn::htcs::Connect(socket[i], &addr));
    }

    int tmpSocket = nn::htcs::Socket();
    EXPECT_LE(0, tmpSocket);
    EXPECT_EQ(-1, nn::htcs::Connect(tmpSocket, &addr));
    EXPECT_EQ(nn::htcs::HTCS_ECONNREFUSED, nn::htcs::GetLastError());
    nn::htcs::Close(tmpSocket);

    for( int i = 0; i < socketCount; ++i )
    {
        char buffer[32];
        sprintf( buffer, "Sending socket[%d]." , i );
        CheckEcho( socket[i], buffer );
        EXPECT_EQ(0, nn::htcs::Close(socket[i]));
    }
}

TEST_F(BasicClient, End)
{
    // 接続したいホスト側サーバのアドレス情報構築
    nn::htcs::SockAddrHtcs addr;
    addr.family = nn::htcs::HTCS_AF_HTCS;
    addr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(addr.portName.name, "ServerInHost");

    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    NN_LOG("Trying to connect...\n");
    while (NN_STATIC_CONDITION(true))
    {
        if (nn::htcs::Connect(socket, &addr) == 0)
        {
            NN_LOG("Connected to server on host.\n");
            break;
        }
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1000) );
    }
    NN_LOG("Connected!\n");

    SendEnd(socket);

    EXPECT_EQ(0, nn::htcs::Close(socket));
}
