﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nnt/nntest.h>
#include <nnt/htcsUtil/testHtcs_util.h>

#include <nn/nn_Log.h>

namespace {
    void ProcessRecvMode(int socket, nnt::htcs::util::ThreadArgs* pArgs)
    {
        NN_LOG("Start %s (size = %lld, count = %lld)\n", __FUNCTION__, pArgs->size, pArgs->count);

        // 準備
        uint8_t* recvBuffer = reinterpret_cast<uint8_t*>(std::malloc(pArgs->size));

        ASSERT_NE(nullptr, recvBuffer);

        std::memset(recvBuffer, 0, pArgs->size);

        // 待機
        if (pArgs->pBarrier != nullptr)
        {
            pArgs->pBarrier->Await();
        }

        // 通信
        double totalDuration = 0.0;
        for (int i = 0; i < pArgs->count; i++)
        {
            NN_LOG("  %s count %d\n", pArgs->address.portName.name, i);
            nn::os::Tick start = nn::os::GetSystemTick();
            nnt::htcs::util::Recv(socket, pArgs->size, recvBuffer, pArgs->recvFlag);
            nn::os::Tick end = nn::os::GetSystemTick();

            double duration = nn::os::ConvertToTimeSpan(end - start).GetNanoSeconds() / (1000.0 * 1000.0 * 1000.0);
            if (pArgs->verbose)
            {
                NN_LOG("nn::htcs::Recv() consumed %f milli seconds.\n", duration * 1000);
            }
            totalDuration += duration;
        }
        if (pArgs->verbose)
        {
            NN_LOG("nn::htcs::Recv() consumed average %f milli seconds.\n", totalDuration / pArgs->count);
        }

        std::free(recvBuffer);

        NN_LOG("End %s\n", __FUNCTION__);
    }

    void ProcessSendMode(int socket, nnt::htcs::util::ThreadArgs* pArgs)
    {
        ASSERT_GE(pArgs->size, (size_t)0);
        ASSERT_GE(pArgs->count, 0);

        NN_LOG("Start %s (size = %lld, count = %lld)\n", __FUNCTION__, pArgs->size, pArgs->count);

        // 準備
        uint8_t* sendBuffer = reinterpret_cast<uint8_t*>(std::malloc(pArgs->size));

        ASSERT_NE(nullptr, sendBuffer);

        nnt::htcs::util::FillBuffer(sendBuffer, pArgs->size);

        // 待機
        if (pArgs->pBarrier != nullptr)
        {
            pArgs->pBarrier->Await();
        }

        // 通信
        double totalDuration = 0.0;
        for (int i = 0; i < pArgs->count; i++)
        {
            NN_LOG("  %s count %d\n", pArgs->address.portName.name, i);
            nn::os::Tick start = nn::os::GetSystemTick();
            nnt::htcs::util::Send(socket, pArgs->size, sendBuffer);
            nn::os::Tick end = nn::os::GetSystemTick();

            double duration = nn::os::ConvertToTimeSpan(end - start).GetNanoSeconds() / (1000.0 * 1000.0 * 1000.0);
            if (pArgs->verbose)
            {
                NN_LOG("nn::htcs::Send() consumed %f milli seconds.\n", duration);
            }
            totalDuration += duration;
        }
        if (pArgs->verbose)
        {
            NN_LOG("nn::htcs::Send() consumed average %f milli seconds.\n", totalDuration / pArgs->count);
        }

        std::free(sendBuffer);

        NN_LOG("End \n", __FUNCTION__);
    }

    void ProcessEchoMode(int socket, nnt::htcs::util::ThreadArgs* pArgs)
    {
        ASSERT_GE(pArgs->size, (size_t)0);
        ASSERT_GE(pArgs->count, 0);

        NN_LOG("Start %s (size = %lld, count = %lld)\n", __FUNCTION__, pArgs->size, pArgs->count);

        // 準備
        uint8_t* sendBuffer = reinterpret_cast<uint8_t*>(std::malloc(pArgs->size));
        uint8_t* recvBuffer = reinterpret_cast<uint8_t*>(std::malloc(pArgs->size));

        ASSERT_NE(nullptr, sendBuffer);
        ASSERT_NE(nullptr, recvBuffer);

        nnt::htcs::util::FillBuffer(sendBuffer, pArgs->size);
        std::memset(recvBuffer, 0, pArgs->size);

        // 待機
        if (pArgs->pBarrier != nullptr)
        {
            pArgs->pBarrier->Await();
        }

        double totalDuration = 0.0;
        for (int i = 0; i < pArgs->count; i++)
        {
            NN_LOG("  %s count %d\n", pArgs->address.portName.name, i);
            nn::os::Tick start = nn::os::GetSystemTick();
            nnt::htcs::util::Echo(socket, pArgs->size, sendBuffer, recvBuffer, pArgs->recvFlag);
            nn::os::Tick end = nn::os::GetSystemTick();

            double duration = nn::os::ConvertToTimeSpan(end - start).GetNanoSeconds() / (1000.0 * 1000.0 * 1000.0);
            if (pArgs->verbose)
            {
                NN_LOG("nnt::htcs::util::Echo() consumed %f milli seconds.\n", duration);
            }
            totalDuration += duration;
        }
        if (pArgs->verbose)
        {
            NN_LOG("nn::htcs::util::Echo() consumed average %f milli seconds.\n", totalDuration / pArgs->count);
        }

        std::free(sendBuffer);
        std::free(recvBuffer);

        NN_LOG("End \n", __FUNCTION__);
    }
}

namespace nnt { namespace htcs { namespace util {
    void ClientThread(void *pArgs)
    {
        ThreadArgs* pThreadArgs = reinterpret_cast<ThreadArgs*>(pArgs);

        int socket = nn::htcs::Socket();
        ASSERT_LE(0, socket);

        util::Connect(socket, &pThreadArgs->address);

        switch (pThreadArgs->mode)
        {
        case ThreadMode_Recv:
            ProcessRecvMode(socket, pThreadArgs);
            break;
        case ThreadMode_Send:
            ProcessSendMode(socket, pThreadArgs);
            break;
        case ThreadMode_Echo:
            ProcessEchoMode(socket, pThreadArgs);
            break;
        default:
            NN_LOG("Mode %d is not implemented.\n", pThreadArgs->mode);
            break;
        }

        EXPECT_EQ(0, nn::htcs::Close(socket));
    }

    void ServerThread(void *pArgs)
    {
        ThreadArgs* pThreadArgs = reinterpret_cast<ThreadArgs*>(pArgs);

        int listenSocket = nn::htcs::Socket();
        ASSERT_LE(0, listenSocket);

        EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &pThreadArgs->address));
        EXPECT_EQ(0, nn::htcs::Listen(listenSocket, 1));

        int socket = nn::htcs::Accept(listenSocket, nullptr);
        EXPECT_LE(0, socket);

        switch (pThreadArgs->mode)
        {
        case ThreadMode_Recv:
            ProcessRecvMode(socket, pThreadArgs);
            break;
        case ThreadMode_Send:
            ProcessSendMode(socket, pThreadArgs);
            break;
        case ThreadMode_Echo:
            ProcessEchoMode(socket, pThreadArgs);
            break;
        default:
            NN_LOG("Mode %d is not implemented.\n", pThreadArgs->mode);
            break;
        }

        EXPECT_EQ(0, nn::htcs::Close(socket));
        EXPECT_EQ(0, nn::htcs::Close(listenSocket));
    }
}}}
