﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/htc.h>
#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/htc/htc_Result.h>
#include <nn/util/util_StringUtil.h>

#include <nnt/base/testBase_Exit.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#if defined(NN_BUILD_CONFIG_OS_WIN32)
#include <nn/nn_Windows.h>
#include <nn/htc/htc_ResultPrivate.h>
#include <nn/htc/tenv/htc_Tenv.h>
#endif

#if defined(NN_BUILD_CONFIG_OS_WIN32)
TEST(Internal, RegisterDefinitionFilePath)
{
    nn::os::ProcessId processId = { 0x1 };
    const char path[] = "C:/test/test.tdf";
    size_t length = sizeof(path) - 1; // NULL 終端含まない
    NNT_EXPECT_RESULT_FAILURE(nn::htc::ResultTooLongFilePath, nn::htc::tenv::RegisterDefinitionFilePath(processId, path, 768));
    NNT_EXPECT_RESULT_SUCCESS(nn::htc::tenv::RegisterDefinitionFilePath(processId, path, length));
    NNT_EXPECT_RESULT_SUCCESS(nn::htc::tenv::RegisterDefinitionFilePath(processId, path, length));
    nn::htc::tenv::UnregisterDefinitionFilePath(processId);
}
#endif

TEST(GetTargetEnvironmentVariableTest, Basic)
{
    std::string variableNames[] =
        {
            std::string("TEST"),
            std::string("TESTFORUSER"),
            std::string("TESTTESTTESTTESTTESTTESTTESTTESTTESTTESTTESTTESTTESTTESTTESTTES"),
            std::string("TEST_VALUE_LENGTH1"),
            std::string("TEST_VALUE_LENGTH2"),
        };
    std::string correctValues[] =
        {
            std::string("TestValue"),
            std::string("TestValueForUser"),
            std::string("TestNameLength"),
            std::string(""),
            std::string("TestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTestTes"), // 定義ファイルの値より 1 文字少ない
        };

    for (int i = 0; i < sizeof(variableNames) / sizeof(variableNames[0]); ++i)
    {
        NN_LOG("# %d\n", i);

        size_t outLength = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::htc::GetTargetEnvironmentVariableLength(&outLength, variableNames[i].c_str()));
        EXPECT_EQ(outLength, correctValues[i].length() + 1);

        char* value = new char[outLength];
        NNT_EXPECT_RESULT_SUCCESS(nn::htc::GetTargetEnvironmentVariable(&outLength, value, outLength, variableNames[i].c_str()));
        EXPECT_EQ(outLength, correctValues[i].length() + 1);
        EXPECT_TRUE(nn::util::Strncmp(value, correctValues[i].c_str(), static_cast<int>(correctValues[i].length())) == 0);
    }
}

TEST(GetTargetEnvironmentVariableTest, Recursive)
{
    std::string variableNames[] =
        {
            std::string("TEST_RECURSIVE1"),
            std::string("TEST_RECURSIVE2"),
            std::string("TEST_RECURSIVE3"),
            std::string("TEST_RECURSIVE4")
        };
    std::string correctValues[] =
        {
            std::string("TestValueTEST"),
            std::string("%TEST"),
            std::string("TEST%"),
            std::string("TestValueTESTTestValue"),
        };

    for (int i = 0; i < sizeof(variableNames) / sizeof(variableNames[0]); ++i)
    {
        NN_LOG("# %d\n", i);

        size_t outLength = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::htc::GetTargetEnvironmentVariableLength(&outLength, variableNames[i].c_str()));
        EXPECT_EQ(outLength, correctValues[i].length() + 1);

        char* value = new char[outLength];
        NNT_EXPECT_RESULT_SUCCESS(nn::htc::GetTargetEnvironmentVariable(&outLength, value, outLength, variableNames[i].c_str()));
        EXPECT_EQ(outLength, correctValues[i].length() + 1);
        EXPECT_TRUE(nn::util::Strncmp(value, correctValues[i].c_str(), static_cast<int>(correctValues[i].length())) == 0);
    }
}

#if defined(NN_BUILD_CONFIG_OS_WIN32)
TEST(GetTargetEnvironmentVariableTest, NullLength)
{
    std::string variableName("TEST");
    EXPECT_DEATH(nn::htc::GetTargetEnvironmentVariableLength(nullptr, variableName.c_str()), "");
}

TEST(GetTargetEnvironmentVariableTest, NullName)
{
    size_t outLength = 0;
    EXPECT_DEATH(nn::htc::GetTargetEnvironmentVariableLength(&outLength, nullptr), "");
}

TEST(GetTargetEnvironmentVariableTest, EmptyName)
{
    std::string variableName("");
    size_t outLength = 0;
    EXPECT_DEATH(nn::htc::GetTargetEnvironmentVariableLength(&outLength, variableName.c_str()), "");
}
#endif

TEST(GetTargetEnvironmentVariableTest, NonExistName)
{
    std::string variableNames[] =
        {
            std::string("NOTEXIST"),
            std::string("TES"),
        };

    for (int i = 0; i < sizeof(variableNames) / sizeof(variableNames[0]); ++i)
    {
        NN_LOG("# %d\n", i);
        size_t outLength = 0;
        NNT_EXPECT_RESULT_FAILURE(nn::htc::ResultNotFound, nn::htc::GetTargetEnvironmentVariableLength(&outLength, variableNames[i].c_str()));
    }
}

#if defined(NN_BUILD_CONFIG_OS_WIN32)
TEST(GetTargetEnvironmentVariableTest, InvalidNameLength)
{
    std::string variableName("TESTTESTTESTTESTTESTTESTTESTTESTTESTTESTTESTTESTTESTTESTTESTTEST");
    size_t outLength = 0;
    EXPECT_DEATH(nn::htc::GetTargetEnvironmentVariableLength(&outLength, variableName.c_str()), "");
}

TEST(GetTargetEnvironmentVariableTest, NullBuffer)
{
    std::string variableName("TEST");
    size_t outLength = 0;
    NNT_EXPECT_RESULT_SUCCESS(nn::htc::GetTargetEnvironmentVariableLength(&outLength, variableName.c_str()));
    EXPECT_DEATH(nn::htc::GetTargetEnvironmentVariable(&outLength, nullptr, 2048, variableName.c_str()), "");
}
#endif

TEST(GetTargetEnvironmentVariableTest, NotEnoughBuffer)
{
    std::string variableName("TEST");
    size_t outLength = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::htc::GetTargetEnvironmentVariableLength(&outLength, variableName.c_str()));

    char* value = new char[outLength - 1];
    NNT_EXPECT_RESULT_FAILURE(nn::htc::ResultNotEnoughBuffer, nn::htc::GetTargetEnvironmentVariable(&outLength, value, outLength - 1, variableName.c_str()));
}

TEST(WaitUntilTargetEnvironmentVariableAvailableTest, Basic)
{
    NNT_EXPECT_RESULT_SUCCESS(nn::htc::WaitUntilTargetEnvironmentVariableAvailable(nn::TimeSpan::FromMilliSeconds(100)));
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    const int exitCode = RUN_ALL_TESTS();

    nnt::Exit(exitCode);
}
