﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_SdkText.h>
#include <nn/htc.h>
#include <nn/htc/htc_Result.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

/*
    ※テスト開始前に以下を環境変数に追加し、TargetManager を起動しておく。
    HTC_GETENVIRONMENT_TEST_PATH=（envTestPath の値）
    HTC_GETENVIRONMENT_TEST_MAX=（envTestMax の値）
    HTC_GETENVIRONMENT_TEST_JAPANESE=（envTestJapanese の値）
    HTC_GETENVIRONMENT_TEST_MULTI=（envTestMulti の値）
    HTC_GETENVIRONMENT_TEST_SYMBOL=（envTestSymbol の値）
*/

namespace
{
    const char envTestPath[] = "D:\\home";
    const char envTestMax[] = // TORIAEZU: 2047 文字
        "phqghumeaylnlfdxfircvscxggbwkfnqduxwfnfozvsrtkjprepggxrpnrvystmwcysyycqpevikeffmznimkkasvwsrenzkycxf"
        "xtlsgypsfadpooefxzbcoejuvpvaboygpoeylfpbnpljvrvipyamyehwqnqrqpmxujjloovaowuxwhmsncbxcoksfzkvatxdknly"
        "jyhfixjswnkkufnuxxzrzbmnmgqooketlyhnkoaugzqrcddiuteiojwayyzpvscmpsajlfvgubfaaovlzylntrkdcpwsrtesjwhd"
        "izcobzcnfwlqijtvdwvxhrcbldvgylwgbusbmborxtlhcsmpxohgmgnkeufdxotogbgxpeyanfetcukepzshkljugggekjdqzjen"
        "pevqgxiepjsrdzjazujllchhbfqmkimwzobiwybxduunfsksrsrtekmqdcyzjeeuhmsrqcozijipfioneeddpszrnavymmtatbdz"
        "qsoemuvnpppsuacbazuxmhecthlegrpunkdmbppweqtgjoparmowzdqyoxytjbbhawdydcprjbxphoohpkwqyuhrqzhnbnfuvqnq"
        "qlrzjpxiogvliexdzuzosrkrusvojbrzmwzpowkjilefraamdigpnpuuhgxpqnjwjmwaxxmnsnhhlqqrzudltfzotcjtnzxuglsd"
        "smzcnockvfajfrmxothowkbjzwucwljfrimpmyhchzriwkbarxbgfcbceyhjugixwtbvtrehbbcpxifbxvfbcgkcfqckcotzgkub"
        "mjrmbsztsshfroefwsjrxjhguzyupzwweiqurpixiqflduuveoowqcudhnefnjhaimuczfskuiduburiswtbrecuykabfcvkdzez"
        "toidukuhjzefczzzbfkqdpqzikfobucdhthxdjgkjelrlpaxamceroswitdptpcclifkeljytihrcqaybnefxnxvgzedyyhngycd"
        "rudmphmeckotrwospofghfozqvlqfxwwkmfxdyygmdcaszsgovsodkjghcwmbmxrmhuyfyqgajqkcklznayxqkqoyzwmyubzazcp"
        "khktkydzivcuypurfmbisgekyrgzvxdhpoamvafyrarxsvkhtqdihersigbhzjzujxmmyspnaraewkegjccvhhrjvbjtsqdjootg"
        "pknfpfycgfieowqrwwwpzsqmetogepspxnvjiupalyynmkmnuvklhsecdwracgfmzkgipdfodkjmjqwiqpuoqhimvfvuzwyvijgf"
        "ullkjduhsjafbtlkmfqrmyjfjnhhssqctydteamdcjbprhtnegyiwxgcjwlgrsmeaearwtvjsjbaoiojlwhypnvruihoswkifygt"
        "ydhacwyhsgewzmtgonzltjhgauhnihreqgjfwkjsmtpjhaefqzaauldrchjccdyrfvvrivuyeegfivdrcygurqdredakubnfgupr"
        "oqylobcwqxkzmausjgmhcmhgdnmphnqkamhurktrffaclvgrzkkldacllteojomonxrqyjzginrnnzwacxxaedrwudxzrfusewjt"
        "boxvynfhkstcenaumnddxfdmvzcautdcckxaaydzsxttobbgqngvvpjgojoglmkxgbfcpypckqchbddzwrxbzmqrlxvobtwhxgin"
        "fgfrcclmznmjugwwbsqfcihubsjollmsqsghmcphelsotflbgsfnpcuzsrupchynvzhcpqugriwniqxdfjpwpxfblkpnpeelfjmt"
        "kuqpzomwnlmbupmktlptndmpdsydsgvfpenemwborifsuqhceskmkhssmvnonwafxwhgbibabvqopqfoviussqfqwehtxdzujtln"
        "txmrjxxwtlggkytbiolydnilqadojskkvfxahhjmbocljarintdwcldvdxropbyjzwyyojuothwmlvrglfzdzdbtubxuoffvncrs"
        "wsaznmoijoivvgobqpnckwvnhkebmtdhvygkjisuxhatmuz";
    const char* envTestJapanese = NN_TEXT("あいうえお");
    const char* envTestMulti = NN_TEXT("あaいbうcえdお");
    const char envTestSymbol[] = " \"#$%&'()*+,-./:;<=>?@[\\]^_`{|}~";

    class GetEnvironmentVariable : public ::testing::Test
    {
    protected:
        virtual void SetUp()
        {
            nn::htc::Initialize();

            size_t tmpSize;
            while( nn::htc::ResultConnectionFailure::Includes(nn::htc::GetEnvironmentVariableLength(&tmpSize, "")) )
            {
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }
        }
        virtual void TearDown()
        {
            nn::htc::Finalize();
        }
    };
}

TEST_F(GetEnvironmentVariable, GetLength)
{
    const char EnvVarName[] = "HTC_GETENVIRONMENT_TEST_PATH";
    size_t length;
    nn::Result result = nn::htc::GetEnvironmentVariableLength(&length, EnvVarName);
    NNT_EXPECT_RESULT_SUCCESS(result);
    if( result.IsSuccess() )
    {
        EXPECT_EQ(sizeof(envTestPath), length);
    }
}

TEST_F(GetEnvironmentVariable, GetLengthMax)
{
    const char EnvVarName[] = "HTC_GETENVIRONMENT_TEST_MAX";
    size_t length;
    nn::Result result = nn::htc::GetEnvironmentVariableLength(&length, EnvVarName);
    NNT_EXPECT_RESULT_SUCCESS(result);
    if( result.IsSuccess() )
    {
        EXPECT_EQ(sizeof(envTestMax), length);
    }
}

TEST_F(GetEnvironmentVariable, GetLengthJapanese)
{
    const char EnvVarName[] = "HTC_GETENVIRONMENT_TEST_JAPANESE";
    size_t length;
    nn::Result result = nn::htc::GetEnvironmentVariableLength(&length, EnvVarName);
    NNT_EXPECT_RESULT_SUCCESS(result);
    if( result.IsSuccess() )
    {
        EXPECT_EQ(strlen(envTestJapanese) + 1, length);
    }
}

TEST_F(GetEnvironmentVariable, GetLengthNotFound)
{
    const char EnvVarName[] = "NULL";
    size_t length;
    nn::Result result = nn::htc::GetEnvironmentVariableLength(&length, EnvVarName);
    NNT_EXPECT_RESULT_FAILURE(nn::htc::ResultNotFound, result);
}

TEST_F(GetEnvironmentVariable, GetVariableBasic)
{
    const char EnvVarName[] = "HTC_GETENVIRONMENT_TEST_PATH";
    size_t length;
    const size_t bufferLength = 128;
    char buffer[bufferLength];
    nn::Result result = nn::htc::GetEnvironmentVariable(&length, buffer, bufferLength, EnvVarName);
    NNT_EXPECT_RESULT_SUCCESS(result);
    if( result.IsSuccess() )
    {
        EXPECT_EQ(sizeof(envTestPath), length);
        EXPECT_STREQ(envTestPath, buffer);
    }
}

TEST_F(GetEnvironmentVariable, GetVariableMax)
{
    const char EnvVarName[] = "HTC_GETENVIRONMENT_TEST_MAX";
    size_t length;
    const size_t bufferLength = 2048;
    char buffer[bufferLength];
    nn::Result result = nn::htc::GetEnvironmentVariable(&length, buffer, bufferLength, EnvVarName);
    NNT_EXPECT_RESULT_SUCCESS(result);
    if( result.IsSuccess() )
    {
        EXPECT_EQ(sizeof(envTestMax), length);
        EXPECT_STREQ(envTestMax, buffer);
    }
}

TEST_F(GetEnvironmentVariable, GetVariableJapanese)
{
    const char EnvVarName[] = "HTC_GETENVIRONMENT_TEST_JAPANESE";
    size_t length;
    const size_t bufferLength = 128;
    char buffer[bufferLength];
    nn::Result result = nn::htc::GetEnvironmentVariable(&length, buffer, bufferLength, EnvVarName);
    NNT_EXPECT_RESULT_SUCCESS(result);
    if( result.IsSuccess() )
    {
        EXPECT_EQ(strlen(envTestJapanese) + 1, length);
        EXPECT_STREQ(envTestJapanese, buffer);
    }
}

TEST_F(GetEnvironmentVariable, GetVariableMultiLanguage)
{
    const char EnvVarName[] = "HTC_GETENVIRONMENT_TEST_MULTI_LANG";
    size_t length;
    const size_t bufferLength = 128;
    char buffer[bufferLength];
    nn::Result result = nn::htc::GetEnvironmentVariable(&length, buffer, bufferLength, EnvVarName);
    NNT_EXPECT_RESULT_SUCCESS(result);
    if( result.IsSuccess() )
    {
        EXPECT_EQ(strlen(envTestMulti) + 1, length);
        EXPECT_STREQ(envTestMulti, buffer);
    }
}

TEST_F(GetEnvironmentVariable, GetVariableSymbol)
{
    const char EnvVarName[] = "HTC_GETENVIRONMENT_TEST_SYMBOL";
    size_t length;
    const size_t bufferLength = 128;
    char buffer[bufferLength];
    nn::Result result = nn::htc::GetEnvironmentVariable(&length, buffer, bufferLength, EnvVarName);
    NNT_EXPECT_RESULT_SUCCESS(result);
    if( result.IsSuccess() )
    {
        EXPECT_EQ(sizeof(envTestSymbol), length);
        EXPECT_STREQ(envTestSymbol, buffer);
    }
}
TEST_F(GetEnvironmentVariable, GetVariableNotFound)
{
    const char EnvVarName[] = "NULL";
    size_t length;
    const size_t bufferLength = 128;
    char buffer[bufferLength];
    nn::Result result = nn::htc::GetEnvironmentVariable(&length, buffer, bufferLength, EnvVarName);
    NNT_EXPECT_RESULT_FAILURE(nn::htc::ResultNotFound, result);
}

TEST_F(GetEnvironmentVariable, GetVariableNotEnoughBuffer)
{
    const char EnvVarName[] = "HTC_GETENVIRONMENT_TEST_PATH";
    size_t length;
    char buffer;
    nn::Result result = nn::htc::GetEnvironmentVariable(&length, &buffer, 1, EnvVarName);
    NNT_EXPECT_RESULT_FAILURE(nn::htc::ResultNotEnoughBuffer, result);
}
