﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @brief
    ツールメニュー別の処理を纏めただけ
 */
#include "MenuFunc.h"

const int NpadIdNum = 9;
uint8_t PollingCount[NpadIdNum];

/////////////////////////////
// エラー出力の文字変換
/////////////////////////////
const char* ToStr(nn::Result result)
{
    if (result.IsSuccess())
    {
        return "Success";
    }
    else if (nn::hidbus::ResultFWUpdateFailed::Includes(result))
    {
        // コントローラーの Firmware Update が発生したが失敗しました。
        // リトライするか、アプリ側で Firmware Update アプレットを立ち上げてください。
        return "Firmware Update Failed(Please retry)";
    }
    else if (nn::hidbus::ResultInvalidHandle::Includes(result))
    {
        // コントローラの切断等が発生し、handle が無効になっています。
        // handle 取得からリトライしてください。
        return "Handle is invalid";
    }
    else if (nn::hidbus::ResultNotInForcus::Includes(result))
    {
        // アプリケーションが InForcus 状態でないため、拡張デバイスを有効化できません。
        // アプリケーション側で nn::oe のライブラリ群を使用してアプリケーションが InForcus になるのを待ってください。
        return "Not InForcus";
    }
    else if (nn::hidbus::ResultActivateFailureNpadBusy::Includes(result))
    {
        // 指定された Npad が拡張バス機能と排他で使用されるべき機能を使用しているため、有効化できませんでした。
        // アプリケーション側で拡張バスと排他で使用されるべき機能を終了して、再度本関数を実行してください。
        return "Please deactivate the dependency function";
    }
    else if (nn::hidbus::ResultExternalDeviceNotEnabled::Includes(result))
    {
        // 拡張バスに接続されたデバイスの取り外しや、アプリケーションが InForcus でなくなった等の理由により、
        // 拡張バスに接続されたデバイスとの通信機能が無効化されています。
        // 再度 Enable からやり直してください。
        return "External Device is not enabled";
    }
    else if (nn::hidbus::ResultExternalDeviceNotAttached::Includes(result))
    {
        // 拡張バスにデバイスが接続されていません。
        return "Waiting External Device Connected..";
    }
    else if (nn::hidbus::ResultExternalDeviceReadyTimeout::Includes(result))
    {
        // 拡張バスに接続されたデバイスとの通信準備中にタイムアウトしました。
        return "Communication Ready Timeout(Please retry)";
    }
    else if (nn::hidbus::ResultExternalDeviceTimeout::Includes(result))
    {
        // 拡張バスに接続されたデバイスとの通信中にタイムアウトしました。
        return "Communication Timeout(Please retry)";
    }
    else if (nn::hidbus::ResultExternalDeviceInvalidId::Includes(result))
    {
        // 指定された Device ID と異なるデバイスが拡張バスに接続されています。
        return "Connected Device is invalid";
    }
    else
    {
        return "Unexpected Result";
    }
}

/////////////////////////////////
// IndexのNpadId変換
/////////////////////////////////
nn::hid::NpadIdType GetNpadIdFromIndex(size_t index)
{
    if (index != 8)
    {
        return index;
    }
    else
    {
        return nn::hid::NpadId::Handheld;
    }
}

/////////////////////////////////////////////////
// Handle取得
/////////////////////////////////////////////////
bool GetHidbusHandle(HidbusState* state, size_t index, nn::hidbus::BusType Type)
{
    nn::hid::NpadIdType id = GetNpadIdFromIndex(index);

    auto timeoutCount = 0;

    while (!nn::hidbus::GetBusHandle(&state->handle, id, Type))
    {
        NN_LOG("Please connect supported controller.\n");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        timeoutCount++;
        // 規定時間経って無理ならタイムアウト
        if (timeoutCount >= 10)
        {
            NN_LOG("Timeout cannot get handle\n");
            return false;
        }
    }
    NN_LOG("GetHandle Result : Success\n");
    return true;
}

/////////////////////////////////////////////////////
// APIコール時のエラーハンドリング
/////////////////////////////////////////////////////
BusStatus ErrorHandling(HidbusState* currentState, nn::Result result)
{
    HidbusState state;

    if (nn::hidbus::ResultInvalidHandle::Includes(result))
    {
        state.mode = HidbusState_NotGotHandle;
        currentState->isPolling = IsPolling_Disable;
        nn::oe::SetUserInactivityDetectionTimeExtendedUnsafe(false);       // 無操作状態時の自動スリープや画面焼け軽減の開始時間延長を無効化
    }
    else if (nn::hidbus::ResultExternalDeviceNotEnabled::Includes(result) || nn::hidbus::ResultExternalDeviceNotAttached::Includes(result))
    {
        state.mode = HidbusState_GotHandle;
        currentState->isPolling = IsPolling_Disable;
        nn::oe::SetUserInactivityDetectionTimeExtendedUnsafe(false);       // 無操作状態時の自動スリープや画面焼け軽減の開始時間延長を無効化
    }
    else
    {
        state.mode = currentState->mode;
    }

    return state.mode;
}

/////////////////////////////////////////////////////////////////
// CalデータのSendAndReceive
// 工程CalとユーザーCal両方に対応しています
/////////////////////////////////////////////////////////////////
void SendAndReceiveRondeCalData(nnt::Cursol* cursol, HidbusState* state, size_t index, nns::hidbus::RondeManufactureCalibrationData* calData, int sendType)
{
    nn::Result result;

    uint8_t writeCommand[8];

    for (auto cnt = 0; cnt < 4; cnt++)
    {
        // DataReadの場合
        if (sendType == 0)
        {
            result = nnt::hidbus::SendAndReceiveRondeData(state->command, state->packetSize, &state->packet[0], state->handle);
            state->mode = (result.IsSuccess() == true) ? state->mode : ErrorHandling(state, result);
            NN_LOG("SendAndReceive Result : %s\n", ToStr(result));
            switch (cnt)
            {
                case 0:
                    calData->osMax = state->packet[1] << 8 | state->packet[0];
                    state->command[1]++;
                    break;
                case 1:
                    calData->hkMax = state->packet[1] << 8 | state->packet[0];
                    state->command[1]++;
                    break;
                case 2:
                    calData->zeroMin = state->packet[1] << 8 | state->packet[0];
                    state->command[1]++;
                    break;
                case 3:
                    calData->zeroMax = state->packet[1] << 8 | state->packet[0];
                    state->command[1] = state->command[1] - 3;
                    break;
                default:
                    break;
            }
        }
        // DataWriteの場合
        else
        {
            switch (cnt)
            {
                case 0:
                    memcpy(&writeCommand[4], &cursol->osMax, 2);
                    break;
                case 1:
                    state->command[1]++;
                    memcpy(&writeCommand[4], &cursol->hkMax, 2);
                    break;
                case 2:
                    state->command[1]++;
                    memcpy(&writeCommand[4], &cursol->zeroMin, 2);
                    break;
                case 3:
                    state->command[1]++;
                    memcpy(&writeCommand[4], &cursol->zeroMax, 2);
                    break;
                default:
                    break;
            }
            memcpy(writeCommand, state->command, 4);
            result = nnt::hidbus::SendAndReceiveRondeData(writeCommand, state->packetSize, &state->packet[0], state->handle);
            state->mode = (result.IsSuccess() == true) ? state->mode : ErrorHandling(state, result);
            NN_LOG("SendAndReceive Result : %s\n", ToStr(result));

        }
    }
}

/////////////////////////////////////////////////////////////////
// PollingデータのTM表示(count数分だけ)
/////////////////////////////////////////////////////////////////
void PrintRondePollingData(HidbusState* state, RondeData* data, size_t index)
{
    NN_LOG("NpadID : %d\n", index);

    for (auto i = 0; i < state->PollingCount; i++)
    {
        // FWVersion
        if (memcmp(&state->PollingCmd, &nnt::hidbus::CmdList.FwVer[0], 4) == 0)
        {
            NN_LOG("PollingData is FWVersion[Count : %d] = %04x\n", i + 1, data->fwVer[i]);
        }
        // PollingFormat
        if (memcmp(&state->PollingCmd, &nnt::hidbus::CmdList.PollingFormat[0], 4) == 0)
        {
            NN_LOG("PollingData is Strain And Thermister[Count : %d] = (%d . %d)\n", i + 1, data->strain[i], data->thermistor[i]);
        }
        // UniqueID
        if (memcmp(&state->PollingCmd, &nnt::hidbus::CmdList.UniqueId[0], 4) == 0)
        {
            NN_LOG("PollingData is UniqueID[Count : %d] = ", i + 1);
            for (int j = 0; j < nnt::hidbus::SizeList.UniqueId; j++)
            {
                NN_LOG("%02x ", data->uniqueId[i][j]);
            }
            NN_LOG("\n");
        }
        NN_LOG("SamplingNumber[Count : %d] = %d\n", i + 1, data->samplingNumber[i]);
        NN_LOG("DataSize[Count : %d] = %d[Byte]\n", i + 1, state->PollingPacketSize[i]);
    }
}

/////////////////////////////////////////////////////////////////
// Rondeデータ通信
/////////////////////////////////////////////////////////////////
void nnt::menuf::PollingReadRondeData(HidbusState* state, RondeData* data, size_t index)
{
    nn::hidbus::JoyPollingReceivedData PollingDataSet[10];

    auto result = nnt::hidbus::GetPollingRondeData(&PollingDataSet[0], state->packetSize, state->PollingCount, state->handle);
    state->mode = (result.IsSuccess() == true) ? state->mode : ErrorHandling(state, result);

    for (auto cnt = 0; cnt < state->PollingCount; cnt++)
    {
        for (auto i = 0; i < sizeof(state->PollingPacket); i++)
        {
            state->PollingPacket[cnt][i] = PollingDataSet[cnt].data[i + 4];
        }
        state->samplingNumber[cnt] = PollingDataSet[cnt].samplingNumber;
        data->samplingNumber[cnt] = state->samplingNumber[cnt];

        state->PollingPacketSize[cnt] = PollingDataSet[cnt].outSize;
    }
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));

    if (PollingCount[index] < 60)
    {
        PollingCount[index]++;
    }
    else
    {
        NN_LOG("GetJoyPollingReceivedData Result : %s\n", ToStr(result));
        PrintRondePollingData(state, data, index);
        PollingCount[index] = 0;
    }

    if (result.IsSuccess() != true)
    {
        NN_LOG("GetJoyPollingReceivedData Result : %s\n", ToStr(result));
    }
}

///////////////////////////////////////////////
// Menu_InitFin時の処理関数
///////////////////////////////////////////////
void nnt::menuf::FuncInitFin(nnt::Cursol* cursol, HidbusState* state, size_t index)
{
    if (cursol->initfin == nnt::InitFin_GetHandle)
    {
        if (cursol->menustate == nnt::Menustate_ForcusRight)
        {
            state->mode = (GetHidbusHandle(state, index, nn::hidbus::BusType_RightJoyRail) == true) ? HidbusState_GotHandle : state->mode;
        }
        else if (cursol->menustate == nnt::Menustate_ForcusLeft)
        {
            state->mode = (GetHidbusHandle(state, index, nn::hidbus::BusType_LeftJoyRail) == true) ? HidbusState_GotHandle : state->mode;
        }
        else
        {
            // do nothing
        }
    }
    if (cursol->initfin == nnt::InitFin_Init)
    {
        auto result = nn::hidbus::Initialize(state->handle);
        state->mode = (result.IsSuccess() == true) ? state->mode : ErrorHandling(state, result);
        NN_LOG("Initialize Result : %s\n", ToStr(result));
    }
    // Finalize
    if (cursol->initfin == nnt::InitFin_Fin)
    {
        nn::hidbus::Finalize(state->handle);
        state->mode = HidbusState_GotHandle;
        state->isPolling = IsPolling_Disable;

        nn::oe::SetUserInactivityDetectionTimeExtendedUnsafe(false);       // 無操作状態時の自動スリープや画面焼け軽減の開始時間延長を無効化
    }
}

///////////////////////////////////////////////
// Menu_Hidbus時の処理関数
///////////////////////////////////////////////
void nnt::menuf::FuncHidbus(nnt::Cursol* cursol, HidbusState* state)
{
    if (cursol->hidbus == nnt::Hidbus_Enable)
    {
        auto result = nn::hidbus::EnableExternalDevice(true, nnt::hidbus::RondeDeviceId, state->handle);
        state->mode = (result.IsSuccess() == true) ? HidbusState_Enabled : ErrorHandling(state, result);
        NN_LOG("EnableRonde Result : %s\n", ToStr(result));
    }
    // EnableExternalDevice(Disable)
    if (cursol->hidbus == nnt::Hidbus_Disable)
    {
        auto result = nn::hidbus::EnableExternalDevice(false, nnt::hidbus::RondeDeviceId, state->handle);
        state->mode = (result.IsSuccess() == true) ? HidbusState_GotHandle : ErrorHandling(state, result);
        state->isPolling = IsPolling_Disable;
        NN_LOG("DisableRonde Result : %s\n", ToStr(result));

        nn::oe::SetUserInactivityDetectionTimeExtendedUnsafe(false);       // 無操作状態時の自動スリープや画面焼け軽減の開始時間延長を無効化
    }
}

///////////////////////////////////////////////
// Menu_TransMode時の処理関数
///////////////////////////////////////////////
void nnt::menuf::FuncTransMode(nnt::Cursol* cursol, HidbusState* state, RondeData* data, size_t index)
{
    if (cursol->trsmode == nnt::TransMode_EnablePolling6Axis)
    {
        auto result = nn::hidbus::EnableJoyPollingReceiveMode(state->handle,
                                                                state->command,
                                                                sizeof(state->command),
                                                                state->workBuffer,
                                                                sizeof(state->workBuffer),
                                                                nn::hidbus::JoyPollingMode_SixAxisSensorEnable
        );
        memcpy(&state->PollingCmd, &state->command[0], 4);
        state->mode = (result.IsSuccess() == true) ? HidbusState_Polling : ErrorHandling(state, result);
        NN_LOG("EnableJoyPollingReceiveMode Result : %s\n", ToStr(result));

        nn::oe::SetUserInactivityDetectionTimeExtendedUnsafe(true);       // 無操作状態時の自動スリープや画面焼け軽減の開始時間を延長
    }
    // EnableJoyPollingReceiveMode(6軸センサー"無効")
    if (cursol->trsmode == nnt::TransMode_EnablePollingNo6Axis)
    {
        auto result = nn::hidbus::EnableJoyPollingReceiveMode(state->handle,
                                                                state->command,
                                                                sizeof(state->command),
                                                                state->workBuffer,
                                                                sizeof(state->workBuffer),
                                                                nn::hidbus::JoyPollingMode_SixAxisSensorDisable
        );
        memcpy(&state->PollingCmd, &state->command[0], 4);
        state->mode = (result.IsSuccess() == true) ? HidbusState_Polling : ErrorHandling(state, result);
        NN_LOG("EnableJoyPollingReceiveMode Result : %s\n", ToStr(result));

        nn::oe::SetUserInactivityDetectionTimeExtendedUnsafe(true);       // 無操作状態時の自動スリープや画面焼け軽減の開始時間を延長
    }
    // DisableJoyPollingReceiveMode
    if (cursol->trsmode == nnt::TransMode_DisablePolling)
    {
        auto result = nn::hidbus::DisableJoyPollingReceiveMode(state->handle);
        state->mode = (result.IsSuccess() == true) ? HidbusState_Enabled : ErrorHandling(state, result);
        state->isPolling = IsPolling_Disable;
        NN_LOG("DisableJoyPollingReceiveMode Result : %s\n", ToStr(result));

        nn::oe::SetUserInactivityDetectionTimeExtendedUnsafe(false);       // 無操作状態時の自動スリープや画面焼け軽減の開始時間延長を無効化
    }
    // SendAndReceive(OneShot通信)
    if (cursol->trsmode == nnt::TransMode_SendAndReceive)
    {
        if (state->command[0] == nnt::hidbus::CmdList.SCalOsMax[0] && state->command[1] == nnt::hidbus::CmdList.SCalOsMax[1])
        {
            SendAndReceiveRondeCalData(cursol, state, index, &data->scal[0], 0);
        }
        else if (state->command[0] == nnt::hidbus::CmdList.UCalOsMax[0]
              && state->command[1] == nnt::hidbus::CmdList.UCalOsMax[1]
              && state->command[2] == nnt::hidbus::CmdList.UCalOsMax[2])
        {
            SendAndReceiveRondeCalData(cursol, state, index, &data->ucal[0], 0);
        }
        else if (state->command[0] == nnt::hidbus::CmdList.WriteUCalOsMax[0]
              && state->command[1] == nnt::hidbus::CmdList.WriteUCalOsMax[1]
              && state->command[2] == nnt::hidbus::CmdList.WriteUCalOsMax[2])
        {
            SendAndReceiveRondeCalData(cursol, state, index, &data->ucal[0], 1);
        }
        else
        {
            NN_LOG("CommandSize = %d\n", state->packetSize);
            auto result = nnt::hidbus::SendAndReceiveRondeData(state->command, state->packetSize, &state->packet[0], state->handle);
            state->mode = (result.IsSuccess() == true) ? state->mode : ErrorHandling(state, result);
            NN_LOG("SendAndReceive Result : %s\n", ToStr(result));
        }
        memcpy(&state->sendingCmd, &state->command[0], 4);
    }

    // GetPollingDataの実行開始(1フレーム毎に取得するように設定)
    if (cursol->trsmode == nnt::TransMode_GetPollingData)
    {
        state->isPolling = IsPolling_Enable;
    }
    // GetPollingDataの実行停止
    if (cursol->trsmode == nnt::TransMode_StopGetPollingData)
    {
        state->isPolling = IsPolling_Disable;
    }
}

///////////////////////////////////////////////
// Menu_Data時の処理関数
///////////////////////////////////////////////
void nnt::menuf::FuncData(nnt::Cursol* cursol, HidbusState* state)
{
    // UniqueID
    if (cursol->data == nnt::Data_UniqueId)
    {
        memcpy(state->command, nnt::hidbus::CmdList.UniqueId, 4);
        state->packetSize = nnt::hidbus::SizeList.UniqueId;
        NN_LOG("CommandSize = %d\n", state->packetSize);
    }
    // PollingData(歪値、Thermister値)
    if (cursol->data == nnt::Data_PollingFormat)
    {
        memcpy(state->command, nnt::hidbus::CmdList.PollingFormat, 4);
        state->packetSize = nnt::hidbus::SizeList.PollingFormat;
    }
    // FWVersion
    if (cursol->data == nnt::Data_FwVersion)
    {
        memcpy(state->command, nnt::hidbus::CmdList.FwVer, 4);
        state->packetSize = nnt::hidbus::SizeList.FwVer;
    }
    // Cal値
    if (cursol->data == nnt::Data_SCal)
    {
        memcpy(state->command, nnt::hidbus::CmdList.SCalOsMax, 4);
        state->packetSize = nnt::hidbus::SizeList.Calibration;
    }
    // ユーザーCal値
    if (cursol->data == nnt::Data_UCal)
    {
        memcpy(state->command, nnt::hidbus::CmdList.UCalOsMax, 4);
        state->packetSize = nnt::hidbus::SizeList.Calibration;
    }
    // ユーザーCal値(Write)
    if (cursol->data == nnt::Data_WriteUCal)
    {
        memcpy(state->command, nnt::hidbus::CmdList.WriteUCalOsMax, 4);
        state->packetSize = 0;
    }
}

///////////////////////////////////////////////
// Menu_PollingNumSelect時の処理関数
///////////////////////////////////////////////
void nnt::menuf::FuncPollingNumSelect(nnt::Cursol* cursol, HidbusState* state)
{
        state->PollingCount = cursol->pollingNumSelect;
}
