﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{HidbusRonde.cpp,PageSampleHidbusRonde}

    @brief
    JoyConのアサイン切替機能
 */

#include "JoyAssignment.h"

namespace {
    const char* FuncText[] =
    {
        "Swap",
        "Merge",
        "Divide(L)",
        "Divide(R)",
        "SetSingle",
        "SetDual",
        "Disconnect",
        "SupportID",
        "SupportStyle",
    };

    const char* NpadText[] =
    {
        "ID：1",
        "ID：2",
        "ID：3",
        "ID：4",
        "ID：5",
        "ID：6",
        "ID：7",
        "ID：8",
        "ID：HH",
    };

    enum SelectState
    {
        SelectState_SelFunc = 0,
        SelectState_SelFirstNpad,
        SelectState_SelSecondNpad,
        SelectState_SelSupportNpadId,
        SelectState_SelSupportNpadStyle,
        SelectStateCount,
    };

    int IdState = 0;
    nn::hid::NpadIdType npadIds[9];
    int count = 0;

    int StyleState = 0;
    nn::hid::NpadStyleSet style;

    enum SelFunc
    {
        SelFunc_Swap = 0,
        SelFunc_Merge,
        SelFunc_DivideL,
        SelFunc_DivideR,
        SelFunc_SetSingle,
        SelFunc_SetDual,
        SelFunc_Disconnect,
        SelFunc_SupportId,
        SelFunc_SupportStyle,
        SelFuncCount,
    };

    struct AssignmentState
    {
        SelectState selstate;
        SelFunc func;
        size_t firstNpad;
        size_t secondNpad;
        bool SupportNpadIdList[9];
        bool SupportNpadStyleList[5];
        bool isSelected;
    };

    AssignmentState state;

    bool up = false;
    bool down = false;
    bool fix = false;
    bool cancel = false;
}

/////////////////////////////////
// IndexのNpadId変換
/////////////////////////////////
nn::hid::NpadIdType ConvertIdFromIndex(size_t index)
{
    if (index != 8)
    {
        return index;
    }
    else
    {
        return nn::hid::NpadId::Handheld;
    }
}
/////////////////////////////////
// Style変換
/////////////////////////////////
const char* ToString(size_t index)
{
    switch (index)
    {
        case 0:
            return "JoyLeft";
        case 1:
            return "JoyRight";
        case 2:
            return "JoyDual";
        case 3:
            return "FullKey";
        case 4:
            return "HandHeld";
        default:
            return "Undefined";
    }
}

/////////////////////////////////////////////
// サポートするNpadIdのダンプ
/////////////////////////////////////////////
void DumpNpadId()
{
    for (auto i = 0; i < 9; i++)
    {
        if (state.SupportNpadIdList[i])
        {
            npadIds[count] = ConvertIdFromIndex(i);
            count++;
        }
    }
}

//////////////////////////////////////////////
// サポートするNpadIdの設定
//////////////////////////////////////////////
void SelectedSupportNpadId()
{
    if (up == true || down == true)
    {
        state.SupportNpadIdList[IdState] = (state.SupportNpadIdList[IdState] ? false : true);
    }
    if (fix == true)
    {
        if (IdState >= 8)
        {
            IdState = 0;
            state.isSelected = true;
            state.selstate = SelectState_SelFunc;
            DumpNpadId();
        }
        else
        {
            IdState++;
        }
    }
    if (cancel == true)
    {
        if (IdState == 0)
        {
            state.selstate = SelectState_SelFunc;
        }
        else
        {
            IdState--;
        }
    }
}

//////////////////////////////////////////////
// サポートするNpadStyleのダンプ
//////////////////////////////////////////////
void DumpNpadStyle()
{
    for (auto i = 0; i < 5; i++)
    {
        if (state.SupportNpadStyleList[i])
        {
            switch (i)
            {
                case 0:
                    style |= nn::hid::NpadStyleJoyLeft::Mask;
                    break;
                case 1:
                    style |= nn::hid::NpadStyleJoyRight::Mask;
                    break;
                case 2:
                    style |= nn::hid::NpadStyleJoyDual::Mask;
                    break;
                case 3:
                    style |= nn::hid::NpadStyleFullKey::Mask;
                    break;
                case 4:
                    style |= nn::hid::NpadStyleHandheld::Mask;
                    break;
                default:
                    // do nothing
                    break;
            }
        }
    }
}

//////////////////////////////////////////////
// サポートするNpadStyleの設定
//////////////////////////////////////////////
void SelectedSupportNpadStyle()
{
    if (up == true || down == true)
    {
        state.SupportNpadStyleList[StyleState] = (state.SupportNpadStyleList[StyleState] ? false : true);
    }
    if (fix == true)
    {
        if (StyleState >= 4)
        {
            StyleState = 0;
            state.isSelected = true;
            state.selstate = SelectState_SelFunc;
            DumpNpadStyle();
        }
        else
        {
            StyleState++;
        }
    }
    if (cancel == true)
    {
        if (StyleState == 0)
        {
            state.selstate = SelectState_SelFunc;
        }
        else
        {
            StyleState--;
        }
    }
}

/////////////////////////////
// アサイン状況の選択
/////////////////////////////
void SelectedNpadAssign()
{
    up = false;
    down = false;
    fix = false;
    cancel = false;
    state.isSelected = false;

    count = 0;

    for (auto i = 0; i < 9; i++)
    {
            npadIds[i] = 0;
    }
    style.Reset();

    // 接続中のコントローラのボタンの押下げ状況を確認します
    for (std::vector<nns::hidfw::hid::Controller*>::iterator it = gController.GetConnectedControllerList().begin();
            it != gController.GetConnectedControllerList().end();
            ++it)
    {
        if ((*it)->IsTrigger(nn::hid::NpadButton::StickLDown::Mask | nn::hid::NpadButton::StickRDown::Mask | nn::hid::NpadButton::Down::Mask | nn::hid::NpadButton::B::Mask))
        {
            down = true;
        }
        if ((*it)->IsTrigger(nn::hid::NpadButton::StickLUp::Mask | nn::hid::NpadButton::StickRUp::Mask | nn::hid::NpadButton::Up::Mask | nn::hid::NpadButton::X::Mask))
        {
            up = true;
        }
        if ((*it)->IsTrigger(nn::hid::NpadButton::L::Mask | nn::hid::NpadButton::R::Mask))
        {
            fix = true;
        }
        if ((*it)->IsTrigger(nn::hid::NpadButton::ZL::Mask | nn::hid::NpadButton::ZR::Mask))
        {
            cancel = true;
        }
    }

    switch (state.selstate)
    {
        case SelectState_SelFunc:
            if (down == true)
            {
                if (state.func == SelFunc_Swap)
                {
                    state.func = SelFunc_Merge;
                }
                else if (state.func == SelFunc_Merge)
                {
                    state.func = SelFunc_DivideL;
                }
                else if (state.func == SelFunc_DivideL)
                {
                    state.func = SelFunc_DivideR;
                }
                else if (state.func == SelFunc_DivideR)
                {
                    state.func = SelFunc_SetSingle;
                }
                else if (state.func == SelFunc_SetSingle)
                {
                    state.func = SelFunc_SetDual;
                }
                else if (state.func == SelFunc_SetDual)
                {
                    state.func = SelFunc_Disconnect;
                }
                else if (state.func == SelFunc_Disconnect)
                {
                    state.func = SelFunc_SupportId;
                }
                else if (state.func == SelFunc_SupportId)
                {
                    state.func = SelFunc_SupportStyle;
                }
                else if (state.func == SelFunc_SupportStyle)
                {
                    state.func = SelFunc_Swap;
                }
                else
                {
                    // do nothing
                }
            }
            if (up == true)
            {
                if (state.func == SelFunc_Swap)
                {
                    state.func = SelFunc_SupportStyle;
                }
                else if (state.func == SelFunc_Merge)
                {
                    state.func = SelFunc_Swap;
                }
                else if (state.func == SelFunc_DivideL)
                {
                    state.func = SelFunc_Merge;
                }
                else if (state.func == SelFunc_DivideR)
                {
                    state.func = SelFunc_DivideL;
                }
                else if (state.func == SelFunc_SetSingle)
                {
                    state.func = SelFunc_DivideR;
                }
                else if (state.func == SelFunc_SetDual)
                {
                    state.func = SelFunc_SetSingle;
                }
                else if (state.func == SelFunc_Disconnect)
                {
                    state.func = SelFunc_SetDual;
                }
                else if (state.func == SelFunc_SupportId)
                {
                    state.func = SelFunc_Disconnect;
                }
                else if (state.func == SelFunc_SupportStyle)
                {
                    state.func = SelFunc_SupportId;
                }
                else
                {
                    // do nothing
                }
            }
            if (fix == true && state.func != SelFunc_SupportId && state.func != SelFunc_SupportStyle)
            {
                state.selstate = SelectState_SelFirstNpad;
            }
            else if (fix == true && state.func == SelFunc_SupportId)
            {
                state.selstate = SelectState_SelSupportNpadId;
            }
            else if (fix == true && state.func == SelFunc_SupportStyle)
            {
                state.selstate = SelectState_SelSupportNpadStyle;
            }
            else
            {
                // do nothing
            }

            if (cancel == true)
            {
                // do nothing
            }
            break;
        case SelectState_SelFirstNpad:
            if (down == true)
            {
                state.firstNpad = ((state.firstNpad < 8) ? (state.firstNpad + 1) : 0);
            }
            if (up == true)
            {
                state.firstNpad = ((state.firstNpad > 0) ? (state.firstNpad - 1) : 8);
            }
            if (fix == true)
            {
                if (state.func == SelFunc_DivideL
                 || state.func == SelFunc_DivideR
                 || state.func == SelFunc_SetSingle
                 || state.func == SelFunc_SetDual
                 || state.func == SelFunc_Disconnect)
                {
                    state.isSelected = true;
                    state.selstate = SelectState_SelFunc;
                }
                else
                {
                    state.selstate = SelectState_SelSecondNpad;
                }
            }
            if (cancel == true)
            {
                state.selstate = SelectState_SelFunc;
            }
            break;
        case SelectState_SelSecondNpad:
            if (down == true)
            {
                state.secondNpad = ((state.secondNpad < 8) ? (state.secondNpad + 1) : 0);
            }
            if (up == true)
            {
                state.secondNpad = ((state.secondNpad > 0) ? (state.secondNpad - 1) : 8);
            }
            if (fix == true)
            {
                state.isSelected = true;
                state.selstate = SelectState_SelFunc;
            }
            if (cancel == true)
            {
                state.selstate = SelectState_SelFirstNpad;
            }
            break;
        case SelectState_SelSupportNpadId:
            SelectedSupportNpadId();
            break;
        case SelectState_SelSupportNpadStyle:
            SelectedSupportNpadStyle();
            break;
        default:
            break;
    }
} // NOLINT(impl/function_size)

/////////////////////////////////
// 選択状況に応じたAPIの実行
/////////////////////////////////
void DoNpadAssign()
{
    if (state.isSelected)
    {
        auto firstId = ConvertIdFromIndex(state.firstNpad);
        auto secondId = ConvertIdFromIndex(state.secondNpad);

        switch (state.func)
        {
            case SelFunc_Swap:
                nn::hid::SwapNpadAssignment(firstId, secondId);
                break;
            case SelFunc_Merge:
                nn::hid::MergeSingleJoyAsDualJoy(firstId, secondId);
                break;
            case SelFunc_DivideL:
                nn::hid::SetNpadJoyAssignmentModeSingle(firstId, nn::hid::NpadJoyDeviceType_Left);
                break;
            case SelFunc_DivideR:
                nn::hid::SetNpadJoyAssignmentModeSingle(firstId, nn::hid::NpadJoyDeviceType_Right);
                break;
            case SelFunc_SetSingle:
                nn::hid::SetNpadJoyAssignmentModeSingle(firstId);
                break;
            case SelFunc_SetDual:
                nn::hid::SetNpadJoyAssignmentModeDual(firstId);
                break;
            case SelFunc_Disconnect:
                nn::hid::DisconnectNpad(firstId);
                break;
            case SelFunc_SupportId:
                nn::hid::SetSupportedNpadIdType(npadIds, count);
                break;
            case SelFunc_SupportStyle:
                nn::hid::SetSupportedNpadStyleSet(style);
                break;
            default:
                break;
        }
    }
}

/////////////////////////////////
// 選択状況に応じた描画
/////////////////////////////////
void DrawNpadAssignMenu(float x, float y)
{
    switch (state.selstate)
    {
        case SelectState_SelSecondNpad:
            gTextWriter.SetCursor(x + 300, y + 22);
            gTextWriter.Print("⇒　%s", NpadText[state.secondNpad]);
            // through down
        case SelectState_SelFirstNpad:
            gTextWriter.SetCursor(x + 200, y + 22);
            gTextWriter.Print("⇒　%s", NpadText[state.firstNpad]);
            // through down
        case SelectState_SelFunc:
            gTextWriter.SetCursor(x, y);
            gTextWriter.Print("<<Change Npad Assignment Menu>>");
            gTextWriter.SetCursor(x + 50, y + 22);
            gTextWriter.Print("%s", FuncText[state.func]);
            break;
        case SelectState_SelSupportNpadId:
            for (auto index = 0; index < 9; index++)
            {
                if (IdState >= index)
                {
                    gTextWriter.SetCursor(x + 150 + (100 * index), y + 22);
                    gTextWriter.Print("⇒　%d：%s", index + 1, (state.SupportNpadIdList[index] ? "ON" : "OFF"));
                }
            }
            gTextWriter.SetCursor(x, y);
            gTextWriter.Print("<<Change Npad Assignment Menu>>");
            gTextWriter.SetCursor(x + 50, y + 22);
            gTextWriter.Print("%s", FuncText[state.func]);
            break;
        case SelectState_SelSupportNpadStyle:
            for (auto index = 0; index < 5; index++)
            {
                if (StyleState >= index)
                {
                    gTextWriter.SetCursor(x + 200 + (170 * index), y + 22);
                    gTextWriter.Print("⇒　%s：%s", ToString(index), (state.SupportNpadStyleList[index] ? "ON" : "OFF"));
                }
            }
            gTextWriter.SetCursor(x, y);
            gTextWriter.Print("<<Change Npad Assignment Menu>>");
            gTextWriter.SetCursor(x + 50, y + 22);
            gTextWriter.Print("%s", FuncText[state.func]);
            break;
        default:
            break;
    }
}
