﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nnt/nntest.h>

#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>

#include <nn/hidbus/hidbus.h>
#include <nns/hidbus/hidbus_Ronde.h>

#include <nn/irsensor.h>
#include <nn/irsensor/irsensor_HeartRateProcessor.h>

namespace
{
    // PollingMode 時に渡すバッファ
    NN_ALIGNAS(nn::os::MemoryPageSize) char s_Buffer[nn::os::MemoryPageSize];

    // hidbus の handle (テストで共通で使用)
    nn::hidbus::BusHandle g_HidbusHandle = {};

    // IR Sensor のカメラハンドル
    nn::irsensor::IrCameraHandle g_IrCameraHandle;

    // HeartRate の Config
    nn::irsensor::HeartRateProcessorConfig g_HeartRateCaptureConfig;

    // HeartRate のバッファ
    nn::irsensor::HeartRateProcessorWorkBuffer s_HeartRateBuffer;

    // NpadID のリスト
    const nn::hid::NpadIdType NpadIds[] =
    {
        nn::hid::NpadId::No1,
    };

    // Npad 周りのセットアップ関数
    void SetUpNpad()
    {
        // Npad の初期化
        nn::hid::InitializeNpad();

        // Style の設定
        nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleJoyRight::Mask);

        // サポートする Npad ID のリスト
        nn::hid::SetSupportedNpadIdType(NpadIds, NN_ARRAY_SIZE(NpadIds));
    }

    nn::TimeSpan totalSwitchTimeFromRondeToIr = 0;
    nn::TimeSpan totalSwitchTimeFromIrToRonde = 0;
}

/**
* RONDE -> 心拍測定、 心拍測定 -> RONDE の遷移をテストします。
* 必ず右ジョイコンを NpadID 1 として接続した状態で、右ジョイコンに RONDE を接続して実行してください。
* エラーハンドリングが雑に ABORT になっていたりするため、電波環境により失敗することがあります。
*/
TEST(SwitchFronRondeToIr, SwitchTest)
{
    SetUpNpad();

    for (int i = 0; i < 10; i++)
    {
        // ハンドルの取得
        while (!nn::hidbus::GetBusHandle(&g_HidbusHandle, nn::hid::NpadId::No1, nn::hidbus::BusType_RightJoyRail))
        {
            NN_LOG("[hidbus] Please Connect Joy-Con\n");

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        // 初期化
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::Initialize(g_HidbusHandle));

        // Enable
        NN_ABORT_UNLESS_RESULT_SUCCESS(nns::hidbus::EnableRonde(g_HidbusHandle, s_Buffer, sizeof(s_Buffer)));

        // 100 回ほど値を取る
        int rondeSensorCount = 0;
        while (rondeSensorCount < 100)
        {
            nns::hidbus::RondeSensorData sensorData;
            auto result = nns::hidbus::GetRondeSensorData(&sensorData, g_HidbusHandle);
            if (result.IsSuccess())
            {
                NN_LOG("Ronde Sensor Value = %d ", sensorData.data);
            }

            int16_t thermistor;
            auto thermistorResult = nns::hidbus::GetRondeThermisterData(&thermistor, g_HidbusHandle);
            if (thermistorResult.IsSuccess())
            {
                NN_LOG("Thermistor = %d\n", thermistor);
            }
            rondeSensorCount++;
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
        }

        // Disable する前から時間を計測
        auto startTick = nn::os::GetSystemTick();
        nns::hidbus::DisableRonde(g_HidbusHandle);

        // 拡張デバイス機能の終了
        nn::hidbus::Finalize(g_HidbusHandle);

        // IR へ切り替え
        //IR カメラのハンドルの取得
        g_IrCameraHandle = nn::irsensor::GetIrCameraHandle(nn::hid::NpadId::No1);

        //IR カメラの初期化
        nn::irsensor::Initialize(g_IrCameraHandle);

        // IrSensor を HeartRate として使用
        g_HeartRateCaptureConfig.measurementDuration = ::nn::TimeSpanType::FromSeconds(30);
        g_HeartRateCaptureConfig.measurementDurationExtension = ::nn::TimeSpanType::FromSeconds(5);
        nn::irsensor::RunHeartRateProcessor(g_IrCameraHandle, g_HeartRateCaptureConfig, s_HeartRateBuffer);

        // モードに入るの待ち
        while (nn::irsensor::GetImageProcessorStatus(g_IrCameraHandle) != nn::irsensor::ImageProcessorStatus_Running)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));
        }

        nn::irsensor::HeartRateProcessorState heartRateProcessorState;

        while (nn::irsensor::GetHeartRateProcessorState(&heartRateProcessorState, g_IrCameraHandle).IsFailure())
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));
        }

        // Ronde -> IR の時間の表示
        auto switchTimeFromRondeToIr = (nn::os::GetSystemTick() - startTick).ToTimeSpan();
        NN_LOG("Ronde -> IR sensor switching time : %d ms\n", switchTimeFromRondeToIr.GetMilliSeconds());
        totalSwitchTimeFromRondeToIr += switchTimeFromRondeToIr;


        // 適当に 1s 程待つ
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

        // IrSensor -> hidbus を計測
        startTick = nn::os::GetSystemTick();

        // IrSensor の終了処理
        nn::irsensor::StopImageProcessor(g_IrCameraHandle);
        nn::irsensor::Finalize(g_IrCameraHandle);

        // hidbus の初期化
        // ハンドルの取得
        while (!nn::hidbus::GetBusHandle(&g_HidbusHandle, nn::hid::NpadId::No1, nn::hidbus::BusType_RightJoyRail))
        {
            NN_LOG("[hidbus] Please Connect Joy-Con\n");

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        // 初期化
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::Initialize(g_HidbusHandle));

        // Enable
        NN_ABORT_UNLESS_RESULT_SUCCESS(nns::hidbus::EnableRonde(g_HidbusHandle, s_Buffer, sizeof(s_Buffer)));

        // Ir -> Ronde の時間の表示
        auto switchTimeFromIrToRonde = (nn::os::GetSystemTick() - startTick).ToTimeSpan();
        NN_LOG("IR sensor -> Ronde switching time : %d ms\n", switchTimeFromIrToRonde.GetMilliSeconds());
        totalSwitchTimeFromIrToRonde += switchTimeFromIrToRonde;

        // hidbus の終了
        nns::hidbus::DisableRonde(g_HidbusHandle);
        nn::hidbus::Finalize(g_HidbusHandle);
    }

    // 10 回の平均を表示
    NN_LOG("[average] Ronde -> IR sensor switching time : %d ms\n", totalSwitchTimeFromRondeToIr.GetMilliSeconds() / 10);
    NN_LOG("[average] IR sensor -> Ronde switching time : %d ms\n", totalSwitchTimeFromIrToRonde.GetMilliSeconds() / 10);
}
