﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>

#include <nn/hid/hid_SixAxisSensor.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>

#include "SixAxisSensorLogger.h"

namespace {

const char* MountName = "sd";

} // namespace

SixAxisSensorLogger::SixAxisSensorLogger() NN_NOEXCEPT
    : m_FileHandle()
    , m_FileOffset(0)
    , m_FileName()
    , m_IsSdCardInserted(false)
    , m_StartTick(nn::os::GetSystemTick())
{
    // 何もしない
}

void SixAxisSensorLogger::Initialize(const char* pFileName) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pFileName);

    strncpy(m_FileName, MountName, FileNameCountMax);
    strncat(m_FileName, ":/", FileNameCountMax);
    strncat(m_FileName, pFileName, FileNameCountMax);

    nn::Result result;

    NN_LOG("Mount SD card as \"%s:/\"\n", MountName);

    // SD カードをマウント名 MountName でマウントします。
    result = nn::fs::MountSdCardForDebug(MountName);
    if (nn::fs::ResultSdCardAccessFailed::Includes(result))
    {
        // SD カードが挿入されていません。
        NN_LOG("SD card not found.");
        return;
    }

    // 上記以外の失敗は実装ミスのため、必ずアボートする
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // SD カードの挿入を確認
    m_IsSdCardInserted = true;

    // ファイルが存在しない場合は作成します。
    {
        nn::fs::DirectoryEntryType directoryEntryType;
        result = nn::fs::GetEntryType(&directoryEntryType, m_FileName);
        if (nn::fs::ResultPathNotFound().Includes(result))
        {
            // 対象ファイルが存在しません。
            // 対象ファイルを作成します。
            NN_LOG("Create \"%s\"\n", m_FileName);
            result = nn::fs::CreateFile(m_FileName, 0);
            if (nn::fs::ResultPathNotFound::Includes(result))
            {
                // 対象ファイルが存在しません。
                // 親ディレクトリが必ず存在する場合は、このエラーハンドリングは不要です。
            }
            else if (nn::fs::ResultPathAlreadyExists::Includes(result) ||
                     nn::fs::ResultTargetLocked::Includes(result))
            {
                // 対象ファイルが既に存在しています。
                // ファイルが既に存在していても構わない場合は、このエラーハンドリングは不要です。
                // エラーハンドリングしない場合、ファイルのサイズが FileSize である保証が無いことに注意してください。
                // 必要であれば対象ファイルを削除してから再度作成してください。
            }
            else if (nn::fs::ResultUsableSpaceNotEnough::Includes(result))
            {
                // 空き領域が不足しています。
                // SD カードの空き領域を確認してください。
                NN_ABORT("Usable space not enough.\n");
                return;
            }
            // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
            // これ以上のエラーハンドリングは不要です。
        }
        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
        // これ以上のエラーハンドリングは不要です。
    }

    // ファイル末尾に追記します。
    {
        NN_LOG("Open \"%s\"\n", m_FileName);
        result = nn::fs::OpenFile(&m_FileHandle,
                                  m_FileName,
                                  nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);

        if (nn::fs::ResultPathNotFound::Includes(result))
        {
            // 対象ファイルが存在しません。
            // 存在するファイルしか開かない場合は、このエラーハンドリングは不要です。
        }
        else if (nn::fs::ResultTargetLocked::Includes(result))
        {
            // 対象ファイルが既にオープンされています。
            // ファイルが既にオープンされている可能性が無い場合は、このエラーハンドリングは不要です。
        }
        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
        // これ以上のエラーハンドリングは不要です。

        // ファイルサイズの取得に失敗した場合はライブラリ内でアボートするため、エラーハンドリングは不要です。
        (void)nn::fs::GetFileSize(&m_FileOffset, m_FileHandle);
    }
}

void SixAxisSensorLogger::Finalize() NN_NOEXCEPT
{
    NN_LOG("Close \"%s\"\n", m_FileName);
    nn::fs::CloseFile(m_FileHandle);

    // アンマウントします。
    NN_LOG("Unmount \"%s:/\"\n", MountName);
    nn::fs::Unmount(MountName);
}

void SixAxisSensorLogger::ResetTick() NN_NOEXCEPT
{
    m_StartTick = nn::os::GetSystemTick();
}

void SixAxisSensorLogger::AppendLine(void* pState) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pState);

    if (m_IsSdCardInserted == false)
    {
        // SD カードが挿入されていないので何もしません。
        // NN_LOG("Do nothing since sd card isn't inserted.\n");
        return;
    }

    char writeBuffer[1024];
    auto pSixAxisSensorState = reinterpret_cast<::nn::hid::SixAxisSensorState*>(pState);

    int length = nn::util::SNPrintf(writeBuffer,
                                    sizeof(writeBuffer),
                                    // timestamp[ms], acc, gyro, angle, direction
                                    "%lld, %f, %f, %f, %f, %f, %f, %f, %f, %f, %f, %f, %f, %f, %f, %f, %f, %f, %f\n",
                                    (nn::os::GetSystemTick() - m_StartTick).ToTimeSpan().GetMilliSeconds(),
                                    pSixAxisSensorState->acceleration.x,
                                    pSixAxisSensorState->acceleration.y,
                                    pSixAxisSensorState->acceleration.z,
                                    pSixAxisSensorState->angularVelocity.x,
                                    pSixAxisSensorState->angularVelocity.y,
                                    pSixAxisSensorState->angularVelocity.z,
                                    pSixAxisSensorState->angle.x,
                                    pSixAxisSensorState->angle.y,
                                    pSixAxisSensorState->angle.z,
                                    pSixAxisSensorState->direction.x.x,
                                    pSixAxisSensorState->direction.x.y,
                                    pSixAxisSensorState->direction.x.z,
                                    pSixAxisSensorState->direction.y.x,
                                    pSixAxisSensorState->direction.y.y,
                                    pSixAxisSensorState->direction.y.z,
                                    pSixAxisSensorState->direction.z.x,
                                    pSixAxisSensorState->direction.z.y,
                                    pSixAxisSensorState->direction.z.z);

    // 確認用にコンソールに SD カードに書き込まれるデータをログ出力します。
    // NN_LOG("%s", writeBuffer);
    auto result = nn::fs::WriteFile(m_FileHandle,
                                    m_FileOffset,
                                    writeBuffer,
                                    length,
                                    nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));

    if (nn::fs::ResultUsableSpaceNotEnough::Includes(result))
    {
        // 空き領域が不足しています。
        // 書き込みサイズが想定範囲内の値となっているか、SD カードの空き領域は十分かを確認してください。
        NN_ABORT("Usable space not enough.\n");
        return;
    }
    // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
    // これ以上のエラーハンドリングは不要です。

    m_FileOffset += length;
}

