﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/util/util_Vector.h>

#include "SpinMotionDetector.h"
#include "SixAxisSensorGestureProcessor.h"

namespace {

enum Sign
{
    Sign_Positive,
    Sign_Negative,
};

Sign GetSign(const float& value) NN_NOEXCEPT
{
    return (value < 0.f) ? Sign::Sign_Negative : Sign::Sign_Positive;
}

} // namespace

SpinMotionDetector::SpinMotionDetector() NN_NOEXCEPT
    : m_SixAxisSensorRawState()
    , m_Configs()
    , m_ThresholdSpinCount(8)
    , m_ThresholdSignInversionSamplingNumberDelta(10) // TODO: OPTIMIZE
    , m_ThresholdTimeoutSamplingNumber(50) // TODO: OPTIMIZE
    , m_ThresholdGyroLengthSquared(0.05f) // TODO: OPTIMIZE
{
    // 何もしない
}

SpinMotionDetector::~SpinMotionDetector() NN_NOEXCEPT
{
    // 何もしない
}

void SpinMotionDetector::Reset() NN_NOEXCEPT
{
    m_Lifo.Clear();
}

void SpinMotionDetector::Update(const SixAxisSensorRawState& rawState) NN_NOEXCEPT
{
    // タイムアウトチェック
    for (auto& config : m_Configs)
    {
        if (rawState.samplingNumber - config.samplingNumber > m_ThresholdTimeoutSamplingNumber)
        {
            config.count = 0;
            config.samplingNumber = rawState.samplingNumber;
            // NN_LOG("Timeout, count cleared\n");
        }
    }

    // 静止時を弾くため
    if (rawState.gyroLengthSquared < m_ThresholdGyroLengthSquared)
    {
        return;
    }

    // X 軸
    auto pConfig = &m_Configs[0];
    if (rawState.samplingNumber - pConfig->samplingNumber > m_ThresholdSignInversionSamplingNumberDelta && // チャタリングを弾くため
        GetSign(rawState.acc.GetX()) != GetSign(m_SixAxisSensorRawState.acc.GetX()))
    {
        pConfig->count++;

        // NN_LOG("X Sign changed: %lld, %f, %lld, %d\n"
        //     , rawState.samplingNumber
        //     , rawState.gyroLengthSquared
        //     , rawState.samplingNumber - pConfig->samplingNumber
        //     , pConfig->count
        // );

        pConfig->samplingNumber = rawState.samplingNumber;
    }

    // Y 軸
    pConfig = &m_Configs[1];
    if (rawState.samplingNumber - pConfig->samplingNumber > m_ThresholdSignInversionSamplingNumberDelta && // チャタリングを弾くため
        GetSign(rawState.acc.GetY()) != GetSign(m_SixAxisSensorRawState.acc.GetY()))
    {
        pConfig->count++;

        // NN_LOG("\tY Sign changed: %lld, %f, %lld, %d\n"
        //     , rawState.samplingNumber
        //     , rawState.gyroLengthSquared
        //     , rawState.samplingNumber - pConfig->samplingNumber
        //     , pConfig->count
        // );

        pConfig->samplingNumber = rawState.samplingNumber;
    }

    // LIFO
    if (IsSpinMotionDetected())
    {
        for (auto& config : m_Configs)
        {
            config.count = 0;
        }

        // NN_LOG("LIFO APPEND : %lld, %lld\n"
        //     , m_SamplingNumber
        //     , rawState.samplingNumber
        // );

        m_Lifo.Append(
            MotionState({
                m_SamplingNumber,
                rawState.samplingNumber,
                static_cast<MotionType>(MotionType_Spin)
            })
        );
        m_SamplingNumber++;

    }

    m_SixAxisSensorRawState = rawState;
}

int SpinMotionDetector::GetMotions(MotionState* pOutStates, int count) NN_NOEXCEPT
{
    return m_Lifo.Read(pOutStates, count);
}

bool SpinMotionDetector::IsSpinMotionDetected() const NN_NOEXCEPT
{
    return (m_Configs[0].count > m_ThresholdSpinCount) &&
           (m_Configs[1].count > m_ThresholdSpinCount);
}
