﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <type_traits>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_StaticAssert.h>
#include <nn/util/util_Vector.h>
#include <nn/TargetConfigs/build_Base.h>

#include "../../../../../Programs/Iris/Sources/Libraries/hid/detail/hid_RingLifo.h"

//!< 生値を表す構造体
struct SixAxisSensorRawState
{
    int64_t samplingNumber;

    ::nn::util::Vector3f acc;
    ::nn::util::Vector3f gyro;

    ::nn::util::Vector3f accDelta;
    ::nn::util::Vector3f gyroDelta;

    ::nn::util::Vector3f gravity;

    ::nn::util::Vector3f accNormalized;
    ::nn::util::Vector3f gyroNormalized;

    float accLengthSquared;
    float gyroLengthSquared;

    void Reset() NN_NOEXCEPT
    {
        samplingNumber = 0;

        ::nn::util::VectorZero(&acc);
        ::nn::util::VectorZero(&gyro);

        ::nn::util::VectorZero(&accDelta);
        ::nn::util::VectorZero(&gyroDelta);

        ::nn::util::VectorZero(&gravity);

        ::nn::util::VectorZero(&accNormalized);
        ::nn::util::VectorZero(&gyroNormalized);

        accLengthSquared = 0.f;
        gyroLengthSquared = 0.f;
    }

    void Print() const NN_NOEXCEPT
    {
        NN_LOG(
            "%lld, "
            "%f, %f, %f, "
            "%f, %f, %f, "
            "%f, %f, %f, "
            "%f, %f, %f, "
            "%f, %f, %f, "
            "%f, %f, %f, "
            "%f, %f, %f, "
            "%f, %f\n"
            , samplingNumber

            , acc.GetX()
            , acc.GetY()
            , acc.GetZ()

            , gyro.GetX()
            , gyro.GetY()
            , gyro.GetZ()

            , accDelta.GetX()
            , accDelta.GetY()
            , accDelta.GetZ()

            , gyroDelta.GetX()
            , gyroDelta.GetY()
            , gyroDelta.GetZ()

            , gravity.GetX()
            , gravity.GetY()
            , gravity.GetZ()

            , accNormalized.GetX()
            , accNormalized.GetY()
            , accNormalized.GetZ()

            , gyroNormalized.GetX()
            , gyroNormalized.GetY()
            , gyroNormalized.GetZ()

            , accLengthSquared
            , gyroLengthSquared
        );

    }
};

const int SixAxisSensorRawStateCountMax = 256; //!< 内部的に保持できる生値のサンプル数

class SixAxisSensorRawStateLifo final :
    public ::nn::hid::detail::RingLifo<SixAxisSensorRawState, SixAxisSensorRawStateCountMax>
{
public:
    SixAxisSensorRawStateLifo() NN_NOEXCEPT { /* 何もしない */ }
};

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_CLANG
NN_STATIC_ASSERT(::std::is_standard_layout<SixAxisSensorRawStateLifo>::value);
#endif

// =============================================

// モーションの組み合わせで表現されるジェスチャを表す列挙体です。
enum GestureType
{
    GestureType_Peak,

    GestureType_Shake,

    GestureType_RotatePlusX,
    GestureType_RotatePlusY,
    GestureType_RotatePlusZ,
    GestureType_RotateMinusX,
    GestureType_RotateMinusY,
    GestureType_RotateMinusZ,

    GestureType_Throw,

    GestureType_PushPlusX,
    GestureType_PushPlusY,
    GestureType_PushPlusZ,
    GestureType_PushMinusX,
    GestureType_PushMinusY,
    GestureType_PushMinusZ,

    GestureType_Spin,

    GestureType_Count,
};

// ジェスチャの入力状態です。
struct GestureState
{
    int64_t samplingNumber;
    int64_t gestureSamplingNumber; //!< ジェスチャーが起きたサンプリング番号
    GestureType gestureType;
};

const int GestureStateCountMax = 16;

class GestureStateLifo final :
    public ::nn::hid::detail::RingLifo<GestureState, GestureStateCountMax>
{
public:
    GestureStateLifo() NN_NOEXCEPT { /* 何もしない */ }
};

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_CLANG
NN_STATIC_ASSERT(::std::is_standard_layout<GestureStateLifo>::value);
#endif

const int MotionDetectorCountMax = 8; //!< 登録できる MotionDetector の最大数

const float FromRadianToDegreeCoef = 180.f / ::nn::util::FloatPi;
