﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <string>
#include <vector>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_NpadJoyCommon.h>
#include <nn/hid/hid_NpadHandheld.h>
#include <nn/hid/hid_NpadSixAxisSensor.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/system/hid_SixAxisSensorCalibration.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/TargetConfigs/build_Platform.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#include "../Common/ApplicationHeap.h"
#include "../Common/Color.h"
#include "../Common/FontSystem.h"
#include "../Common/GraphicsSystem.h"
#include "../Common/WindowMessage.h"

#include "INpadStyleSixAxisSensor.h"
#include "FullKeySixAxisSensor.h"
#include "JoyDualSixAxisSensor.h"
#include "JoyLeftSixAxisSensor.h"
#include "JoyRightSixAxisSensor.h"
#include "HandheldSixAxisSensor.h"
#include "SixAxisSensorLogger.h"
#include "SixAxisSensorPointer.h"

// #define LOGGING // ロギング機能
namespace {

SixAxisSensorLogger g_Logger;

const size_t ApplicationHeapSize = 128 * 1024 * 1024;

const int FrameRate = 60;

template<typename T, std::size_t U>
std::size_t GetArrayLength(const T (&staticArray)[U])
{
    NN_UNUSED(staticArray);
    return U;
}

const nn::hid::NpadIdType NpadIds[] = {nn::hid::NpadId::No1,
                                       nn::hid::NpadId::No2,
                                       nn::hid::NpadId::No3,
                                       nn::hid::NpadId::No4,
                                       nn::hid::NpadId::Handheld };
const int NpadIdCountMax = static_cast<int>(GetArrayLength(NpadIds));

const nn::util::Unorm8x4 ColorArray[]  = {Color::Red,
                                          Color::Orange,
                                          Color::Yellow,
                                          Color::Green};

const int StyleIndices[] = {nn::hid::NpadStyleFullKey::Index,
                            nn::hid::NpadStyleJoyDual::Index,
                            nn::hid::NpadStyleJoyLeft::Index,
                            nn::hid::NpadStyleJoyRight::Index,
                            nn::hid::NpadStyleHandheld::Index};

const int StyleIndicesCountMax = static_cast<int>(GetArrayLength(StyleIndices));
int g_Mode = 0;

const int CommunicationModeCountMax = 4;
int g_CommunicationModeIndex = 3;
const ::nn::hid::NpadCommunicationMode g_CommunicationModes[] = {
    ::nn::hid::NpadCommunicationMode_5ms,
    ::nn::hid::NpadCommunicationMode_10ms,
    ::nn::hid::NpadCommunicationMode_15ms,
    ::nn::hid::NpadCommunicationMode_Default
};

bool IsStyleToDraw(const nn::hid::NpadStyleSet& style)
{
    return style.Test(StyleIndices[g_Mode]);
}

bool IsStyleToControl(const nn::hid::NpadStyleSet& style)
{
    return (style.Test<nn::hid::NpadStyleJoyDual>() ||
            style.Test<nn::hid::NpadStyleJoyLeft>() ||
            style.Test<nn::hid::NpadStyleJoyRight>());
}

void SetModeName(std::string* pOutName) NN_NOEXCEPT
{
    switch(g_Mode)
    {
    case 0:
        *pOutName = "FullKey";
        break;
    case 1:
        *pOutName = "JoyDual";
        break;
    case 2:
        *pOutName = "JoyLeft";
        break;
    case 3:
        *pOutName = "JoyRight";
        break;
    case 4:
        *pOutName = "Handheld";
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void SetCommunicationModeName(std::string* pOutName) NN_NOEXCEPT
{
    switch (g_CommunicationModeIndex)
    {
    case 0:
        *pOutName = "5ms";
        break;
    case 1:
        *pOutName = "10ms";
        break;
    case 2:
        *pOutName = "15ms";
        break;
    case 3:
        *pOutName = "Default";
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void WriteMode(nn::gfx::util::DebugFontTextWriter* pTextWriter) NN_NOEXCEPT
{
    const nn::util::Unorm8x4& textColor = Color::White;
    std::string mode;

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(0.8f, 0.8f);
    pTextWriter->SetCursor(0.0f, 0.0f);

    SetModeName(&mode);
    pTextWriter->Print("DrawMode = %s", mode.c_str());
}

void WriteCommunicationMode(nn::gfx::util::DebugFontTextWriter* pTextWriter) NN_NOEXCEPT
{
    const nn::util::Unorm8x4& textColor = Color::White;
    std::string mode;

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(0.8f, 0.8f);
    pTextWriter->SetCursor(200.0f, 0.0f);

    SetCommunicationModeName(&mode);
    pTextWriter->Print("CommunicationMode = %s", mode.c_str());
}

/**
 * @brief       描画する座標系の識別子です。
 */
enum CoordinateId
{
    CoordinateId_AccelerometerXy = 0,
    CoordinateId_AccelerometerYz,
    CoordinateId_AccelerometerZx,
    CoordinateId_AngularVelocityXy,
    CoordinateId_AngularVelocityYz,
    CoordinateId_AngularVelocityZx,
    CoordinateId_AngleX,
    CoordinateId_AngleY,
    CoordinateId_AngleZ,
    CoordinateId_DirectionXy,
    CoordinateId_DirectionYz,
    CoordinateId_DirectionZx,
};

//!< 座標系の原点座標を取得します。
void GetCoordinateOrigin(nn::util::Float2* pOutOrigin, CoordinateId id)
{
    const float X = 350;
    const float Y = 150;
    const float DeltaX = 250;
    const float DeltaY = 230;

    pOutOrigin->x = X + static_cast<int>(id) / 3 * DeltaX;
    pOutOrigin->y = Y + (static_cast<int>(id) % 3) * DeltaY;
}

//!< 座標系の名前を取得します。
void GetCoordinateName(char* pOutAxis1,
                       char* pOutAxis2,
                       ::std::string* pOutTitle,
                       CoordinateId id)
{
    switch (id)
    {
    case CoordinateId_AccelerometerXy :
        *pOutTitle = "AccelerometerXy";
        *pOutAxis1 = 'x';
        *pOutAxis2 = 'y';
        break;
    case CoordinateId_AccelerometerYz :
        *pOutTitle = "AccelerometerYz";
        *pOutAxis1 = 'y';
        *pOutAxis2 = 'z';
        break;
    case CoordinateId_AccelerometerZx :
        *pOutTitle = "AccelerometerZx";
        *pOutAxis1 = 'z';
        *pOutAxis2 = 'x';
        break;
    case CoordinateId_AngularVelocityXy:
        *pOutTitle = "AngularVelocityXy";
        *pOutAxis1 = 'x';
        *pOutAxis2 = 'y';
        break;
    case CoordinateId_AngularVelocityYz:
        *pOutTitle = "AngularVelocityYz";
        *pOutAxis1 = 'y';
        *pOutAxis2 = 'z';
        break;
    case CoordinateId_AngularVelocityZx:
        *pOutTitle = "AngularVelocityZx";
        *pOutAxis1 = 'z';
        *pOutAxis2 = 'x';
        break;
    case CoordinateId_AngleX:
        *pOutTitle = "AngleX";
        *pOutAxis1 = ' ';
        *pOutAxis2 = ' ';
        break;
    case CoordinateId_AngleY:
        *pOutTitle = "AngleY";
        *pOutAxis1 = ' ';
        *pOutAxis2 = ' ';
        break;
    case CoordinateId_AngleZ:
        *pOutTitle = "AngleZ";
        *pOutAxis1 = ' ';
        *pOutAxis2 = ' ';
        break;
    case CoordinateId_DirectionXy:
        *pOutTitle = "DirectionXy";
        *pOutAxis1 = 'x';
        *pOutAxis2 = 'y';
        break;
    case CoordinateId_DirectionYz:
        *pOutTitle = "DirectionYz";
        *pOutAxis1 = 'y';
        *pOutAxis2 = 'z';
        break;
    case CoordinateId_DirectionZx:
        *pOutTitle = "DirectionZx";
        *pOutAxis1 = 'z';
        *pOutAxis2 = 'x';
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }
}

//!< 6軸センサーの状態を描画します。
void WriteSixAxisSensorState(nn::gfx::util::DebugFontTextWriter* pTextWriter,
                             const INpadStyleSixAxisSensor* const pSixAxisSensor,
                             const float& offsetX,
                             const float& offsetY,
                             const nn::util::Unorm8x4& Color,
                             const int& playerNumber) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    const auto state = pSixAxisSensor->GetSixAxisSensorState(0);
    const nn::util::Unorm8x4& textColor = state.attributes.Test<nn::hid::SixAxisSensorAttribute::IsConnected>()
                                        ? Color : Color::Gray;

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(1, 1);
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print("Player: %d", playerNumber);

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(0.7f, 0.7f);

    float posY = offsetY + 15.0f;
    const float DeltaY = 15.0f;

    pTextWriter->SetCursor(offsetX + 5, posY);
    pTextWriter->Print("acceleration");
    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 10, posY);
    pTextWriter->Print("(%8.5f,%8.5f,%8.5f)", state.acceleration.x,
                                              state.acceleration.y,
                                              state.acceleration.z);

    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 5, posY);
    pTextWriter->Print("angularVelocity");
    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 10, posY);
    pTextWriter->Print("(%8.5f,%8.5f,%8.5f)", state.angularVelocity.x,
                                              state.angularVelocity.y,
                                              state.angularVelocity.z);

    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 5, posY);
    pTextWriter->Print("angle");
    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 10, posY);
    pTextWriter->Print("(%8.5f,%8.5f,%8.5f)", state.angle.x,
                                              state.angle.y,
                                              state.angle.z);

    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 5, posY);
    pTextWriter->Print("samplingNumber");
    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 10, posY);
    pTextWriter->Print("(%d)", state.samplingNumber);

    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 5, posY);
    pTextWriter->Print("AtRest : %s", pSixAxisSensor->IsAtRest(0) ? "Y" : "N");

    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 5, posY);
    if (pSixAxisSensor->IsMeasuringStarted(0))
    {
        pTextWriter->Print("Measuring...");
    }
    else
    {
        pTextWriter->Print("PER[%] : %3.2f", pSixAxisSensor->GetPacketErrorRate(0) * 100);
    }

}

//!< 6軸センサーの状態を座標系上に描画します。
void WriteSixAxisSensorStateFigure(nn::gfx::util::DebugFontTextWriter* pTextWriter,
                                   const nn::hid::SixAxisSensorState& state,
                                   const nn::util::Unorm8x4& Color,
                                   const int playerNumber) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    const nn::util::Unorm8x4& textColor = state.attributes.Test<nn::hid::SixAxisSensorAttribute::IsConnected>()
                                        ? Color : Color::Gray;

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(1, 1);

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(0.8f, 0.8f);

    const float Coefficient  = 50.0f;

    nn::util::Float2 coordinateOrigin;

    // Accelerometer
    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AccelerometerXy);
    pTextWriter->SetCursor(coordinateOrigin.x + state.acceleration.x * Coefficient,
                           coordinateOrigin.y - state.acceleration.y * Coefficient);
    pTextWriter->Print("%d", playerNumber);

    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AccelerometerYz);
    pTextWriter->SetCursor(coordinateOrigin.x + state.acceleration.y * Coefficient,
                           coordinateOrigin.y - state.acceleration.z * Coefficient);
    pTextWriter->Print("%d", playerNumber);

    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AccelerometerZx);
    pTextWriter->SetCursor(coordinateOrigin.x + state.acceleration.z * Coefficient,
                           coordinateOrigin.y - state.acceleration.x * Coefficient);
    pTextWriter->Print("%d", playerNumber);

    // AngularVelocity
    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AngularVelocityXy);
    pTextWriter->SetCursor(coordinateOrigin.x + state.angularVelocity.x * Coefficient,
                           coordinateOrigin.y - state.angularVelocity.y * Coefficient);
    pTextWriter->Print("%d", playerNumber);

    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AngularVelocityYz);
    pTextWriter->SetCursor(coordinateOrigin.x + state.angularVelocity.y * Coefficient,
                           coordinateOrigin.y - state.angularVelocity.z * Coefficient);
    pTextWriter->Print("%d", playerNumber);

    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AngularVelocityZx);
    pTextWriter->SetCursor(coordinateOrigin.x + state.angularVelocity.z * Coefficient,
                           coordinateOrigin.y - state.angularVelocity.x * Coefficient);
    pTextWriter->Print("%d", playerNumber);

    // Angle
    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AngleX);
    pTextWriter->SetCursor(coordinateOrigin.x + cos(2 * nn::util::FloatPi * state.angle.x) * Coefficient,
                           coordinateOrigin.y - sin(2 * nn::util::FloatPi * state.angle.x) * Coefficient);
    pTextWriter->Print("%d", playerNumber);

    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AngleY);
    pTextWriter->SetCursor(coordinateOrigin.x + cos(2 * nn::util::FloatPi * state.angle.y) * Coefficient,
                           coordinateOrigin.y - sin(2 * nn::util::FloatPi * state.angle.y) * Coefficient);
    pTextWriter->Print("%d", playerNumber);

    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AngleZ);
    pTextWriter->SetCursor(coordinateOrigin.x + cos(2 * nn::util::FloatPi * state.angle.z) * Coefficient,
                           coordinateOrigin.y - sin(2 * nn::util::FloatPi * state.angle.z) * Coefficient);
    pTextWriter->Print("%d", playerNumber);

    // Direction
    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_DirectionXy);
    pTextWriter->SetCursor(coordinateOrigin.x + state.direction.x.x * Coefficient,
                           coordinateOrigin.y - state.direction.x.y * Coefficient);
    pTextWriter->Print("%dx", playerNumber);
    pTextWriter->SetCursor(coordinateOrigin.x + state.direction.y.x * Coefficient,
                           coordinateOrigin.y - state.direction.y.y * Coefficient);
    pTextWriter->Print("%dy", playerNumber);
    pTextWriter->SetCursor(coordinateOrigin.x + state.direction.z.x * Coefficient,
                           coordinateOrigin.y - state.direction.z.y * Coefficient);
    pTextWriter->Print("%dz", playerNumber);

    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_DirectionYz);
    pTextWriter->SetCursor(coordinateOrigin.x + state.direction.x.y * Coefficient,
                           coordinateOrigin.y - state.direction.x.z * Coefficient);
    pTextWriter->Print("%dx", playerNumber);
    pTextWriter->SetCursor(coordinateOrigin.x + state.direction.y.y * Coefficient,
                           coordinateOrigin.y - state.direction.y.z * Coefficient);
    pTextWriter->Print("%dy", playerNumber);
    pTextWriter->SetCursor(coordinateOrigin.x + state.direction.z.y * Coefficient,
                           coordinateOrigin.y - state.direction.z.z * Coefficient);
    pTextWriter->Print("%dz", playerNumber);

    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_DirectionZx);
    pTextWriter->SetCursor(coordinateOrigin.x + state.direction.x.z * Coefficient,
                           coordinateOrigin.y - state.direction.x.x * Coefficient);
    pTextWriter->Print("%dx", playerNumber);
    pTextWriter->SetCursor(coordinateOrigin.x + state.direction.y.z * Coefficient,
                           coordinateOrigin.y - state.direction.y.x * Coefficient);
    pTextWriter->Print("%dy", playerNumber);
    pTextWriter->SetCursor(coordinateOrigin.x + state.direction.z.z * Coefficient,
                           coordinateOrigin.y - state.direction.z.x * Coefficient);
    pTextWriter->Print("%dz", playerNumber);
}


//!< 座標系を描画します。
void WriteCoordinateAxes(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    CoordinateId id) NN_NOEXCEPT
{
    const nn::util::Unorm8x4& textColor = Color::White;

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(0.7f, 0.7f);

    const int NumberOfCharacters = 10;
    const int Interval = 10;
    nn::util::Float2 origin;
    char axis1;
    char axis2;
    ::std::string title;
    GetCoordinateName(&axis1, &axis2, &title, id);
    GetCoordinateOrigin(&origin, id);

    for (int i = 0; i < NumberOfCharacters; i++)
    {
        pTextWriter->SetCursor(origin.x + Interval * i, origin.y);
        pTextWriter->Print("-");
        pTextWriter->SetCursor(origin.x - Interval * i, origin.y);
        pTextWriter->Print("-");

        pTextWriter->SetCursor(origin.x, origin.y + Interval * i);
        pTextWriter->Print("|");
        pTextWriter->SetCursor(origin.x, origin.y - Interval * i);
        pTextWriter->Print("|");
    }
    pTextWriter->SetCursor(origin.x + NumberOfCharacters * Interval, origin.y);
    pTextWriter->Print(">%c", axis1);
    pTextWriter->SetCursor(origin.x, origin.y - NumberOfCharacters * Interval);
    pTextWriter->Print("^%c", axis2);
    pTextWriter->SetCursor(origin.x - NumberOfCharacters * Interval,
        origin.y - NumberOfCharacters * Interval - 1.0f);
    pTextWriter->Print("%s", title.c_str());
}

//!< 座標系を描画します。
void WriteCoordinateAxes(nn::gfx::util::DebugFontTextWriter* pTextWriter) NN_NOEXCEPT
{
    WriteCoordinateAxes(pTextWriter, CoordinateId_AccelerometerXy);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AccelerometerYz);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AccelerometerZx);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngularVelocityXy);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngularVelocityYz);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngularVelocityZx);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngleX);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngleY);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngleZ);
    WriteCoordinateAxes(pTextWriter, CoordinateId_DirectionXy);
    WriteCoordinateAxes(pTextWriter, CoordinateId_DirectionYz);
    WriteCoordinateAxes(pTextWriter, CoordinateId_DirectionZx);

    WriteCoordinateAxes(pTextWriter, CoordinateId_AccelerometerXy);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AccelerometerYz);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AccelerometerZx);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngularVelocityXy);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngularVelocityYz);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngularVelocityZx);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngleX);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngleY);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngleZ);
    WriteCoordinateAxes(pTextWriter, CoordinateId_DirectionXy);
    WriteCoordinateAxes(pTextWriter, CoordinateId_DirectionYz);
    WriteCoordinateAxes(pTextWriter, CoordinateId_DirectionZx);
}

//!< 6軸センサーの入力状態を出力します。
void PrintSixAxisSensorState(const nn::hid::SixAxisSensorState& state)
{
    NN_LOG("SamplingNumber:%lld, DeltaTime[us]:%lld\n", state.samplingNumber,
                                                        state.deltaTime.GetMicroSeconds());
    NN_LOG("IsConnected:%s\n",
        state.attributes.Test<::nn::hid::SixAxisSensorAttribute::IsConnected>() ? "True" : "False");
    NN_LOG("Acc: %f, %f, %f\n",         state.acceleration.x,
                                        state.acceleration.y,
                                        state.acceleration.z);
    NN_LOG("Gyro: %f, %f, %f\n",        state.angularVelocity.x,
                                        state.angularVelocity.y,
                                        state.angularVelocity.z);
    NN_LOG("Angle: %f, %f, %f\n",       state.angle.x,
                                        state.angle.y,
                                        state.angle.z);
    NN_LOG("Direction.x: %f, %f, %f\n", state.direction.x.x,
                                        state.direction.x.y,
                                        state.direction.x.z);
    NN_LOG("Direction.y: %f, %f, %f\n", state.direction.y.x,
                                        state.direction.y.y,
                                        state.direction.y.z);
    NN_LOG("Direction.z: %f, %f, %f\n", state.direction.z.x,
                                        state.direction.z.y,
                                        state.direction.z.z);
    NN_LOG("\n");
}

//!< 接続中のデバイス全てに対してユーザーキャリブレーションを実行します
void ExecuteUserCalibration() NN_NOEXCEPT
{
    // UniquePadId の取得
    ::nn::hid::system::UniquePadId ids[::nn::hid::system::UniquePadIdCountMax];
    const int IdCount = ::nn::hid::system::ListUniquePads(ids, sizeof(ids) / sizeof(ids[0]));

    // UniquePadSixAxisSensorHandle を取得
    ::nn::hid::system::UniqueSixAxisSensorHandle handles[::nn::hid::system::UniqueSixAxisSensorHandleCountMax];

    int handleCount = 0;
    for(int i = 0; i < IdCount; i++)
    {
        int count = ::nn::hid::system::ListSixAxisSensorHandles(&handles[handleCount],
                                                                ids[i],
                                                                ::nn::hid::system::UniquePadIdCountMax);
        handleCount += count;
    }

    for(int i = 0; i < handleCount; i++)
    {
        if (::nn::hid::system::IsSixAxisSensorUserCalibrationSupported(handles[i]) == false)
        {
            continue;
        }

        NN_LOG("[%d] ResetSixAxisSensorCalibrationValues\n", i);
        auto result = ::nn::hid::system::ResetSixAxisSensorCalibrationValues(handles[i]);

        if (result.IsFailure())
        {
            continue;
        }

        NN_LOG("[%d] ExecuteSixAxisSensorUserCalibration\n", i);
        result = ::nn::hid::system::StartSixAxisSensorUserCalibration(handles[i]);

        if (result.IsFailure())
        {
            if (::nn::hid::system::ResultSixAxisSensorDisconnected::Includes(result))
            {
                NN_LOG("ResultSixAxisSensorDisconnected\n");
            }
            else if (::nn::hid::system::ResultSixAxisSensorNotSupported::Includes(result))
            {
                NN_LOG("ResultSixAxisSensorNotSupported\n");
            }
            else if (::nn::hid::system::ResultSixAxisSensorNotHorizontal::Includes(result))
            {
                NN_LOG("ResultSixAxisSensorNotHorizontal\n");
            }
            else if (::nn::hid::system::ResultSixAxisSensorNotStable::Includes(result))
            {
                NN_LOG("ResultSixAxisSensorNotStable\n");
            }
            else if (::nn::hid::system::ResultSixAxisSensorWriteFailure::Includes(result))
            {
                NN_LOG("ResultSixAxisSensorWriteFailure\n");
            }
            else
            {
                NN_LOG("ResultSixAxisSensorDisconnected\n");
            }
        }
    }
}

//!< アプリ全体に影響するコマンドを実行します。
void ExecuteCommands(const nn::hid::NpadButtonSet& buttons)
{
    if ((buttons.Test<nn::hid::NpadButton::X>()) ||
        (buttons.Test<nn::hid::NpadButton::Up>() ))
    {
        ExecuteUserCalibration();
    }

    if(buttons.Test<nn::hid::NpadButton::StickR>())
    {
        g_Mode = (g_Mode + 1) % StyleIndicesCountMax;
    }
    else if (buttons.Test<nn::hid::NpadButton::StickL>())
    {
        g_Mode = (g_Mode == 0) ? StyleIndicesCountMax - 1
                               : g_Mode - 1;
    }
    else if (buttons.Test<nn::hid::NpadButton::ZL>() &&
             buttons.Test<nn::hid::NpadButton::StickL>())
    {
        for (const auto& id : NpadIds)
        {
            if (id == nn::hid::NpadId::Handheld)
            {
                continue;
            }
            nn::hid::SetNpadJoyAssignmentModeSingle(id);
        }
    }
    else if (buttons.Test<nn::hid::NpadButton::ZR>() &&
             buttons.Test<nn::hid::NpadButton::StickR>())
    {
        for (const auto& id : NpadIds)
        {
            if (id == nn::hid::NpadId::Handheld)
            {
                continue;
            }
            nn::hid::SetNpadJoyAssignmentModeDual(id);
        }
    }

    if (buttons.Test<nn::hid::NpadJoyButton::LeftSL>() ||
        buttons.Test<nn::hid::NpadJoyButton::RightSL>())
    {
        g_CommunicationModeIndex = (g_CommunicationModeIndex == 0) ? CommunicationModeCountMax - 1
            : g_CommunicationModeIndex - 1;
        ::nn::hid::SetNpadCommunicationMode(g_CommunicationModes[g_CommunicationModeIndex]);
        NN_LOG("CommunicationMode is set to = %x\n", ::nn::hid::GetNpadCommunicationMode());
    }
    else if (buttons.Test<nn::hid::NpadJoyButton::LeftSR>() ||
             buttons.Test<nn::hid::NpadJoyButton::RightSR>())
    {
        g_CommunicationModeIndex = (g_CommunicationModeIndex + 1) % CommunicationModeCountMax;
        ::nn::hid::SetNpadCommunicationMode(g_CommunicationModes[g_CommunicationModeIndex]);
        NN_LOG("CommunicationMode is set to = %x\n", ::nn::hid::GetNpadCommunicationMode());
    }

}

//!< アプリの終了判定をします。
bool IsQuitRequired(const nn::hid::NpadButtonSet& buttons)
{
    return (buttons.Test<nn::hid::NpadButton::Plus>() ||
            buttons.Test<nn::hid::NpadButton::Minus>());
}

//!< メインのループ処理を実行します。
void RunLoop(nn::gfx::util::DebugFontTextWriter* pTextWriter,
             GraphicsSystem* pGraphicsSystem,
             FontSystem* pFontSystem,
             std::vector<INpadStyleSixAxisSensor*>* pNpadStyleSixAxisSensors)
{
    while (NN_STATIC_CONDITION(true))
    {
        bool quits = false;

        switch (GetWindowMessage(pGraphicsSystem->GetNativeWindowHandle()))
        {
        case WindowMessage_Close:
            quits = true;
            break;
        default:
            break;
        }

        WriteMode(pTextWriter);
        WriteCommunicationMode(pTextWriter);
        WriteCoordinateAxes(pTextWriter);

        for (std::vector<INpadStyleSixAxisSensor*>::iterator it = pNpadStyleSixAxisSensors->begin();
            it != pNpadStyleSixAxisSensors->end();
            ++it)
        {
            if (!(*it)->IsConnected())
            {
                continue;
            }

            // 状態更新
            (*it)->Update();

            nn::hid::NpadButtonSet triggerButtons = (*it)->GetTriggerButtons();
            nn::hid::NpadStyleSet style = (*it)->GetNpadStyleSet();

            if (IsStyleToControl(style))
            {
                ExecuteCommands(triggerButtons);
            }

            if (IsQuitRequired(triggerButtons))
            {
                quits = true;
            }

            // 描画対象の操作形態か
            if (!IsStyleToDraw(style))
            {
                continue;
            }

            int count = (*it)->GetSixAxisSensorHandleCount();

            for (int i = 0; i < count; i++)
            {
                SixAxisSensorPointer* pPointer = (*it)->GetSixAxisSensorPointer(i);
                nn::hid::SixAxisSensorState state = (*it)->GetSixAxisSensorState(i);
                nn::hid::NpadIdType id = (*it)->GetNpadIdType();


                // 各操作形態で反映されるコマンド
                if (triggerButtons.Test<nn::hid::NpadButton::A>() ||
                    triggerButtons.Test<nn::hid::NpadButton::Right>())
                {
                    PrintSixAxisSensorState(state);
                    pPointer->Reset();
                    (*it)->SetGyroscopeZeroDriftMode(i, nn::hid::GyroscopeZeroDriftMode_Tight);
                }

                if (triggerButtons.Test<nn::hid::NpadButton::B>() ||
                    triggerButtons.Test<nn::hid::NpadButton::Down>())
                {
                    (*it)->SetGyroscopeZeroDriftMode(i, nn::hid::GyroscopeZeroDriftMode_Standard);
                }

                if (triggerButtons.Test<nn::hid::NpadButton::X>() ||
                    triggerButtons.Test<nn::hid::NpadButton::Up>())
                {
                    (*it)->StartMeasuringPacketErrorRate(i);
                }
                if (triggerButtons.Test<nn::hid::NpadButton::Y>() ||
                    triggerButtons.Test<nn::hid::NpadButton::Left>())
                {
                    (*it)->StopMeasuringPacketErrorRate(i);
                }

                int idx = static_cast<uint32_t>(id);
                if (idx >= NpadIdCountMax)
                {
                    idx = 0;
                }

                if (i == 0)
                {
                    WriteSixAxisSensorState(pTextWriter,
                                            (*it),
                                            25, 15 + 175 * static_cast<float>(idx),
                                            ColorArray[idx],
                                            idx + 1);
                }

                WriteSixAxisSensorStateFigure(pTextWriter,
                    state,
                    ColorArray[idx],
                    idx + 1);

                pTextWriter->SetScale(3.f, 3.f);
                ::nn::util::Vector3f cursor = pPointer->GetCursor();
                pTextWriter->SetTextColor(ColorArray[idx]);
                pTextWriter->SetCursor(cursor.GetX(), cursor.GetY());
                pTextWriter->Print("(-+-)");

#ifdef LOGGING
                if ((*it)->GetNpadIdType() == nn::hid::NpadId::No1 &&
                    (*it)->GetNpadStyleSet() == nn::hid::NpadStyleFullKey::Mask)
                {
                    g_Logger.AppendLine(&state);
                }
#endif
            }
        }

        pGraphicsSystem->BeginDraw();
        pFontSystem->Draw();
        pGraphicsSystem->EndDraw();

        pGraphicsSystem->Synchronize(
            nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / FrameRate));

        if (quits)
        {
            break;
        }
    }
} // NOLINT(impl/function_size)

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
const size_t GraphicsMemorySize = 8 * 1024 * 1024;

void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}

void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}

void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

} // namespace

extern "C" void nnMain()
{
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    nv::SetGraphicsAllocator(NvAllocate, NvFree,NvReallocate,NULL);
    nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree,NvReallocate,NULL);
    nv::InitializeGraphics(std::malloc(GraphicsMemorySize), GraphicsMemorySize);
#endif

    std::vector<INpadStyleSixAxisSensor*> npadStyleSixAxisSensors; //!< 管理対象の操作形態を格納するコンテナーです。
    NN_LOG("Npad SixAxisSensor Sample Start.\n");

    for(int i = 0; i < 4; i++)
    {
        // Add respective NpadStyles to the managed list
        npadStyleSixAxisSensors.push_back(reinterpret_cast<INpadStyleSixAxisSensor*>(new FullKeySixAxisSensor(NpadIds[i])));
        npadStyleSixAxisSensors.push_back(reinterpret_cast<INpadStyleSixAxisSensor*>(new JoyDualSixAxisSensor(NpadIds[i])));
        npadStyleSixAxisSensors.push_back(reinterpret_cast<INpadStyleSixAxisSensor*>(new JoyLeftSixAxisSensor(NpadIds[i])));
        npadStyleSixAxisSensors.push_back(reinterpret_cast<INpadStyleSixAxisSensor*>(new JoyRightSixAxisSensor(NpadIds[i])));
    }
    npadStyleSixAxisSensors.push_back(reinterpret_cast<INpadStyleSixAxisSensor*>(new HandheldSixAxisSensor(NpadIds[NpadIdCountMax - 1])));

    nn::hid::InitializeNpad();

    ::nn::hid::SetNpadHandheldActivationMode(::nn::hid::NpadHandheldActivationMode_None);

    g_Logger.Initialize("SixAxisSensorLogger.csv");

    //使用する操作形態を設定
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask |
                                      nn::hid::NpadStyleJoyDual::Mask |
                                      nn::hid::NpadStyleJoyLeft::Mask |
                                      nn::hid::NpadStyleJoyRight::Mask|
                                      nn::hid::NpadStyleHandheld::Mask);

    nn::hid::SetSupportedNpadIdType(NpadIds, NpadIdCountMax);

    NN_LOG("If you push any button, button state log will appear on the console.\n");
    NN_LOG("Push (+) and (-) Button to shutdown this application.\n");

    // 管理対象の操作形態に関する初期化
    for(std::vector<INpadStyleSixAxisSensor*>::iterator it = npadStyleSixAxisSensors.begin();
        it != npadStyleSixAxisSensors.end();
        ++it)
    {
        (*it)->Initialize();
    }

    ApplicationHeap applicationHeap;
    applicationHeap.Initialize(ApplicationHeapSize);

    GraphicsSystem* pGraphicsSystem = new GraphicsSystem();
    pGraphicsSystem->SetApplicationHeap(&applicationHeap);
    pGraphicsSystem->Initialize();

    EnableWindowMessage(pGraphicsSystem->GetNativeWindowHandle());

    FontSystem* pFontSystem = new FontSystem();
    pFontSystem->SetApplicationHeap(&applicationHeap);
    pFontSystem->SetGraphicsSystem(pGraphicsSystem);
    pFontSystem->Initialize();

    nn::gfx::util::DebugFontTextWriter& textWriter =
        pFontSystem->GetDebugFontTextWriter();

    g_Logger.ResetTick();

    ::nn::hid::SetNpadHandheldActivationMode(::nn::hid::NpadHandheldActivationMode::NpadHandheldActivationMode_None);

    RunLoop(&textWriter,
            pGraphicsSystem,
            pFontSystem,
            &npadStyleSixAxisSensors);

    pFontSystem->Finalize();
    delete pFontSystem;

    pGraphicsSystem->Finalize();
    delete pGraphicsSystem;

    applicationHeap.Finalize();

    g_Logger.Finalize();
}
