﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once
#include <cstdlib>
#include <string>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/oe.h>
#include <nn/fs.h>

#include <nn/hid/hid_Palma.h>
#include <nn/hid/hid_Result.palma.h>
#include <nn/result/result_HandlingUtility.h>

#include "framework/Applet.h"
#include "framework/Npad.h"
#include "framework/CursolControl.h"
#include "framework/FileSystem.h"



namespace util{

    const int PalmaActivityIndexNum = 512;                //!< Activityのインデックス最大数

    struct PalmaActivity
    {
        int32_t dataBaseIdversion;                      //!< データベース識別バージョン
        nn::hid::PalmaActivityEntry activityEntry;      //!< Activityのエントリー
    };

    struct PalmaApiResult
    {
        uint8_t viewCounter = 0;
        char*   string;
        bool    isUpdate = false;
    };

    //!< Palmaの状態をひとまとめにした構造体定義
    struct PalmaController
    {
        nn::hid::PalmaConnectionHandle                  Handle;                         //!< Palmaの制御用ハンドル
        nn::os::SystemEventType                         OperationEvent;                 //!< Palmaのイベント格納用変数
        nn::hid::PalmaApplicationSectionAccessBuffer    applicationSectionbuffer;       //!< ReadしたApplicationSectionのデータ
        int32_t                                         applicationSectionSize;         //!< ReadしたApplicationSectionのサイズ
        uint8_t                                         AppDataAllFillCount;            //!< 全領域Fill中に用いるカウンタ
        bool                                            IsAllAppData;                   //!< ApplicationSectionのAllReadかどうか
        char*                                           AppDataDumpfileName;            //!< ファイル名
        bool                                            IsEnable;                       //!< Palmaが現在操作可能かどうかを示すフラグ
        nn::hid::PalmaFeatureSet                        feature;                        //!< PalmaのfertureSet(どの機能がONしているか)フラグ群
        uint8_t                                         uniqueCode[16];                 //!< PalmaのuniqueCode
        uint32_t                                        step;                           //!< Step値(歩数)
        int32_t                                         databaseIdVersion;              //!< データベース識別バージョン
        uint8_t                                         fwVersion[16];                  //!< PalmaのFwVersion(Ble経由で取得)
        size_t                                          fwVersionLength;                //!< PalmaのFwVersion(Ble経由で取得)の長さ
        uint8_t                                         playLog[8];                     //!< PlayLog
        PalmaApiResult                                  apiResult;                      //!< PalmaのAPIたたいたときのResult
        bool                                            IsWriting;                      //!< Activity書き込み中を示すフラグ
    };

    static const int            RgbPattern_Max = 30;
    static const int            WavePattern_Max = 30;

    //!< Activityファイルを纏めた構造体定義
    struct ActivityFileList
    {
        std::string wave[WavePattern_Max];
        std::string rgbLed[RgbPattern_Max];
        int rgbLedCount;
        int waveCount;
    };

    //!< AppDataファイルを纏めた構造体定義
    struct AppDataList
    {
        std::string data[10];
        int count;
    };

    // ApplicationSectionの全領域Fill用テーブル
    static NN_ALIGNAS(nn::os::MemoryPageSize) nn::hid::PalmaApplicationSectionAccessBuffer WorkBufferFillTable = {{
        '0','0','1','0','2','0','3','0','4','0','5','0','6','0','7','0','8','0','9','0','A','0','B','0','C','0','D','0','E','0','F','\n',
        '0','1','1','1','2','1','3','1','4','1','5','1','6','1','7','1','8','1','9','1','A','1','B','1','C','1','D','1','E','1','F','\n',
        '0','2','1','2','2','2','3','2','4','2','5','2','6','2','7','2','8','2','9','2','A','2','B','2','C','2','D','2','E','2','F','\n',
        '0','3','1','3','2','3','3','3','4','3','5','3','6','3','7','3','8','3','9','3','A','3','B','3','C','3','D','3','E','3','F','\n',
        '0','4','1','4','2','4','3','4','4','4','5','4','6','4','7','4','8','4','9','4','A','4','B','4','C','4','D','4','E','4','F','\n',
        '0','5','1','5','2','5','3','5','4','5','5','5','6','5','7','5','8','5','9','5','A','5','B','5','C','5','D','5','E','5','F','\n',
        '0','6','1','6','2','6','3','6','4','6','5','6','6','6','7','6','8','6','9','6','A','6','B','6','C','6','D','6','E','6','F','\n',
        '0','7','1','7','2','7','3','7','4','7','5','7','6','7','7','7','8','7','9','7','A','7','B','7','C','7','D','7','E','7','F','\n'
    }};

    const int ReservedIndex[] =
    {17,  18,  19,
     41,  42,  43,  44,  45,  46,  47,  48,  49,
     71,  72,  73,  74,  75,  76,  77,  78,  79,
     101, 102, 103, 104, 105, 106, 107, 108, 109,
     131, 132, 133, 134, 135, 136, 137, 138, 139,
     161, 162, 163, 164, 165, 166, 167, 168, 169,
     181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 192,193, 194, 195, 196, 197,
     214, 215, 216, 217, 218, 219,
     224, 225, 226, 227, 228, 229,
     232, 233, 234, 235, 236, 237, 238, 239,
     255, 256, 257, 258, 259, 260,
     261, 262, 263, 264, 265, 266, 267, 268, 269, 270,
     271, 272, 273, 274, 275, 276, 277, 278, 279, 280,
     281, 282, 283, 284, 285, 286, 287, 288, 289, 290,
     291, 292, 293, 294, 295, 296, 297, 298, 299, 300,
     301, 302, 303, 304, 305, 306, 307, 308, 309, 310,
     311, 312, 313, 314, 315, 316, 317, 318, 319, 320,
     321, 322, 323, 324, 325, 326, 327, 328, 329, 330,
     331, 332, 333, 334, 335, 336, 337, 338, 339, 340,
     341, 342, 343, 344, 345, 346, 347, 348, 349, 350,
     351, 352, 353, 354, 355, 356, 357, 358, 359, 360,
     361, 362, 363, 364, 365, 366, 367, 368, 369, 370,
     371, 372, 373, 374, 375, 376, 377, 378, 379, 380,
     381, 382, 383, 384, 385, 386, 387, 388, 389, 390,
     391, 392, 393, 394, 395, 396, 397, 398, 399, 400,
     401, 402, 403, 404, 405, 406, 407, 408, 409, 410,
     411, 412, 413, 414, 415, 416, 417, 418, 419, 420,
     421, 422, 423, 424, 425, 426, 427, 428, 429, 430,
     431, 432, 433, 434, 435, 436, 437, 438, 439, 440,
     441, 442, 443, 444, 445, 446, 447, 448, 449, 450,
     451, 452, 453, 454, 455, 456, 457, 458, 459, 460,
     461, 462, 463, 464, 465, 466, 467, 468, 469, 470,
     471, 472, 473, 474, 475, 476, 477, 478, 479, 480,
     481, 482, 483, 484, 485, 486, 487, 488, 489, 490,
     491, 492, 493, 494, 495, 496, 497, 498, 499, 500,
     501, 502, 503, 504, 505, 506, 507, 508, 509, 510,
     511
    };
    const int ReservedIndexCount = sizeof(ReservedIndex) / sizeof(int);

    struct ReservedCheckInfo
    {
        bool    isChecking = false;
        int     checkIndex = 0;
        int     startArray = 0;
        int     endArray = 0;
        int     array    = 0;
    };

    class Palma
    {
    private:
        PalmaController m_controler[frm::NpadIdNum];                //!< 最新の状態
        PalmaController m_controlerOld[frm::NpadIdNum];             //!< 1フレーム前の状態

    public:
        bool                        m_isScaning;                                //!< Scan中かどうかを示すフラグ
        ReservedCheckInfo           m_reservedCheckInfo;
        ActivityFileList            m_ActivityFile;
        AppDataList                 m_Appdata;

    private:
        void AllWritePalmaApplicationSection(size_t index);
        void AllReadPalmaApplicationSection(size_t index);
        void DumpPalmaApplicationSection(size_t index);
        void GetResult(size_t index, const char * apiString, const char * resultString);

    public:
        //----------------------------------------------------------------
        // @brief   インスタンス取得
        //----------------------------------------------------------------
        static Palma& GetInstance() NN_NOEXCEPT;

        //----------------------------------------------------------------
        // @brief   PalmaOperationTypeの文字列変換
        //----------------------------------------------------------------
        const char* ToString(nn::hid::PalmaOperationType type);

        //----------------------------------------------------------------
        // @brief   PalmaWaveSetの文字列変換
        //----------------------------------------------------------------
        const char * ToString(nn::hid::PalmaWaveSet waveSet);

        //----------------------------------------------------------------
        // @brief   PalmaResultの文字列変換
        //----------------------------------------------------------------
        const char* ToString(nn::Result result);

        //----------------------------------------------------------------
        // @brief   PalmaFrModeTypeの文字列変換
        //----------------------------------------------------------------
        const char * ToString(nn::hid::PalmaFrModeType type);

        //----------------------------------------------------------------
        // @brief   Palmaの固有機能を有効化します
        // @detail  Handle取得、Initialize、イベント通知のバインド設定を行います
        //          本API成功後、IsEnableがTRUEとなります
        //----------------------------------------------------------------
        void Enable(size_t index);

        //----------------------------------------------------------------
        // @brief   PalmaのStep機能を有効化します
        //----------------------------------------------------------------
        void EnableStep(size_t index);

        //----------------------------------------------------------------
        // @brief   PalmaのStep機能を無効化します
        //----------------------------------------------------------------
        void DisableStep(size_t index);

        //----------------------------------------------------------------
        // @brief   PalmaのScan、接続を開始します
        //----------------------------------------------------------------
        void StartScan(void);

        //----------------------------------------------------------------
        // @brief   PalmaのScan、接続を終了します
        //----------------------------------------------------------------
        void StopScan(void);

        //----------------------------------------------------------------
        // @brief   指定したindexのPalmaをペアリングします
        //----------------------------------------------------------------
        void SetPairing(size_t index);

        //----------------------------------------------------------------
        // @brief   ペアリング済のPalmaの自動接続機能を有効化します
        //----------------------------------------------------------------
        void EnablePairedConnect(void);

        //----------------------------------------------------------------
        // @brief   ペアリング済のPalmaの自動接続機能を無効化します
        //----------------------------------------------------------------
        void DisablePairedConnect(void);

        //----------------------------------------------------------------
        // @brief   Palmaの高速転送モードの有効/無効を切り替えます
        //----------------------------------------------------------------
        void SetBoostMode(bool isSet);

        //----------------------------------------------------------------
        // @brief   PalmaのStep数をクリアします
        //----------------------------------------------------------------
        void ResetStep(size_t index);

        //----------------------------------------------------------------
        // @brief   PalmaのFeatureSetを中断・再開制御します
        //----------------------------------------------------------------
        void SuspendFeatureSet(size_t index, nn::hid::PalmaFeatureSet featureSet);

        //----------------------------------------------------------------
        // @brief   PalmaのUniqueCodeを呼び出します
        //----------------------------------------------------------------
        void ReadUniqueCode(size_t index);

        //----------------------------------------------------------------
        // @brief   PalmaのUniqueCodeを無効化します
        //----------------------------------------------------------------
        void DisableUniqueCode(size_t index);

        //----------------------------------------------------------------
        // @brief   Palma内に格納されたPlayLogを読み込みます
        //----------------------------------------------------------------
        void ReadPlayLog(size_t index, uint16_t playLogIndex);

        //----------------------------------------------------------------
        // @brief   Palma内に格納されたPlayLogをクリアします
        //----------------------------------------------------------------
        void ResetPlayLog(size_t index, uint16_t playLogIndex);

        //----------------------------------------------------------------
        // @brief   Palmaのデータベース識別バージョンを取得します
        //----------------------------------------------------------------
        void GetDataBaseIdVer(size_t index);

        //----------------------------------------------------------------
        // @brief   Palmaのデータベース識別バージョンを書き込みます
        //----------------------------------------------------------------
        void SetDataBaseIdVer(size_t index, int idVer);

        //----------------------------------------------------------------
        // @brief   PalmaのApplicationSectionを読み込みます
        //----------------------------------------------------------------
        void ReadPalmaApplicationSection(size_t index, frm::AppSectionElement element);

        //----------------------------------------------------------------
        // @brief   PalmaのApplicationSectionを書き込みます
        //----------------------------------------------------------------
        void WritePalmaApplicationSection(size_t index, frm::AppSectionElement element);

        //----------------------------------------------------------------
        // @brief   PalmaのFrModeの中断、再開を切り替えます
        //----------------------------------------------------------------
        void SetFrModeType(size_t index, int frMode);

        //----------------------------------------------------------------
        // @brief   PalmaのStep数(歩数)を読み込みます
        //----------------------------------------------------------------
        void ReadStep(size_t index);

        //----------------------------------------------------------------
        // @brief   Palmaのコントロール状態を更新します
        //----------------------------------------------------------------
        void Update();

        //----------------------------------------------------------------
        // @brief   Palmaのコントロール状態を取得します
        //----------------------------------------------------------------
        PalmaController GetController(size_t index);

        //----------------------------------------------------------------
        // @brief   PalmaへActivityを書き込みます
        //----------------------------------------------------------------
        nn::Result WriteActivity(size_t index, frm::ActivityElement activity);

        //----------------------------------------------------------------
        // @brief   PalmaのActivityを再生します
        //----------------------------------------------------------------
        nn::Result PlayActivity(size_t index, frm::ActivityElement activity);

        //----------------------------------------------------------------
        // @brief   PalmaのActivityIndexのReverved領域全てを再生します
        //----------------------------------------------------------------
        void CheckReserved(size_t index, int chechArea);

        void CheckReserved();

        //----------------------------------------------------------------
        // @brief   指定したPalmaのインデックスへ、音声データを書き込みます
        //----------------------------------------------------------------
        void WriteWave(size_t pathIndex, size_t handleIndex, frm::WriteElement activity, int fileIndex);

        //----------------------------------------------------------------
        // @brief   指定したPalmaのインデックスへ、LEDデータを書き込みます
        //----------------------------------------------------------------
        void WriteRgbLed(size_t pathIndex, size_t handleIndex, frm::WriteElement activity, int fileIndex);
    };
}
#ifndef gPalma
#define gPalma (util::Palma::GetInstance())
#endif
