﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once
#include <cstdlib>
#include <string>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/util/util_Quaternion.h>

#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_NpadPalma.h>
#include <nn/hid/hid_NpadSixAxisSensor.h>

#include "Gesture.h"

namespace frm {

    // サポートするNpadのID
    static nn::hid::NpadIdType NpadIds[] = {
                                    nn::hid::NpadId::No1,         //!< [0]：コントローラ番号1
                                    nn::hid::NpadId::No2,         //!< [1]：コントローラ番号2
                                    nn::hid::NpadId::No3,         //!< [2]：コントローラ番号3
                                    nn::hid::NpadId::No4,         //!< [3]：コントローラ番号4
                                    nn::hid::NpadId::No5,         //!< [4]：コントローラ番号5
                                    nn::hid::NpadId::No6,         //!< [5]：コントローラ番号6
                                    nn::hid::NpadId::No7,         //!< [6]：コントローラ番号7
                                    nn::hid::NpadId::No8,         //!< [7]：コントローラ番号8
                                    nn::hid::NpadId::Handheld,    //!< [8]：携帯モード
    };
    // サポートするNpadの数
    const int NpadIdNum = sizeof(NpadIds) / sizeof(nn::hid::NpadIdType);

    //!< NPadの状態をひとまとめにした構造体定義
    struct NpadState
    {
        int64_t                     samplingNumber;             //!< サンプリングカウンタ
        float                       packetLossRatio;            //!< パケットロス率(サンプリングカウンタ - 1[s]前のサンプリングカウンタ) / 200 * 100
        nn::hid::NpadButtonSet      press;                      //!< 現在のボタン押下状態
        nn::hid::NpadButtonSet      trigger;                    //!< 現在のボタンのトリガ入力(押されたとき一発だけ立つ)
        nn::hid::NpadButtonSet      longPress;                  //!< 現在のボタン長押し状態(一定期間長押しすると、pressと同様の出力になる)
        nn::hid::NpadButtonSet      longPressTrigger;           //!< 現在のボタン長押し状態のトリガ入力
        nn::hid::AnalogStickState   analogLeft;                 //!< Lアナログスティックのアナログ値
        nn::hid::AnalogStickState   analogRight;                //!< Rアナログスティックのアナログ値
        nn::hid::NpadAttributesSet  attributes;                 //!< Npadの属性
        bool                        isConnected;                //!< Npadの接続状態

        nn::hid::NpadIdType         id;                         //!< NpadのID
        nn::hid::NpadStyleSet       style;                      //!< NpadのStyle

        NpadState& operator= (const nn::hid::NpadFullKeyState& state);
        NpadState& operator= (const nn::hid::NpadHandheldState& state);
        NpadState& operator= (const nn::hid::NpadJoyDualState& state);
        NpadState& operator= (const nn::hid::NpadJoyLeftState& state);
        NpadState& operator= (const nn::hid::NpadJoyRightState& state);
        NpadState& operator= (const nn::hid::NpadPalmaState& state);
    };

    //!< コントローラ6軸の状態をひとまとめにした構造体定義
    struct SixAxis
    {
        nn::hid::SixAxisSensorHandle    handles[2];               //!< コントローラー6軸のハンドル(最大2個)
        int                             handleCount;              //!< コントローラー6軸のハンドル数
        nn::hid::SixAxisSensorState     state[2];                 //!< コントローラー6軸のステート(最大2個)
        nn::util::Quaternion            relationalQuaternion[2];  //!< コントローラーの相対(取得 / 基準)クオータニオン(最大2個)
        nn::util::Quaternion            quaternion[2];            //!< コントローラー6軸ステートのクオータニオン(最大2個)
    };

    class Npad
    {
    private:
        NpadState                       m_state[NpadIdNum];                             //!< 最新の状態
        NpadState                       m_stateOld[NpadIdNum];                          //!< 1フレーム前の状態
        SixAxis                         m_sixAxis[NpadIdNum];                           //!< コントローラ6軸に関する情報
        nn::util::Vector4fType          m_baseQuaternion[NpadIdNum][2];                 //!< コントローラーの基準クオータニオン(最大2個)
        int                             m_LongPressCount[NpadIdNum];                    //!< 長押しのカウンタ
        int64_t                         m_samplingNumberPrev1sec[NpadIdNum];            //!< 1[s]前のサンプリングカウンタ
        int64_t                         m_calculatePacketLossRatioCount[NpadIdNum];     //!< 1[s]前のサンプリングカウンタ

    public:

    private:
        void Latch(size_t index) NN_NOEXCEPT;
        void Reset(size_t index) NN_NOEXCEPT;
        void UpdateState(size_t index) NN_NOEXCEPT;
        void UpdatePacketLossRatio(size_t index) NN_NOEXCEPT;
        void UpdateTrigger(size_t index) NN_NOEXCEPT;
        void UpdateBeat(size_t index) NN_NOEXCEPT;
        void UpdateLongPress(size_t index) NN_NOEXCEPT;
        void UpdateLongPressTrigger(size_t index) NN_NOEXCEPT;
        void SetSixAxisConfigration(size_t index);      //!< コントローラ6軸の初期化
        void UpdateSixAxis(size_t index);               //!< コントローラ6軸のステート取得

    public:
        //----------------------------------------------------------------
        // @brief   インスタンス取得
        //----------------------------------------------------------------
        static Npad& GetInstance() NN_NOEXCEPT;

        //----------------------------------------------------------------
        // @brief   indexのNpadId変換
        //----------------------------------------------------------------
        nn::hid::NpadIdType ConvertNpadIdFromIndex(size_t index) NN_NOEXCEPT;

        //----------------------------------------------------------------
        // @brief   NpadIdのindex変換
        //----------------------------------------------------------------
        size_t ConvertIndexFromNpadId(nn::hid::NpadIdType id) NN_NOEXCEPT;

        //----------------------------------------------------------------
        // @brief   NpadStyleの文字列変換
        //----------------------------------------------------------------
        const char* ToString(nn::hid::NpadStyleSet style) NN_NOEXCEPT;

        //----------------------------------------------------------------
        // @brief   NpadIdの文字列変換
        //----------------------------------------------------------------
        const char* ToString(nn::hid::NpadIdType id) NN_NOEXCEPT;

        //----------------------------------------------------------------
        // @brief   ボタン押下状況の文字列変換
        //----------------------------------------------------------------
        const std::string ToString(nn::hid::NpadButtonSet button);

        //----------------------------------------------------------------
        // @brief   Npadの初期化
        //----------------------------------------------------------------
        void Initialize() NN_NOEXCEPT;

        //----------------------------------------------------------------
        // @brief   NPadの状態更新
        //----------------------------------------------------------------
        void Update() NN_NOEXCEPT;

        //----------------------------------------------------------------
        // @brief   NpadIdを起点としたNPadの状態取得
        // @comment なぜかオーバーロードが出来ません
        //----------------------------------------------------------------
        NpadState GetFromId(nn::hid::NpadIdType id) NN_NOEXCEPT;

        //----------------------------------------------------------------
        // @brief   Indexを起点としたNPadの状態取得
        // @comment なぜかオーバーロードが出来ません
        //----------------------------------------------------------------
        NpadState GetFromIndex(size_t index) NN_NOEXCEPT;

        //----------------------------------------------------------------
        // @brief   NpadIdを起点としたコントローラ6軸の状態取得
        // @comment なぜかオーバーロードが出来ません
        //----------------------------------------------------------------
        SixAxis GetSixAxisFromId(nn::hid::NpadIdType id);

        //----------------------------------------------------------------
        // @brief   Indexを起点としたコントローラ6軸の状態取得
        // @comment なぜかオーバーロードが出来ません
        //----------------------------------------------------------------
        SixAxis GetSixAxisFromIndex(size_t index);
    };
}

#ifndef gNpad
#define gNpad (frm::Npad::GetInstance())
#endif
