﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Npad.h"

namespace frm {

    NpadState& NpadState::operator= (const nn::hid::NpadFullKeyState& state)
    {
        samplingNumber = state.samplingNumber;
        press = state.buttons;
        analogLeft = state.analogStickL;
        analogRight = state.analogStickR;
        attributes = state.attributes;
        isConnected = state.attributes.Test<nn::hid::NpadAttribute::IsConnected>();
        return (*this);
    }
    NpadState& NpadState::operator= (const nn::hid::NpadHandheldState& state)
    {
        samplingNumber = state.samplingNumber;
        press = state.buttons;
        analogLeft = state.analogStickL;
        analogRight = state.analogStickR;
        attributes = state.attributes;
        isConnected = state.attributes.Test<nn::hid::NpadAttribute::IsConnected>();
        return (*this);
    }
    NpadState& NpadState::operator= (const nn::hid::NpadJoyDualState& state)
    {
        samplingNumber = state.samplingNumber;
        press = state.buttons;
        analogLeft = state.analogStickL;
        analogRight = state.analogStickR;
        attributes = state.attributes;
        isConnected = state.attributes.Test<nn::hid::NpadAttribute::IsConnected>();
        return (*this);
    }
    NpadState& NpadState::operator= (const nn::hid::NpadJoyLeftState& state)
    {
        samplingNumber = state.samplingNumber;
        press = state.buttons;
        analogLeft = state.analogStickL;
        analogRight = state.analogStickR;
        attributes = state.attributes;
        isConnected = state.attributes.Test<nn::hid::NpadAttribute::IsConnected>();
        return (*this);
    }
    NpadState& NpadState::operator= (const nn::hid::NpadJoyRightState& state)
    {
        samplingNumber = state.samplingNumber;
        press = state.buttons;
        analogLeft = state.analogStickL;
        analogRight = state.analogStickR;
        attributes = state.attributes;
        isConnected = state.attributes.Test<nn::hid::NpadAttribute::IsConnected>();
        return (*this);
    }
    NpadState& NpadState::operator= (const nn::hid::NpadPalmaState& state)
    {
        samplingNumber = state.samplingNumber;
        press = state.buttons;
        analogLeft = state.analogStickL;
        analogRight = state.analogStickR;
        attributes = state.attributes;
        isConnected = state.attributes.Test<nn::hid::NpadAttribute::IsConnected>();
        return (*this);
    }

    /* ------------------------------------------------------------ */
    // PRIVATE関数
    /* ------------------------------------------------------------ */
    void Npad::Latch(size_t index) NN_NOEXCEPT
    {
       m_stateOld[index] = m_state[index];
    }

    void Npad::Reset(size_t index) NN_NOEXCEPT
    {
        // ボタン押下状態のリセット
        m_state[index].press.Reset();
        m_state[index].trigger.Reset();
        m_state[index].longPress.Reset();
        m_state[index].longPressTrigger.Reset();

        m_state[index].attributes.IsAllOff();
        m_state[index].isConnected = false;
    }

    void Npad::UpdateState(size_t index) NN_NOEXCEPT
    {
        nn::hid::NpadHandheldState  handheldState;
        nn::hid::NpadFullKeyState   fullKeyState;
        nn::hid::NpadJoyDualState   joyDualState;
        nn::hid::NpadJoyLeftState   joyLeftState;
        nn::hid::NpadJoyRightState  joyRightState;
        nn::hid::NpadPalmaState     palmaState;

        //現在有効な操作形態(NpadStyleSet)を取得
        m_state[index].id = ConvertNpadIdFromIndex(index);
        m_state[index].style = nn::hid::GetNpadStyleSet(m_state[index].id);

        if (m_state[index].style.Test<nn::hid::NpadStyleHandheld>() == true)
        {
            // 状態取得
            nn::hid::GetNpadState(&handheldState, m_state[index].id);
            // 状態更新
            m_state[index] = handheldState;
        }
        else if (m_state[index].style.Test<nn::hid::NpadStyleFullKey>() == true)
        {
            // 状態取得
            nn::hid::GetNpadState(&fullKeyState, m_state[index].id);
            // 状態更新
            m_state[index] = fullKeyState;
        }
        else if (m_state[index].style.Test<nn::hid::NpadStyleJoyDual>() == true)
        {
            // 状態取得
            nn::hid::GetNpadState(&joyDualState, m_state[index].id);
            // 状態更新
            m_state[index] = joyDualState;
        }
        else if (m_state[index].style.Test<nn::hid::NpadStyleJoyLeft>() == true)
        {
            // 状態取得
            nn::hid::GetNpadState(&joyLeftState, m_state[index].id);

            if (nn::hid::GetNpadJoyHoldType() == nn::hid::NpadJoyHoldType_Horizontal)
            {
                const nn::hid::NpadButtonSet tempPress = joyLeftState.buttons;

                joyLeftState.buttons.Set<nn::hid::NpadButton::A>(tempPress.Test<nn::hid::NpadButton::Down>());
                joyLeftState.buttons.Set<nn::hid::NpadButton::B>(tempPress.Test<nn::hid::NpadButton::Left>());
                joyLeftState.buttons.Set<nn::hid::NpadButton::X>(tempPress.Test<nn::hid::NpadButton::Right>());
                joyLeftState.buttons.Set<nn::hid::NpadButton::Y>(tempPress.Test<nn::hid::NpadButton::Up>());

                joyLeftState.buttons.Set<nn::hid::NpadButton::Right>(tempPress.Test<nn::hid::NpadButton::StickLDown>());
                joyLeftState.buttons.Set<nn::hid::NpadButton::Down>(tempPress.Test<nn::hid::NpadButton::StickLLeft>());
                joyLeftState.buttons.Set<nn::hid::NpadButton::Up>(tempPress.Test<nn::hid::NpadButton::StickLRight>());
                joyLeftState.buttons.Set<nn::hid::NpadButton::Left>(tempPress.Test<nn::hid::NpadButton::StickLUp>());

                joyLeftState.buttons.Set<nn::hid::NpadButton::L>(tempPress.Test<nn::hid::NpadJoyButton::LeftSL>());
                joyLeftState.buttons.Set<nn::hid::NpadButton::R>(tempPress.Test<nn::hid::NpadJoyButton::LeftSR>());

                joyLeftState.buttons.Set<nn::hid::NpadButton::StickLRight>(tempPress.Test<nn::hid::NpadButton::StickLDown>());
                joyLeftState.buttons.Set<nn::hid::NpadButton::StickLDown>(tempPress.Test<nn::hid::NpadButton::StickLLeft>());
                joyLeftState.buttons.Set<nn::hid::NpadButton::StickLUp>(tempPress.Test<nn::hid::NpadButton::StickLRight>());
                joyLeftState.buttons.Set<nn::hid::NpadButton::StickLLeft>(tempPress.Test<nn::hid::NpadButton::StickLUp>());

                joyLeftState.buttons.Set<nn::hid::NpadJoyButton::LeftSL>(false);
                joyLeftState.buttons.Set<nn::hid::NpadJoyButton::LeftSR>(false);
            }
            // 状態更新
            m_state[index] = joyLeftState;
        }
        else if (m_state[index].style.Test<nn::hid::NpadStyleJoyRight>() == true)
        {
            // 状態取得
            nn::hid::GetNpadState(&joyRightState, m_state[index].id);

            if (nn::hid::GetNpadJoyHoldType() == nn::hid::NpadJoyHoldType_Horizontal)
            {
                const nn::hid::NpadButtonSet tempPress = joyRightState.buttons;

                joyRightState.buttons.Set<nn::hid::NpadButton::A>(tempPress.Test<nn::hid::NpadButton::X>());
                joyRightState.buttons.Set<nn::hid::NpadButton::B>(tempPress.Test<nn::hid::NpadButton::A>());
                joyRightState.buttons.Set<nn::hid::NpadButton::X>(tempPress.Test<nn::hid::NpadButton::Y>());
                joyRightState.buttons.Set<nn::hid::NpadButton::Y>(tempPress.Test<nn::hid::NpadButton::B>());

                joyRightState.buttons.Set<nn::hid::NpadButton::Right>(tempPress.Test<nn::hid::NpadButton::StickRUp>());
                joyRightState.buttons.Set<nn::hid::NpadButton::Down>(tempPress.Test<nn::hid::NpadButton::StickRRight>());
                joyRightState.buttons.Set<nn::hid::NpadButton::Up>(tempPress.Test<nn::hid::NpadButton::StickRLeft>());
                joyRightState.buttons.Set<nn::hid::NpadButton::Left>(tempPress.Test<nn::hid::NpadButton::StickRDown>());

                joyRightState.buttons.Set<nn::hid::NpadButton::L>(tempPress.Test<nn::hid::NpadJoyButton::RightSL>());
                joyRightState.buttons.Set<nn::hid::NpadButton::R>(tempPress.Test<nn::hid::NpadJoyButton::RightSR>());

                joyRightState.buttons.Set<nn::hid::NpadButton::StickRRight>(tempPress.Test<nn::hid::NpadButton::StickRUp>());
                joyRightState.buttons.Set<nn::hid::NpadButton::StickRDown>(tempPress.Test<nn::hid::NpadButton::StickRRight>());
                joyRightState.buttons.Set<nn::hid::NpadButton::StickRUp>(tempPress.Test<nn::hid::NpadButton::StickRLeft>());
                joyRightState.buttons.Set<nn::hid::NpadButton::StickRLeft>(tempPress.Test<nn::hid::NpadButton::StickRDown>());

                joyRightState.buttons.Set<nn::hid::NpadJoyButton::RightSL>(false);
                joyRightState.buttons.Set<nn::hid::NpadJoyButton::RightSR>(false);
            }
            // 状態更新
            m_state[index] = joyRightState;
        }
        else if (m_state[index].style.Test<nn::hid::NpadStylePalma>() == true)
        {
            // 状態取得
            nn::hid::GetNpadState(&palmaState, m_state[index].id);
            // 状態更新
            m_state[index] = palmaState;
        }
    }

    // パケットロス率を取得します
    void Npad::UpdatePacketLossRatio(size_t index) NN_NOEXCEPT
    {
        m_calculatePacketLossRatioCount[index] = m_calculatePacketLossRatioCount[index] >= 30 ? 0 : m_calculatePacketLossRatioCount[index] + 1;

        if (m_calculatePacketLossRatioCount[index] == 0)
        {
            m_state[index].packetLossRatio = (static_cast<float>(m_state[index].samplingNumber - m_samplingNumberPrev1sec[index]));
            if (m_state[index].packetLossRatio > 100.f)
            {
                m_state[index].packetLossRatio = 100.f;
            }
            m_samplingNumberPrev1sec[index] = m_state[index].samplingNumber;
        }
    }

    //!< ボタンのトリガー入力を取得します。
    void Npad::UpdateTrigger(size_t index)
    {
        m_state[index].trigger = m_state[index].press & (m_state[index].press ^ m_stateOld[index].press);
    }

    //!< ボタンの長押しを取得します。
    void Npad::UpdateLongPress(size_t index)
    {
        if (m_state[index].press == m_stateOld[index].press)
        {
            if (m_LongPressCount[index] >= 30)
            {
                m_state[index].longPress = m_state[index].press;
            }
            else
            {
                m_LongPressCount[index]++;
            }
        }
        else
        {
            m_LongPressCount[index] = 0;
        }
    }

    //!< ボタンの長押しの際、一発だけたつトリガ入力を取得します。
    void Npad::UpdateLongPressTrigger(size_t index)
    {
        m_state[index].longPressTrigger = m_state[index].longPress & (m_state[index].longPress ^ m_stateOld[index].longPress);
    }

    //!< ボタン長押しによる周期入力を取得します。
    void Npad::UpdateBeat(size_t index)
    {
        // todo
    }

    void Npad::SetSixAxisConfigration(size_t index) NN_NOEXCEPT
    {
        if (m_state[index].isConnected == false)
        {
            m_sixAxis[index].handleCount = 0;
            return;
        }
        if (m_state[index].style != m_stateOld[index].style)
        {
            m_sixAxis[index].handleCount = nn::hid::GetSixAxisSensorHandles(&m_sixAxis[index].handles[0],
                                                                            nn::hid::NpadSixAxisSensorHandleCountMax,
                                                                            ConvertNpadIdFromIndex(index),
                                                                            m_state[index].style);
            for (int count = 0; count < m_sixAxis[index].handleCount; count++)
            {
                nn::hid::StartSixAxisSensor(m_sixAxis[index].handles[count]);
            }
        }
    }

    void Npad::UpdateSixAxis(size_t index) NN_NOEXCEPT
    {
        for (int count = 0; count < m_sixAxis[index].handleCount; count++)
        {
            nn::hid::GetSixAxisSensorStates(&m_sixAxis[index].state[count], 1, m_sixAxis[index].handles[count]);
            m_sixAxis[index].state[count].GetQuaternion(&m_sixAxis[index].quaternion[count]);

            if (m_state[index].style.Test<nn::hid::NpadStylePalma>())
            {
                nn::util::Quaternion t_quat;
                t_quat.Set(1.f, 0.f, 0.f, 1.f);
                t_quat.Normalize();
                m_sixAxis[index].quaternion[count] *= t_quat;
            }

            // 基準クオータニオンの更新判定、更新処理
            if (gGesture.GetTouchTriggerInRange(20, 40, 300, 300))
            {
                m_baseQuaternion[index][count] = m_sixAxis[index].quaternion[count];
            }
            // 相対クオータニオンの算出
            m_sixAxis[index].relationalQuaternion[count] = m_sixAxis[index].quaternion[count] / m_baseQuaternion[index][count];
        }
    }

    /* ------------------------------------------------------------ */
    // PUBLIC関数
    /* ------------------------------------------------------------ */
    Npad& Npad::GetInstance()
    {
        static Npad Instance;
        return Instance;
    }

    nn::hid::NpadIdType Npad::ConvertNpadIdFromIndex(size_t index)
    {
        if (index != 8)
        {
            return index;
        }
        else
        {
            return nn::hid::NpadId::Handheld;
        }
    }

    size_t Npad::ConvertIndexFromNpadId(nn::hid::NpadIdType id)
    {
        if (id != nn::hid::NpadId::Handheld)
        {
            return id;
        }
        else
        {
            return 8;
        }
    }

    const char* Npad::ToString(nn::hid::NpadStyleSet style)
    {
        if (style == nn::hid::NpadStyleJoyLeft::Mask)
        {
            return "JoyLeft";
        }
        else if (style == nn::hid::NpadStyleJoyRight::Mask)
        {
            return "JoyRight";
        }
        else if (style == nn::hid::NpadStyleJoyDual::Mask)
        {
            return "JoyDual";
        }
        else if (style == nn::hid::NpadStyleFullKey::Mask)
        {
            return "FullKey";
        }
        else if (style == nn::hid::NpadStyleHandheld::Mask)
        {
            return "Handheld";
        }
        else if (style == nn::hid::NpadStylePalma::Mask)
        {
            return "Palma";
        }
        else
        {
            return "Undefined";
        }
    }

    const char* Npad::ToString(nn::hid::NpadIdType id)
    {
        if (id == nn::hid::NpadId::No1)
        {
            return "No1";
        }
        else if (id == nn::hid::NpadId::No2)
        {
            return "No2";
        }
        else if (id == nn::hid::NpadId::No3)
        {
            return "No3";
        }
        else if (id == nn::hid::NpadId::No4)
        {
            return "No4";
        }
        else if (id == nn::hid::NpadId::No5)
        {
            return "No5";
        }
        else if (id == nn::hid::NpadId::No6)
        {
            return "No6";
        }
        else if (id == nn::hid::NpadId::No7)
        {
            return "No7";
        }
        else if (id == nn::hid::NpadId::No8)
        {
            return "No8";
        }
        else if (id == nn::hid::NpadId::Handheld)
        {
            return "Handheld";
        }
        else
        {
            return "Undefined";
        }
    }

    const std::string Npad::ToString(nn::hid::NpadButtonSet button)
    {
        char buttons[38];
        buttons[0] = (button.Test<nn::hid::NpadJoyButton::A>()) ? 'A' : '.';
        buttons[1] = (button.Test<nn::hid::NpadJoyButton::B>()) ? 'B' : '.';
        buttons[2] = (button.Test<nn::hid::NpadJoyButton::X>()) ? 'X' : '.';
        buttons[3] = (button.Test<nn::hid::NpadJoyButton::Y>()) ? 'Y' : '.';
        buttons[4] = (button.Test<nn::hid::NpadJoyButton::StickL>()) ? 'L' : '.';
        buttons[5] = (button.Test<nn::hid::NpadJoyButton::StickL>()) ? 'S' : '.';
        buttons[6] = (button.Test<nn::hid::NpadJoyButton::StickR>()) ? 'R' : '.';
        buttons[7] = (button.Test<nn::hid::NpadJoyButton::StickR>()) ? 'S' : '.';
        buttons[8] = (button.Test<nn::hid::NpadJoyButton::L>()) ? 'L' : '.';
        buttons[9] = (button.Test<nn::hid::NpadJoyButton::R>()) ? 'R' : '.';
        buttons[10] = (button.Test<nn::hid::NpadJoyButton::ZL>()) ? 'Z' : '.';
        buttons[11] = (button.Test<nn::hid::NpadJoyButton::ZL>()) ? 'L' : '.';
        buttons[12] = (button.Test<nn::hid::NpadJoyButton::ZR>()) ? 'Z' : '.';
        buttons[13] = (button.Test<nn::hid::NpadJoyButton::ZR>()) ? 'R' : '.';
        buttons[14] = (button.Test<nn::hid::NpadJoyButton::Plus>()) ? '+' : '.';
        buttons[15] = (button.Test<nn::hid::NpadJoyButton::Minus>()) ? '-' : '.';
        buttons[16] = (button.Test<nn::hid::NpadJoyButton::Left>()) ? '<' : '.';
        buttons[17] = (button.Test<nn::hid::NpadJoyButton::Up>()) ? '^' : '.';
        buttons[18] = (button.Test<nn::hid::NpadJoyButton::Right>()) ? '>' : '.';
        buttons[19] = (button.Test<nn::hid::NpadJoyButton::Down>()) ? 'v' : '.';
        buttons[20] = (button.Test<nn::hid::NpadJoyButton::LeftSL>()) ? 'S' : '.';
        buttons[21] = (button.Test<nn::hid::NpadJoyButton::LeftSL>()) ? 'L' : '.';
        buttons[22] = (button.Test<nn::hid::NpadJoyButton::LeftSR>()) ? 'S' : '.';
        buttons[23] = (button.Test<nn::hid::NpadJoyButton::LeftSR>()) ? 'R' : '.';
        buttons[24] = (button.Test<nn::hid::NpadJoyButton::RightSL>()) ? 'S' : '.';
        buttons[25] = (button.Test<nn::hid::NpadJoyButton::RightSL>()) ? 'L' : '.';
        buttons[26] = (button.Test<nn::hid::NpadJoyButton::RightSR>()) ? 'S' : '.';
        buttons[27] = (button.Test<nn::hid::NpadJoyButton::RightSR>()) ? 'R' : '.';
        buttons[28] = (button.Test<nn::hid::NpadJoyButton::StickLRight>()) ? '>' : '.';
        buttons[29] = (button.Test<nn::hid::NpadJoyButton::StickLUp>()) ? '^' : '.';
        buttons[30] = (button.Test<nn::hid::NpadJoyButton::StickLLeft>()) ? '<' : '.';
        buttons[31] = (button.Test<nn::hid::NpadJoyButton::StickLDown>()) ? 'v' : '.';
        buttons[32] = (button.Test<nn::hid::NpadJoyButton::StickRRight>()) ? '>' : '.';
        buttons[33] = (button.Test<nn::hid::NpadJoyButton::StickRUp>()) ? '^' : '.';
        buttons[34] = (button.Test<nn::hid::NpadJoyButton::StickRLeft>()) ? '<' : '.';
        buttons[35] = (button.Test<nn::hid::NpadJoyButton::StickRDown>()) ? 'v' : '.';
        buttons[36] = (button.Test<nn::hid::NpadPalmaButton::Palma>()) ? 'P' : '.';
        buttons[37] = '\0';

        return buttons;
    }

    void Npad::Initialize() NN_NOEXCEPT
    {
        nn::hid::InitializeNpad();
        // 使用するID、Styleを設定
        nn::hid::SetSupportedNpadIdType(NpadIds, NpadIdNum);
        nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask
                                        | nn::hid::NpadStyleJoyDual::Mask
                                        | nn::hid::NpadStyleHandheld::Mask
                                        | nn::hid::NpadStyleJoyLeft::Mask
                                        | nn::hid::NpadStyleJoyRight::Mask
                                        | nn::hid::NpadStylePalma::Mask
                                        );

        // 常に操作形態の変更を行えるようにする
        nn::hid::StartLrAssignmentMode();

        for (auto index = 0; index < NpadIdNum; index++)
        {
            for (int count = 0; count < 2; count++)
            {
                m_sixAxis[index].relationalQuaternion[count] = nn::util::Quaternion::Identity();
                m_baseQuaternion[index][count] = nn::util::Quaternion::Identity();
            }
            m_samplingNumberPrev1sec[index] = 0;
        }
    }

    void Npad::Update() NN_NOEXCEPT
    {
        for (auto index = 0; index < NpadIdNum; index++)
        {
            // Button押下前回値のラッチ
            Latch(index);
            // 状態のリセット
            Reset(index);
            // Npad状態の取得
            UpdateState(index);
            // パケットロス率の取得
            UpdatePacketLossRatio(index);
            // Button押下のトリガ入力取得
            UpdateTrigger(index);
            // Button押下の長押し入力取得
            UpdateLongPress(index);
            // Button押下の長押しトリガ入力取得
            UpdateLongPressTrigger(index);
            // Button押下の連続入力取得
            UpdateBeat(index);

            // コントローラ6軸の設定変更
            SetSixAxisConfigration(index);
            // コントローラ6軸のステート取得
            UpdateSixAxis(index);
        }
    }

    NpadState Npad::GetFromId(nn::hid::NpadIdType id)
    {
        auto index = ConvertIndexFromNpadId(id);

        return m_state[index];
    }

    NpadState Npad::GetFromIndex(size_t index)
    {
        return m_state[index];
    }

    SixAxis Npad::GetSixAxisFromId(nn::hid::NpadIdType id)
    {
        auto index = ConvertIndexFromNpadId(id);

        return m_sixAxis[index];
    }

    SixAxis Npad::GetSixAxisFromIndex(size_t index)
    {
        return m_sixAxis[index];
    }
}
